<?php

namespace AutoCraftPlayer\App\Hooks\Handlers;

use AutoCraftPlayer\App\Helpers\MediaHelper;
use AutoCraftPlayer\App\Models\Medium;
use AutoCraftPlayer\App\Models\Playlist;
use AutoCraftPlayer\App\Vite;
use AutoCraftPlayer\Framework\Support\Arr;
use AutoCraftPlayer\Framework\Support\Facades\View;
use AutoCraftPlayer\App\App;

class ShortCodeHandler
{
    protected $playerInstances = [];
    protected $playlistInstances = [];

    public function register()
    {
        add_shortcode('autocraft_player', [$this, 'addShortcode']);
        add_shortcode('autocraft_player_playlist', [$this, 'addPlaylistShortcode']);
        add_shortcode('autocraft_player_share_page', [$this, 'addSharePageShortcode']);
    }

    public function addShortcode($atts, $content = null)
    {
        $shortcodeAtts = shortcode_atts($this->defaultShortCodeAtts(), $atts);

        $id = intval(Arr::get($shortcodeAtts, 'id'));

        if ($id === 0) {
            return '';
        }

        if (!$player = Medium::find($id)) {
            return '';
        }

        return $this->renderPlayer($player, $shortcodeAtts, $content);
    }

    protected function defaultShortCodeAtts()
    {
        return [
            'id'       => null,
            'autoplay' => 'muted',
            'controls' => true,
            'height'   => 264,
            'loop'     => false,
            'muted'    => false,
            'poster'   => Vite::getAssetsUrl() . 'images/video.png',
            'preload'  => 'auto',
            'src'      => '',
            'width'    => 640
        ];
    }

    public function renderPlayer($player, $shortcodeAtts, $content = null)
    {
        $this->commonAssets();
        $this->enqueuePlayerAssets($player);
        $playerInstance          = 'autocraft_player_instance_' . count($this->playerInstances);
        $this->playerInstances[] = $playerInstance;

        $player['shortcode_atts'] = $shortcodeAtts;
        $player['instance']       = $playerInstance;
        $this->setInstanceToWindow($playerInstance, $player);

        return View::make('public/player/player', compact('player', 'shortcodeAtts'));
    }

    private function commonAssets()
    {
        Vite::enqueueStaticStyle(
            'autocraft_player-video-js-css',
            'libs/videojs/video-js.css',
            [],
            AUTOCRAFT_PLAYER_VERSION
        );
        Vite::enqueueStaticScript(
            'autocraft_player-video-js',
            'libs/videojs/video.min.js',
            ['jquery'],
            AUTOCRAFT_PLAYER_VERSION
        );

        $data = apply_filters('autocraft_player/localized_public_script', []);
        wp_localize_script('autocraft_player-video-js', 'autoCraftPlayerInfo', $data);

        // Todo: need to render conditionally
        Vite::enqueueStaticScript(
            'autocraft_player-youtube-video-js',
            'libs/videojs/Youtube.min.js',
            ['autocraft_player-video-js'],
            AUTOCRAFT_PLAYER_VERSION
        );
        Vite::enqueueStaticScript(
            'autocraft_player-vimeo-video-js',
            'libs/videojs/vimeo.min.js',
            ['autocraft_player-video-js'],
            AUTOCRAFT_PLAYER_VERSION
        );
    }

    private function enqueuePlayerAssets($player)
    {
        do_action('autocraft_player/after_enqueue_player_assets', $player);
    }

    private function setInstanceToWindow($instance, $data)
    {
        wp_add_inline_script(
            'autocraft_player-video-js',
            'window["' . esc_js($instance) . '"] = ' . wp_json_encode($data) . ';'
        );
    }

    public function addPlaylistShortcode($atts, $content = null)
    {
        $shortcodeAtts = shortcode_atts($this->defaultPlaylistShortCodeAtts(), $atts);

        $id = intval(Arr::get($shortcodeAtts, 'id'));

        if ($id === 0) {
            return '';
        }

        if (!$playlist = Playlist::find($id)) {
            return '';
        }

        return $this->renderPlaylist($playlist, $shortcodeAtts, $content);
    }

    protected function defaultPlaylistShortCodeAtts()
    {
        return [
            'id' => null,
        ];
    }

    public function renderPlaylist($playlist, $shortcodeAtts, $content = null)
    {
        $this->commonAssets();

        $playlistInstance          = 'autocraft_player_playlist_instance_' . count($this->playlistInstances);
        $this->playlistInstances[] = $playlistInstance;

        if (!isset($playlist['meta']['playerConfig'])) {
            $playlist['meta'] = array_merge($playlist['meta'], [
                'playerConfig' => MediaHelper::getConfigByType(Arr::get($playlist, 'type', 'video'))
            ]);
        }

        $playlist['shortcode_atts'] = $shortcodeAtts;
        $playlist['instance']       = $playlistInstance;
        $this->enqueuePlaylistAssets($playlist);
        $this->setInstanceToWindow($playlistInstance, $playlist);

        return View::make('public/playlist/playlist', compact('playlist', 'shortcodeAtts'));
    }

    protected function enqueuePlaylistAssets($playlist)
    {
        Vite::enqueueScript(
            'autocraft_player-playlist-init-js',
            'public/js/playlist.init.js',
            ['autocraft_player-video-js'],
            AUTOCRAFT_PLAYER_VERSION,
            true
        );

        do_action('autocraft_player/after_enqueue_playlist_assets', $playlist);
    }

    public function addSharePageShortcode($atts, $content = null)
    {
        $request = \AutoCraftPlayer\request();

        $uid = sanitize_text_field($request->get('autocraft_player_share'));
        if (!$uid) {
            return '';
        }

        $type = sanitize_text_field($request->get('type'));
        if (!$type) {
            return '';
        }

        $id = intval(explode('_', $uid)[0]);

        if (!$id) {
            return __('Invalid shared url', 'autocraft-player');
        }

        if ($type === 'media') {
            $shortcode = "[autocraft_player id=\"$id\"]";
        } elseif ($type === 'playlist') {
            $app = App::getInstance();
            wp_enqueue_script('jquery');
            wp_localize_script('jquery', 'autoCraftPlayer', $app->localizedScripts());
            $shortcode = "[autocraft_player_playlist id=\"$id\"]";
        } else {
            return '';
        }

        return do_shortcode($shortcode);
    }
}
