<?php

namespace AutoCraftPlayer\App\Hooks\Handlers;

use AutoCraftPlayer\App\Helpers\MediaHelper;
use AutoCraftPlayer\App\Helpers\Translation;
use AutoCraftPlayer\App\Services\Helper;
use AutoCraftPlayer\App\Vite;
use AutoCraftPlayer\App\App;
use AutoCraftPlayer\Framework\Support\Arr;
use AutoCraftPlayer\Framework\Support\Facades\Config;

class AdminMenuHandler
{
    public function addMenu()
    {
        $slug       = Config::get('app.slug');
        $capability = \AutoCraftPlayer\adminRole();
        $menuName   = __('AutoCraft Player', 'autocraft-player');
        $subMenus   = [
            ''                       => __('Dashboard', 'autocraft-player'),
            'videos'                 => __('Videos', 'autocraft-player'),
            'audios'                 => __('Audios', 'autocraft-player'),
            'playlists'              => __('Playlists', 'autocraft-player'),
            'global-settings/videos' => __('Global Settings', 'autocraft-player')
        ];

        $subMenus = apply_filters('autocraft_player/before_render_submenus', $subMenus);

        add_menu_page(
            $menuName,
            $menuName,
            $capability,
            $slug,
            [$this, 'render'],
            $this->getIcon(),
            5
        );

        foreach ($subMenus as $subSlug => $subMenuName) {
            $subMenuSlug = $subSlug === '' ? $slug : "{$slug}#/{$subSlug}";
            add_submenu_page(
                $slug,
                $subMenuName,
                $subMenuName,
                $capability,
                $subMenuSlug,
                [$this, 'render']
            );
        }
    }

    public function simpleRender()
    {
        echo "<div id='autocraft-player-app-wrapper'></div>";
    }

    public function render()
    {
        $app = App::getInstance();

        $baseUrl   = admin_url('admin.php?page=' . $app->config->get('app.slug') . '#/');
        $menuItems = [
            [
                'key'       => 'dashboard',
                'label'     => __('Dashboard', 'autocraft-player'),
                'permalink' => $baseUrl,
            ],
            [
                'key'       => 'media',
                'label'     => __('Media', 'autocraft-player'),
                'permalink' => $baseUrl . 'media',
            ],
            [
                'key'       => 'playlists',
                'label'     => __('Playlists', 'autocraft-player'),
                'permalink' => $baseUrl . 'playlists',
            ]
        ];

        $menuItems = apply_filters('autocraft_player/before_render_menu_items', $menuItems, $baseUrl);


        $app->view->render('admin/menu', [
            'slug'      => $app->config->get('app.slug'),
            'menuItems' => $menuItems,
            'baseUrl'   => $baseUrl,
            'logo'      => Vite::getAssetsUrl() . 'images/logo.svg',
        ]);
    }

    protected function getIcon()
    {
        return Helper::svgToDataUrl(
            '<svg version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
     viewBox="0 0 20 20" xml:space="preserve">
  <g>
    <!-- Video screen outline with white fill and empty interior -->
    <path d="M3.5,2h13C17.3,2,18,2.7,18,3.5v8c0,0.8-0.7,1.5-1.5,1.5h-13C2.7,13,2,12.3,2,11.5v-8C2,2.7,2.7,2,3.5,2z
             M3.5,3C3.2,3,3,3.2,3,3.5v8C3,11.8,3.2,12,3.5,12h13c0.3,0,0.5-0.2,0.5-0.5v-8C17,3.2,16.8,3,16.5,3H3.5z" fill="#ffffff" />
    
    <!-- Play triangle -->
    <path d="M6,4.5 L13.5,7.5 L6,10.5 Z" fill="#ffffff" />
    
    <!-- Audio waves -->
    <path d="M2,15 Q6,13 10,15 Q14,17 18,15" stroke="#ffffff" stroke-width="1.5" fill="none" stroke-linecap="round" />
    <path d="M4,17 Q7,16 10,17 Q13,18 16,17" stroke="#ffffff" stroke-width="1.2" fill="none" stroke-linecap="round" />
  </g>
</svg>'
        );
    }

    public function enqueueAssets()
    {
        wp_enqueue_media();
        $app  = App::getInstance();
        $page = sanitize_text_field(Arr::get($_GET, 'page')); // phpcs:ignore

        if ($page === $app->config->get('app.slug')) {
            do_action('autocraft_player/before_enqueue_admin_assets');
            $this->enqueueStyles($app);
            $this->enqueueScripts($app);
        }
    }

    protected function enqueueStyles($app)
    {
        Vite::enqueueStyle(
            $app->config->get('app.slug') . '_admin_app',
            'scss/admin/admin.scss',
            [],
            $app->config->get('app.version')
        );
    }

    protected function enqueueScripts($app)
    {
        $slug = $app->config->get('app.slug');
        Vite::enqueueScript($slug . '_admin_app', 'admin/app.js', ['jquery'], $app->config->get('app.version'), true);
        $this->localizeScript($app, $slug);
    }

    protected function localizeScript($app, $slug): void
    {
        $data = [
            'nonce'      => wp_create_nonce($slug),
            'slug'       => $slug,
            'assets_url' => Vite::getAssetsUrl(),
            'i18n'       => Translation::getStrings(),
            'site_url'   => get_site_url(),
            'scripts'    => [
                'video-js-css'     => Vite::getAssetsUrl() . 'libs/videojs/video-js.css',
                'video-js'         => Vite::getAssetsUrl() . 'libs/videojs/video.min.js',
                'video-js-youtube' => Vite::getAssetsUrl() . 'libs/videojs/Youtube.min.js',
                'video-js-vimeo'   => Vite::getAssetsUrl() . 'libs/videojs/vimeo.min.js',
                'themes'           => [
                    'city'    => Vite::getAssetsUrl() . 'libs/videojs/themes/city.css',
                    'forest'  => Vite::getAssetsUrl() . 'libs/videojs/themes/forest.css',
                    'sea'     => Vite::getAssetsUrl() . 'libs/videojs/themes/sea.css',
                    'fantasy' => Vite::getAssetsUrl() . 'libs/videojs/themes/fantasy.css',
                    'acp'     => Vite::getAssetsUrl() . 'libs/videojs/themes/acp.css',
                    'text'    => Vite::getAssetsUrl() . 'libs/videojs/themes/text.css',
                ],
            ],
            'meta'       => [
                'video'       => MediaHelper::getConfigByType('video'), // dynamic config
                'audio'       => MediaHelper::getConfigByType('audio'), // dynamic config
                'videoConfig' => Config::get('video'), // static config
                'audioConfig' => Config::get('audio'), // static config

                'audioPlaylist'       => MediaHelper::getConfigByType('audio_playlist'), // dynamic config
                'videoPlaylist'       => MediaHelper::getConfigByType('video_playlist'), // dynamic config
                'audioPlaylistConfig' => Config::get('audio_playlist'), // static config
                'videoPlaylistConfig' => Config::get('video_playlist'), // static config
            ]
        ];

        $localizedScripts = apply_filters('autocraft_player/localized_admin_script', $data);

        wp_localize_script($slug . '_admin_app', 'autoCraftPlayerAdmin', $localizedScripts);
    }
}
