<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// --- CONSTANTS (Mirroring the primary plugin file for scope safety) ---
define( 'ADBWD_SETTINGS_NAME_VAL', 'adbwd-style-options' );
define( 'ADBWD_PLUGIN_SLUG_VAL', 'adbwd-download-buttons' );


// --- 1. ADMIN SETTING: MAX BUTTONS (1-4) ---

/**
 * Registers the new setting field for the number of buttons on the existing admin page.
 */
function adbwd_register_multi_setting() {
    // Add a new setting field to the existing main section
    add_settings_field( 
        'adbwd_max_image_buttons', 
        __( 'Max Download Buttons per Image (1-4)', 'auto-download-buttons' ), 
        'adbwd_max_image_buttons_callback', 
        ADBWD_PLUGIN_SLUG_VAL, 
        'adbwd-main-section' // Must match the section slug from adbwd-settings.php
    );
}
add_action( 'admin_init', 'adbwd_register_multi_setting' );


/**
 * Renders the Max Buttons input field.
 */
function adbwd_max_image_buttons_callback() {
    $options = get_option( ADBWD_SETTINGS_NAME_VAL );
    $value = isset( $options['max-image-buttons'] ) ? $options['max-image-buttons'] : 1;
    
    // The max attribute is now 4
    echo '<input type="number" min="1" max="4" step="1" id="adbwd_max_image_buttons" name="' . esc_attr(ADBWD_SETTINGS_NAME_VAL) . '[max-image-buttons]" value="' . esc_attr( $value ) . '" class="small-text"/>';
    echo '<p class="description">Sets the maximum number of download URL fields that will appear when editing an image.</p>';
}


// --- 2. MULTI-FIELD LOGIC & ATTACHMENT HOOKS ---

// Remove hooks from the old single-URL feature (if they were present in a previous file)
remove_filter( 'attachment_fields_to_edit', 'adbwd_add_image_download_url_field', 10 );
remove_filter( 'attachment_fields_to_save', 'adbwd_save_image_download_url_field', 10 );
remove_filter( 'render_block', 'adbwd_render_image_download_button', 10 );


/**
 * Adds multiple custom fields (URL and Text) to the Attachment Edit screen.
 */
function adbwd_add_multi_attachment_fields( $fields, $post ) {
    $global_options = get_option( ADBWD_SETTINGS_NAME_VAL );
    $max_buttons = isset( $global_options['max-image-buttons'] ) ? intval( $global_options['max-image-buttons'] ) : 1;
    $max_buttons = max( 1, min( 4, $max_buttons ) ); 

    if ( strpos( $post->post_mime_type, 'image/' ) !== 0 ) {
        return $fields;
    }

    for ( $i = 1; $i <= $max_buttons; $i++ ) {
        $meta_key_url = 'adbwd_download_url_' . $i;
        $meta_key_text = 'adbwd_button_text_' . $i;

        // URL Field
        $fields[$meta_key_url] = array(
            /* translators: %1$d: The number of the download button URL field (1, 2, 3, etc.). */
            'label' => sprintf( __( 'Download Button URL #%1$d', 'auto-download-buttons' ), $i ),
            'input' => 'text',
            'value' => get_post_meta( $post->ID, $meta_key_url, true ),
            /* translators: %1$d: The number of the download button (1, 2, 3, etc.) for a file like ZIP or PDF. */
            'helps' => sprintf( __( '(e.g. JPG, ZIP, PDF, etc.)', 'auto-download-buttons' ), $i ),
            'show_in_edit' => true,
        );
        
        // TEXT Field
        $fields[$meta_key_text] = array(
            /* translators: %1$d: The number of the download button text field (1, 2, 3, etc.). */
            'label' => sprintf( __( 'Custom Button Text #%1$d', 'auto-download-buttons' ), $i ),
            'input' => 'text',
            'value' => get_post_meta( $post->ID, $meta_key_text, true ),
            'helps' => __( 'Optional: Overrides the Global setting.', 'auto-download-buttons' ),
            'show_in_edit' => true,
        );
    }
    
    return $fields;
}
add_filter( 'attachment_fields_to_edit', 'adbwd_add_multi_attachment_fields', 10, 2 );


/**
 * Saves all custom URL and Text fields and deletes keys that exceed the current max.
 */
function adbwd_save_multi_attachment_fields( $post, $attachment ) {
    $global_options = get_option( ADBWD_SETTINGS_NAME_VAL );
    $current_max_buttons = isset( $global_options['max-image-buttons'] ) ? intval( $global_options['max-image-buttons'] ) : 1;
    $current_max_buttons = max( 1, min( 4, $current_max_buttons ) );

    $absolute_max = 4;
    // List of field key prefixes to check
    $fields_to_check = array( 'adbwd_download_url_', 'adbwd_button_text_' ); 

    for ( $i = 1; $i <= $absolute_max; $i++ ) {
        
        foreach ( $fields_to_check as $key_prefix ) {
            $meta_key = $key_prefix . $i;

            if ( $i <= $current_max_buttons ) {
                // SAVE: This is a visible field, save its submitted value
                if ( isset( $attachment[$meta_key] ) ) {
                    $sanitized_value = ( 'adbwd_download_url_' === $key_prefix ) ? sanitize_url( $attachment[$meta_key] ) : sanitize_text_field( $attachment[$meta_key] );
                    update_post_meta( $post['ID'], $meta_key, $sanitized_value );
                }
            } else {
                // DELETE: This field is ABOVE the current max, delete any existing meta data.
                delete_post_meta( $post['ID'], $meta_key );
            }
        }
    }
    return $post;
}
add_filter( 'attachment_fields_to_save', 'adbwd_save_multi_attachment_fields', 10, 2 );


// --- 3. FRONT-END RENDERING (Multiple Buttons) ---

/**
 * Injects multiple download button shortcodes after the image block output (Gutenberg).
 */
function adbwd_render_multi_image_download_buttons( $block_content, $block ) {
    if ( 'core/image' !== $block['blockName'] ) {
        return $block_content;
    }

    if ( empty( $block['attrs']['id'] ) ) {
        return $block_content;
    }

    $image_id = absint( $block['attrs']['id'] );
    $button_output = '';
    
    // Check up to the maximum possible download URLs (4)
    for ( $i = 1; $i <= 4; $i++ ) {
        $download_url = get_post_meta( $image_id, 'adbwd_download_url_' . $i, true );
        $custom_text = get_post_meta( $image_id, 'adbwd_button_text_' . $i, true );

        if ( ! empty( $download_url ) ) {
            // Build the shortcode attributes string
            $button_attr = 'url="' . esc_url( $download_url ) . '"';
            
            // Pass the custom text only if it is not empty
            if ( ! empty( $custom_text ) ) {
                 $button_attr .= ' text="' . esc_attr( $custom_text ) . '"';
            }
            
            // Render the shortcode with attributes
            $button_output .= do_shortcode( '[auto-download-button ' . $button_attr . ']' );
        }
    }

    // If any buttons were generated, append them after the block content
    if ( ! empty( $button_output ) ) {
        // Use the wrapper for horizontal alignment
        $wrapper_start = '<div class="adbwd-multi-button-wrapper">';
        $wrapper_end = '</div>';
        return $block_content . $wrapper_start . $button_output . $wrapper_end;
    }

    return $block_content;
}
add_filter( 'render_block', 'adbwd_render_multi_image_download_buttons', 10, 2 );


// --- 4. ADMIN CSS INJECTION FOR FIELD SPACING ---

/**
 * Injects custom CSS into the admin panel to reduce the vertical gap between URL and Text fields 
 * in the attachment editor screen.
 */
function adbwd_reduce_attachment_field_spacing() {
    // The CSS targets the container and help text of the custom fields.
    $css = '
/* Target the custom button text fields to reduce their top margin */
.compat-field-adbwd_button_text_1,
.compat-field-adbwd_button_text_2,
.compat-field-adbwd_button_text_3,
.compat-field-adbwd_button_text_4 {
    /* Reduce the default top margin applied to the wrapper element */
    margin-top: 1px !important; 
}

/* Reduce the bottom margin of the help text description of the URL field */
.compat-field-adbwd_download_url_1 p.description,
.compat-field-adbwd_download_url_2 p.description,
.compat-field-adbwd_download_url_3 p.description,
.compat-field-adbwd_download_url_4 p.description {
    /* Remove the default margin below the help text for the URL box */
    margin-bottom: 1px;
}
    ';

    // Output the CSS directly in the admin header.
    echo '<style type="text/css">' . $css . '</style>';
}
add_action( 'admin_head', 'adbwd_reduce_attachment_field_spacing' );