<?php
/**
 * Plugin Name: Auto-Disable Unused Image Sizes
 * Plugin URI: https://wordpress.org/plugins/auto-disable-unused-image-sizes/
 * Description: A lightweight plugin that lets users disable unused image sizes to save storage space and improve performance.
 * Version: 1.0.0
 * Author: Faizan
 * License: GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: auto-disable-unused-image-sizes
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 *
 * @package Auto_Disable_Unused_Image_Sizes
 * @version 1.0.0
 * @author Faizan
 * @license GPL2
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants
define( 'ADIS_PLUGIN_VERSION', '1.0.0' );
define( 'ADIS_PLUGIN_FILE', __FILE__ );
define( 'ADIS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'ADIS_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'ADIS_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Main plugin class
 */
class Auto_Disable_Unused_Image_Sizes {

	/**
	 * Plugin instance
	 *
	 * @var Auto_Disable_Unused_Image_Sizes
	 */
	private static $instance = null;

	/**
	 * Get plugin instance
	 *
	 * @return Auto_Disable_Unused_Image_Sizes
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 */
	private function init_hooks() {
		add_action( 'init', array( $this, 'init' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
	}

	/**
	 * Initialize plugin
	 */
	public function init() {
		// Add admin menu
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );

		// Register settings
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// Hook into image generation
		add_filter( 'intermediate_image_sizes_advanced', array( $this, 'filter_image_sizes' ), 10, 2 );

		// Add settings link to plugins page
		add_filter( 'plugin_action_links_' . ADIS_PLUGIN_BASENAME, array( $this, 'add_settings_link' ) );

		// Add plugin row meta
		add_filter( 'plugin_row_meta', array( $this, 'add_plugin_row_meta' ), 10, 2 );
	}

	/**
	 * Enqueue admin scripts and styles
	 */
	public function enqueue_admin_scripts( $hook ) {
		if ( 'settings_page_auto-disable-unused-image-sizes' !== $hook ) {
			return;
		}

		wp_enqueue_script(
			'adis-admin-script',
			ADIS_PLUGIN_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			ADIS_PLUGIN_VERSION,
			true
		);

		wp_enqueue_style(
			'adis-admin-style',
			ADIS_PLUGIN_URL . 'assets/css/admin.css',
			array(),
			ADIS_PLUGIN_VERSION
		);

		wp_localize_script(
			'adis-admin-script',
			'adis_ajax',
			array(
				'ajax_url' => admin_url( 'admin-ajax.php' ),
				'nonce'    => wp_create_nonce( 'adis_nonce' ),
				'strings'  => array(
					'confirm_bulk_disable' => __( 'Are you sure you want to disable the selected image sizes?', 'auto-disable-unused-image-sizes' ),
					'confirm_bulk_enable'  => __( 'Are you sure you want to enable the selected image sizes?', 'auto-disable-unused-image-sizes' ),
				),
			)
		);
	}

	/**
	 * Add admin menu
	 */
	public function add_admin_menu() {
		add_options_page(
			__( 'Image Sizes', 'auto-disable-unused-image-sizes' ),
			__( 'Image Sizes', 'auto-disable-unused-image-sizes' ),
			'manage_options',
			'auto-disable-unused-image-sizes',
			array( $this, 'admin_page' )
		);
	}

	/**
	 * Register plugin settings
	 */
	public function register_settings() {
		register_setting(
			'adis_image_sizes_group',
			'adis_disabled_image_sizes',
			array(
				'type'              => 'array',
				'description'       => __( 'Disabled image sizes', 'auto-disable-unused-image-sizes' ),
				'sanitize_callback' => array( $this, 'sanitize_disabled_sizes' ),
				'default'           => array(),
			)
		);

		// Add settings section
		add_settings_section(
			'adis_image_sizes_section',
			__( 'Image Size Management', 'auto-disable-unused-image-sizes' ),
			array( $this, 'settings_section_callback' ),
			'adis_image_sizes_group'
		);
	}

	/**
	 * Settings section callback
	 */
	public function settings_section_callback() {
		echo '<p>' . esc_html__( 'Select the image sizes you want to disable. Disabled sizes will not be generated when uploading new images.', 'auto-disable-unused-image-sizes' ) . '</p>';
	}

	/**
	 * Sanitize disabled sizes array
	 *
	 * @param array $input The input array to sanitize.
	 * @return array Sanitized array.
	 */
	public function sanitize_disabled_sizes( $input ) {
		if ( ! is_array( $input ) ) {
			return array();
		}

		$sanitized = array();
		$available_sizes = $this->get_all_image_sizes();

		foreach ( $input as $size ) {
			$size = sanitize_text_field( $size );
			if ( in_array( $size, $available_sizes, true ) ) {
				$sanitized[] = $size;
			}
		}

		// Set success message
		set_transient( 'adis_settings_updated', true, 30 );

		return $sanitized;
	}

	/**
	 * Get all available image sizes
	 *
	 * @return array Array of image size names.
	 */
	public function get_all_image_sizes() {
		global $_wp_additional_image_sizes;

		$sizes = array();

		// Get default sizes
		$default_sizes = array( 'thumbnail', 'medium', 'medium_large', 'large' );
		foreach ( $default_sizes as $size ) {
			if ( has_image_size( $size ) ) {
				$sizes[] = $size;
			}
		}

		// Get custom sizes
		if ( ! empty( $_wp_additional_image_sizes ) && is_array( $_wp_additional_image_sizes ) ) {
			foreach ( $_wp_additional_image_sizes as $size => $data ) {
				$sizes[] = $size;
			}
		}

		// Get intermediate sizes (WordPress 5.3+)
		if ( function_exists( 'get_intermediate_image_sizes' ) ) {
			$intermediate_sizes = get_intermediate_image_sizes();
			if ( is_array( $intermediate_sizes ) ) {
				foreach ( $intermediate_sizes as $size ) {
					if ( ! in_array( $size, $sizes, true ) ) {
						$sizes[] = $size;
					}
				}
			}
		}

		// Remove duplicates and sort
		$sizes = array_unique( $sizes );
		sort( $sizes );

		return $sizes;
	}

	/**
	 * Get image size information
	 *
	 * @param string $size Image size name.
	 * @return array|false Image size data or false if not found.
	 */
	public function get_image_size_info( $size ) {
		global $_wp_additional_image_sizes;

		// Default sizes
		$default_sizes = array(
			'thumbnail'     => array( 'width' => get_option( 'thumbnail_size_w' ), 'height' => get_option( 'thumbnail_size_h' ) ),
			'medium'        => array( 'width' => get_option( 'medium_size_w' ), 'height' => get_option( 'medium_size_h' ) ),
			'medium_large'  => array( 'width' => get_option( 'medium_large_size_w' ), 'height' => get_option( 'medium_large_size_h' ) ),
			'large'         => array( 'width' => get_option( 'large_size_w' ), 'height' => get_option( 'large_size_h' ) ),
		);

		if ( isset( $default_sizes[ $size ] ) ) {
			return $default_sizes[ $size ];
		}

		if ( isset( $_wp_additional_image_sizes[ $size ] ) ) {
			return $_wp_additional_image_sizes[ $size ];
		}

		return false;
	}

	/**
	 * Filter image sizes to remove disabled ones
	 *
	 * @param array $sizes Array of image sizes.
	 * @param array $metadata Image metadata.
	 * @return array Filtered array of image sizes.
	 */
	public function filter_image_sizes( $sizes, $metadata ) {
		$disabled_sizes = get_option( 'adis_disabled_image_sizes', array() );

		if ( ! empty( $disabled_sizes ) && is_array( $disabled_sizes ) ) {
			foreach ( $disabled_sizes as $disabled_size ) {
				$key = array_search( $disabled_size, $sizes, true );
				if ( false !== $key ) {
					unset( $sizes[ $key ] );
				}
			}
		}

		return $sizes;
	}

	/**
	 * Admin page content
	 */
	public function admin_page() {
		// Check user capabilities
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'auto-disable-unused-image-sizes' ) );
		}

		$available_sizes = $this->get_all_image_sizes();
		$disabled_sizes = get_option( 'adis_disabled_image_sizes', array() );

		?>
		<div class="wrap">
			<h1 class="wp-heading-inline"><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<div class="adis-header-info">
				<p class="description">
					<?php esc_html_e( 'Manage which image sizes are generated when uploading new images. This helps save storage space and improve performance.', 'auto-disable-unused-image-sizes' ); ?>
				</p>
			</div>

			<?php if ( ! empty( $available_sizes ) ) : ?>
				<div class="adis-stats">
					<div class="adis-stat-box">
						<span class="adis-stat-number"><?php echo esc_html( count( $available_sizes ) ); ?></span>
						<span class="adis-stat-label"><?php esc_html_e( 'Total Image Sizes', 'auto-disable-unused-image-sizes' ); ?></span>
					</div>
					<div class="adis-stat-box">
						<span class="adis-stat-number"><?php echo esc_html( count( $disabled_sizes ) ); ?></span>
						<span class="adis-stat-label"><?php esc_html_e( 'Disabled Sizes', 'auto-disable-unused-image-sizes' ); ?></span>
					</div>
					<div class="adis-stat-box">
						<span class="adis-stat-number"><?php echo esc_html( count( $available_sizes ) - count( $disabled_sizes ) ); ?></span>
						<span class="adis-stat-label"><?php esc_html_e( 'Active Sizes', 'auto-disable-unused-image-sizes' ); ?></span>
					</div>
				</div>

				<form method="post" action="options.php" id="adis-settings-form">
					<?php
					settings_fields( 'adis_image_sizes_group' );
					do_settings_sections( 'adis_image_sizes_group' );
					?>

					<div class="adis-bulk-actions">
						<button type="button" class="button" id="adis-select-all"><?php esc_html_e( 'Select All', 'auto-disable-unused-image-sizes' ); ?></button>
						<button type="button" class="button" id="adis-deselect-all"><?php esc_html_e( 'Deselect All', 'auto-disable-unused-image-sizes' ); ?></button>
						<button type="button" class="button" id="adis-disable-selected"><?php esc_html_e( 'Disable Selected', 'auto-disable-unused-image-sizes' ); ?></button>
						<button type="button" class="button" id="adis-enable-selected"><?php esc_html_e( 'Enable Selected', 'auto-disable-unused-image-sizes' ); ?></button>
					</div>

					<div class="adis-image-sizes-grid">
						<?php foreach ( $available_sizes as $size ) : ?>
							<?php
							$size_info = $this->get_image_size_info( $size );
							$is_disabled = in_array( $size, $disabled_sizes, true );
							$size_label = ucfirst( str_replace( '_', ' ', $size ) );
							?>
							<div class="adis-size-card <?php echo $is_disabled ? esc_attr( 'adis-disabled' ) : esc_attr( 'adis-enabled' ); ?>">
								<div class="adis-size-header">
									<label class="adis-size-checkbox">
										<input type="checkbox" 
											   name="adis_disabled_image_sizes[]" 
											   value="<?php echo esc_attr( $size ); ?>"
											   <?php checked( $is_disabled ); ?>
											   class="adis-size-checkbox-input" />
										<span class="adis-checkmark"></span>
									</label>
									<h3 class="adis-size-title"><?php echo esc_html( $size_label ); ?></h3>
									<span class="adis-size-status <?php echo $is_disabled ? esc_attr( 'adis-status-disabled' ) : esc_attr( 'adis-status-enabled' ); ?>">
										<?php echo $is_disabled ? esc_html__( 'Disabled', 'auto-disable-unused-image-sizes' ) : esc_html__( 'Enabled', 'auto-disable-unused-image-sizes' ); ?>
									</span>
								</div>
								
								<?php if ( $size_info && isset( $size_info['width'] ) && isset( $size_info['height'] ) ) : ?>
									<div class="adis-size-details">
										<span class="adis-size-dimensions">
											<?php echo esc_html( $size_info['width'] ); ?> × <?php echo esc_html( $size_info['height'] ); ?> px
										</span>
									</div>
								<?php endif; ?>
								
								<div class="adis-size-description">
									<?php if ( in_array( $size, array( 'thumbnail', 'medium', 'large' ), true ) ) : ?>
										<span class="adis-size-badge adis-default"><?php esc_html_e( 'Default', 'auto-disable-unused-image-sizes' ); ?></span>
									<?php else : ?>
										<span class="adis-size-badge adis-custom"><?php esc_html_e( 'Custom', 'auto-disable-unused-image-sizes' ); ?></span>
									<?php endif; ?>
								</div>
							</div>
						<?php endforeach; ?>
					</div>

					<?php submit_button( __( 'Save Changes', 'auto-disable-unused-image-sizes' ), 'primary', 'submit', false ); ?>
				</form>

			<?php else : ?>
				<div class="notice notice-warning">
					<p><?php esc_html_e( 'No image sizes found. This might indicate an issue with your WordPress installation.', 'auto-disable-unused-image-sizes' ); ?></p>
				</div>
			<?php endif; ?>

			<div class="adis-info-section">
				<div class="adis-info-box">
					<h3><?php esc_html_e( 'How It Works', 'auto-disable-unused-image-sizes' ); ?></h3>
					<p><?php esc_html_e( 'When you upload an image, WordPress automatically generates multiple sizes (thumbnail, medium, large, etc.). This plugin prevents the disabled sizes from being generated, saving storage space and improving performance.', 'auto-disable-unused-image-sizes' ); ?></p>
				</div>

				<div class="adis-info-box">
					<h3><?php esc_html_e( 'Important Notes', 'auto-disable-unused-image-sizes' ); ?></h3>
					<ul>
						<li><?php esc_html_e( 'This plugin only affects new image uploads.', 'auto-disable-unused-image-sizes' ); ?></li>
						<li><?php esc_html_e( 'Existing images and sizes remain untouched.', 'auto-disable-unused-image-sizes' ); ?></li>
						<li><?php esc_html_e( 'Disabling all sizes may break your theme layout.', 'auto-disable-unused-image-sizes' ); ?></li>
						<li><?php esc_html_e( 'Changes take effect immediately after saving.', 'auto-disable-unused-image-sizes' ); ?></li>
					</ul>
				</div>

				<div class="adis-info-box">
					<h3><?php esc_html_e( 'Tips for Best Results', 'auto-disable-unused-image-sizes' ); ?></h3>
					<ul>
						<li><?php esc_html_e( 'Keep thumbnail and medium sizes enabled for basic functionality.', 'auto-disable-unused-image-sizes' ); ?></li>
						<li><?php esc_html_e( 'Test your site after making changes to ensure everything works correctly.', 'auto-disable-unused-image-sizes' ); ?></li>
						<li><?php esc_html_e( 'Monitor your storage usage to see the space savings.', 'auto-disable-unused-image-sizes' ); ?></li>
					</ul>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Display admin notices
	 */
	public function admin_notices() {
		if ( get_transient( 'adis_settings_updated' ) ) {
			delete_transient( 'adis_settings_updated' );
			?>
			<div class="notice notice-success is-dismissible">
				<p><?php esc_html_e( 'Image size settings updated successfully!', 'auto-disable-unused-image-sizes' ); ?></p>
			</div>
			<?php
		}
	}

	/**
	 * Add settings link to plugins page
	 *
	 * @param array $links Plugin action links.
	 * @return array Modified plugin action links.
	 */
	public function add_settings_link( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			admin_url( 'options-general.php?page=auto-disable-unused-image-sizes' ),
			esc_html__( 'Settings', 'auto-disable-unused-image-sizes' )
		);
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Add plugin row meta
	 *
	 * @param array  $links Plugin row meta.
	 * @param string $file  Plugin file.
	 * @return array Modified plugin row meta.
	 */
	public function add_plugin_row_meta( $links, $file ) {
		if ( ADIS_PLUGIN_BASENAME === $file ) {
			$row_meta = array(
				'docs'    => sprintf(
					'<a href="%s" aria-label="%s">%s</a>',
					esc_url( 'https://wordpress.org/plugins/auto-disable-unused-image-sizes/' ),
					esc_attr__( 'View Auto-Disable Unused Image Sizes Documentation', 'auto-disable-unused-image-sizes' ),
					esc_html__( 'Documentation', 'auto-disable-unused-image-sizes' )
				),
				'support' => sprintf(
					'<a href="%s" aria-label="%s">%s</a>',
					esc_url( 'https://wordpress.org/support/plugin/auto-disable-unused-image-sizes/' ),
					esc_attr__( 'Visit support forum', 'auto-disable-unused-image-sizes' ),
					esc_html__( 'Support', 'auto-disable-unused-image-sizes' )
				),
			);

			return array_merge( $links, $row_meta );
		}

		return $links;
	}
}

// Initialize plugin
Auto_Disable_Unused_Image_Sizes::get_instance();

// Activation hook
register_activation_hook( __FILE__, 'adis_activate' );

/**
 * Plugin activation function
 */
function adis_activate() {
	// Add default options if they don't exist
	if ( ! get_option( 'adis_disabled_image_sizes' ) ) {
		add_option( 'adis_disabled_image_sizes', array() );
	}

	// Flush rewrite rules
	flush_rewrite_rules();
}

// Deactivation hook
register_deactivation_hook( __FILE__, 'adis_deactivate' );

/**
 * Plugin deactivation function
 */
function adis_deactivate() {
	// Flush rewrite rules
	flush_rewrite_rules();
}

// Uninstall hook is handled by uninstall.php
