/**
 * Auto-Disable Unused Image Sizes - Admin JavaScript
 * Version: 1.0.0
 */

(function($) {
    'use strict';

    // Plugin namespace
    window.ADIS = window.ADIS || {};

    // Main admin functionality
    ADIS.Admin = {
        init: function() {
            this.bindEvents();
            this.initializeTooltips();
            this.updateBulkActionButtons();
        },

        bindEvents: function() {
            // Select all functionality
            $('#adis-select-all').on('click', this.selectAll);
            $('#adis-deselect-all').on('click', this.deselectAll);
            
            // Bulk actions
            $('#adis-disable-selected').on('click', this.disableSelected);
            $('#adis-enable-selected').on('click', this.enableSelected);
            
            // Individual checkbox changes
            $('.adis-size-checkbox-input').on('change', this.onCheckboxChange);
            
            // Form submission
            $('#adis-settings-form').on('submit', this.onFormSubmit);
            
            // Keyboard navigation
            this.bindKeyboardEvents();
        },

        selectAll: function(e) {
            e.preventDefault();
            $('.adis-size-checkbox-input').prop('checked', true).trigger('change');
            ADIS.Admin.updateBulkActionButtons();
            ADIS.Admin.showMessage('All image sizes selected.', 'success');
        },

        deselectAll: function(e) {
            e.preventDefault();
            $('.adis-size-checkbox-input').prop('checked', false).trigger('change');
            ADIS.Admin.updateBulkActionButtons();
            ADIS.Admin.showMessage('All image sizes deselected.', 'success');
        },

        disableSelected: function(e) {
            e.preventDefault();
            var selectedSizes = $('.adis-size-checkbox-input:checked');
            
            if (selectedSizes.length === 0) {
                ADIS.Admin.showMessage('Please select at least one image size to disable.', 'warning');
                return;
            }

            if (confirm(adis_ajax.strings.confirm_bulk_disable)) {
                selectedSizes.each(function() {
                    var $card = $(this).closest('.adis-size-card');
                    $card.removeClass('adis-enabled').addClass('adis-disabled');
                    $card.find('.adis-size-status')
                        .removeClass('adis-status-enabled')
                        .addClass('adis-status-disabled')
                        .text('Disabled');
                });
                
                ADIS.Admin.updateBulkActionButtons();
                ADIS.Admin.showMessage(selectedSizes.length + ' image size(s) disabled.', 'success');
            }
        },

        enableSelected: function(e) {
            e.preventDefault();
            var selectedSizes = $('.adis-size-checkbox-input:checked');
            
            if (selectedSizes.length === 0) {
                ADIS.Admin.showMessage('Please select at least one image size to enable.', 'warning');
                return;
            }

            if (confirm(adis_ajax.strings.confirm_bulk_enable)) {
                selectedSizes.each(function() {
                    var $card = $(this).closest('.adis-size-card');
                    $card.removeClass('adis-disabled').addClass('adis-enabled');
                    $card.find('.adis-size-status')
                        .removeClass('adis-status-disabled')
                        .addClass('adis-status-enabled')
                        .text('Enabled');
                });
                
                ADIS.Admin.updateBulkActionButtons();
                ADIS.Admin.showMessage(selectedSizes.length + ' image size(s) enabled.', 'success');
            }
        },

        onCheckboxChange: function() {
            var $checkbox = $(this);
            var $card = $checkbox.closest('.adis-size-card');
            var isChecked = $checkbox.is(':checked');
            
            if (isChecked) {
                $card.removeClass('adis-enabled').addClass('adis-disabled');
                $card.find('.adis-size-status')
                    .removeClass('adis-status-enabled')
                    .addClass('adis-status-disabled')
                    .text('Disabled');
            } else {
                $card.removeClass('adis-disabled').addClass('adis-enabled');
                $card.find('.adis-size-status')
                    .removeClass('adis-status-disabled')
                    .addClass('adis-status-enabled')
                    .text('Enabled');
            }
            
            ADIS.Admin.updateBulkActionButtons();
            ADIS.Admin.updateStats();
        },

        updateBulkActionButtons: function() {
            var checkedCount = $('.adis-size-checkbox-input:checked').length;
            var totalCount = $('.adis-size-checkbox-input').length;
            
            // Update button states
            $('#adis-disable-selected').prop('disabled', checkedCount === 0);
            $('#adis-enable-selected').prop('disabled', checkedCount === 0);
            
            // Update button text with counts
            if (checkedCount > 0) {
                $('#adis-disable-selected').text('Disable Selected (' + checkedCount + ')');
                $('#adis-enable-selected').text('Enable Selected (' + checkedCount + ')');
            } else {
                $('#adis-disable-selected').text('Disable Selected');
                $('#adis-enable-selected').text('Enable Selected');
            }
            
            // Update select all button
            if (checkedCount === totalCount) {
                $('#adis-select-all').text('Deselect All').removeClass('button').addClass('button-secondary');
            } else {
                $('#adis-select-all').text('Select All').removeClass('button-secondary').addClass('button');
            }
        },

        updateStats: function() {
            var totalSizes = $('.adis-size-checkbox-input').length;
            var disabledSizes = $('.adis-size-checkbox-input:checked').length;
            var activeSizes = totalSizes - disabledSizes;
            
            $('.adis-stat-box:nth-child(1) .adis-stat-number').text(totalSizes);
            $('.adis-stat-box:nth-child(2) .adis-stat-number').text(disabledSizes);
            $('.adis-stat-box:nth-child(3) .adis-stat-number').text(activeSizes);
        },

        onFormSubmit: function(e) {
            var $form = $(this);
            var $submitButton = $form.find('input[type="submit"]');
            
            // Show loading state
            $submitButton.prop('disabled', true).val('Saving...');
            $form.addClass('adis-loading');
            
            // Remove loading state after a short delay to show the loading effect
            setTimeout(function() {
                $form.removeClass('adis-loading');
            }, 500);
        },

        bindKeyboardEvents: function() {
            // Keyboard shortcuts
            $(document).on('keydown', function(e) {
                // Ctrl/Cmd + A to select all
                if ((e.ctrlKey || e.metaKey) && e.keyCode === 65) {
                    e.preventDefault();
                    $('#adis-select-all').click();
                }
                
                // Ctrl/Cmd + D to deselect all
                if ((e.ctrlKey || e.metaKey) && e.keyCode === 68) {
                    e.preventDefault();
                    $('#adis-deselect-all').click();
                }
                
                // Escape to deselect all
                if (e.keyCode === 27) {
                    $('.adis-size-checkbox-input').prop('checked', false).trigger('change');
                }
            });
            
            // Enter key on cards to toggle checkbox
            $('.adis-size-card').on('keydown', function(e) {
                if (e.keyCode === 13 || e.keyCode === 32) { // Enter or Space
                    e.preventDefault();
                    $(this).find('.adis-size-checkbox-input').click();
                }
            });
        },

        initializeTooltips: function() {
            // Add tooltips to size cards
            $('.adis-size-card').each(function() {
                var $card = $(this);
                var sizeName = $card.find('.adis-size-title').text();
                var dimensions = $card.find('.adis-size-dimensions').text();
                var isDefault = $card.find('.adis-size-badge.adis-default').length > 0;
                
                var tooltipText = sizeName;
                if (dimensions) {
                    tooltipText += '\nDimensions: ' + dimensions;
                }
                if (isDefault) {
                    tooltipText += '\nThis is a default WordPress image size';
                } else {
                    tooltipText += '\nThis is a custom image size';
                }
                
                $card.attr('title', tooltipText);
            });
        },

        showMessage: function(message, type) {
            type = type || 'info';
            
            // Remove existing messages
            $('.adis-message').remove();
            
            // Create new message
            var $message = $('<div class="adis-message adis-' + type + '">' + message + '</div>');
            
            // Insert after the header
            $('.adis-header-info').after($message);
            
            // Auto-hide after 5 seconds
            setTimeout(function() {
                $message.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        },

        // Utility functions
        debounce: function(func, wait, immediate) {
            var timeout;
            return function() {
                var context = this, args = arguments;
                var later = function() {
                    timeout = null;
                    if (!immediate) func.apply(context, args);
                };
                var callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(context, args);
            };
        }
    };

    // Initialize when document is ready
    $(document).ready(function() {
        ADIS.Admin.init();
        
        // Add some helpful hints
        if ($('.adis-size-card').length > 0) {
            ADIS.Admin.showMessage('Tip: Use Ctrl+A to select all image sizes, or Ctrl+D to deselect all.', 'info');
        }
    });

    // Expose to global scope for debugging
    window.ADIS_Admin = ADIS.Admin;

})(jQuery);
