<?php
/**
 * Authyo Google Authenticator Fallback Module
 *
 * Handles integration with Authyo SDK for Google Authenticator fallback
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Authyo_Google_Auth')) {
    class Authyo_Google_Auth {

        /**
         * Instance of this class
         */
        private static $instance = null;

        /**
         * Get instance of this class
         */
        public static function get_instance() {
            if (null === self::$instance) {
                self::$instance = new self();
            }
            return self::$instance;
        }

        /**
         * Constructor
         */
        private function __construct() {
            $this->init_hooks();
        }

        /**
         * Initialize hooks
         */
        private function init_hooks() {
            // Enqueue scripts specifically for the login page
            add_action('login_enqueue_scripts', array($this, 'enqueue_assets'));
            
            // Also support custom login pages where the shortcode might be used
            add_action('wp_enqueue_scripts', array($this, 'enqueue_assets'));

            // Enqueue admin assets
            add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

            // AJAX handlers for verification
            add_action('wp_ajax_authyo_google_auth_verify', array($this, 'ajax_verify_google_auth'));
            add_action('wp_ajax_nopriv_authyo_google_auth_verify', array($this, 'ajax_verify_google_auth'));
        }

        /**
         * Enqueue scripts and styles
         */
        public function enqueue_assets() {
            // Only enqueue if passwordless login is enabled
            if (class_exists('Authyo_Settings') && !Authyo_Settings::is_enabled()) {
                return;
            }

            // Get API Credentials
            $app_id = '';
            if (class_exists('Authyo_Settings') && is_callable(array('Authyo_Settings', 'get_settings'))) {
                $settings = Authyo_Settings::get_settings();
                $app_id = isset($settings['app_id']) ? $settings['app_id'] : '';
            } else {
                $settings = get_option('authyo_passwordless_login_settings', array());
                $app_id = isset($settings['app_id']) ? $settings['app_id'] : '';
            }
            
            if (empty($app_id)) {
                return;
            }

            wp_enqueue_script(
                'authyo-google-auth-fallback',
                plugin_dir_url(__FILE__) . 'assets/js/google-auth-fallback.js',
                array('jquery', 'authyo-passwordless-login'), // Depend on main login script
                AUTHYO_PASSWORDLESS_LOGIN_VERSION,
                true
            );

            wp_localize_script('authyo-google-auth-fallback', 'authyoGoogleAuth', array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('authyo_google_auth_nonce'),
                'appKey' => $app_id,
                'strings' => array(
                    'tryGoogleAuth' => __('Try Google Authenticator', 'authyo-passwordless-login'),
                    'fallbackTitle' => __('Google Authenticator Verification', 'authyo-passwordless-login'),
                    'verifying' => __('Verifying...', 'authyo-passwordless-login'),
                    'success' => __('Verification successful! Logging in...', 'authyo-passwordless-login'),
                    'error' => __('Verification failed. Please try again.', 'authyo-passwordless-login'),
                )
            ));
        }

        /**
         * Enqueue assets for admin settings
         */
        public function enqueue_admin_assets($hook) {
             if ($hook !== 'settings_page_authyo-passwordless-login') {
                return;
            }

            // Get API Credentials
            $app_id = '';
            if (class_exists('Authyo_Settings') && is_callable(array('Authyo_Settings', 'get_settings'))) {
                $settings = Authyo_Settings::get_settings();
                $app_id = isset($settings['app_id']) ? $settings['app_id'] : '';
            }

            if (empty($app_id)) {
                return;
            }
            
            // Current User Email
            $current_user = wp_get_current_user();
            $user_email = $current_user->user_email;

            // Enqueue SDK for Admin Setup
            // We use a separate localized script variable for admin context
            // Enqueue SDK for Admin Setup
            // Load external SDK
            wp_enqueue_script(
                'authyo-2fasdk',
                'https://app.authyo.io/js/v1/auth-2fasdk.js',
                array(),
                AUTHYO_PASSWORDLESS_LOGIN_VERSION,
                true
            );

            // Add attributes to the script tag using the standard WordPress filter
            add_filter('script_loader_tag', function($tag, $handle) use ($app_id, $user_email) {
                if ('authyo-2fasdk' !== $handle) {
                    return $tag;
                }
                
                // Inject data attributes directly into the script tag
                // This ensures they are present immediately when the script is parsed
                return str_replace(
                    ' src',
                    sprintf(
                        ' data-appkey="%s" data-emailid="%s" src',
                        esc_attr($app_id),
                        esc_attr($user_email)
                    ),
                    $tag
                );
            }, 10, 2);
            
            // Add inline script to handle response and configuration
            $inline_script = "
                (function() {
                    // Simple admin response handler
                    window.authyoResponse = function(response) {
                        console.log('Authyo Admin SDK Response:', response);
                        var messageContainer = document.getElementById('authyo-admin-message');
                        
                        if (response.success) {
                             // Always show the success message to the user for better feedback
                             var successMsg = response.message || 'Operation successful';
                             if (messageContainer) {
                                 messageContainer.innerHTML = '<p style=\"color:green; font-weight:bold;\">' + successMsg + '</p>';
                             }
                        } else {
                            if (messageContainer) messageContainer.innerHTML = '<p style=\"color:red;\">Error: ' + response.message + '</p>';
                        }
                    };
                })();
            ";
            
            wp_add_inline_script('authyo-2fasdk', $inline_script);
        }

        /**
         * AJAX handler for verifying Google Auth response
         */
        public function ajax_verify_google_auth() {
            check_ajax_referer('authyo_google_auth_nonce', 'nonce');

            $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
            $sdk_response = isset($_POST['sdk_response']) ? array_map('sanitize_text_field', wp_unslash($_POST['sdk_response'])) : array();

            if (empty($email) || !is_email($email)) {
                wp_send_json_error(array('message' => __('Invalid email address.', 'authyo-passwordless-login')));
                wp_die();
            }

            // Verify user exists
            $user = get_user_by('email', $email);
            if (!$user) {
                wp_send_json_error(array('message' => __('User not found.', 'authyo-passwordless-login')));
                wp_die();
            }

            // Verify the SDK response
            // Ideally, we would verify a cryptographic proof here.
            // Based on current SDK docs, we are trusting the client-side success callback as requested.
            // Ensure the response indicates success
            if (!isset($sdk_response['success']) || $sdk_response['success'] !== 'true') {
                 // Double check boolean vs string 'true'
                 if (!isset($sdk_response['success']) || ($sdk_response['success'] !== true && $sdk_response['success'] !== 'true')) {
                    wp_send_json_error(array('message' => __('Authentication failed.', 'authyo-passwordless-login')));
                    wp_die();
                 }
            }
            
            // Generate login token using the main plugin's logic
            if (class_exists('Authyo_Login') && method_exists('Authyo_Login', 'generate_login_token')) {
                $token = Authyo_Login::generate_login_token($user->ID);
                
                $login_url = add_query_arg(
                    array('authyo_token' => $token),
                    wp_login_url()
                );
                
                $login_url = apply_filters('authyo_passwordless_login_token_redirect', $login_url, $user);

                wp_send_json_success(array(
                    'message' => __('Verified', 'authyo-passwordless-login'),
                    'redirect' => $login_url
                ));
            } else {
                wp_send_json_error(array('message' => __('Internal plugin error.', 'authyo-passwordless-login')));
            }

            wp_die();
        }
    }

    // Initialize the module
    Authyo_Google_Auth::get_instance();
}
