(function ($) {
    'use strict';

    $(document).ready(function () {
        const $container = $('#authyo-login-container');
        const $sendOtpBtn = $('#authyo-send-otp-btn');
        const $emailInput = $('#authyo-email');
        const $messages = $('#authyo-messages'); // Ensure this selector matches your main plugin's message container

        let googleAuthScriptLoaded = false;

        // Create and append the "Try Google Authenticator" button
        const $fallbackBtn = $('<button>', {
            type: 'button',
            id: 'authyo-google-auth-btn',
            class: 'button button-secondary', // Use secondary style to distinct from primary
            text: authyoGoogleAuth.strings.tryGoogleAuth,
            text: authyoGoogleAuth.strings.tryGoogleAuth,
            style: 'margin-top: 10px; width: 100%; display: none;' // Hidden by default
        });

        // Add container for the SDK
        const $sdkContainer = $('<div>', {
            id: 'authyo-2fasdk',
            style: 'margin-top: 20px; text-align: center;'
        });

        // Function to determine current step and append items
        function updateFallbackPosition() {
            var currentStep = $('#authyo-otp-step').is(':visible') ? $('#authyo-otp-step') : $('#authyo-email-step');
            // Move button to current step
            currentStep.append($fallbackBtn);
            // Note: SDK Container should stay in the main container so it doesn't get hidden
            // whenever we hide the steps.
        }

        // Initially append to email step (hidden anyway)
        // We will move it to OTP step on 3rd click, or keep it on Email step if error occurs there
        if ($container.length) {
            $('#authyo-email-step').append($fallbackBtn);
            // Append SDK container to MAIN container, not the steps
            $container.append($sdkContainer);
        }

        // Monitor messages for errors (e.g. API Timeout) to trigger fallback immediately
        const observer = new MutationObserver(function (mutations) {
            mutations.forEach(function (mutation) {
                if (mutation.addedNodes.length) {
                    // Check if error message was added
                    const $errorDiv = $messages.find('.error, .notice-error');
                    if ($errorDiv.length > 0) {
                        const errorText = $errorDiv.text().trim();

                        // Define exclusion list for errors where we should NOT show the fallback
                        // because they imply the identity is unknown or invalid.
                        const excludedErrors = [
                            authyoPasswordlessLogin.strings.userNotFound,
                            authyoPasswordlessLogin.strings.invalidEmail
                        ];

                        if (excludedErrors.includes(errorText)) {
                            console.log('Identity error detected, keeping Google Auth fallback hidden.');
                            return;
                        }

                        console.log('Generic error detected, offering Google Auth fallback...');
                        updateFallbackPosition();
                        $fallbackBtn.show();
                        // Add hint text
                        if ($messages.find('.google-auth-hint').length === 0) {
                            $messages.append('<p class="info google-auth-hint" style="font-size:small; color:#666; margin-top:5px;">Connection issues? Try Google Authenticator below.</p>');
                        }
                    }
                }
            });
        });

        if ($messages.length) {
            observer.observe($messages[0], { childList: true, subtree: true });
        }

        // Counter for resend clicks
        let resendClickCount = 0;

        // Monitor clicks on the Resend OTP button from the main plugin
        $(document).on('click', '#authyo-resend-otp-btn', function () {
            resendClickCount++;
            console.log('Resend OTP clicked. Count: ' + resendClickCount);

            if (resendClickCount >= 3) {
                // Move to OTP step and show
                updateFallbackPosition();
                $fallbackBtn.show();
            }
        });

        // Ensure button stays hidden initially
        $emailInput.on('input change', function () {
            // Do not show button immediately on input anymore
            // Wait for resend clicks
        });

        $fallbackBtn.on('click', function () {
            const email = $emailInput.val().trim();

            if (!email) {
                alert(authyoPasswordlessLogin.strings.emailRequired); // Reuse main string if available or fallback
                return;
            }

            // Ensure SDK container is in the main container
            if (!$('#authyo-2fasdk').parent().is($container)) {
                $container.append($sdkContainer);
            }

            // ROBUST VISIBILITY: Hide all other children of the main container
            // This ensures Email Step, OTP Step, and any other debris are hidden
            $container.children().not($sdkContainer).hide();

            // Show loading or title in the SDK container (since messages might be hidden if inside steps)
            // Actually messages ($messages) is usually top level.
            // If $messages is hidden, we can't see errors.
            // Let's check if $messages is a child of $container.
            // If so, we should exclude it from hiding if possible, OR move it to SDK container context?
            // Usually messages are part of the steps or separate.
            // Let's assume we want to hide form steps but keep messages if they are separate.
            // SAFEST: Explicitly hide the known steps.
            $('#authyo-email-step').hide();
            $('#authyo-otp-step').hide();
            // AND hide siblings just in case
            $sdkContainer.siblings('.authyo-step, #authyo-email-step, #authyo-otp-step').hide();

            // Make sure SDK container is visible
            $sdkContainer.show();

            // Show loading or title
            // We append title to sdkContainer to be sure it's seen
            $sdkContainer.prepend('<p class="info" id="ga-fallback-title" style="margin-bottom:15px; font-weight:bold;">' + authyoGoogleAuth.strings.fallbackTitle + '</p>');
            // Also try to update main messages if visible
            if ($messages.is(':visible')) {
                $messages.html('');
            }

            loadAuthyoSdk(email);
        });

        function loadAuthyoSdk(email) {
            console.log('Initiating Authyo SDK load...');
            console.log('Email:', email);
            console.log('App Key:', authyoGoogleAuth.appKey ? 'Present' : 'Missing');

            // Remove existing script if any to re-initialize with new email
            $('script[src*="auth-2fasdk.js"]').remove();

            // Clean container
            $('#authyo-2fasdk').empty();

            // Create script element
            const script = document.createElement('script');
            script.src = 'https://app.authyo.io/js/v1/auth-2fasdk.js';
            script.setAttribute('data-appkey', authyoGoogleAuth.appKey);
            script.setAttribute('data-emailid', email);
            script.async = true;

            script.onload = function () {
                console.log('Authyo SDK Script loaded successfully.');
            };

            script.onerror = function () {
                console.error('Failed to load Authyo SDK Script.');
                $messages.html('<p class="error">Failed to load Google Authenticator service. Please check your connection.</p>');
                addBackLink();
            };

            document.body.appendChild(script);
        }

        // Define the global callback
        window.authyoResponse = function (response) {
            console.log("Authyo SDK Response:", response);

            // SECURITY FIX: Strictly check for "verified" message.
            // Other events (like AuthState or Enable 2FA) also return success:true but should NOT log the user in.
            // Updated to include "2fa successful!" as seen in user testing.
            // CRITICAL: Use strict equality to avoid false positives (e.g. "not verified" containing "verified")
            const validMessages = ['verified', '2fa successful', '2fa successful!'];
            const msg = (response.message || '').toLowerCase().trim();

            console.log("Checking response message: '" + msg + "'");

            if (response.success && validMessages.includes(msg)) {
                $messages.html('<p class="success">' + authyoGoogleAuth.strings.success + '</p>');

                // Verify with backend
                $.ajax({
                    url: authyoGoogleAuth.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'authyo_google_auth_verify',
                        nonce: authyoGoogleAuth.nonce,
                        email: $emailInput.val().trim(),
                        sdk_response: response
                    },
                    success: function (res) {
                        if (res.success) {
                            // Use window.location.replace to prevent back-button loops
                            window.location.replace(res.data.redirect);
                        } else {
                            $messages.html('<p class="error">' + res.data.message + '</p>');
                            addBackLink();
                        }
                    },
                    error: function () {
                        $messages.html('<p class="error">' + authyoGoogleAuth.strings.error + '</p>');
                        addBackLink();
                    }
                });

            } else if (response.success) {
                // Success is true, but not verified (e.g. AuthState loaded, or QR code generated)
                // Just log it, do not take action that implies authentication
                console.log("Authyo SDK Event: " + response.message);

                if (response.message === 'qrcode generated successfully') {
                    $messages.html('<p class="info">Please scan the QR code with your Google Authenticator app.</p>');
                }

            } else {
                $messages.html('<p class="error">' + (response.message || authyoGoogleAuth.strings.error) + '</p>');
                addBackLink();
            }
        };

        function addBackLink() {
            if ($('#authyo-back-to-email').length === 0) {
                const $backLink = $('<a href="#" id="authyo-back-to-email" style="display:block; margin-top:10px;">&larr; Back to Email Login</a>');
                $sdkContainer.append($backLink);
                $backLink.on('click', function (e) {
                    e.preventDefault();
                    $('#authyo-email-step').show();
                    $sdkContainer.empty(); // Clear SDK
                    $messages.empty();
                    $(this).remove();
                });
            }
        }

    });

})(jQuery);
