<?php
/**
 * Authyo Settings Handler
 *
 * Handles plugin settings page and options
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Authyo_Settings')) {
class Authyo_Settings {
    
    /**
     * Option name
     */
    const OPTION_NAME = 'authyo_passwordless_login_settings';
    
    /**
     * Add settings page to WordPress admin
     */
    public static function add_settings_page() {
        add_options_page(
            __('Authyo Passwordless Login Settings', 'authyo-passwordless-login'),
            __('Authyo Passwordless Login', 'authyo-passwordless-login'),
            'manage_options',
            'authyo-passwordless-login',
            array(__CLASS__, 'render_settings_page')
        );
    }

    /**
     * Enqueue admin scripts and styles
     */
    public static function enqueue_admin_assets($hook) {
        // Only enqueue on our settings page
        if ($hook !== 'settings_page_authyo-passwordless-login') {
            return;
        }

        wp_enqueue_style(
            'authyo-passwordless-login-admin',
            AUTHYO_PASSWORDLESS_LOGIN_URL . 'assets/css/admin.css',
            array(),
            AUTHYO_PASSWORDLESS_LOGIN_VERSION
        );
    }
    
    /**
     * Register settings
     */
    public static function register_settings() {
        register_setting(
            'authyo_passwordless_login_settings_group',
            self::OPTION_NAME,
            array(__CLASS__, 'sanitize_settings')
        );
        
        add_settings_section(
            'authyo_passwordless_login_general_section',
            __('General Settings', 'authyo-passwordless-login'),
            array(__CLASS__, 'render_general_section_description'),
            'authyo-passwordless-login'
        );
        
        add_settings_field(
            'enable_passwordless_login',
            __('Enable Passwordless Login', 'authyo-passwordless-login'),
            array(__CLASS__, 'render_enable_field'),
            'authyo-passwordless-login',
            'authyo_passwordless_login_general_section'
        );
        
        add_settings_section(
            'authyo_passwordless_login_main_section',
            __('Authyo API Credentials', 'authyo-passwordless-login'),
            array(__CLASS__, 'render_section_description'),
            'authyo-passwordless-login'
        );

        // Google Authenticator Setup Section (Admin Only)
        add_settings_section(
            'authyo_google_auth_setup_section',
            __('Google Authenticator Setup', 'authyo-passwordless-login'),
            array(__CLASS__, 'render_google_auth_section'),
            'authyo-passwordless-login'
        );
        
        add_settings_field(
            'app_id',
            __('Authyo App ID', 'authyo-passwordless-login'),
            array(__CLASS__, 'render_app_id_field'),
            'authyo-passwordless-login',
            'authyo_passwordless_login_main_section'
        );
        
        add_settings_field(
            'client_id',
            __('Authyo Client ID', 'authyo-passwordless-login'),
            array(__CLASS__, 'render_client_id_field'),
            'authyo-passwordless-login',
            'authyo_passwordless_login_main_section'
        );
        
        add_settings_field(
            'client_secret',
            __('Authyo Client Secret', 'authyo-passwordless-login'),
            array(__CLASS__, 'render_client_secret_field'),
            'authyo-passwordless-login',
            'authyo_passwordless_login_main_section'
        );
    }
    
    /**
     * Sanitize settings
     */
    public static function sanitize_settings($input) {
        $sanitized = array();
        
        // Get existing settings to check credentials
        $existing_settings = self::get_settings();
        
        // Sanitize API credentials first
        if (isset($input['app_id'])) {
            $sanitized['app_id'] = sanitize_text_field($input['app_id']);
        } elseif (isset($existing_settings['app_id'])) {
            $sanitized['app_id'] = $existing_settings['app_id'];
        }
        
        if (isset($input['client_id'])) {
            $sanitized['client_id'] = sanitize_text_field($input['client_id']);
        } elseif (isset($existing_settings['client_id'])) {
            $sanitized['client_id'] = $existing_settings['client_id'];
        }
        
        if (isset($input['client_secret'])) {
            $sanitized['client_secret'] = sanitize_text_field($input['client_secret']);
        } elseif (isset($existing_settings['client_secret'])) {
            $sanitized['client_secret'] = $existing_settings['client_secret'];
        }
        
        // Check if all credentials are present
        $has_all_credentials = !empty($sanitized['app_id']) && !empty($sanitized['client_id']) && !empty($sanitized['client_secret']);
        
        // Enable/Disable toggle (default to disabled if not set)
        if (isset($input['enable_passwordless_login'])) {
            // Only allow enabling if all credentials are present
            if ($has_all_credentials) {
                $sanitized['enable_passwordless_login'] = absint($input['enable_passwordless_login']);
            } else {
                // Force disable if credentials are missing
                $sanitized['enable_passwordless_login'] = 0;
            }
        } else {
            $sanitized['enable_passwordless_login'] = 0;
        }

        // Set transient to trigger tracking in the tracker class
        set_transient('authyo_passwordless_trigger_tracking', $sanitized, 60);
        
        return $sanitized;
    }
    
    /**
     * Render general section description
     */
    public static function render_general_section_description() {
        $has_all_credentials = self::has_all_credentials();
        if ($has_all_credentials) {
            echo '<p>' . esc_html__('Configure general plugin settings. All API credentials are configured.', 'authyo-passwordless-login') . '</p>';
        } else {
            echo '<p>' . esc_html__('Configure general plugin settings. Please add all API credentials below before enabling passwordless login.', 'authyo-passwordless-login') . '</p>';
        }
    }
    
    /**
     * Render section description
     */
    public static function render_section_description() {
        echo '<p>' . esc_html__('Enter your Authyo API credentials below. These can be found in your Authyo dashboard.', 'authyo-passwordless-login') . '</p>';

        echo '<div class="authyo-help-text">';
        echo '<span class="dashicons dashicons-info"></span>';
        echo esc_html__('Get these credentials from your Authyo dashboard at ', 'authyo-passwordless-login');
        echo '<a href="https://app.authyo.io/account/welcome?utm_source=plugin-authyo-passwordless-login" target="_blank" rel="noopener" style="cursor: pointer;">authyo.io</a>';
        echo '</div>';
    }
    
    /**
     * Render Enable/Disable field
     */
    public static function render_enable_field() {
        $settings = self::get_settings();
        $enabled = isset($settings['enable_passwordless_login']) ? absint($settings['enable_passwordless_login']) : 0; // Default to disabled
        $has_all_credentials = self::has_all_credentials();
        ?>
        <label>
            <input type="checkbox" name="authyo_passwordless_login_settings[enable_passwordless_login]" value="1" <?php checked($enabled, 1); ?> <?php disabled(!$has_all_credentials); ?> />
            <?php esc_html_e('Enable passwordless login on the WordPress login page', 'authyo-passwordless-login'); ?>
        </label>
        <?php if (!$has_all_credentials) : ?>
            <p class="description" style="color: #d63638;">
                <strong><?php esc_html_e('Please configure all API credentials (App ID, Client ID, and Client Secret) before enabling passwordless login.', 'authyo-passwordless-login'); ?></strong>
            </p>
        <?php else : ?>
            <p class="description"><?php esc_html_e('When enabled, the passwordless login form will be displayed and when disabled, only the default WordPress login form will be shown.', 'authyo-passwordless-login'); ?></p>
        <?php endif; ?>
        <?php
    }
    
    /**
     * Render App ID field
     */
    public static function render_app_id_field() {
        $settings = self::get_settings();
        $value = isset($settings['app_id']) ? $settings['app_id'] : '';
        ?>
        <input type="text" name="authyo_passwordless_login_settings[app_id]" value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e('Your Authyo App ID', 'authyo-passwordless-login'); ?></p>
        <?php
    }
    
    /**
     * Render Client ID field
     */
    public static function render_client_id_field() {
        $settings = self::get_settings();
        $value = isset($settings['client_id']) ? $settings['client_id'] : '';
        ?>
        <input type="text" name="authyo_passwordless_login_settings[client_id]" value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e('Your Authyo Client ID', 'authyo-passwordless-login'); ?></p>
        <?php
    }
    
    /**
     * Render Client Secret field
     */
    public static function render_client_secret_field() {
        $settings = self::get_settings();
        $value = isset($settings['client_secret']) ? $settings['client_secret'] : '';
        ?>
        <input type="password" name="authyo_passwordless_login_settings[client_secret]" value="<?php echo esc_attr($value); ?>" class="regular-text" />
        <p class="description"><?php esc_html_e('Your Authyo Client Secret', 'authyo-passwordless-login'); ?></p>
        <?php
    }
    
    /**
     * Render Google Authenticator Setup Section
     */
    public static function render_google_auth_section() {
        echo '<p>' . esc_html__('Scan the QR code below with your Google Authenticator app. This will act as your fallback login method.', 'authyo-passwordless-login') . '</p>';
        echo '<div id="authyo-admin-message" style="margin-bottom: 10px;"></div>';
        echo '<div id="authyo-2fasdk" style="background: white; padding: 20px; display: inline-block; border: 1px solid #ccc;"></div>';
        echo '<p class="description">' . esc_html__('If you do not see a QR code, please ensure your API credentials are correct and saved.', 'authyo-passwordless-login') . '</p>';
        echo '<p class="description" style="margin-top: 10px; font-weight: bold;">' . esc_html__('Note: After attempting 3 unsuccessful login, you can login with two factor authentication app.', 'authyo-passwordless-login') . '</p>';
    }
    
    /**
     * Render settings page
     */
    public static function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Check if settings were saved
        // Nonce verification is handled by WordPress settings API via settings_fields()
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        if (isset($_GET['settings-updated'])) {
            add_settings_error('authyo_passwordless_login_messages', 'authyo_passwordless_login_message', __('Settings saved successfully.', 'authyo-passwordless-login'), 'updated');
        }
        
        settings_errors('authyo_passwordless_login_messages');
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <form action="options.php" method="post">
                <?php
                settings_fields('authyo_passwordless_login_settings_group');
                do_settings_sections('authyo-passwordless-login');
                submit_button(esc_html__('Save Settings', 'authyo-passwordless-login'));
                ?>
            </form>
        </div>
        <?php
    }
    
    /**
     * Get settings
     *
     * @return array Settings array
     */
    public static function get_settings() {
        return get_option(self::OPTION_NAME, array());
    }
    
    /**
     * Check if all API credentials are configured
     *
     * @return bool True if all credentials are present, false otherwise
     */
    public static function has_all_credentials() {
        $settings = self::get_settings();
        return !empty($settings['app_id']) && !empty($settings['client_id']) && !empty($settings['client_secret']);
    }
    
    /**
     * Check if passwordless login is enabled
     *
     * @return bool True if enabled, false otherwise
     */
    public static function is_enabled() {
        $settings = self::get_settings();
        // Default to disabled if not set
        // Also check that all credentials are present before enabling
        if (!self::has_all_credentials()) {
            return false;
        }
        return isset($settings['enable_passwordless_login']) ? absint($settings['enable_passwordless_login']) === 1 : false;
    }
}
} // End if !class_exists

