<?php
/**
 * Feedback Handler for Authyo Passwordless Login
 * 
 * Handles deactivation feedback collection, settings tracking, and email subscriptions.
 */
if ( ! defined( 'ABSPATH' ) ) exit;

class Authyo_Passwordless_Feedback {
	
	const FEEDBACK_API_URL = 'https://app.authyo.io/api/v1/user/WordpressWebhook';
	const PLUGIN_SLUG = 'authyo-passwordless-login';
	const SUBJECT_PREFIX = 'Passwordless_Login_';
	
	/**
	 * Constructor
	 */
	public function __construct() {
		// Hook into admin scripts to enqueue scripts and styles
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
		
		// Register AJAX endpoint for feedback submission (admin only)
		add_action( 'wp_ajax_authyo_passwordless_submit_deactivation_feedback', [ $this, 'handle_feedback_submission' ] );
		
		// Register REST API endpoints
		add_action( 'rest_api_init', [ $this, 'register_rest_endpoints' ] );
		
		// Hook into settings save to trigger tracking
		add_action( 'update_option_authyo_passwordless_login_settings', [ $this, 'handle_settings_save_tracking_hook' ], 10, 3 );
		add_action( 'add_option_authyo_passwordless_login_settings', [ $this, 'handle_settings_add_tracking_hook' ], 10, 2 );
		
		// Email subscription popup hooks
		add_action( 'admin_notices', [ $this, 'maybe_show_email_subscription_popup' ] );
		add_action( 'wp_ajax_authyo_passwordless_submit_email_subscription', [ $this, 'handle_email_subscription' ] );
		add_action( 'wp_ajax_authyo_passwordless_dismiss_email_subscription', [ $this, 'handle_email_subscription_dismiss' ] );
	}
	
	/**
	 * Handle the update_option hook for settings save tracking
	 */
	public function handle_settings_save_tracking_hook( $old_value, $value, $option ) {
		// Check if we have the value from sanitize callback
		$unencrypted_value = get_transient( 'authyo_passwordless_trigger_tracking' );
		if ( $unencrypted_value !== false ) {
			delete_transient( 'authyo_passwordless_trigger_tracking' );
			$value = $unencrypted_value;
		}
		
		// Send tracking whenever settings are saved (non-blocking)
		$this->send_settings_save_to_api( $value );
	}
	
	/**
	 * Handle the add_option hook for settings save tracking (first time save)
	 */
	public function handle_settings_add_tracking_hook( $option, $value ) {
		// Check if we have the value from sanitize callback
		$unencrypted_value = get_transient( 'authyo_passwordless_trigger_tracking' );
		if ( $unencrypted_value !== false ) {
			delete_transient( 'authyo_passwordless_trigger_tracking' );
			$value = $unencrypted_value;
		}
		
		// Send tracking when settings are added for the first time
		$this->send_settings_save_to_api( $value );
	}
	
	/**
	 * Enqueue scripts and styles
	 */
	public function enqueue_scripts( $hook ) {
		// Load deactivation popup scripts on plugins page
		if ( $hook === 'plugins.php' ) {
			$this->enqueue_deactivation_scripts();
		}
		
		// Load email subscription popup scripts on plugin settings page
		if ( $hook === 'settings_page_authyo-passwordless-login' ) {
			$this->enqueue_email_subscription_scripts();
		}
	}
	
	/**
	 * Enqueue scripts and styles for deactivation popup
	 */
	private function enqueue_deactivation_scripts() {
		// Note: CSS and JS files would need to be created or adapted
		wp_enqueue_style(
			'authyo-passwordless-deactivation',
			AUTHYO_PASSWORDLESS_LOGIN_URL . 'assets/css/deactivation.css',
			[],
			AUTHYO_PASSWORDLESS_LOGIN_VERSION
		);
		
		wp_enqueue_script(
			'authyo-passwordless-deactivation',
			AUTHYO_PASSWORDLESS_LOGIN_URL . 'assets/js/deactivation.js',
			[ 'jquery' ],
			AUTHYO_PASSWORDLESS_LOGIN_VERSION,
			true
		);
		
		// Get settings to pass app_id/client_id to JavaScript
		$settings = Authyo_Settings::get_settings();
		$app_id = $settings['app_id'] ?? '';
		
		$plugin_basename = AUTHYO_PASSWORDLESS_LOGIN_BASENAME;
		
		wp_localize_script( 'authyo-passwordless-deactivation', 'AUTHYO_PASSWORDLESS_DEACTIVATION', [
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'rest_url' => rest_url( 'authyo-passwordless/v1/deactivation-feedback' ),
			'rest_nonce' => wp_create_nonce( 'wp_rest' ),
			'nonce' => wp_create_nonce( 'authyo_passwordless_deactivation_feedback' ),
			'app_id' => $app_id,
			'plugin_slug' => self::PLUGIN_SLUG,
			'plugin_basename' => $plugin_basename,
			'plugin_version' => AUTHYO_PASSWORDLESS_LOGIN_VERSION,
			'i18n' => [
				'title' => __( 'Quick Feedback', 'authyo-passwordless-login' ),
				'question' => __( 'Would you mind sharing why you are deactivating the Authyo Passwordless Login plugin?', 'authyo-passwordless-login' ),
				'reason_temporary' => __( 'This is temporary — I will re-enable it later.', 'authyo-passwordless-login' ),
				'reason_complicated' => __( 'It is too complicated to use.', 'authyo-passwordless-login' ),
				'reason_alternative' => __( 'I found a better alternative.', 'authyo-passwordless-login' ),
				'reason_other' => __( 'Other (please share below)', 'authyo-passwordless-login' ),
				'other_placeholder' => __( 'Please share your feedback...', 'authyo-passwordless-login' ),
				'skip' => __( 'Skip', 'authyo-passwordless-login' ),
				'submit' => __( 'Submit & Deactivate', 'authyo-passwordless-login' ),
				'submitting' => __( 'Submitting...', 'authyo-passwordless-login' ),
				'thank_you' => __( 'Thank you for your feedback!', 'authyo-passwordless-login' ),
			],
		] );
	}
	
	/**
	 * Enqueue scripts and styles for email subscription popup
	 */
	private function enqueue_email_subscription_scripts() {
		wp_enqueue_style(
			'authyo-passwordless-email-subscription',
			AUTHYO_PASSWORDLESS_LOGIN_URL . 'assets/css/email-subscription.css',
			[],
			AUTHYO_PASSWORDLESS_LOGIN_VERSION
		);
		
		wp_enqueue_script(
			'authyo-passwordless-email-subscription',
			AUTHYO_PASSWORDLESS_LOGIN_URL . 'assets/js/email-subscription.js',
			[ 'jquery' ],
			AUTHYO_PASSWORDLESS_LOGIN_VERSION,
			true
		);
		
		wp_localize_script( 'authyo-passwordless-email-subscription', 'AUTHYO_PASSWORDLESS_EMAIL_SUBSCRIPTION', [
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce' => wp_create_nonce( 'authyo_passwordless_email_subscription' ),
			'plugin_slug' => self::PLUGIN_SLUG,
			'plugin_version' => AUTHYO_PASSWORDLESS_LOGIN_VERSION,
			'i18n' => [
				'submitting' => __( 'Submitting...', 'authyo-passwordless-login' ),
			],
		] );
	}
	
	/**
	 * Register REST API endpoints
	 */
	public function register_rest_endpoints() {
		register_rest_route( 'authyo-passwordless/v1', '/deactivation-feedback', [
			'methods' => 'POST',
			'callback' => [ $this, 'handle_feedback_submission_rest' ],
			'permission_callback' => [ $this, 'verify_permission' ],
		] );
	}
	
	/**
	 * Verify permission for REST endpoint
	 */
	public function verify_permission() {
		return current_user_can( 'activate_plugins' );
	}
	
	/**
	 * Handle feedback submission via AJAX
	 */
	public function handle_feedback_submission() {
		check_ajax_referer( 'authyo_passwordless_deactivation_feedback', 'nonce' );
		
		if ( ! current_user_can( 'activate_plugins' ) ) {
			wp_send_json_error( [ 'message' => __( 'Permission denied.', 'authyo-passwordless-login' ) ] );
		}
		
		$reason = isset( $_POST['reason'] ) ? sanitize_text_field( wp_unslash( $_POST['reason'] ) ) : 'skipped';
		$details = isset( $_POST['details'] ) ? sanitize_textarea_field( wp_unslash( $_POST['details'] ) ) : '';
		
		$this->send_feedback_to_api( $reason, $details );
		
		wp_send_json_success( [
			'message' => __( 'Thank you for your feedback!', 'authyo-passwordless-login' ),
		] );
	}
	
	/**
	 * Handle feedback submission via REST API
	 */
	public function handle_feedback_submission_rest( WP_REST_Request $request ) {
		$params = $request->get_params();
		
		$reason = isset( $params['reason'] ) ? sanitize_text_field( $params['reason'] ) : 'skipped';
		$details = isset( $params['details'] ) ? sanitize_textarea_field( $params['details'] ) : '';
		
		$this->send_feedback_to_api( $reason, $details );
		
		return new WP_REST_Response( [
			'success' => true,
			'message' => __( 'Thank you for your feedback!', 'authyo-passwordless-login' ),
		], 200 );
	}
	
	/**
	 * Send settings save tracking data
	 */
	public function send_settings_save_to_api( $saved_value = null ) {
		$app_id = '';
		$client_id = '';
		$client_secret = '';
		
		if ( ! empty( $saved_value ) && is_array( $saved_value ) ) {
			$app_id = $saved_value['app_id'] ?? '';
			$client_id = $saved_value['client_id'] ?? '';
			$client_secret = $saved_value['client_secret'] ?? '';
		}
		
		if ( empty( $app_id ) || empty( $client_id ) || empty( $client_secret ) ) {
			$settings = Authyo_Settings::get_settings();
			$app_id = $app_id ?: ( $settings['app_id'] ?? '' );
			$client_id = $client_id ?: ( $settings['client_id'] ?? '' );
			$client_secret = $client_secret ?: ( $settings['client_secret'] ?? '' );
		}
		
		// If any credential is empty, we can't send activation tracking
		if ( empty( $app_id ) || empty( $client_id ) || empty( $client_secret ) ) {
			return false;
		}
		
		$pageurl = admin_url( 'options-general.php?page=authyo-passwordless-login' );
		
		$payload = [
			'subject' => self::SUBJECT_PREFIX . 'Plugin_Activation',
			'api_key' => $app_id,
			'data' => [
				'plugin' => self::PLUGIN_SLUG,
				'version' => AUTHYO_PASSWORDLESS_LOGIN_VERSION,
				'timestamp' => gmdate( 'c' ),
				'pageurl' => $pageurl,
				'plugin-category' => 'authyo-passwordless',
			],
		];

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
			error_log( '[Authyo Passwordless Feedback] Sending settings save tracking: ' . wp_json_encode( $payload ) );
		}
		
		$this->make_async_request( $payload );
		return true;
	}
	
	/**
	 * Send feedback data
	 */
	private function send_feedback_to_api( $reason, $details = '' ) {
		$settings = Authyo_Settings::get_settings();
		$app_id = $settings['app_id'] ?? '';
		
		$payload = [
			'subject' => self::SUBJECT_PREFIX . 'Plugin_Deactivation_Popup',
			'api_key' => $app_id,
			'data' => [
				'reason' => $reason,
				'details' => $details,
				'plugin' => self::PLUGIN_SLUG,
				'version' => AUTHYO_PASSWORDLESS_LOGIN_VERSION,
				'timestamp' => gmdate( 'c' ),
				'pageurl' => admin_url( 'plugins.php' ),
			],
		];
		
		$this->make_async_request( $payload );
		return true;
	}
	
	/**
	 * Check if email subscription popup should be shown
	 */
	public function maybe_show_email_subscription_popup() {
		$screen = get_current_screen();
		if ( ! $screen || $screen->id !== 'settings_page_authyo-passwordless-login' ) {
			return;
		}
		
		if ( get_option( 'authyo_passwordless_email_subscription_dismissed' ) || get_option( 'authyo_passwordless_email_subscription_completed' ) ) {
			return;
		}
		
		$show_popup = get_transient( 'authyo_passwordless_show_email_subscription' );
		if ( ! $show_popup ) {
			return;
		}
		
		delete_transient( 'authyo_passwordless_show_email_subscription' );
		
		$this->render_email_subscription_popup();
	}
	
	/**
	 * Render email subscription popup
	 */
	private function render_email_subscription_popup() {
		?>
		<div id="authyo-passwordless-email-subscription-popup" class="authyo-cf7-email-subscription-popup" style="display: none;">
			<div class="authyo-cf7-email-subscription-overlay"></div>
			<div class="authyo-cf7-email-subscription-content">
				<button type="button" class="authyo-cf7-email-subscription-close" aria-label="<?php esc_attr_e( 'Close', 'authyo-passwordless-login' ); ?>">
					<span class="dashicons dashicons-no-alt"></span>
				</button>
				<div class="authyo-cf7-email-subscription-header">
					<h2><?php esc_html_e( 'Stay Updated with Authyo', 'authyo-passwordless-login' ); ?></h2>
				</div>
				<div class="authyo-cf7-email-subscription-body">
					<p class="authyo-cf7-email-subscription-description">
						<?php esc_html_e( 'Optional: Enter your email to receive important updates, security notices, and new features related to Authyo plugins.', 'authyo-passwordless-login' ); ?>
					</p>
					<form id="authyo-passwordless-email-subscription-form">
						<div class="authyo-cf7-email-subscription-field">
							<input type="email" id="authyo-passwordless-email-subscription-input" name="email" placeholder="<?php esc_attr_e( 'Enter your email address', 'authyo-passwordless-login' ); ?>" required />
							<span class="authyo-cf7-email-subscription-error" style="display: none;"></span>
						</div>
						<div class="authyo-cf7-email-subscription-actions">
							<button type="button" class="authyo-passwordless-email-subscription-skip button"><?php esc_html_e( 'Skip for now', 'authyo-passwordless-login' ); ?></button>
							<button type="submit" class="authyo-passwordless-email-subscription-submit button button-primary"><?php esc_html_e( 'Subscribe', 'authyo-passwordless-login' ); ?></button>
						</div>
					</form>
				</div>
			</div>
		</div>
		<?php
	}
	
	/**
	 * Handle AJAX request for email subscription
	 */
	public function handle_email_subscription() {
		check_ajax_referer( 'authyo_passwordless_email_subscription', 'nonce' );
		
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( [ 'message' => __( 'Permission denied.', 'authyo-passwordless-login' ) ] );
		}
		
		$email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
		
		if ( empty( $email ) || ! is_email( $email ) ) {
			wp_send_json_error( [ 'message' => __( 'Valid email required.', 'authyo-passwordless-login' ) ] );
		}
		
		$this->send_email_subscription_to_api( $email );
		update_option( 'authyo_passwordless_email_subscription_completed', true );
		
		wp_send_json_success( [ 'message' => __( 'Subscribed!', 'authyo-passwordless-login' ) ] );
	}
	
	/**
	 * Handle AJAX request for dismissal
	 */
	public function handle_email_subscription_dismiss() {
		check_ajax_referer( 'authyo_passwordless_email_subscription', 'nonce' );
		update_option( 'authyo_passwordless_email_subscription_dismissed', true );
		wp_send_json_success();
	}
	
	/**
	 * Send email subscription to API
	 */
	private function send_email_subscription_to_api( $email ) {
		$settings = Authyo_Settings::get_settings();
		$app_id = $settings['app_id'] ?? '';
		
		$payload = [
			'subject' => self::SUBJECT_PREFIX . 'email_subscription',
			'api_key' => $app_id,
			'data' => [
				'event_type' => 'email_subscription',
				'email' => $email,
				'site_url' => home_url(),
				'plugin_slug' => self::PLUGIN_SLUG,
				'plugin_version' => AUTHYO_PASSWORDLESS_LOGIN_VERSION,
				'wp_version' => get_bloginfo( 'version' ),
				'php_version' => PHP_VERSION,
				'timestamp' => gmdate( 'c' ),
			],
		];
		
		$this->make_async_request( $payload );
		return true;
	}
	
	/**
	 * Make asynchronous (non-blocking) request to the API
	 */
	private function make_async_request( $payload ) {
		$args = [
			'method' => 'POST',
			'headers' => [
				'Content-Type' => 'application/json',
				'Accept' => 'application/json',
			],
			'body' => wp_json_encode( $payload ),
			'timeout' => 5,
			'blocking' => false,
			'redirection' => 0,
		];
		
		wp_remote_request( self::FEEDBACK_API_URL, $args );
	}
}
