<?php
/**
 * Authyo Login Handler
 *
 * Handles login page integration and OTP flow
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Authyo_Login')) {
class Authyo_Login {
    
    /**
     * Initialize login integration
     */
    public static function init() {
        // Handle passwordless login token first (before any output)
        // This must be called directly, not hooked, since we're already on login_init
        self::handle_passwordless_login_token();
        
        // Add custom login form after the WordPress login form
        add_action('login_footer', array(__CLASS__, 'add_authyo_login_form'));
    }
    
    public static function generate_login_token($user_id) {
        // Generate secure random token using WordPress function
        $token = wp_generate_password(32, false);
        $transient_key = 'authyo_login_token_' . $token;
        
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        
        /**
         * SECURITY: Bind the token to the requester's User-Agent hash to prevent 
         * cross-browser token usage (session fixation protection).
         */
        $token_data = array(
            'user_id' => absint($user_id),
            'ua_hash' => wp_hash($user_agent), // Bind to browser signature
        );
        
        // Store token data, expire in 5 minutes max
        set_transient($transient_key, $token_data, 300); // 300 seconds = 5 min
        
        return $token;
    }
    
    /**
     * Handle passwordless login token validation and authentication
     * Intercepts wp-login.php to validate token and log user in
     */
    public static function handle_passwordless_login_token() {
        // Check if passwordless login is enabled
        if (!Authyo_Settings::is_enabled()) {
            return;
        }
        
        // Check for token in URL
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $token = isset($_GET['authyo_token']) ? sanitize_text_field(wp_unslash($_GET['authyo_token'])) : '';
        
        if (empty($token)) {
            return; // No token, proceed with normal login
        }
        
        // Validate token format (32 characters, alphanumeric)
        // wp_generate_password generates alphanumeric tokens
        if (empty($token) || strlen($token) !== 32 || !preg_match('/^[a-zA-Z0-9]+$/', $token)) {
            // Invalid token format, redirect to login with error
            wp_safe_redirect(add_query_arg(
                array('authyo_error' => 'invalid_token'),
                wp_login_url()
            ));
            exit;
        }
        
        // Get token data from transient
        $token_transient_key = 'authyo_login_token_' . $token;
        $token_data = get_transient($token_transient_key);
        
        if (!$token_data || !is_array($token_data)) {
            // Token expired or invalid, redirect to login with error
            wp_safe_redirect(add_query_arg(
                array('authyo_error' => 'expired_token'),
                wp_login_url()
            ));
            exit;
        }
        
        $user_id = isset($token_data['user_id']) ? $token_data['user_id'] : 0;
        $stored_ua_hash = isset($token_data['ua_hash']) ? $token_data['ua_hash'] : '';
        
        $user_agent = isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '';
        $current_ua_hash = wp_hash($user_agent);
        
        /**
         * SECURITY: Prevent Replay / Cross-Browser Token Usage.
         * Ensure that the browser validating the token is the same one that requested it.
         */
        if ($stored_ua_hash !== $current_ua_hash) {
            // Security mismatch: probable session hijacking or cross-browser attempt
            delete_transient($token_transient_key);
            wp_safe_redirect(add_query_arg(
                array('authyo_error' => 'ua_mismatch'),
                wp_login_url()
            ));
            exit;
        }
        
        // Validate user exists and is valid
        $user = get_user_by('ID', absint($user_id));
        if (!$user) {
            // User not found, delete token and redirect
            delete_transient($token_transient_key);
            wp_safe_redirect(add_query_arg(
                array('authyo_error' => 'invalid_user'),
                wp_login_url()
            ));
            exit;
        }
        
        // Token is valid - delete it immediately to prevent replay attacks (single-use enforcement)
        delete_transient($token_transient_key);
        
        // Log the user in
        wp_set_current_user($user_id, $user->user_login);
        wp_set_auth_cookie($user_id, true); // Remember user
        
        // Fire login actions for compatibility
        do_action('authyo_passwordless_login_authenticated', $user->user_login, $user);
        
        // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound
        do_action('wp_login', $user->user_login, $user);
        
        // Refactored redirect filters for clarity and purpose-specific control
        $redirect_url = apply_filters('authyo_passwordless_login_success_redirect', admin_url(), $user);
        
        // Ensure no previous output interferes with redirect
        if (!headers_sent()) {
            wp_safe_redirect($redirect_url);
            exit;
        } else {
            // Headers already sent, use JavaScript fallback
            // Headers already sent, use HTML fallback
            $link = sprintf(
                // translators: %s: Login redirect URL
                __('Login successful. If you are not redirected automatically, <a href="%s">click here</a>.', 'authyo-passwordless-login'),
                esc_url($redirect_url)
            );
            echo '<p>' . wp_kses_post($link) . '</p>';
            exit;
        }
    }
    

    
    /**
     * Add Authyo login form to WordPress login page
     */
    public static function add_authyo_login_form() {
        // Only show form if passwordless login is enabled
        if (!Authyo_Settings::is_enabled()) {
            return;
        }
        
        // Display error messages for token validation failures
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $authyo_error = isset($_GET['authyo_error']) ? sanitize_text_field(wp_unslash($_GET['authyo_error'])) : '';
        $error_message = '';
        if ($authyo_error === 'invalid_token') {
            $error_message = __('Invalid login token. Please request a new OTP.', 'authyo-passwordless-login');
        } elseif ($authyo_error === 'expired_token') {
            $error_message = __('Login token has expired. Please request a new OTP.', 'authyo-passwordless-login');
        } elseif ($authyo_error === 'invalid_user') {
            $error_message = __('User account not found. Please try again.', 'authyo-passwordless-login');
        } elseif ($authyo_error === 'ua_mismatch') {
            $error_message = __('Security check failed: Browser session mismatch. Please try again from the same browser.', 'authyo-passwordless-login');
        }
        ?>
        <div id="authyo-login-wrapper">
            <div id="authyo-login-container">
                <h3><?php esc_html_e('Passwordless Login', 'authyo-passwordless-login'); ?></h3>
                <p class="authyo-description"><?php esc_html_e('Enter your email address to receive a one-time password (OTP).', 'authyo-passwordless-login'); ?></p>
                
                <?php if (!empty($error_message)) : ?>
                    <div id="authyo-token-error" class="error">
                        <p><?php echo esc_html($error_message); ?></p>
                    </div>
                <?php endif; ?>
                
                <div id="authyo-email-step">
                    <p>
                        <label for="authyo-email"><?php esc_html_e('Email Address', 'authyo-passwordless-login'); ?></label>
                        <input type="email" id="authyo-email" name="authyo_email" class="input" value="" size="20" autocomplete="email" />
                    </p>
                    <p>
                        <button type="button" id="authyo-send-otp-btn" class="button button-primary button-large">
                            <?php esc_html_e('Send OTP', 'authyo-passwordless-login'); ?>
                        </button>
                    </p>
                </div>
                
                <div id="authyo-otp-step" style="display: none;">
                    <p id="authyo-email-display"></p>
                    <p>
                        <label for="authyo-otp"><?php esc_html_e('Enter OTP Code', 'authyo-passwordless-login'); ?></label>
                        <input type="text" id="authyo-otp" name="authyo_otp" class="input" value="" size="20" maxlength="6" autocomplete="off" />
                    </p>
                    <p id="authyo-timer-wrapper" style="margin: 10px 0;">
                        <span id="authyo-timer-display" style="display: none; font-weight: bold; color: #2271b1;"></span>
                        <button type="button" id="authyo-resend-otp-btn" class="button" style="display: none;">
                            <?php esc_html_e('Resend OTP', 'authyo-passwordless-login'); ?>
                        </button>
                    </p>
                    <p>
                        <button type="button" id="authyo-verify-otp-btn" class="button button-primary button-large">
                            <?php esc_html_e('Verify OTP', 'authyo-passwordless-login'); ?>
                        </button>
                        <button type="button" id="authyo-back-btn" class="button button-large">
                            <?php esc_html_e('Back', 'authyo-passwordless-login'); ?>
                        </button>
                    </p>
                </div>
                
                <div id="authyo-messages"></div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render login form (for custom login pages)
     */
    public static function render_login_form() {
        // This can be used for custom login page integration
        self::add_authyo_login_form();
    }
    
    /**
     * Shortcode to display Authyo login form
     * Usage: [authyo_login]
     *
     * @param array $atts Shortcode attributes
     * @return string HTML output
     */
    public static function shortcode_login_form($atts = array()) {
        // Only show on frontend, not on admin pages
        if (is_admin()) {
            return '';
        }
        
        // Check if passwordless login is enabled
        if (!Authyo_Settings::is_enabled()) {
            return '';
        }
        
        // Enqueue assets if not already enqueued
        if (!wp_script_is('authyo-passwordless-login', 'enqueued')) {
            wp_enqueue_script(
                'authyo-passwordless-login',
                AUTHYO_PASSWORDLESS_LOGIN_URL . 'assets/js/login.js',
                array('jquery'),
                AUTHYO_PASSWORDLESS_LOGIN_VERSION,
                true
            );
            
            wp_enqueue_style(
                'authyo-passwordless-login',
                AUTHYO_PASSWORDLESS_LOGIN_URL . 'assets/css/login.css',
                array(),
                AUTHYO_PASSWORDLESS_LOGIN_VERSION
            );
            
            wp_localize_script('authyo-passwordless-login', 'authyoPasswordlessLogin', array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('authyo_passwordless_login_nonce'),
                'strings' => array(
                    'emailRequired' => __('Please enter your email address.', 'authyo-passwordless-login'),
                    'otpRequired' => __('Please enter the OTP code.', 'authyo-passwordless-login'),
                    'sendingOtp' => __('Sending OTP...', 'authyo-passwordless-login'),
                    'verifyingOtp' => __('Verifying OTP...', 'authyo-passwordless-login'),
                    'invalidEmail' => __('Invalid email address.', 'authyo-passwordless-login'),
                    'userNotFound' => __('No account found with this email address.', 'authyo-passwordless-login'),
                    'otpSent' => __('OTP sent successfully. Please check your email.', 'authyo-passwordless-login'),
                    'otpVerified' => __('OTP verified successfully. Logging you in...', 'authyo-passwordless-login'),
                    'invalidOtp' => __('Invalid OTP code. Please try again.', 'authyo-passwordless-login'),
                    'error' => __('An error occurred. Please try again.', 'authyo-passwordless-login'),
                    'sendOtp' => __('Send OTP', 'authyo-passwordless-login'),
                    'verifyOtp' => __('Verify OTP', 'authyo-passwordless-login'),
                    'resendOtp' => __('Resend OTP', 'authyo-passwordless-login'),
                    'emailSentTo' => __('OTP sent to', 'authyo-passwordless-login'),
                )
            ));
        }
        
        ob_start();
        self::add_authyo_login_form();
        return ob_get_clean();
    }
    
    /**
     * AJAX handler for sending OTP
     */
    public static function ajax_send_otp() {
        // Check nonce
        $nonce_check = check_ajax_referer('authyo_passwordless_login_nonce', 'nonce', false);
        
        if (!$nonce_check) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login] ERROR: Nonce verification failed');
            }
            wp_send_json_error(array(
                'message' => __('Security check failed. Please refresh the page and try again.', 'authyo-passwordless-login')
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        // Check if passwordless login is enabled
        if (!Authyo_Settings::is_enabled()) {
            wp_send_json_error(array(
                'message' => __('Passwordless login is currently disabled.', 'authyo-passwordless-login')
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        
        if (empty($email) || !is_email($email)) {
            wp_send_json_error(array(
                'message' => __('Please enter a valid email address.', 'authyo-passwordless-login')
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        // Check if user exists
        $user = get_user_by('email', $email);
        if (!$user) {
            wp_send_json_error(array(
                'message' => __('No account found with this email address.', 'authyo-passwordless-login')
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        // Send OTP via Authyo API
        try {
            $response = Authyo_Passwordless_Login_API::send_otp($email);
        } catch (Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login] Exception: ' . $e->getMessage());
            }
            wp_send_json_error(array(
                'message' => __('An error occurred while sending OTP. Please try again.', 'authyo-passwordless-login')
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login] API Error: ' . $response->get_error_message());
            }
            wp_send_json_error(array(
                'message' => $response->get_error_message()
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
            error_log('[Authyo Passwordless Login] API Response: ' . print_r($response, true));
        }
        
        // Store mask_id in transient for verification
        // Handle different possible response field names
        $mask_id = '';
        
        // First check if response has a 'data' object (most common format)
        if (isset($response['data']) && is_array($response['data'])) {
            // Check if maskId is in data.results[0].maskId (Authyo API format)
            if (isset($response['data']['results']) && is_array($response['data']['results']) && !empty($response['data']['results'])) {
                $first_result = $response['data']['results'][0];
                if (isset($first_result['maskId'])) {
                    $mask_id = $first_result['maskId'];
                } elseif (isset($first_result['mask_id'])) {
                    $mask_id = $first_result['mask_id'];
                }
            }
            
            // Check all possible field names directly in data object
            if (empty($mask_id)) {
                if (isset($response['data']['maskId'])) {
                    $mask_id = $response['data']['maskId'];
                } elseif (isset($response['data']['mask_id'])) {
                    $mask_id = $response['data']['mask_id'];
                } elseif (isset($response['data']['mask'])) {
                    $mask_id = $response['data']['mask'];
                } elseif (isset($response['data']['id'])) {
                    $mask_id = $response['data']['id'];
                } else {
                    // Try to find any field that might contain the mask ID
                    foreach ($response['data'] as $key => $value) {
                        if (is_string($value) && !empty($value)) {
                            // Check if key name suggests it's an ID
                            if (stripos($key, 'mask') !== false || 
                                stripos($key, 'id') !== false || 
                                stripos($key, 'token') !== false ||
                                stripos($key, 'reference') !== false) {
                                $mask_id = $value;
                                break;
                            }
                            // Also check if value looks like an ID (alphanumeric string)
                            if (preg_match('/^[a-zA-Z0-9_-]+$/', $value) && strlen($value) > 10) {
                                $mask_id = $value;
                                break;
                            }
                        }
                    }
                }
            }
        }
        
        // If not found in data, check top level
        if (empty($mask_id)) {
            if (isset($response['maskId'])) {
                $mask_id = $response['maskId'];
            } elseif (isset($response['mask_id'])) {
                $mask_id = $response['mask_id'];
            } elseif (isset($response['mask'])) {
                $mask_id = $response['mask'];
            } elseif (isset($response['id'])) {
                $mask_id = $response['id'];
            }
        }
        
        // Check in result object
        if (empty($mask_id) && isset($response['result']) && is_array($response['result'])) {
            if (isset($response['result']['maskId'])) {
                $mask_id = $response['result']['maskId'];
            } elseif (isset($response['result']['mask_id'])) {
                $mask_id = $response['result']['mask_id'];
            }
        }
        
        if (empty($mask_id)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login] Error: Mask ID not found in API response');
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
                error_log('[Authyo Passwordless Login] Response structure: ' . print_r($response, true));
            }
            
            $error_message = __('Invalid response from Authyo API. Mask ID not found.', 'authyo-passwordless-login');
            
            if (current_user_can('manage_options')) {
                $error_message .= ' ' . __('Please check your API credentials and try again.', 'authyo-passwordless-login');
            }
            
            wp_send_json_error(array(
                'message' => $error_message,
                'response_keys' => is_array($response) ? array_keys($response) : array(),
                'response_preview' => (current_user_can('manage_options') && is_array($response)) ? $response : null
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        $transient_key = 'authyo_passwordless_login_otp_' . md5($email . $mask_id . time());
        
        set_transient($transient_key, array(
            'email' => $email,
            'user_id' => $user->ID,
            'mask_id' => $mask_id
        ), 600); // 10 minutes expiry
        
        $success_data = array(
            'message' => __('OTP sent successfully. Please check your email.', 'authyo-passwordless-login'),
            'mask_id' => $mask_id,
            'transient_key' => $transient_key
        );
        
        wp_send_json_success($success_data);
        wp_die();
    }
    
    /**
     * AJAX handler for verifying OTP
     */
    public static function ajax_verify_otp() {
        check_ajax_referer('authyo_passwordless_login_nonce', 'nonce');
        
        // Check if passwordless login is enabled
        if (!Authyo_Settings::is_enabled()) {
            wp_send_json_error(array(
                'message' => __('Passwordless login is currently disabled.', 'authyo-passwordless-login')
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        $otp = isset($_POST['otp']) ? sanitize_text_field(wp_unslash($_POST['otp'])) : '';
        $mask_id = isset($_POST['mask_id']) ? sanitize_text_field(wp_unslash($_POST['mask_id'])) : '';
        $transient_key = isset($_POST['transient_key']) ? sanitize_text_field(wp_unslash($_POST['transient_key'])) : '';
        
        if (empty($otp) || empty($mask_id) || empty($transient_key)) {
            wp_send_json_error(array(
                'message' => __('Missing required parameters.', 'authyo-passwordless-login')
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        // Get stored data from transient
        $stored_data = get_transient($transient_key);
        
        if (!$stored_data) {
            wp_send_json_error(array(
                'message' => __('OTP session expired. Please request a new OTP.', 'authyo-passwordless-login')
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        // Verify OTP via Authyo API
        $response = Authyo_Passwordless_Login_API::verify_otp($mask_id, $otp);
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login] API Error: ' . $response->get_error_message());
            }
            wp_send_json_error(array(
                'message' => $response->get_error_message()
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        // Check if OTP is verified
        // Handle different possible response formats
        $verified = false;
        if (isset($response['verified']) && $response['verified'] === true) {
            $verified = true;
        } elseif (isset($response['success']) && $response['success'] === true) {
            $verified = true;
        } elseif (isset($response['status']) && $response['status'] === 'success') {
            $verified = true;
        } elseif (isset($response['valid']) && $response['valid'] === true) {
            $verified = true;
        }
        
        if (!$verified) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
                error_log('[Authyo Passwordless Login] OTP verification failed. Response: ' . print_r($response, true));
            }
            $error_message = isset($response['message']) ? $response['message'] : __('Invalid OTP code. Please try again.', 'authyo-passwordless-login');
            wp_send_json_error(array(
                'message' => $error_message
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        // Success - Generate secure one-time token for passwordless login
        // Delete OTP transient
        delete_transient($transient_key);
        
        // Get user from stored data
        $user_id = isset($stored_data['user_id']) ? absint($stored_data['user_id']) : 0;
        $user = $user_id ? get_user_by('ID', $user_id) : null;
        
        if (!$user || !$user_id) {
            wp_send_json_error(array(
                'message' => __('User not found. Please try again.', 'authyo-passwordless-login')
            ));
            wp_die(); // SECURITY: Ensure script termination after AJAX error
        }
        
        // Generate secure one-time token for passwordless login
        $token = self::generate_login_token($user_id);
        
        // Build login URL with token
        $login_url = add_query_arg(
            array('authyo_token' => $token),
            wp_login_url()
        );
        
        /**
         * Separate redirect filters by purpose as required by Plugin Review Team.
         * 1. authyo_passwordless_login_token_redirect: Filter the URL containing the login token.
         */
        $login_url = apply_filters('authyo_passwordless_login_token_redirect', $login_url, $user);
        
        wp_send_json_success(array(
            'message' => __('OTP verified successfully. Logging you in...', 'authyo-passwordless-login'),
            'redirect' => esc_url_raw($login_url)
        ));
        wp_die(); // SECURITY: Ensure script termination after AJAX success
    }
    
}
} // End if !class_exists

