<?php
/**
 * Authyo API Handler
 *
 * Handles all API communication with Authyo service
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Authyo_Passwordless_Login_API')) {
class Authyo_Passwordless_Login_API {
    
    /**
     * API Base URL
     */
    const API_BASE_URL = 'https://app.authyo.io/api/v1';
    
    /**
     * Send OTP to user
     *
     * @param string $email User email address
     * @param int $expiry OTP expiry time in seconds (optional)
     * @param int $otp_length OTP length (optional)
     * @param string $auth_way Authentication way (optional)
     * @return array|WP_Error Response data or error
     */
    public static function send_otp($email, $expiry = 300, $otp_length = 6, $auth_way = 'email') {
        $settings = Authyo_Settings::get_settings();
        
        if (empty($settings['client_id']) || empty($settings['client_secret'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login API] Error: Missing API credentials');
            }
            return new WP_Error('missing_credentials', __('Authyo credentials are not configured.', 'authyo-passwordless-login'));
        }
        
        // Use GET method - data goes in query string
        $url = add_query_arg(array(
            'to' => sanitize_email($email),
            'expiry' => absint($expiry),
            'otpLength' => absint($otp_length),
            'authWay' => sanitize_text_field($auth_way)
        ), self::API_BASE_URL . '/auth/sendotp');
        
        $headers = array(
            'clientId' => $settings['client_id'],
            'clientSecret' => $settings['client_secret']
        );
        
        $response = wp_remote_get($url, array(
            'headers' => $headers,
            'timeout' => 30,
            'sslverify' => true
        ));
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login API] HTTP Error: ' . $response->get_error_message());
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login API] Error Code: ' . $response->get_error_code());
            }
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $status_code = wp_remote_retrieve_response_code($response);
        
        $data = json_decode($body, true);
        
        // Handle non-JSON responses
        if (json_last_error() !== JSON_ERROR_NONE) {
            $error_msg = __('Invalid JSON response from Authyo API.', 'authyo-passwordless-login');
            if (defined('WP_DEBUG') && WP_DEBUG) {
                $error_msg .= ' Response: ' . substr($body, 0, 200);
            }
            return new WP_Error('api_error', $error_msg, array('status' => $status_code, 'body' => $body));
        }
        
        // Check for success indicators in response
        $is_success = false;
        if ($status_code === 200) {
            $is_success = true;
        } elseif (isset($data['success']) && $data['success'] === true) {
            $is_success = true;
        } elseif (isset($data['status']) && $data['status'] === 'success') {
            $is_success = true;
        }
        
        if (!$is_success) {
            $error_message = __('Failed to send OTP.', 'authyo-passwordless-login');
            if (isset($data['message'])) {
                $error_message = $data['message'];
            } elseif (isset($data['error'])) {
                $error_message = is_array($data['error']) ? implode(', ', $data['error']) : $data['error'];
            } elseif (isset($data['errors'])) {
                $error_message = is_array($data['errors']) ? implode(', ', $data['errors']) : $data['errors'];
            }
            return new WP_Error('api_error', $error_message, array('status' => $status_code, 'data' => $data));
        }
        
        return $data;
    }
    
    /**
     * Verify OTP
     *
     * @param string $mask_id Mask ID returned from send_otp
     * @param string $otp OTP code to verify
     * @return array|WP_Error Response data or error
     */
    public static function verify_otp($mask_id, $otp) {
        $settings = Authyo_Settings::get_settings();
        
        if (empty($settings['client_id']) || empty($settings['client_secret'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login API] Error: Missing API credentials');
            }
            return new WP_Error('missing_credentials', __('Authyo credentials are not configured.', 'authyo-passwordless-login'));
        }
        
        // Use GET method - data goes in query string
        $url = add_query_arg(array(
            'maskId' => sanitize_text_field($mask_id),
            'otp' => sanitize_text_field($otp)
        ), self::API_BASE_URL . '/auth/verifyotp');
        
        $headers = array(
            'clientId' => $settings['client_id'],
            'clientSecret' => $settings['client_secret']
        );
        
        $response = wp_remote_get($url, array(
            'headers' => $headers,
            'timeout' => 30,
            'sslverify' => true
        ));
        
        if (is_wp_error($response)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login API] HTTP Error: ' . $response->get_error_message());
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login API] Error Code: ' . $response->get_error_code());
            }
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $status_code = wp_remote_retrieve_response_code($response);
        
        $data = json_decode($body, true);
        
        // Handle non-JSON responses
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('api_error', __('Invalid response from Authyo API.', 'authyo-passwordless-login'), array('status' => $status_code, 'body' => $body));
        }
        
        if ($status_code !== 200) {
            $error_message = isset($data['message']) ? $data['message'] : (isset($data['error']) ? $data['error'] : __('Failed to verify OTP.', 'authyo-passwordless-login'));
            return new WP_Error('api_error', $error_message, array('status' => $status_code, 'data' => $data));
        }
        
        return $data;
    }
}
} // End if !class_exists
