<?php
/**
 * Plugin Name: Authyo Passwordless Login
 * Description: Passwordless login for WordPress using Authyo OTP service. Users can log in with their email address and receive an OTP for verification.
 * Version:     1.0.3
 * Author:      konceptwise
 * Author URI:  https://authyo.io/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: authyo-passwordless-login
 * Domain Path: /languages
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Check minimum PHP version
if (version_compare(PHP_VERSION, '7.2', '<')) {
    add_action('admin_notices', function() {
        if (current_user_can('manage_options')) {
            echo '<div class="notice notice-error"><p>';
            echo esc_html(sprintf(
                'Authyo Passwordless Login requires PHP 7.2 or higher. You are running PHP %s. Please upgrade PHP.',
                PHP_VERSION
            ));
            echo '</p></div>';
        }
    });
    return;
}

// Define plugin constants
if (!defined('AUTHYO_PASSWORDLESS_LOGIN_VERSION')) {
    define('AUTHYO_PASSWORDLESS_LOGIN_VERSION', '1.0.3');
}
if (!defined('AUTHYO_PASSWORDLESS_LOGIN_DIR')) {
    define('AUTHYO_PASSWORDLESS_LOGIN_DIR', plugin_dir_path(__FILE__));
}
if (!defined('AUTHYO_PASSWORDLESS_LOGIN_URL')) {
    define('AUTHYO_PASSWORDLESS_LOGIN_URL', plugin_dir_url(__FILE__));
}
if (!defined('AUTHYO_PASSWORDLESS_LOGIN_BASENAME')) {
    define('AUTHYO_PASSWORDLESS_LOGIN_BASENAME', plugin_basename(__FILE__));
}

// Include required files with existence checks
$authyo_passwordless_login_includes_dir = AUTHYO_PASSWORDLESS_LOGIN_DIR . 'includes' . DIRECTORY_SEPARATOR;

$authyo_passwordless_login_api_file = $authyo_passwordless_login_includes_dir . 'class-authyo-api.php';
$authyo_passwordless_login_settings_file = $authyo_passwordless_login_includes_dir . 'class-authyo-settings.php';
$authyo_passwordless_login_login_file = $authyo_passwordless_login_includes_dir . 'class-authyo-login.php';
$authyo_passwordless_login_feedback_file = $authyo_passwordless_login_includes_dir . 'class-authyo-passwordless-feedback.php';

// Check file existence without using translation functions (avoid early translation loading)
if (!file_exists($authyo_passwordless_login_api_file)) {
    if (function_exists('wp_die')) {
        wp_die('Authyo Passwordless Login plugin error: Required file not found: ' . esc_html($authyo_passwordless_login_api_file));
    } else {
        die('Authyo Passwordless Login plugin error: Required file not found: ' . esc_html($authyo_passwordless_login_api_file));
    }
}
if (!file_exists($authyo_passwordless_login_settings_file)) {
    if (function_exists('wp_die')) {
        wp_die('Authyo Passwordless Login plugin error: Required file not found: ' . esc_html($authyo_passwordless_login_settings_file));
    } else {
        die('Authyo Passwordless Login plugin error: Required file not found: ' . esc_html($authyo_passwordless_login_settings_file));
    }
}
if (!file_exists($authyo_passwordless_login_login_file)) {
    if (function_exists('wp_die')) {
        wp_die('Authyo Passwordless Login plugin error: Required file not found: ' . esc_html($authyo_passwordless_login_login_file));
    } else {
        die('Authyo Passwordless Login plugin error: Required file not found: ' . esc_html($authyo_passwordless_login_login_file));
    }
}
if (!file_exists($authyo_passwordless_login_feedback_file)) {
    if (function_exists('wp_die')) {
        wp_die('Authyo Passwordless Login plugin error: Required file not found: ' . esc_html($authyo_passwordless_login_feedback_file));
    } else {
        die('Authyo Passwordless Login plugin error: Required file not found: ' . esc_html($authyo_passwordless_login_feedback_file));
    }
}

require_once $authyo_passwordless_login_api_file;
require_once $authyo_passwordless_login_settings_file;
require_once $authyo_passwordless_login_login_file;

require_once $authyo_passwordless_login_feedback_file;

// Include Google Authenticator Fallback Module (Modular)
$authyo_google_auth_file = AUTHYO_PASSWORDLESS_LOGIN_DIR . 'includes/modules/google-auth/class-authyo-google-auth.php';
if (file_exists($authyo_google_auth_file)) {
    require_once $authyo_google_auth_file;
}

// Initialize feedback handler globally to ensure all hooks are registered early
if (class_exists('Authyo_Passwordless_Feedback')) {
    new Authyo_Passwordless_Feedback();
}

/**
 * Main plugin class
 * Using unique class name to avoid conflicts with other Authyo plugins
 */
if (!class_exists('Authyo_Passwordless_Login')) {
class Authyo_Passwordless_Login {
    
    /**
     * Instance of this class
     */
    private static $instance = null;
    
    /**
     * Get instance of this class
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Initialize settings
        add_action('admin_menu', array('Authyo_Settings', 'add_settings_page'));
        add_action('admin_init', array('Authyo_Settings', 'register_settings'));
        add_action('admin_enqueue_scripts', array('Authyo_Settings', 'enqueue_admin_assets'));
        
        // Initialize login integration
        add_action('login_init', array('Authyo_Login', 'init'));
        
        // AJAX handlers
        add_action('wp_ajax_authyo_passwordless_login_send_otp', array('Authyo_Login', 'ajax_send_otp'));
        add_action('wp_ajax_nopriv_authyo_passwordless_login_send_otp', array('Authyo_Login', 'ajax_send_otp'));
        add_action('wp_ajax_authyo_passwordless_login_verify_otp', array('Authyo_Login', 'ajax_verify_otp'));
        add_action('wp_ajax_nopriv_authyo_passwordless_login_verify_otp', array('Authyo_Login', 'ajax_verify_otp'));
        
        // Shortcode for custom login pages
        add_shortcode('authyo_login', array('Authyo_Login', 'shortcode_login_form'));
        
        // Enqueue scripts and styles (only when enabled)
        add_action('login_enqueue_scripts', array($this, 'enqueue_login_assets'));

        // Add settings link to plugins page
        $plugin_file = plugin_basename(AUTHYO_PASSWORDLESS_LOGIN_DIR . 'authyo-passwordless-login.php');
        add_filter("plugin_action_links_{$plugin_file}", array($this, 'add_plugin_action_links'));
    }
    
    /**
     * Enqueue scripts and styles for login page
     */
    public function enqueue_login_assets() {
        // Only enqueue assets if passwordless login is enabled
        if (!Authyo_Settings::is_enabled()) {
            return;
        }
        
        // Check if jQuery is available
        if (!wp_script_is('jquery', 'registered')) {
            wp_enqueue_script('jquery');
        }
        
        wp_enqueue_script(
            'authyo-passwordless-login',
            AUTHYO_PASSWORDLESS_LOGIN_URL . 'assets/js/login.js',
            array('jquery'),
            AUTHYO_PASSWORDLESS_LOGIN_VERSION,
            true
        );
        
        wp_enqueue_style(
            'authyo-passwordless-login',
            AUTHYO_PASSWORDLESS_LOGIN_URL . 'assets/css/login.css',
            array(),
            AUTHYO_PASSWORDLESS_LOGIN_VERSION
        );
        
        // Localize script - ensure nonce is generated correctly
        $nonce = wp_create_nonce('authyo_passwordless_login_nonce');
        
        wp_localize_script('authyo-passwordless-login', 'authyoPasswordlessLogin', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => $nonce,
            'strings' => array(
                'emailRequired' => __('Please enter your email address.', 'authyo-passwordless-login'),
                'otpRequired' => __('Please enter the OTP code.', 'authyo-passwordless-login'),
                'sendingOtp' => __('Sending OTP...', 'authyo-passwordless-login'),
                'verifyingOtp' => __('Verifying OTP...', 'authyo-passwordless-login'),
                'invalidEmail' => __('Invalid email address.', 'authyo-passwordless-login'),
                'userNotFound' => __('No account found with this email address.', 'authyo-passwordless-login'),
                'otpSent' => __('OTP sent successfully. Please check your email.', 'authyo-passwordless-login'),
                    'otpVerified' => __('OTP verified successfully. Logging you in...', 'authyo-passwordless-login'),
                'invalidOtp' => __('Invalid OTP code. Please try again.', 'authyo-passwordless-login'),
                'error' => __('An error occurred. Please try again.', 'authyo-passwordless-login'),
                'sendOtp' => __('Send OTP', 'authyo-passwordless-login'),
                'verifyOtp' => __('Verify OTP', 'authyo-passwordless-login'),
                'resendOtp' => __('Resend OTP', 'authyo-passwordless-login'),
                'emailSentTo' => __('OTP sent to', 'authyo-passwordless-login'),
            )
        ));
    }

    /**
     * Add settings link to the plugins page action links
     *
     * @param array $links Existing action links
     * @return array Modified action links
     */
    public function add_plugin_action_links($links) {
        $settings_link = '<a href="' . admin_url('options-general.php?page=authyo-passwordless-login') . '">' . __('Settings', 'authyo-passwordless-login') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Plugin activation hook
     */
    public static function activate() {
        set_transient('authyo_passwordless_show_email_subscription', true, HOUR_IN_SECONDS);
    }
}
} // End if !class_exists

// Register activation hook
register_activation_hook(__FILE__, array('Authyo_Passwordless_Login', 'activate'));

// Initialize the plugin
if (!function_exists('authyo_passwordless_login_init')) {
    function authyo_passwordless_login_init() {
        try {
            if (class_exists('Authyo_Passwordless_Login')) {
                return Authyo_Passwordless_Login::get_instance();
            } else {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                    error_log('[Authyo Passwordless Login] Error: Authyo_Passwordless_Login class not found');
                }
                return false;
            }
        } catch (Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login] Fatal Error: ' . $e->getMessage());
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login] Stack Trace: ' . $e->getTraceAsString());
            }
            add_action('admin_notices', function() use ($e) {
                if (current_user_can('manage_options')) {
                    echo '<div class="notice notice-error"><p>';
                    echo esc_html__('Authyo Passwordless Login plugin encountered an error during initialization.', 'authyo-passwordless-login');
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        echo ' <strong>Error:</strong> ' . esc_html($e->getMessage());
                    }
                    echo '</p></div>';
                }
            });
            return false;
        }
    }
}

// Start the plugin with error handling - delay initialization until init hook
// Using 'init' hook ensures WordPress is fully loaded and translations are ready
add_action('init', function() {
    try {
        if (class_exists('Authyo_Passwordless_Login') && method_exists('Authyo_Passwordless_Login', 'get_instance')) {
            authyo_passwordless_login_init();
        } else {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('[Authyo Passwordless Login] Error: Authyo_Passwordless_Login class or get_instance method not found');
            }
        }
    } catch (Exception $e) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('[Authyo Passwordless Login] Plugin initialization failed: ' . $e->getMessage());
        }
        add_action('admin_notices', function() use ($e) {
            if (current_user_can('manage_options')) {
                echo '<div class="notice notice-error"><p>';
                echo esc_html__('Authyo Passwordless Login plugin failed to initialize.', 'authyo-passwordless-login');
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    echo ' <strong>Error:</strong> ' . esc_html($e->getMessage());
                }
                echo '</p></div>';
            }
        });
    }
}, 1); // Priority 1 to load early but after WordPress core

/**
 * Helper function to display Authyo login form in themes
 * Usage: <?php authyo_passwordless_login_form(); ?>
 */
function authyo_passwordless_login_form() {
    echo do_shortcode('[authyo_login]');
}

