/**
 * Authyo Passwordless Login Script
 */

(function ($) {
    'use strict';

    $(document).ready(function () {
        // Check if authyoPasswordlessLogin object exists
        if (typeof authyoPasswordlessLogin === 'undefined') {
            return;
        }

        // Validate required objects
        if (!authyoPasswordlessLogin.ajaxUrl || !authyoPasswordlessLogin.nonce) {
            return;
        }

        var $emailStep = $('#authyo-email-step');
        var $otpStep = $('#authyo-otp-step');
        var $emailInput = $('#authyo-email');
        var $otpInput = $('#authyo-otp');
        var $sendOtpBtn = $('#authyo-send-otp-btn');
        var $verifyOtpBtn = $('#authyo-verify-otp-btn');
        var $backBtn = $('#authyo-back-btn');
        var $resendOtpBtn = $('#authyo-resend-otp-btn');
        var $messages = $('#authyo-messages');
        var $emailDisplay = $('#authyo-email-display');
        var $timerDisplay = $('#authyo-timer-display');

        var currentEmail = '';
        var currentMaskId = '';
        var currentTransientKey = '';
        var timerInterval = null;
        var timerSeconds = 60;

        /**
         * Show message
         */
        function showMessage(message, type) {
            type = type || 'info';
            var className = 'notice notice-' + type;
            if (type === 'error') {
                className = 'error';
            } else if (type === 'success') {
                className = 'success';
            }

            $messages.html('<div class="' + className + '"><p>' + message + '</p></div>');
        }

        /**
         * Clear messages
         */
        function clearMessages() {
            $messages.html('');
        }

        /**
         * Format time display
         */
        function formatTime(seconds) {
            var mins = Math.floor(seconds / 60);
            var secs = seconds % 60;
            return (mins < 10 ? '0' : '') + mins + ':' + (secs < 10 ? '0' : '') + secs;
        }

        /**
         * Start timer
         */
        function startTimer() {
            timerSeconds = 5;
            $resendOtpBtn.hide();
            $timerDisplay.show();
            updateTimerDisplay();

            if (timerInterval) {
                clearInterval(timerInterval);
            }

            timerInterval = setInterval(function () {
                timerSeconds--;
                updateTimerDisplay();

                if (timerSeconds <= 0) {
                    clearInterval(timerInterval);
                    timerInterval = null;
                    $timerDisplay.hide();
                    $resendOtpBtn.show();
                }
            }, 1000);
        }

        /**
         * Update timer display
         */
        function updateTimerDisplay() {
            if ($timerDisplay.length) {
                $timerDisplay.text(formatTime(timerSeconds));
            }
        }

        /**
         * Stop timer
         */
        function stopTimer() {
            if (timerInterval) {
                clearInterval(timerInterval);
                timerInterval = null;
            }
            $timerDisplay.hide();
            $resendOtpBtn.hide();
        }

        /**
         * Show email step
         */
        function showEmailStep() {
            $emailStep.show();
            $otpStep.hide();
            $emailInput.val('');
            $otpInput.val('');
            clearMessages();
            stopTimer();

            // Reset stored values
            currentEmail = '';
            currentMaskId = '';
            currentTransientKey = '';
        }

        /**
         * Show OTP step
         */
        function showOtpStep(email, maskId, transientKey) {
            currentEmail = email;
            currentMaskId = maskId;
            currentTransientKey = transientKey;

            $emailStep.hide();
            $otpStep.show();
            var emailSentText = authyoPasswordlessLogin.strings.emailSentTo || 'OTP sent to';
            $emailDisplay.text(emailSentText + ': ' + email);
            $otpInput.focus();
            clearMessages();

            // Start timer
            startTimer();
        }

        /**
         * Send OTP
         */
        function sendOtp() {
            var email = $emailInput.val().trim();

            if (!email) {
                showMessage(authyoPasswordlessLogin.strings.emailRequired, 'error');
                $emailInput.focus();
                return;
            }

            if (!isValidEmail(email)) {
                showMessage(authyoPasswordlessLogin.strings.invalidEmail, 'error');
                $emailInput.focus();
                return;
            }

            $sendOtpBtn.prop('disabled', true).text(authyoPasswordlessLogin.strings.sendingOtp);
            clearMessages();

            // Prepare request data
            var requestData = {
                action: 'authyo_passwordless_login_send_otp',
                nonce: authyoPasswordlessLogin.nonce,
                email: email
            };

            $.ajax({
                url: authyoPasswordlessLogin.ajaxUrl,
                type: 'POST',
                data: requestData,
                timeout: 30000,
                dataType: 'json',
                contentType: 'application/x-www-form-urlencoded; charset=UTF-8',
                success: function (response) {
                    // Handle case where response might be a string
                    if (typeof response === 'string') {
                        try {
                            response = JSON.parse(response);
                        } catch (e) {
                            showMessage(authyoPasswordlessLogin.strings.error + ' (Invalid response format)', 'error');
                            $sendOtpBtn.prop('disabled', false).text(authyoPasswordlessLogin.strings.sendOtp || 'Send OTP');
                            return;
                        }
                    }

                    if (response && response.success) {
                        if (response.data) {
                            showMessage(response.data.message, 'success');
                            showOtpStep(email, response.data.mask_id, response.data.transient_key);
                        } else {
                            showMessage(authyoPasswordlessLogin.strings.error, 'error');
                            $sendOtpBtn.prop('disabled', false).text(authyoPasswordlessLogin.strings.sendOtp || 'Send OTP');
                        }
                    } else {
                        if (response && response.data) {
                            showMessage(response.data.message || authyoPasswordlessLogin.strings.error, 'error');
                        } else {
                            showMessage(authyoPasswordlessLogin.strings.error + ' (Unexpected response)', 'error');
                        }

                        $sendOtpBtn.prop('disabled', false).text(authyoPasswordlessLogin.strings.sendOtp || 'Send OTP');
                    }
                },
                error: function (xhr, status, error) {
                    // Try to parse error response
                    if (xhr.responseText) {
                        try {
                            var errorResponse = JSON.parse(xhr.responseText);
                            if (errorResponse.data && errorResponse.data.message) {
                                showMessage(errorResponse.data.message, 'error');
                            } else {
                                showMessage(authyoPasswordlessLogin.strings.error + ' (HTTP ' + xhr.status + ')', 'error');
                            }
                        } catch (e) {
                            showMessage(authyoPasswordlessLogin.strings.error + ' (HTTP ' + xhr.status + ')', 'error');
                        }
                    } else {
                        if (status === 'timeout') {
                            showMessage('Request timed out. Please try again.', 'error');
                        } else {
                            showMessage(authyoPasswordlessLogin.strings.error + ' (Status: ' + status + ')', 'error');
                        }
                    }

                    $sendOtpBtn.prop('disabled', false).text(authyoPasswordlessLogin.strings.sendOtp || 'Send OTP');
                },
                complete: function (xhr, status) {
                    // Ensure button is re-enabled if something went wrong
                    if (status === 'error' || status === 'timeout') {
                        $sendOtpBtn.prop('disabled', false).text(authyoPasswordlessLogin.strings.sendOtp || 'Send OTP');
                    }
                }
            });
        }

        /**
         * Verify OTP
         */
        function verifyOtp() {
            var otp = $otpInput.val().trim();

            if (!otp) {
                showMessage(authyoPasswordlessLogin.strings.otpRequired, 'error');
                $otpInput.focus();
                return;
            }

            $verifyOtpBtn.prop('disabled', true).text(authyoPasswordlessLogin.strings.verifyingOtp);
            clearMessages();

            // Prepare request data
            var requestData = {
                action: 'authyo_passwordless_login_verify_otp',
                nonce: authyoPasswordlessLogin.nonce,
                otp: otp,
                mask_id: currentMaskId,
                transient_key: currentTransientKey
            };

            $.ajax({
                url: authyoPasswordlessLogin.ajaxUrl,
                type: 'POST',
                data: requestData,
                success: function (response) {
                    if (response.success) {
                        showMessage(response.data.message, 'success');
                        stopTimer();

                        // Redirect to login page with token for automatic authentication
                        // Token is included in redirect URL from backend
                        setTimeout(function () {
                            if (response.data.redirect) {
                                window.location.href = response.data.redirect;
                            } else {
                                // Fallback to login URL if redirect not provided
                                window.location.href = window.location.origin + '/wp-login.php';
                            }
                        }, 1000);
                    } else {
                        showMessage(response.data.message || authyoPasswordlessLogin.strings.error, 'error');
                        $verifyOtpBtn.prop('disabled', false).text(authyoPasswordlessLogin.strings.verifyOtp || 'Verify OTP');
                        $otpInput.val('').focus();
                    }
                },
                error: function (xhr, status, error) {
                    showMessage(authyoPasswordlessLogin.strings.error, 'error');
                    $verifyOtpBtn.prop('disabled', false).text(authyoPasswordlessLogin.strings.verifyOtp || 'Verify OTP');
                }
            });
        }

        /**
         * Resend OTP
         */
        function resendOtp() {
            if (!currentEmail) {
                showEmailStep();
                return;
            }

            // Use the stored email to resend
            $emailInput.val(currentEmail);
            sendOtp();
        }

        /**
         * Validate email
         */
        function isValidEmail(email) {
            var emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
            return emailRegex.test(email);
        }

        // Event handlers
        $sendOtpBtn.on('click', function (e) {
            e.preventDefault();
            sendOtp();
        });

        $verifyOtpBtn.on('click', function (e) {
            e.preventDefault();
            verifyOtp();
        });

        $backBtn.on('click', function (e) {
            e.preventDefault();
            showEmailStep();
        });

        $resendOtpBtn.on('click', function (e) {
            e.preventDefault();
            resendOtp();
        });

        // Allow Enter key to submit
        $emailInput.on('keypress', function (e) {
            if (e.which === 13) {
                e.preventDefault();
                sendOtp();
            }
        });

        $otpInput.on('keypress', function (e) {
            if (e.which === 13) {
                e.preventDefault();
                verifyOtp();
            }
        });

        // Auto-focus email input
        $emailInput.focus();

        // Cleanup on page unload
        $(window).on('beforeunload', function () {
            stopTimer();
        });
    });

})(jQuery);
