<?php
/**
 * Template Mapping Helper
 *
 * This file contains helper functions for template mapping.
 *
 * @package Authyo_OTP_Auth
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get template mapping for an event
 *
 * @param string $event WooCommerce event
 * @return string Template name
 */
function authyo_get_template_for_event($event) {
    $template_map = array(
        'processing' => 'wp_processing_order',
        'completed' => 'wp_completed_order',
        'cancelled' => 'wp_order_cancelled',
        'payment_failed' => 'wp_order_failed',
        'refunded' => 'wp_refunded_order',
        'on_hold' => 'wp_order_on_hold',
        'pending' => 'wp_pending_order',
    );
    
    $template = isset($template_map[$event]) ? $template_map[$event] : '';
    
    // Debug log: Template mapping
    authyo_debug_log(
        sprintf('Template mapping: event "%s" -> template "%s"', $event, $template ?: 'none'),
        array('event' => $event, 'template' => $template),
        'template'
    );
    
    return $template;
}

/**
 * Get required variables for a template
 *
 * @param string $template Template name
 * @return array Required variables
 */
function authyo_get_required_variables($template) {
    // All templates require the same 5 variables
    $default_required = array(
        'billingName',
        'orderNumber',
        'orderTotal',
        'status',
        'orderDate',
    );
    
    $required_vars = array(
        'wp_processing_order' => $default_required,
        'wp_completed_order' => $default_required,
        'wp_order_cancelled' => $default_required, // Fixed case to match template mapping
        'wp_order_failed' => $default_required,
        'wp_refunded_order' => $default_required,
        'wp_order_on_hold' => $default_required,
        'wp_pending_order' => $default_required,
    );
    
    return isset($required_vars[$template]) ? $required_vars[$template] : array();
}

/**
 * Validate required variables
 *
 * @param array $variables Available variables
 * @param array $required Required variables
 * @return array Validation result with 'valid' and 'missing' keys
 */
function authyo_validate_required_variables($variables, $required) {
    $missing = array();
    
    if (empty($required)) {
        return array(
            'valid' => true,
            'missing' => array(),
        );
    }
    
    foreach ($required as $var) {
        // Check if variable exists and is not empty
        // Use strict check: must be set and not null/empty string
        if (!isset($variables[$var])) {
            $missing[] = $var;
        } elseif ($var === 'orderTotal' && is_array($variables[$var])) {
            // orderTotal is now a structured array with 'amount' and 'currency'
            // Validate that it has the required structure
            if (!isset($variables[$var]['amount']) || empty($variables[$var]['amount'])) {
                $missing[] = $var;
            }
        } elseif ($variables[$var] === null || $variables[$var] === '') {
            // Variable exists but is null or empty string
            $missing[] = $var;
        }
    }
    
    // Debug logging if WP_DEBUG is enabled
    if (defined('WP_DEBUG') && WP_DEBUG && !empty($missing)) {
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log('Authyo Notification: Missing required variables: ' . implode(', ', $missing));
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
        error_log('Authyo Notification: Available variables: ' . print_r(array_keys($variables), true));
    }
    
    return array(
        'valid' => empty($missing),
        'missing' => $missing,
    );
}

/**
 * Validate required variables for SMS templates
 *
 * Special validation for SMS that allows APPNAME to be empty (as per SMS rules).
 *
 * @param array $variables Available variables
 * @param array $required Required variables/tags
 * @return array Validation result with 'valid' and 'missing' keys
 */
function authyo_validate_sms_required_variables($variables, $required) {
    $missing = array();
    
    if (empty($required)) {
        return array(
            'valid' => true,
            'missing' => array(),
        );
    }
    
    foreach ($required as $var) {
        // Check if variable exists
        if (!isset($variables[$var])) {
            $missing[] = $var;
        } elseif ($var === 'orderTotal' && is_array($variables[$var])) {
            // orderTotal is now a structured array with 'amount' and 'currency'
            // Validate that it has the required structure
            if (!isset($variables[$var]['amount']) || empty($variables[$var]['amount'])) {
                $missing[] = $var;
            }
        } elseif ($variables[$var] === null) {
            // Variable exists but is null (not allowed except for APPNAME)
            if ($var !== 'APPNAME') {
                $missing[] = $var;
            }
        } elseif ($variables[$var] === '' && $var !== 'APPNAME') {
            // Variable exists but is empty string (not allowed except for APPNAME)
            // APPNAME is intentionally empty for SMS, so it's valid
            $missing[] = $var;
        }
        // If APPNAME is empty string, it's valid (intentional for SMS)
    }
    
    // Debug logging if WP_DEBUG is enabled
    if (defined('WP_DEBUG') && WP_DEBUG && !empty($missing)) {
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
        error_log('Authyo Notification: Missing required SMS variables: ' . implode(', ', $missing));
        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
        error_log('Authyo Notification: Available SMS variables: ' . print_r(array_keys($variables), true));
    }
    
    return array(
        'valid' => empty($missing),
        'missing' => $missing,
    );
}

/**
 * Get all available template variables for an event
 *
 * @param string $event WooCommerce event
 * @return array Available variables (required and optional)
 */
function authyo_get_template_variables($event) {
    $template = authyo_get_template_for_event($event);
    $required = authyo_get_required_variables($template);
    
    // All available variables (required + optional)
    $all_variables = array(
        'customerName',
        'orderId',
        'orderDate',
        'amountPaid',
        'paymentMethod',
        'refundedAmount',
        'trackingLink',
        'trackingNo',
        'estimatedDelivery',
        'orderDetailsLink',
        'cancelReason',
        'failureReason',
        'retryPaymentLink',
    );
    
    return array(
        'required' => $required,
        'optional' => array_diff($all_variables, $required),
        'all' => $all_variables,
    );
}

/**
 * Render template with variables
 *
 * @param string $template Template string
 * @param array $variables Template variables
 * @return string Rendered template
 */
function authyo_render_template($template, $variables = array()) {
    // Placeholder for Phase 2 - actual rendering will be handled by API
    return $template;
}

/**
 * Get SMS template mapping for an event
 *
 * @param string $event WooCommerce event
 * @return string SMS template name
 */
function authyo_get_sms_template_for_event($event) {
    $sms_template_map = array(
        'pending' => 'order_updated',
        'processing' => 'order_placed',
        'on_hold' => 'order_updated',
        'completed' => 'order_delivered',
        'cancelled' => 'order_cancelled',
        'refunded' => 'order_refunded',
        'failed' => 'order_updated',
        'payment_failed' => 'order_updated', // Alias for failed
    );
    
    $template = isset($sms_template_map[$event]) ? $sms_template_map[$event] : '';
    
    // Debug log: SMS template mapping
    authyo_debug_log(
        sprintf('SMS template mapping: event "%s" -> template "%s"', $event, $template ?: 'none'),
        array('event' => $event, 'template' => $template),
        'template'
    );
    
    return $template;
}

/**
 * Get required dynamic tags for an SMS template
 *
 * @param string $template SMS template name
 * @return array Required dynamic tags
 */
function authyo_get_sms_template_tags($template) {
    $sms_template_tags = array(
        'order_updated' => array(
            'billingName',
            'orderNumber',
            'APPNAME',
            'status',
            'orderTotal',
        ),
        'order_placed' => array(
            'billingName',
            'orderNumber',
            'APPNAME',
            'orderDate',
            'estimatedDelivery',
        ),
        'order_delivered' => array(
            'billingName',
            'orderNumber',
            'APPNAME',
        ),
        'order_cancelled' => array(
            'billingName',
            'orderNumber',
            'APPNAME',
            'status',
        ),
        'order_refunded' => array(
            'billingName',
            'orderTotal',
            'orderNumber',
            'APPNAME',
        ),
    );
    
    return isset($sms_template_tags[$template]) ? $sms_template_tags[$template] : array();
}

/**
 * Prepare variables for SMS template
 *
 * Applies SMS-specific variable rules, including setting APPNAME to empty string.
 * Maps internal variable names to API-expected variable names.
 *
 * @param array $variables Original variables from order
 * @param string $template SMS template name
 * @return array Prepared variables for SMS (with API-expected names)
 */
function authyo_prepare_sms_variables($variables, $template) {
    // Get required tags for this SMS template
    $required_tags = authyo_get_sms_template_tags($template);
    
    // Prepare SMS variables according to rules
    // Map our internal variable names to API-expected variable names
    $sms_variables = array();
    
    // Map variables according to SMS rules
    foreach ($required_tags as $tag) {
        switch ($tag) {
            case 'billingName':
                // Map billingName to customerName (API expects customerName)
                $sms_variables['customerName'] = isset($variables['billingName']) ? $variables['billingName'] : '';
                break;
            
            case 'orderNumber':
                // Map orderNumber to orderId (API expects orderId)
                $sms_variables['orderId'] = isset($variables['orderNumber']) ? $variables['orderNumber'] : '';
                break;
            
            case 'orderTotal':
                // Map orderTotal to orderTotal (same name, but ensure it's included)
                // For SMS, use ISO currency codes instead of symbols to avoid encoding issues
                // This prevents HTML entities like &#8377; from appearing as ? in SMS messages
                if (isset($variables['orderTotal'])) {
                    if (is_array($variables['orderTotal']) && isset($variables['orderTotal']['amount']) && isset($variables['orderTotal']['currency'])) {
                        // New structured format: format with ISO currency code for SMS
                        $currency_code = $variables['orderTotal']['currency']; // e.g., 'INR', 'USD', 'EUR'
                        $amount_str = $variables['orderTotal']['amount'];
                        
                        // Extract numeric value: remove all non-numeric characters except digits, comma, and period
                        // This handles cases like "₹2,000.00", "$100.00", "€50.00", etc.
                        $numeric_amount = preg_replace('/[^\d.,]/', '', $amount_str);
                        
                        // Format: "INR 2,000.00" (currency code before amount with space)
                        // This format works reliably with all SMS providers
                        $sms_variables['orderTotal'] = $currency_code . ' ' . $numeric_amount;
                    } else {
                        // Legacy string format: decode HTML entities as fallback
                        // Note: This may still have encoding issues, but it's better than nothing
                        $sms_variables['orderTotal'] = html_entity_decode($variables['orderTotal'], ENT_QUOTES | ENT_HTML5, 'UTF-8');
                    }
                } else {
                    $sms_variables['orderTotal'] = '';
                }
                break;
            
            case 'status':
                // Map status to orderStatus (API expects orderStatus)
                $sms_variables['orderStatus'] = isset($variables['status']) ? $variables['status'] : '';
                break;
            
            case 'orderDate':
                // Map orderDate to orderDate (same name)
                $sms_variables['orderDate'] = isset($variables['orderDate']) ? $variables['orderDate'] : '';
                break;
            
            case 'estimatedDelivery':
                // Map estimatedDelivery to estimatedDelivery (same name)
                $sms_variables['estimatedDelivery'] = isset($variables['estimatedDelivery']) ? $variables['estimatedDelivery'] : '';
                break;
            
            case 'APPNAME':
                // Special SMS rule: always send blank for APPNAME
                // API might expect APPNAME or appName - send both to be safe
                $sms_variables['APPNAME'] = '';
                $sms_variables['appName'] = '';
                break;
        }
    }
    
    // For order_updated template, add special variables
    // The API template expects 'status' and 'description' instead of 'orderStatus'
    if ($template === 'order_updated') {
        // Map orderStatus to status (API expects 'status' for order_updated template)
        if (isset($sms_variables['orderStatus'])) {
            $sms_variables['status'] = $sms_variables['orderStatus'];
            // Keep orderStatus for backward compatibility, but API uses 'status'
        } elseif (isset($variables['status'])) {
            $sms_variables['status'] = $variables['status'];
        }
        
        // Add description variable with formatted message
        // Format: "Your Order Is \"orderStatus\""
        $status_value = isset($sms_variables['status']) ? $sms_variables['status'] : (isset($variables['status']) ? $variables['status'] : '');
        // translators: %s: Order status (e.g., "Pending", "Processing", "Completed")
        $sms_variables['description'] = sprintf(__('Your Order Is "%s"', 'authyo-otp-authentication-for-woocommerce'), $status_value);
        
        // Set updateMessage to "Quick update"
        $sms_variables['updateMessage'] = 'Quick update';
    }
    
    // For order_cancelled template, add cancelReason
    if ($template === 'order_cancelled') {
        // Add cancelReason with value "Order Issue"
        $sms_variables['cancelReason'] = 'Order Issue';
        
        // Also ensure status is sent (in addition to orderStatus)
        if (isset($sms_variables['orderStatus'])) {
            $sms_variables['status'] = $sms_variables['orderStatus'];
        } elseif (isset($variables['status'])) {
            $sms_variables['status'] = $variables['status'];
        }
    }
    
    // For order_refunded template, add refundAmount from orderTotal
    if ($template === 'order_refunded') {
        // Add refundAmount with value from orderTotal
        // Always use the same source as orderTotal to ensure consistency
        // For SMS, use ISO currency codes (same format as orderTotal)
        if (isset($sms_variables['orderTotal'])) {
            // Use the already processed orderTotal (which uses ISO currency code for SMS)
            $sms_variables['refundAmount'] = $sms_variables['orderTotal'];
        } elseif (isset($variables['orderTotal'])) {
            // Handle both structured format (array with amount/currency) and legacy string format
            if (is_array($variables['orderTotal']) && isset($variables['orderTotal']['amount']) && isset($variables['orderTotal']['currency'])) {
                // Format with ISO currency code for SMS (same as orderTotal)
                $currency_code = $variables['orderTotal']['currency'];
                $amount_str = $variables['orderTotal']['amount'];
                $numeric_amount = preg_replace('/[^\d.,]/', '', $amount_str);
                $sms_variables['refundAmount'] = $currency_code . ' ' . $numeric_amount;
            } else {
                // Legacy string format: decode HTML entities to ensure raw Unicode characters
                $sms_variables['refundAmount'] = html_entity_decode($variables['orderTotal'], ENT_QUOTES | ENT_HTML5, 'UTF-8');
            }
        } else {
            $sms_variables['refundAmount'] = '';
        }
    }
    
    return $sms_variables;
}