<?php
/**
 * Order Variables Extraction Helper
 *
 * Extracts variables from WooCommerce order object.
 *
 * @package Authyo_OTP_Auth
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Order Variables Helper Class
 */
class Authyo_Order_Variables {
    
    /**
     * Extract all variables from WooCommerce order
     *
     * @param WC_Order|int $order WooCommerce order object or order ID
     * @param string $event Order event/status
     * @return array Extracted variables
     */
    public static function extract($order, $event = '') {
        if (is_numeric($order)) {
            $order = wc_get_order($order);
        }
        
        if (!$order || !is_a($order, 'WC_Order')) {
            authyo_debug_log(
                'Order variables extraction failed: Invalid order object',
                array('order_id' => is_numeric($order) ? $order : 'invalid', 'event' => $event),
                'template'
            );
            return array();
        }
        
        $order_id = $order->get_id();
        
        // Debug log: Starting variable extraction
        authyo_debug_log(
            sprintf('Extracting variables for order #%d, event: %s', $order_id, $event),
            array('order_id' => $order_id, 'event' => $event),
            'template'
        );
        
        $variables = array();
        
        // Required variables (new naming)
        $variables['billingName'] = self::get_customer_name($order);
        $variables['orderNumber'] = self::get_order_id($order);
        // Get orderTotal as structured object with amount and currency
        $order_total_data = self::get_order_total_with_currency($order);
        $variables['orderTotal'] = $order_total_data;
        $variables['status'] = self::get_order_status($order);
        $variables['orderDate'] = self::get_order_date($order);
        
        // Backward compatibility - keep old variable names as aliases
        $variables['customerName'] = $variables['billingName'];
        $variables['orderId'] = $variables['orderNumber'];
        // amountPaid should be a string with currency symbol prefix, not an object
        $variables['amountPaid'] = isset($order_total_data['amount']) ? $order_total_data['amount'] : '';
        
        // Optional variables
        $variables['paymentMethod'] = self::get_payment_method($order);
        $variables['refundedAmount'] = self::get_refunded_amount($order);
        $variables['trackingLink'] = self::get_tracking_link($order);
        $variables['trackingNo'] = self::get_tracking_number($order);
        $variables['estimatedDelivery'] = self::get_estimated_delivery($order, $event);
        $variables['orderDetailsLink'] = self::get_order_details_link($order);
        $variables['cancelReason'] = self::get_cancel_reason($order, $event);
        $variables['failureReason'] = self::get_failure_reason($order, $event);
        $variables['retryPaymentLink'] = self::get_retry_payment_link($order);
        
        // Required variables that must always be present (even if empty)
        $required_vars = array('billingName', 'orderNumber', 'orderTotal', 'status', 'orderDate');
        
        // Separate required and optional variables
        $required = array();
        $optional = array();
        
        foreach ($variables as $key => $value) {
            if (in_array($key, $required_vars)) {
                // Handle orderTotal as structured array (with amount and currency)
                if ($key === 'orderTotal' && is_array($value) && isset($value['amount'])) {
                    $required[$key] = $value; // Keep the structured format
                } else {
                    $required[$key] = $value !== null ? $value : '';
                }
            } else {
                // Only include optional variables if they're not empty
                if ($value !== null && $value !== '') {
                    $optional[$key] = $value;
                }
            }
        }
        
        // Merge required (always included) with optional (only non-empty)
        $result = array_merge($required, $optional);
        
        // Debug log: Variables extracted
        authyo_debug_log(
            sprintf('Variables extracted for order #%d: %d total variables', $order_id, count($result)),
            array('order_id' => $order_id, 'event' => $event, 'variable_count' => count($result), 'variables' => $result),
            'template'
        );
        
        return $result;
    }
    
    /**
     * Get customer name
     *
     * @param WC_Order $order Order object
     * @return string Customer name (always returns a value, never empty)
     */
    private static function get_customer_name($order) {
        $first_name = $order->get_billing_first_name();
        $last_name = $order->get_billing_last_name();
        
        if ($first_name || $last_name) {
            $name = trim($first_name . ' ' . $last_name);
            if (!empty($name)) {
                return $name;
            }
        }
        
        // Fallback to display name
        $user = $order->get_user();
        if ($user && !empty($user->display_name)) {
            return $user->display_name;
        }
        
        // Fallback to email if name not available
        $email = $order->get_billing_email();
        if (!empty($email)) {
            return $email;
        }
        
        // Last resort: return "Customer"
        return __('Customer', 'authyo-otp-authentication-for-woocommerce');
    }
    
    /**
     * Get order ID
     *
     * @param WC_Order $order Order object
     * @return string Order ID
     */
    private static function get_order_id($order) {
        return (string) $order->get_order_number();
    }
    
    /**
     * Get order date
     *
     * @param WC_Order $order Order object
     * @return string Formatted order date
     */
    private static function get_order_date($order) {
        $date = $order->get_date_created();
        if ($date) {
            return $date->date_i18n(get_option('date_format'));
        }
        return '';
    }
    
    /**
     * Get order total with currency information
     *
     * Returns structured data with amount and currency code for global compatibility.
     *
     * @param WC_Order $order Order object
     * @return array Structured array with 'amount' (formatted string) and 'currency' (currency code)
     */
    private static function get_order_total_with_currency($order) {
        $total = $order->get_total();
        $currency = $order->get_currency();
        
        // Get currency symbol (may be HTML entity in some WooCommerce configurations)
        $symbol = get_woocommerce_currency_symbol($currency);
        
        // CRITICAL: Decode HTML entities to ensure we get the raw currency character (₹, $, €, etc.)
        // Some WooCommerce configurations return HTML entities like &#8377; instead of ₹
        $symbol = html_entity_decode($symbol, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        
        // Format the number with proper decimal and thousand separators
        $formatted_number = number_format((float) $total, 2, '.', ',');
        
        // Get currency position from WooCommerce settings (left or right)
        // Default is 'left' if not set
        $currency_position = get_option('woocommerce_currency_pos', 'left');
        
        // Format based on currency position setting
        // 'left' = symbol before amount (e.g., "$100.00", "₹4,000.00")
        // 'right' = symbol after amount (e.g., "100.00 kr", "100.00 R")
        // 'left_space' = symbol before with space (e.g., "$ 100.00")
        // 'right_space' = symbol after with space (e.g., "100.00 kr")
        switch ($currency_position) {
            case 'right':
                $formatted_amount = $formatted_number . $symbol;
                break;
            case 'left_space':
                $formatted_amount = $symbol . ' ' . $formatted_number;
                break;
            case 'right_space':
                $formatted_amount = $formatted_number . ' ' . $symbol;
                break;
            case 'left':
            default:
                // Default: symbol before amount (most common: $, ₹, €, £, etc.)
                $formatted_amount = $symbol . $formatted_number;
                break;
        }
        
        // Return structured data with amount and currency code
        return array(
            'amount' => $formatted_amount,
            'currency' => $currency, // Currency code (USD, INR, EUR, etc.)
        );
    }
    
    /**
     * Get amount paid (legacy method for backward compatibility)
     *
     * @param WC_Order $order Order object
     * @return string Formatted amount (plain text, not HTML)
     * @deprecated Use get_order_total_with_currency() instead
     */
    private static function get_amount_paid($order) {
        $order_total_data = self::get_order_total_with_currency($order);
        // Return just the formatted amount string for backward compatibility
        return $order_total_data['amount'];
    }
    
    /**
     * Get payment method
     *
     * @param WC_Order $order Order object
     * @return string Payment method title
     */
    private static function get_payment_method($order) {
        $payment_method = $order->get_payment_method();
        if ($payment_method) {
            $gateways = WC()->payment_gateways->payment_gateways();
            if (isset($gateways[$payment_method])) {
                return $gateways[$payment_method]->get_title();
            }
            return $payment_method;
        }
        return '';
    }
    
    /**
     * Get refunded amount
     *
     * @param WC_Order $order Order object
     * @return string Formatted refunded amount
     */
    private static function get_refunded_amount($order) {
        $refunded = $order->get_total_refunded();
        if ($refunded > 0) {
            return wc_price($refunded, array('currency' => $order->get_currency()));
        }
        return '';
    }
    
    /**
     * Get tracking link
     *
     * @param WC_Order $order Order object
     * @return string Tracking link URL
     */
    private static function get_tracking_link($order) {
        // Check for tracking plugins or custom meta
        $tracking_link = $order->get_meta('_tracking_link');
        if ($tracking_link) {
            return esc_url($tracking_link);
        }
        
        // Check for tracking number and build link if possible
        $tracking_no = self::get_tracking_number($order);
        if ($tracking_no) {
            // Generic tracking link - can be customized
            return '';
        }
        
        return '';
    }
    
    /**
     * Get tracking number
     *
     * @param WC_Order $order Order object
     * @return string Tracking number
     */
    private static function get_tracking_number($order) {
        // Check for tracking plugins or custom meta
        $tracking_no = $order->get_meta('_tracking_number');
        if ($tracking_no) {
            return $tracking_no;
        }
        
        // Check for shipping tracking
        $tracking_no = $order->get_meta('_wc_shipment_tracking_number');
        if ($tracking_no) {
            return $tracking_no;
        }
        
        return '';
    }
    
    /**
     * Get estimated delivery date
     *
     * @param WC_Order $order Order object
     * @param string $event Order event/status
     * @return string Estimated delivery date
     */
    private static function get_estimated_delivery($order, $event = '') {
        // For 'processing' event (order_placed template), return static text
        if ($event === 'processing') {
            return 'will be notified soon';
        }
        
        $estimated_delivery = $order->get_meta('_estimated_delivery');
        if ($estimated_delivery) {
            return $estimated_delivery;
        }
        
        return '';
    }
    
    /**
     * Get order details link
     *
     * @param WC_Order $order Order object
     * @return string Order details URL
     */
    private static function get_order_details_link($order) {
        if ($order->get_customer_id()) {
            return esc_url(wc_get_endpoint_url('view-order', $order->get_id(), wc_get_page_permalink('myaccount')));
        }
        
        // For guest orders, return empty or a generic link
        return '';
    }
    
    /**
     * Get cancel reason
     *
     * @param WC_Order $order Order object
     * @param string $event Order event
     * @return string Cancel reason
     */
    private static function get_cancel_reason($order, $event) {
        if ($event === 'cancelled') {
            $reason = $order->get_meta('_cancellation_reason');
            if ($reason) {
                return $reason;
            }
            
            // Check order notes for cancellation
            $notes = wc_get_order_notes(array(
                'order_id' => $order->get_id(),
                'limit' => 1,
            ));
            
            if (!empty($notes)) {
                $note = reset($notes);
                if (stripos($note->content, 'cancel') !== false) {
                    return $note->content;
                }
            }
        }
        
        return '';
    }
    
    /**
     * Get failure reason
     *
     * @param WC_Order $order Order object
     * @param string $event Order event
     * @return string Failure reason
     */
    private static function get_failure_reason($order, $event) {
        if ($event === 'payment_failed' || $event === 'failed') {
            $reason = $order->get_meta('_payment_failure_reason');
            if ($reason) {
                return $reason;
            }
            
            // Check order notes for payment failure
            $notes = wc_get_order_notes(array(
                'order_id' => $order->get_id(),
                'limit' => 1,
            ));
            
            if (!empty($notes)) {
                $note = reset($notes);
                if (stripos($note->content, 'payment') !== false || stripos($note->content, 'fail') !== false) {
                    return $note->content;
                }
            }
        }
        
        return '';
    }
    
    /**
     * Get retry payment link
     *
     * @param WC_Order $order Order object
     * @return string Retry payment URL
     */
    private static function get_retry_payment_link($order) {
        if ($order->needs_payment()) {
            return esc_url($order->get_checkout_payment_url());
        }
        
        return '';
    }
    
    /**
     * Get order status
     *
     * @param WC_Order $order Order object
     * @return string Order status
     */
    private static function get_order_status($order) {
        $status = $order->get_status();
        
        // Map WooCommerce status to readable format
        $status_map = array(
            'pending' => __('Pending', 'authyo-otp-authentication-for-woocommerce'),
            'processing' => __('Processing', 'authyo-otp-authentication-for-woocommerce'),
            'on-hold' => __('On Hold', 'authyo-otp-authentication-for-woocommerce'),
            'completed' => __('Completed', 'authyo-otp-authentication-for-woocommerce'),
            'cancelled' => __('Cancelled', 'authyo-otp-authentication-for-woocommerce'),
            'refunded' => __('Refunded', 'authyo-otp-authentication-for-woocommerce'),
            'failed' => __('Failed', 'authyo-otp-authentication-for-woocommerce'),
        );
        
        // Return translated status if available, otherwise return the status code
        return isset($status_map[$status]) ? $status_map[$status] : ucfirst($status);
    }
    
    /**
     * Get products array for email notifications
     * 
     * Returns nested array of products with name, quantity, price, subtotal, and image URL.
     * Only used for email notifications on processing orders.
     *
     * @param WC_Order $order Order object
     * @return array Products array with product details
     */
    public static function get_products_array($order) {
        $products = array();
        $currency = $order->get_currency();
        $items = $order->get_items();
        
        if (empty($items)) {
            return $products;
        }
        
        foreach ($items as $item_id => $item) {
            // Skip if item is not a product line item
            if (!is_a($item, 'WC_Order_Item_Product')) {
                continue;
            }
            
            $product = $item->get_product();
            if (!$product) {
                continue;
            }
            
            // Get product name
            $name = $item->get_name();
            
            // Get quantity
            $quantity = (int) $item->get_quantity();
            
            // Get subtotal (total for this line item)
            $subtotal = (float) $item->get_subtotal();
            
            // Calculate per-item price (subtotal divided by quantity)
            $price = $quantity > 0 ? $subtotal / $quantity : 0;
            
            // Format prices with currency symbol
            $symbol = get_woocommerce_currency_symbol($currency);
            // Decode HTML entities to get raw currency character
            $symbol = html_entity_decode($symbol, ENT_QUOTES | ENT_HTML5, 'UTF-8');
            
            // Format numbers with proper decimal and thousand separators
            $formatted_price = number_format((float) $price, 2, '.', ',');
            $formatted_subtotal = number_format((float) $subtotal, 2, '.', ',');
            
            // Get currency position
            $currency_position = get_option('woocommerce_currency_pos', 'left');
            
            // Format price based on currency position
            $formatted_price_str = self::format_price_with_currency($formatted_price, $symbol, $currency_position);
            $formatted_subtotal_str = self::format_price_with_currency($formatted_subtotal, $symbol, $currency_position);
            
            // Get product image URL (full size)
            $image_url = '';
            $image_id = $product->get_image_id();
            if ($image_id && $image_id > 0) {
                $image_url = wp_get_attachment_image_url($image_id, 'full');
            }
            // If no image found, use placeholder
            if (!$image_url) {
                $image_url = wc_placeholder_img_src('full');
            }
            
            // Build product array
            $products[] = array(
                'name' => $name,
                'quantity' => $quantity,
                'price' => $formatted_price_str,
                'subtotal' => $formatted_subtotal_str,
                'image' => $image_url ? esc_url($image_url) : '',
            );
        }
        
        return $products;
    }
    
    /**
     * Get billing address details for email notifications
     * 
     * Returns nested array with all billing address fields including formatted address.
     * Only used for email notifications on processing orders.
     *
     * @param WC_Order $order Order object
     * @return array Billing address details
     */
    public static function get_billing_address_array($order) {
        // Get individual address fields
        $firstName = $order->get_billing_first_name();
        $lastName = $order->get_billing_last_name();
        $email = $order->get_billing_email();
        $phone = $order->get_billing_phone();
        $address1 = $order->get_billing_address_1();
        $address2 = $order->get_billing_address_2();
        $city = $order->get_billing_city();
        $state = $order->get_billing_state();
        $postcode = $order->get_billing_postcode();
        $country = $order->get_billing_country();
        
        // Get country name from country code
        $countries = WC()->countries->get_countries();
        $country_name = isset($countries[$country]) ? $countries[$country] : $country;
        
        // Build formatted address (HTML format for email)
        $formatted_parts = array();
        if ($firstName || $lastName) {
            $formatted_parts[] = trim($firstName . ' ' . $lastName);
        }
        if ($address1) {
            $formatted_parts[] = $address1;
        }
        if ($address2) {
            $formatted_parts[] = $address2;
        }
        
        // City, state, postcode line
        $city_line_parts = array();
        if ($city) {
            $city_line_parts[] = $city;
        }
        if ($state) {
            $state_name = $state;
            // Try to get state name from country states
            $states = WC()->countries->get_states($country);
            if ($states && isset($states[$state])) {
                $state_name = $states[$state];
            }
            if ($city) {
                $city_line_parts[] = $state_name;
            } else {
                $city_line_parts[] = $state_name;
            }
        }
        if ($postcode) {
            $city_line_parts[] = $postcode;
        }
        if (!empty($city_line_parts)) {
            $formatted_parts[] = implode(', ', $city_line_parts);
        }
        
        // Country line
        if ($country_name) {
            $formatted_parts[] = $country_name . ($country ? ' (' . $country . ')' : '');
        }
        
        $formatted = !empty($formatted_parts) ? implode('<br>', $formatted_parts) : '';
        
        return array(
            'firstName' => $firstName,
            'lastName' => $lastName,
            'email' => $email,
            'phone' => $phone,
            'address1' => $address1,
            'address2' => $address2,
            'city' => $city,
            'state' => $state,
            'postcode' => $postcode,
            'country' => $country,
            'formatted' => $formatted,
        );
    }
    
    /**
     * Format price with currency symbol based on WooCommerce currency position setting
     *
     * @param string $formatted_number Formatted number (with thousand separators)
     * @param string $symbol Currency symbol
     * @param string $currency_position Currency position setting (left, right, left_space, right_space)
     * @return string Formatted price with currency symbol
     */
    private static function format_price_with_currency($formatted_number, $symbol, $currency_position) {
        switch ($currency_position) {
            case 'right':
                return $formatted_number . $symbol;
            case 'left_space':
                return $symbol . ' ' . $formatted_number;
            case 'right_space':
                return $formatted_number . ' ' . $symbol;
            case 'left':
            default:
                return $symbol . $formatted_number;
        }
    }
}
