<?php
/**
 * Notification Logger Helper
 *
 * Logs notification events, payloads, and API responses.
 *
 * @package Authyo_OTP_Auth
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Notification Logger class
 */
class Authyo_Notification_Logger {
    
    /**
     * Log notification event
     *
     * @param int $order_id Order ID
     * @param string $event Event type
     * @param string $method Notification method
     * @param array $payload Notification payload
     * @param array|WP_Error $response API response or error
     * @return void
     */
    public static function log($order_id, $event, $method, $payload, $response) {
        $log_entry = array(
            'order_id' => $order_id,
            'event' => $event,
            'method' => $method,
            'payload' => $payload,
            'response' => is_wp_error($response) ? array(
                'error' => true,
                'message' => $response->get_error_message(),
                'code' => $response->get_error_code(),
            ) : $response,
            'timestamp' => current_time('mysql'),
            'success' => !is_wp_error($response),
        );
        
        // Store in order meta for easy access
        $order = wc_get_order($order_id);
        if ($order) {
            $existing_logs = $order->get_meta('_authyo_notification_logs', true);
            if (!is_array($existing_logs)) {
                $existing_logs = array();
            }
            
            $existing_logs[] = $log_entry;
            
            // Keep only last 50 logs per order
            if (count($existing_logs) > 50) {
                $existing_logs = array_slice($existing_logs, -50);
            }
            
            $order->update_meta_data('_authyo_notification_logs', $existing_logs);
            $order->save();
        }
        
        // Also log to debug log if WP_DEBUG is enabled
        if (defined('WP_DEBUG') && WP_DEBUG) {
            $log_message = sprintf(
                'Authyo Notification: Order #%s | Event: %s | Method: %s | Success: %s',
                $order_id,
                $event,
                $method,
                $log_entry['success'] ? 'Yes' : 'No'
            );
            
            if (is_wp_error($response)) {
                $log_message .= ' | Error: ' . $response->get_error_message();
            }
            
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log($log_message);
            
            // Log full payload and response in debug mode
            if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('Payload: ' . wp_json_encode($payload, JSON_PRETTY_PRINT));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('Response: ' . wp_json_encode($log_entry['response'], JSON_PRETTY_PRINT));
            }
        }
    }
    
    /**
     * Log warning (for missing variables, coming soon templates, etc.)
     *
     * @param int $order_id Order ID
     * @param string $event Event type
     * @param string $method Notification method
     * @param string $warning_message Warning message
     * @return void
     */
    public static function log_warning($order_id, $event, $method, $warning_message) {
        $log_entry = array(
            'order_id' => $order_id,
            'event' => $event,
            'method' => $method,
            'warning' => $warning_message,
            'timestamp' => current_time('mysql'),
            'type' => 'warning',
        );
        
        // Store in order meta
        $order = wc_get_order($order_id);
        if ($order) {
            $existing_logs = $order->get_meta('_authyo_notification_logs', true);
            if (!is_array($existing_logs)) {
                $existing_logs = array();
            }
            
            $existing_logs[] = $log_entry;
            
            // Keep only last 50 logs per order
            if (count($existing_logs) > 50) {
                $existing_logs = array_slice($existing_logs, -50);
            }
            
            $order->update_meta_data('_authyo_notification_logs', $existing_logs);
            $order->save();
        }
        
        // Log to debug log
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log(sprintf(
                'Authyo Notification Warning: Order #%s | Event: %s | Method: %s | %s',
                $order_id,
                $event,
                $method,
                $warning_message
            ));
        }
    }
    
    /**
     * Get logs for an order
     *
     * @param int $order_id Order ID
     * @return array Log entries
     */
    public static function get_logs($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return array();
        }
        
        $logs = $order->get_meta('_authyo_notification_logs', true);
        return is_array($logs) ? $logs : array();
    }
}
