<?php
/**
 * Notification Flow Module
 *
 * @package Authyo_OTP_Auth
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Main Notification Flow class
 */
class Authyo_Notification_Flow {
    
    /**
     * The single instance of the class.
     *
     * @var Authyo_Notification_Flow
     */
    private static $instance = null;
    
    /**
     * Plugin version
     *
     * @var string
     */
    private $version;
    
    /**
     * Main Instance.
     *
     * @return Authyo_Notification_Flow
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
            self::$instance->version = defined('AUTHYO_OTP_AUTH_VERSION') ? AUTHYO_OTP_AUTH_VERSION : '1.0.10';
        }
        return self::$instance;
    }
    
    /**
     * Constructor.
     */
    private function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Use priority 20 to ensure parent menu is registered first (default is 10)
        add_action('admin_menu', array($this, 'add_admin_menu'), 20);
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('wp_ajax_authyo_save_notification_flow', array($this, 'ajax_save_flow'));
        add_action('wp_ajax_authyo_get_notification_flows', array($this, 'ajax_get_flows'));
        add_action('wp_ajax_authyo_delete_notification_flow', array($this, 'ajax_delete_flow'));
        add_action('wp_ajax_authyo_update_notification_flow', array($this, 'ajax_update_flow'));
        add_action('wp_ajax_authyo_toggle_notification_flow', array($this, 'ajax_toggle_flow'));
        add_action('wp_ajax_authyo_delete_all_notification_flows', array($this, 'ajax_delete_all_flows'));
        add_action('wp_ajax_authyo_update_all_notification_flows', array($this, 'ajax_update_all_flows'));
        
        // Load helper classes
        require_once AUTHYO_OTP_AUTH_PLUGIN_DIR . 'modules/notification-flow/helpers/class-order-variables.php';
        require_once AUTHYO_OTP_AUTH_PLUGIN_DIR . 'modules/notification-flow/helpers/template-mapping.php';
        require_once AUTHYO_OTP_AUTH_PLUGIN_DIR . 'modules/notification-flow/helpers/class-notification-logger.php';
        require_once AUTHYO_OTP_AUTH_PLUGIN_DIR . 'modules/notification-flow/class-notification-api.php';
        
        // WooCommerce hooks - only if WooCommerce is active
        if (class_exists('WooCommerce')) {
            $this->init_woocommerce_hooks();
        }
    }
    
    /**
     * Initialize WooCommerce hooks
     */
    private function init_woocommerce_hooks() {
        // Status hooks (fire when order reaches a specific status)
        add_action('woocommerce_order_status_processing', array($this, 'handle_order_status_processing'), 10, 1);
        add_action('woocommerce_order_status_completed', array($this, 'handle_order_status_completed'), 10, 1);
        add_action('woocommerce_order_status_cancelled', array($this, 'handle_order_status_cancelled'), 10, 1);
        add_action('woocommerce_order_status_failed', array($this, 'handle_order_status_failed'), 10, 1);
        add_action('woocommerce_order_status_refunded', array($this, 'handle_order_status_refunded'), 10, 1);
        add_action('woocommerce_order_status_on-hold', array($this, 'handle_order_status_on_hold'), 10, 1);
        add_action('woocommerce_order_status_pending', array($this, 'handle_order_status_pending'), 10, 1);
        
        // Transition hooks (fire when status changes from one to another - more reliable)
        // This is a backup to catch status changes that might be missed by status hooks
        add_action('woocommerce_order_status_changed', array($this, 'handle_order_status_transition'), 10, 4);
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'authyo-otp-authentication-for-woocommerce',
            __('Notification Flow', 'authyo-otp-authentication-for-woocommerce'),
            __('Notification Flow', 'authyo-otp-authentication-for-woocommerce'),
            'manage_options',
            'authyo-notification-flow',
            array($this, 'render_notification_flow_page')
        );
    }
    
    /**
     * Enqueue admin scripts and styles
     *
     * @param string $hook Current admin page hook
     */
    public function enqueue_admin_scripts($hook) {
        // Check if we're on the notification flow page
        // WordPress creates hook like: authyo-otp-authentication-for-woocommerce_page_authyo-notification-flow
        // or sometimes: wc-otp-auth_page_authyo-notification-flow
        if (strpos($hook, 'authyo-notification-flow') === false) {
            return;
        }
        
        // Enqueue styles
        wp_enqueue_style(
            'authyo-notification-flow-admin',
            AUTHYO_OTP_AUTH_PLUGIN_URL . 'modules/notification-flow/assets/css/notification-flow-admin.css',
            array(),
            $this->version
        );
        
        // Enqueue scripts
        wp_enqueue_script(
            'authyo-notification-flow-admin',
            AUTHYO_OTP_AUTH_PLUGIN_URL . 'modules/notification-flow/assets/js/notification-flow-admin.js',
            array('jquery'),
            $this->version,
            true
        );
        
        // Localize script
        wp_localize_script('authyo-notification-flow-admin', 'authyoNotificationFlow', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('authyo_notification_flow_nonce'),
            'messages' => array(
                'save_success' => esc_html__('Flow saved successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'save_error' => esc_html__('Error saving flow. Please try again.', 'authyo-otp-authentication-for-woocommerce'),
                'delete_success' => esc_html__('Flow deleted successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'delete_error' => esc_html__('Error deleting flow. Please try again.', 'authyo-otp-authentication-for-woocommerce'),
                'confirm_delete' => esc_html__('Are you sure you want to delete this flow?', 'authyo-otp-authentication-for-woocommerce'),
                'flow_enabled' => esc_html__('Notification flow enabled', 'authyo-otp-authentication-for-woocommerce'),
                'flow_disabled' => esc_html__('Notification flow disabled', 'authyo-otp-authentication-for-woocommerce'),
                'toggle_error' => esc_html__('Failed to update flow status.', 'authyo-otp-authentication-for-woocommerce'),
            )
        ));
    }
    
    /**
     * Render notification flow page
     */
    public function render_notification_flow_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        // Verify constant is defined
        if (!defined('AUTHYO_OTP_AUTH_PLUGIN_DIR')) {
            wp_die(esc_html__('Plugin directory constant is not defined.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        // Build view file path
        $view_file = AUTHYO_OTP_AUTH_PLUGIN_DIR . 'modules/notification-flow/views/notification-flow-page.php';
        
        // Check if file exists
        if (!file_exists($view_file)) {
            wp_die(sprintf(
                /* translators: %s: View file path */
                esc_html__('View file not found: %s', 'authyo-otp-authentication-for-woocommerce'),
                esc_html($view_file)
            ));
        }
        
        // Include the view file
        include $view_file;
    }
    
    /**
     * Get WooCommerce events list
     *
     * @return array
     */
    public function get_woocommerce_events() {
        return array(
            'processing' => __('Processing', 'authyo-otp-authentication-for-woocommerce'),
            'completed' => __('Completed', 'authyo-otp-authentication-for-woocommerce'),
            'cancelled' => __('Cancelled', 'authyo-otp-authentication-for-woocommerce'),
            'payment_failed' => __('Payment Failed', 'authyo-otp-authentication-for-woocommerce'),
            'refunded' => __('Refunded', 'authyo-otp-authentication-for-woocommerce'),
            'on_hold' => __('On-hold', 'authyo-otp-authentication-for-woocommerce'),
            'pending' => __('Pending', 'authyo-otp-authentication-for-woocommerce'),
        );
    }
    
    /**
     * Get notification methods
     *
     * @return array
     */
    public function get_notification_methods() {
        return array(
            'email' => __('Email', 'authyo-otp-authentication-for-woocommerce'),
            'sms' => __('SMS', 'authyo-otp-authentication-for-woocommerce'),
            'whatsapp' => __('WhatsApp', 'authyo-otp-authentication-for-woocommerce'),
        );
    }
    
    /**
     * Get all saved flows
     *
     * @return array
     */
    public function get_saved_flows() {
        $flows = get_option('authyo_notification_flows', array());
        
        if (!is_array($flows)) {
            return array();
        }
        
        // Check if flows are already in new structure (keyed by event with methods array)
        $needs_conversion = false;
        foreach ($flows as $flow_id => $flow) {
            // If flow has 'method' (singular) instead of 'methods' (plural), it's old structure
            if (isset($flow['method']) && !isset($flow['methods'])) {
                $needs_conversion = true;
                break;
            }
            // If flow_id contains underscore (old format: event_method), it's old structure
            if (strpos($flow_id, '_') !== false && !isset($flow['methods'])) {
                $needs_conversion = true;
                break;
            }
        }
        
        // Convert old structure to new structure if needed (backward compatibility)
        if ($needs_conversion) {
            $converted_flows = array();
            foreach ($flows as $flow_id => $flow) {
                $event = isset($flow['event']) ? $flow['event'] : '';
                $method = isset($flow['method']) ? $flow['method'] : '';
                
                // If flow_id is in old format (event_method), extract event
                if (empty($event) && strpos($flow_id, '_') !== false) {
                    $parts = explode('_', $flow_id, 2);
                    $event = $parts[0];
                    if (empty($method)) {
                        $method = isset($parts[1]) ? $parts[1] : '';
                    }
                }
                
                if (empty($event) || empty($method)) {
                    // Skip if we can't determine event/method, might be new structure
                    if (isset($flow['methods']) && is_array($flow['methods'])) {
                        $converted_flows[$flow_id] = $flow;
                    }
                    continue;
                }
                
                // Old structure - convert to new structure
                if (!isset($converted_flows[$event])) {
                    $converted_flows[$event] = array(
                        'event' => $event,
                        'methods' => array(),
                        'enabled' => true, // Default to enabled for converted flows
                        'created_at' => isset($flow['created_at']) ? $flow['created_at'] : current_time('mysql'),
                        'updated_at' => current_time('mysql'),
                    );
                }
                if (!in_array($method, $converted_flows[$event]['methods'])) {
                    $converted_flows[$event]['methods'][] = $method;
                }
            }
            
            // Save converted structure
            if (!empty($converted_flows)) {
                update_option('authyo_notification_flows', $converted_flows);
                return $converted_flows;
            }
        }
        
        return $flows;
    }
    
    /**
     * Save a notification flow
     *
     * @param string $event WooCommerce event
     * @param string|array $method Notification method(s) - can be a single method string or array of methods
     * @return bool|WP_Error
     */
    public function save_flow($event, $method) {
        if (empty($event) || empty($method)) {
            return new WP_Error('invalid_data', __('Event and method are required.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        $events = $this->get_woocommerce_events();
        $available_methods = $this->get_notification_methods();
        
        if (!isset($events[$event])) {
            return new WP_Error('invalid_event', __('Invalid event selected.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        // Convert single method to array for consistent processing
        $methods = is_array($method) ? $method : array($method);
        
        // Validate all methods
        foreach ($methods as $method_item) {
            if (!isset($available_methods[$method_item])) {
                /* translators: %s: Method name */
                return new WP_Error('invalid_method', sprintf(__('Invalid method selected: %s', 'authyo-otp-authentication-for-woocommerce'), $method_item));
            }
        }
        
        $flows = $this->get_saved_flows();
        
        // Store methods as array in a single flow per event
        if (!isset($flows[$event])) {
            $flows[$event] = array(
                'event' => sanitize_text_field($event),
                'methods' => array(),
                'enabled' => true, // Default to enabled
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql'),
            );
        }
        
        // Merge new methods with existing ones (avoid duplicates)
        $existing_methods = isset($flows[$event]['methods']) ? $flows[$event]['methods'] : array();
        $all_methods = array_unique(array_merge($existing_methods, $methods));
        $flows[$event]['methods'] = array_values($all_methods);
        $flows[$event]['updated_at'] = current_time('mysql');
        
        $result = update_option('authyo_notification_flows', $flows);
        
        if ($result) {
            return true;
        }
        
        return new WP_Error('save_failed', __('Failed to save flow.', 'authyo-otp-authentication-for-woocommerce'));
    }
    
    /**
     * Delete a notification flow
     *
     * @param string $event Event name (flow ID is now the event)
     * @return bool|WP_Error
     */
    public function delete_flow($event) {
        if (empty($event)) {
            return new WP_Error('invalid_id', __('Event is required.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        $flows = $this->get_saved_flows();
        
        if (!isset($flows[$event])) {
            return new WP_Error('not_found', __('Flow not found.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        unset($flows[$event]);
        
        $result = update_option('authyo_notification_flows', $flows);
        
        if ($result) {
            return true;
        }
        
        return new WP_Error('delete_failed', __('Failed to delete flow.', 'authyo-otp-authentication-for-woocommerce'));
    }
    
    /**
     * Update a notification flow
     *
     * @param string $event WooCommerce event
     * @param array $methods Notification methods array
     * @return bool|WP_Error
     */
    public function update_flow($event, $methods) {
        if (empty($event) || empty($methods) || !is_array($methods)) {
            return new WP_Error('invalid_data', __('Event and methods are required.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        $events = $this->get_woocommerce_events();
        $available_methods = $this->get_notification_methods();
        
        if (!isset($events[$event])) {
            return new WP_Error('invalid_event', __('Invalid event selected.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        // Validate all methods
        foreach ($methods as $method_item) {
            if (!isset($available_methods[$method_item])) {
                /* translators: %s: Method name */
                return new WP_Error('invalid_method', sprintf(__('Invalid method selected: %s', 'authyo-otp-authentication-for-woocommerce'), $method_item));
            }
        }
        
        $flows = $this->get_saved_flows();
        
        if (!isset($flows[$event])) {
            return new WP_Error('not_found', __('Flow not found.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        // Update methods
        $flows[$event]['methods'] = array_values(array_unique($methods));
        $flows[$event]['updated_at'] = current_time('mysql');
        
        $result = update_option('authyo_notification_flows', $flows);
        
        if ($result) {
            return true;
        }
        
        return new WP_Error('update_failed', __('Failed to update flow.', 'authyo-otp-authentication-for-woocommerce'));
    }
    
    /**
     * AJAX handler to save flow
     */
    public function ajax_save_flow() {
        // Verify nonce
        if (!check_ajax_referer('authyo_notification_flow_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Get and sanitize data
        $event = isset($_POST['event']) ? sanitize_text_field(wp_unslash($_POST['event'])) : '';
        
        // Handle multiple methods from checkboxes
        $methods = array();
        if (isset($_POST['methods']) && is_array($_POST['methods'])) {
            $unslashed_methods = wp_unslash($_POST['methods']); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            foreach ($unslashed_methods as $method) {
                $methods[] = sanitize_text_field($method);
            }
        }
        
        // Fallback to single method for backward compatibility
        if (empty($methods) && isset($_POST['method'])) {
            $methods = array(sanitize_text_field(wp_unslash($_POST['method'])));
        }
        
        if (empty($methods)) {
            wp_send_json_error(array('message' => esc_html__('Please select at least one notification method.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        $result = $this->save_flow($event, $methods);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
            return;
        }
        
        wp_send_json_success(array(
            'message' => esc_html__('Flow saved successfully!', 'authyo-otp-authentication-for-woocommerce'),
            'flows' => $this->get_saved_flows()
        ));
    }
    
    /**
     * AJAX handler to get flows
     */
    public function ajax_get_flows() {
        // Verify nonce
        if (!check_ajax_referer('authyo_notification_flow_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        wp_send_json_success(array('flows' => $this->get_saved_flows()));
    }
    
    /**
     * AJAX handler to delete flow
     */
    public function ajax_delete_flow() {
        // Verify nonce
        if (!check_ajax_referer('authyo_notification_flow_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Get and sanitize data
        $event = isset($_POST['flow_id']) ? sanitize_text_field(wp_unslash($_POST['flow_id'])) : '';
        // Also support 'event' parameter for consistency
        if (empty($event) && isset($_POST['event'])) {
            $event = sanitize_text_field(wp_unslash($_POST['event']));
        }
        
        $result = $this->delete_flow($event);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
            return;
        }
        
        wp_send_json_success(array(
            'message' => esc_html__('Flow deleted successfully!', 'authyo-otp-authentication-for-woocommerce'),
            'flows' => $this->get_saved_flows()
        ));
    }
    
    /**
     * AJAX handler to update flow
     */
    public function ajax_update_flow() {
        // Verify nonce
        if (!check_ajax_referer('authyo_notification_flow_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Get and sanitize data
        $event = isset($_POST['event']) ? sanitize_text_field(wp_unslash($_POST['event'])) : '';
        
        // Handle multiple methods from checkboxes
        $methods = array();
        if (isset($_POST['methods']) && is_array($_POST['methods'])) {
            $unslashed_methods = wp_unslash($_POST['methods']); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            foreach ($unslashed_methods as $method) {
                $methods[] = sanitize_text_field($method);
            }
        }
        
        if (empty($methods)) {
            wp_send_json_error(array('message' => esc_html__('Please select at least one notification method.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        $result = $this->update_flow($event, $methods);
        
        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()));
            return;
        }
        
        wp_send_json_success(array(
            'message' => esc_html__('Flow updated successfully!', 'authyo-otp-authentication-for-woocommerce'),
            'flows' => $this->get_saved_flows()
        ));
    }
    
    /**
     * AJAX handler to toggle flow enabled/disabled state
     */
    public function ajax_toggle_flow() {
        // Verify nonce
        if (!check_ajax_referer('authyo_notification_flow_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Get and sanitize data
        $event = isset($_POST['event']) ? sanitize_text_field(wp_unslash($_POST['event'])) : '';
        $enabled = isset($_POST['enabled']) ? (int) $_POST['enabled'] : 1;
        $enabled = $enabled === 1 ? true : false;
        
        if (empty($event)) {
            wp_send_json_error(array('message' => esc_html__('Event is required.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        $flows = $this->get_saved_flows();
        
        if (!isset($flows[$event])) {
            wp_send_json_error(array('message' => esc_html__('Flow not found.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Update enabled status
        $flows[$event]['enabled'] = $enabled;
        $flows[$event]['updated_at'] = current_time('mysql');
        
        $result = update_option('authyo_notification_flows', $flows);
        
        if ($result) {
            $message = $enabled 
                ? esc_html__('Notification flow enabled successfully!', 'authyo-otp-authentication-for-woocommerce')
                : esc_html__('Notification flow disabled successfully!', 'authyo-otp-authentication-for-woocommerce');
            
            wp_send_json_success(array(
                'message' => $message,
                'enabled' => $enabled
            ));
        } else {
            wp_send_json_error(array('message' => esc_html__('Failed to update flow status.', 'authyo-otp-authentication-for-woocommerce')));
        }
    }
    
    /**
     * AJAX handler to delete all flows
     */
    public function ajax_delete_all_flows() {
        // Verify nonce
        if (!check_ajax_referer('authyo_notification_flow_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Delete all flows by setting empty array
        $result = update_option('authyo_notification_flows', array());
        
        if ($result) {
            wp_send_json_success(array(
                'message' => esc_html__('All flows deleted successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'flows' => array()
            ));
        } else {
            wp_send_json_error(array('message' => esc_html__('Failed to delete all flows.', 'authyo-otp-authentication-for-woocommerce')));
        }
    }
    
    /**
     * AJAX handler to update all flows at once
     */
    public function ajax_update_all_flows() {
        // Verify nonce
        if (!check_ajax_referer('authyo_notification_flow_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Get events from form
        $events = array();
        if (isset($_POST['events']) && is_array($_POST['events'])) {
            $unslashed_events = wp_unslash($_POST['events']); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            foreach ($unslashed_events as $event) {
                $events[] = sanitize_text_field($event);
            }
        }
        
        // Get methods for each event
        $methods_data = array();
        if (isset($_POST['methods']) && is_array($_POST['methods'])) {
            $unslashed_methods = wp_unslash($_POST['methods']); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            foreach ($unslashed_methods as $event_key => $method_array) {
                if (is_array($method_array)) {
                    $sanitized_methods = array();
                    foreach ($method_array as $method) {
                        $sanitized_methods[] = sanitize_text_field($method);
                    }
                    $methods_data[sanitize_text_field($event_key)] = $sanitized_methods;
                }
            }
        }
        
        $events_list = $this->get_woocommerce_events();
        $available_methods = $this->get_notification_methods();
        
        // Validate events
        foreach ($events as $event) {
            if (!isset($events_list[$event])) {
                wp_send_json_error(array(
                    /* translators: %s: Event name */
                    'message' => sprintf(esc_html__('Invalid event: %s', 'authyo-otp-authentication-for-woocommerce'), $event)
                ));
                return;
            }
        }
        
        // Validate methods for each event
        foreach ($methods_data as $event => $methods) {
            foreach ($methods as $method) {
                if (!isset($available_methods[$method])) {
                    wp_send_json_error(array(
                        /* translators: %s: Method name */
                        'message' => sprintf(esc_html__('Invalid method: %s', 'authyo-otp-authentication-for-woocommerce'), $method)
                    ));
                    return;
                }
            }
        }
        
        // Get current flows
        $flows = $this->get_saved_flows();
        
        // Update or create flows for selected events
        foreach ($events as $event) {
            if (isset($methods_data[$event]) && !empty($methods_data[$event])) {
                // Update or create flow with methods
                if (!isset($flows[$event])) {
                    $flows[$event] = array(
                        'event' => sanitize_text_field($event),
                        'methods' => array(),
                        'enabled' => true,
                        'created_at' => current_time('mysql'),
                        'updated_at' => current_time('mysql'),
                    );
                }
                $flows[$event]['methods'] = array_values(array_unique($methods_data[$event]));
                $flows[$event]['updated_at'] = current_time('mysql');
            } else {
                // Remove flow if no methods selected
                if (isset($flows[$event])) {
                    unset($flows[$event]);
                }
            }
        }
        
        // Remove flows for events not in the selected list
        foreach ($flows as $event_key => $flow) {
            if (!in_array($event_key, $events, true)) {
                unset($flows[$event_key]);
            }
        }
        
        // Save all flows
        $result = update_option('authyo_notification_flows', $flows);
        
        if ($result !== false) {
            wp_send_json_success(array(
                'message' => esc_html__('All flows updated successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'flows' => $this->get_saved_flows()
            ));
        } else {
            wp_send_json_error(array('message' => esc_html__('Failed to update flows.', 'authyo-otp-authentication-for-woocommerce')));
        }
    }
    
    /**
     * Handle order status: processing
     *
     * @param int $order_id Order ID
     */
    public function handle_order_status_processing($order_id) {
        $this->process_order_notification($order_id, 'processing');
    }
    
    /**
     * Handle order status: completed
     *
     * @param int $order_id Order ID
     */
    public function handle_order_status_completed($order_id) {
        $this->process_order_notification($order_id, 'completed');
    }
    
    /**
     * Handle order status: cancelled
     *
     * @param int $order_id Order ID
     */
    public function handle_order_status_cancelled($order_id) {
        $this->process_order_notification($order_id, 'cancelled');
    }
    
    /**
     * Handle order status: failed
     *
     * @param int $order_id Order ID
     */
    public function handle_order_status_failed($order_id) {
        $this->process_order_notification($order_id, 'payment_failed');
    }
    
    /**
     * Handle order status: refunded
     *
     * @param int $order_id Order ID
     */
    public function handle_order_status_refunded($order_id) {
        $this->process_order_notification($order_id, 'refunded');
    }
    
    /**
     * Handle order status: on-hold
     *
     * @param int $order_id Order ID
     */
    public function handle_order_status_on_hold($order_id) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("Authyo Notification: handle_order_status_on_hold called for order #{$order_id}");
        }
        $this->process_order_notification($order_id, 'on_hold');
    }
    
    /**
     * Handle order status transition (backup hook for status changes)
     * This catches status changes that might be missed by individual status hooks
     *
     * @param int $order_id Order ID
     * @param string $old_status Old order status
     * @param string $new_status New order status
     * @param WC_Order $order Order object
     */
    public function handle_order_status_transition($order_id, $old_status, $new_status, $order) {
        // Only process if status actually changed
        if ($old_status === $new_status) {
            return;
        }
        
        // Map WooCommerce status to our event names
        $status_to_event = array(
            'processing' => 'processing',
            'completed' => 'completed',
            'cancelled' => 'cancelled',
            'failed' => 'payment_failed',
            'refunded' => 'refunded',
            'on-hold' => 'on_hold', // WooCommerce uses hyphen, we use underscore
            'pending' => 'pending',
        );
        
        // Check if this status change should trigger a notification
        if (isset($status_to_event[$new_status])) {
            $event = $status_to_event[$new_status];
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: Status transition detected - Order #{$order_id} from '{$old_status}' to '{$new_status}' (event: {$event})");
            }
            
            // Process notification for the new status
            // Note: We check if status hook already fired to avoid duplicates
            // Status hooks fire before transition hooks, so we use a small delay check
            $this->process_order_notification($order_id, $event);
        }
    }
    
    /**
     * Handle order status: pending
     *
     * @param int $order_id Order ID
     */
    public function handle_order_status_pending($order_id) {
        $this->process_order_notification($order_id, 'pending');
    }
    
    /**
     * Process order notification for a specific event
     *
     * @param int $order_id Order ID
     * @param string $event Order event/status
     */
    private function process_order_notification($order_id, $event) {
        // Debug log: Start processing
        authyo_debug_log(
            sprintf('Processing notification for order #%d, event: %s', $order_id, $event),
            array('order_id' => $order_id, 'event' => $event),
            'notification'
        );
        
        // Deduplication: Prevent processing the same event twice within 5 seconds
        // This helps when both status hook and transition hook fire
        $cache_key = 'authyo_notif_' . $order_id . '_' . $event;
        $cache_group = 'authyo_notifications';
        $cached = wp_cache_get($cache_key, $cache_group);
        
        if ($cached !== false) {
            authyo_debug_log(
                sprintf('Skipping duplicate notification for order #%d, event: %s (processed within last 5 seconds)', $order_id, $event),
                array('order_id' => $order_id, 'event' => $event),
                'notification'
            );
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: Skipping duplicate notification for order #{$order_id}, event: {$event} (processed within last 5 seconds)");
            }
            return; // Already processed recently, skip
        }
        
        // Set cache for 5 seconds to prevent duplicates
        wp_cache_set($cache_key, time(), $cache_group, 5);
        // Check if order notifications are enabled
        $order_notifications_enabled = get_option('authyo_otp_auth_order_notifications_enabled', 'no');
        if ($order_notifications_enabled !== 'yes') {
            authyo_debug_log(
                sprintf('Order notifications disabled for order #%d, event: %s', $order_id, $event),
                array('order_id' => $order_id, 'event' => $event),
                'notification'
            );
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: Order notifications disabled for order #{$order_id}, event: {$event}");
            }
            return;
        }
        
        // Get order object
        $order = wc_get_order($order_id);
        if (!$order || !is_a($order, 'WC_Order')) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: Invalid order object for order #{$order_id}, event: {$event}");
            }
            return;
        }
        
        // Get saved flows
        $flows = $this->get_saved_flows();
        if (empty($flows)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: No flows found for order #{$order_id}, event: {$event}");
            }
            return;
        }
        
        // Find flow matching this event
        if (!isset($flows[$event])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: No flow found for event '{$event}' (order #{$order_id}). Available events: " . implode(', ', array_keys($flows)));
            }
            return;
        }
        
        $flow = $flows[$event];
        
        // Check if flow is enabled (default to enabled for backward compatibility)
        $is_enabled = !isset($flow['enabled']) || $flow['enabled'] !== false;
        if (!$is_enabled) {
            return; // Flow is disabled, skip notification
        }
        
        $methods = isset($flow['methods']) && is_array($flow['methods']) ? $flow['methods'] : array();
        
        if (empty($methods)) {
            return;
        }
        
        // Process each method for this event
        foreach ($methods as $method) {
            $this->process_single_flow($order, array(
                'event' => $event,
                'method' => $method,
            ));
        }
    }
    
    /**
     * Process a single notification flow
     *
     * @param WC_Order $order Order object
     * @param array $flow Flow configuration
     */
    private function process_single_flow($order, $flow) {
        $event = isset($flow['event']) ? $flow['event'] : '';
        $method = isset($flow['method']) ? $flow['method'] : '';
        
        if (empty($event) || empty($method)) {
            return;
        }
        
        // Map event to template
        // WhatsApp uses email templates, SMS uses SMS-specific templates
        if ($method === 'sms') {
            $template = authyo_get_sms_template_for_event($event);
        } else {
            // Email and WhatsApp both use email templates
            $template = authyo_get_template_for_event($event);
        }
        
        if (empty($template)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: No template mapped for event '{$event}' (order #{$order->get_id()}, method: {$method})");
            }
            return;
        }
        
        // Skip "coming_soon" templates - log warning and skip
        if ($template === 'coming_soon') {
            $warning_message = sprintf(
                /* translators: %s: Template name */
                __('Template "%s" is not yet available. Notification skipped.', 'authyo-otp-authentication-for-woocommerce'),
                $template
            );
            Authyo_Notification_Logger::log_warning($order->get_id(), $event, $method, $warning_message);
            return;
        }
        
        // Extract order variables
        $variables = Authyo_Order_Variables::extract($order, $event);
        
        // Add products and billing address for email notifications on processing orders
        if ($method === 'email' && $event === 'processing') {
            // Add products array
            $products = Authyo_Order_Variables::get_products_array($order);
            if (!empty($products)) {
                $variables['products'] = $products;
            }
            
            // Add billing address details
            $billing_address = Authyo_Order_Variables::get_billing_address_array($order);
            if (!empty($billing_address)) {
                $variables['billingAddress'] = $billing_address;
            }
            
            // Debug log: Extended variables added for email/processing
            authyo_debug_log(
                sprintf('Added products and billing address for email/processing order #%d', $order->get_id()),
                array(
                    'order_id' => $order->get_id(),
                    'products_count' => count($products),
                    'has_billing_address' => !empty($billing_address)
                ),
                'template'
            );
        }
        
        // Debug log: Variables extracted
        authyo_debug_log(
            sprintf('Extracted variables for order #%d, event: %s, template: %s', $order->get_id(), $event, $template),
            array(
                'order_id' => $order->get_id(),
                'event' => $event,
                'template' => $template,
                'method' => $method,
                'variables' => $variables
            ),
            'template'
        );
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("Authyo Notification: Extracted variables for order #{$order->get_id()}, event: {$event}, template: {$template}");
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
            error_log("Authyo Notification: Variables: " . print_r($variables, true));
        }
        
        // For SMS, prepare variables according to SMS rules
        if ($method === 'sms') {
            // Add APPNAME to variables (required by SMS templates, always empty for SMS)
            // This must be added before validation since it's required but not extracted from order
            $variables['APPNAME'] = '';
            
            // Get required tags for SMS template (internal names)
            $required_tags = authyo_get_sms_template_tags($template);
            
            // Validate original variables BEFORE mapping (validate against internal names)
            $validation = authyo_validate_sms_required_variables($variables, $required_tags);
            
            // Debug log: SMS variable validation
            authyo_debug_log(
                sprintf('Validating %s variables for template "%s", order #%d', $method, $template, $order->get_id()),
                array(
                    'method' => $method,
                    'template' => $template,
                    'order_id' => $order->get_id(),
                    'required_tags' => $required_tags,
                    'validation_result' => $validation
                ),
                'template'
            );
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: Required SMS tags for template '{$template}': " . implode(', ', $required_tags));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
                error_log("Authyo Notification: Original variables before mapping: " . print_r($variables, true));
            }
            
            // Only proceed if validation passes
            if ($validation['valid']) {
                // Prepare SMS variables (applies APPNAME = "" rule and maps to API-expected names)
                $variables = authyo_prepare_sms_variables($variables, $template);
                
                // Debug log: Variables prepared for SMS
                authyo_debug_log(
                    sprintf('Prepared %s variables for template "%s", order #%d', $method, $template, $order->get_id()),
                    array(
                        'method' => $method,
                        'template' => $template,
                        'order_id' => $order->get_id(),
                        'prepared_variables' => $variables
                    ),
                    'template'
                );
                
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
                    error_log("Authyo Notification: Prepared SMS variables (mapped to API names): " . print_r($variables, true));
                }
            }
        } else {
            // For email and WhatsApp, prepare variables
            // CRITICAL: Format orderTotal FIRST, before any validation or other processing
            // This ensures orderTotal is ALWAYS set correctly for all events, especially pending
            $total = $order->get_total();
            $currency = $order->get_currency();
            
            // Handle edge case: ensure total is always a valid number (even if 0)
            // Pending orders might have 0 total, but we still need to show the formatted amount (e.g., "$0.00" or "₹0.00")
            if ($total === null || $total === '') {
                $total = 0;
            }
            
            // Ensure total is a valid number (convert to float)
            $total = (float) $total;
            
            // Format the number with proper decimal and thousand separators
            $formatted_number = number_format((float) $total, 2, '.', ',');
            
            // For WhatsApp, use ISO currency codes instead of symbols to avoid encoding issues
            // For Email, use currency symbols (they work fine in HTML emails)
            if ($method === 'whatsapp') {
                // Format with ISO currency code: "INR 2,000.00"
                $formatted_order_total = $currency . ' ' . $formatted_number;
            } else {
                // For Email, use currency symbols
                $symbol = get_woocommerce_currency_symbol($currency); // Gets symbol (may be HTML entity in some cases)
                
                // CRITICAL: Decode HTML entities to ensure we get the raw currency character (₹, $, €, etc.)
                // Some WooCommerce configurations return HTML entities like &#8377; instead of ₹
                $symbol = html_entity_decode($symbol, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                
                // Get currency position from WooCommerce settings (respects user's currency format)
                $currency_position = get_option('woocommerce_currency_pos', 'left');
                
                // Format based on currency position setting to support all currencies
                $formatted_order_total = '';
                switch ($currency_position) {
                    case 'right':
                        $formatted_order_total = $formatted_number . $symbol; // e.g., "100.00 kr"
                        break;
                    case 'left_space':
                        $formatted_order_total = $symbol . ' ' . $formatted_number; // e.g., "$ 100.00"
                        break;
                    case 'right_space':
                        $formatted_order_total = $formatted_number . ' ' . $symbol; // e.g., "100.00 kr"
                        break;
                    case 'left':
                    default:
                        // Default: symbol before amount (most common: $, ₹, €, £, etc.)
                        $formatted_order_total = $symbol . $formatted_number; // e.g., "₹4,000.00" or "$100.00"
                        break;
                }
            }
            
            // FORCE set orderTotal in variables array (overwrite any existing value)
            // This ensures consistency and prevents any issues with array format from extraction
            $variables['orderTotal'] = $formatted_order_total;
            
            // Debug log: Final orderTotal value
            authyo_debug_log(
                sprintf('Final orderTotal for order #%d, method: %s', $order->get_id(), $method),
                array(
                    'order_id' => $order->get_id(),
                    'method' => $method,
                    'orderTotal_value' => $variables['orderTotal'],
                    'orderTotal_type' => gettype($variables['orderTotal']),
                    'orderTotal_length' => strlen($variables['orderTotal']),
                    'orderTotal_hex' => bin2hex($variables['orderTotal']), // For debugging encoding issues
                    'orderTotal_json' => json_encode($variables['orderTotal'], JSON_UNESCAPED_UNICODE)
                ),
                'template'
            );
            
            // Ensure amountPaid is also a string (it should already be, but double-check)
            // For WhatsApp, format with ISO currency codes; for Email, use currency symbols
            if (isset($variables['amountPaid'])) {
                if (is_array($variables['amountPaid']) && isset($variables['amountPaid']['amount'])) {
                    if ($method === 'whatsapp') {
                        // For WhatsApp, format with ISO currency code
                        $currency_code = isset($variables['amountPaid']['currency']) ? $variables['amountPaid']['currency'] : $currency;
                        $amount_str = $variables['amountPaid']['amount'];
                        $numeric_amount = preg_replace('/[^\d.,]/', '', $amount_str);
                        $variables['amountPaid'] = $currency_code . ' ' . $numeric_amount;
                    } else {
                        // For Email, use the formatted amount with currency symbol
                        $variables['amountPaid'] = $variables['amountPaid']['amount'];
                    }
                } elseif ($method === 'whatsapp' && is_string($variables['amountPaid'])) {
                    // If it's already a string for WhatsApp, convert to ISO code format
                    $amount_str = $variables['amountPaid'];
                    $numeric_amount = preg_replace('/[^\d.,]/', '', $amount_str);
                    $variables['amountPaid'] = $currency . ' ' . $numeric_amount;
                }
            }
            
            // Format refundedAmount for WhatsApp with ISO currency codes
            if (isset($variables['refundedAmount']) && $method === 'whatsapp' && is_string($variables['refundedAmount']) && !empty($variables['refundedAmount'])) {
                // Extract numeric amount and format with ISO currency code
                $amount_str = $variables['refundedAmount'];
                $numeric_amount = preg_replace('/[^\d.,]/', '', $amount_str);
                if (!empty($numeric_amount)) {
                    $variables['refundedAmount'] = $currency . ' ' . $numeric_amount;
                }
            }
            
            // CRITICAL: Ensure orderTotal is set BEFORE validation
            // This prevents validation from failing due to missing orderTotal
            // Note: Don't use empty() as it treats "0.00" as empty - use explicit checks
            if (!isset($variables['orderTotal']) || 
                !is_string($variables['orderTotal']) || 
                trim($variables['orderTotal']) === '' ||
                $variables['orderTotal'] === null) {
                $total = $order->get_total();
                $currency = $order->get_currency();
                $symbol = get_woocommerce_currency_symbol($currency);
                // Decode HTML entities to get raw currency character
                $symbol = html_entity_decode($symbol, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                $formatted_number = number_format((float) $total, 2, '.', ',');
                $currency_position = get_option('woocommerce_currency_pos', 'left');
                
                switch ($currency_position) {
                    case 'right':
                        $variables['orderTotal'] = $formatted_number . $symbol;
                        break;
                    case 'left_space':
                        $variables['orderTotal'] = $symbol . ' ' . $formatted_number;
                        break;
                    case 'right_space':
                        $variables['orderTotal'] = $formatted_number . ' ' . $symbol;
                        break;
                    case 'left':
                    default:
                        $variables['orderTotal'] = $symbol . $formatted_number;
                        break;
                }
            }
            
            // For email and WhatsApp, use standard validation
            $required = authyo_get_required_variables($template);
            $validation = authyo_validate_required_variables($variables, $required);
            
            // Debug log: Email/WhatsApp variable validation
            authyo_debug_log(
                sprintf('Validating %s variables for template "%s", order #%d', $method, $template, $order->get_id()),
                array(
                    'method' => $method,
                    'template' => $template,
                    'order_id' => $order->get_id(),
                    'event' => $event,
                    'required_variables' => $required,
                    'available_variables' => array_keys($variables),
                    'orderTotal_before_validation' => isset($variables['orderTotal']) ? $variables['orderTotal'] : 'MISSING',
                    'validation_result' => $validation,
                    'variables' => $variables
                ),
                'template'
            );
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: Required variables for template '{$template}': " . implode(', ', $required));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
                error_log("Authyo Notification: orderTotal before validation: " . (isset($variables['orderTotal']) ? $variables['orderTotal'] : 'MISSING'));
            }
        }
        
        if (!$validation['valid']) {
            // Log warning and stop - do not proceed
            $warning_message = sprintf(
                /* translators: %s: Comma-separated list of missing variable names */
                __('Missing required variables: %s', 'authyo-otp-authentication-for-woocommerce'),
                implode(', ', $validation['missing'])
            );
            Authyo_Notification_Logger::log_warning($order->get_id(), $event, $method, $warning_message);
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: Validation failed - {$warning_message}");
            }
            return;
        }
        
        // Ensure orderTotal is definitely in variables before building payload
        // Use the same direct formatting approach as above to ensure consistency across all events
        // Note: Don't use empty() check as it treats "0.00" as empty - use explicit checks instead
        if (!isset($variables['orderTotal']) || 
            !is_string($variables['orderTotal']) || 
            trim($variables['orderTotal']) === '' ||
            $variables['orderTotal'] === null) {
            // Format directly - use ISO codes for WhatsApp, symbols for Email
            $total = $order->get_total();
            $currency = $order->get_currency();
            $formatted_number = number_format((float) $total, 2, '.', ',');
            
            if ($method === 'whatsapp') {
                // For WhatsApp, use ISO currency code: "INR 2,000.00"
                $variables['orderTotal'] = $currency . ' ' . $formatted_number;
            } else {
                // For Email, use currency symbols
                $symbol = get_woocommerce_currency_symbol($currency);
                // CRITICAL: Decode HTML entities to ensure we get the raw currency character
                // Some WooCommerce configurations return HTML entities like &#8377; instead of ₹
                $symbol = html_entity_decode($symbol, ENT_QUOTES | ENT_HTML5, 'UTF-8');
                $currency_position = get_option('woocommerce_currency_pos', 'left');
                
                switch ($currency_position) {
                    case 'right':
                        $variables['orderTotal'] = $formatted_number . $symbol;
                        break;
                    case 'left_space':
                        $variables['orderTotal'] = $symbol . ' ' . $formatted_number;
                        break;
                    case 'right_space':
                        $variables['orderTotal'] = $formatted_number . ' ' . $symbol;
                        break;
                    case 'left':
                    default:
                        $variables['orderTotal'] = $symbol . $formatted_number;
                        break;
                }
            }
        }
        
        // CRITICAL: Ensure orderTotal is ALWAYS set in variables before building payload
        // This is especially important for pending orders which might have edge cases
        $total = $order->get_total();
        $currency = $order->get_currency();
        $formatted_number = number_format((float) $total, 2, '.', ',');
        
        // For WhatsApp, use ISO currency codes; for Email, use currency symbols
        if ($method === 'whatsapp') {
            // Format with ISO currency code: "INR 2,000.00"
            $formatted_order_total = $currency . ' ' . $formatted_number;
        } else {
            // For Email, use currency symbols
            $symbol = get_woocommerce_currency_symbol($currency); // Raw symbol: ₹, $, €, etc.
            // CRITICAL: Decode HTML entities to ensure we get the raw currency character
            $symbol = html_entity_decode($symbol, ENT_QUOTES | ENT_HTML5, 'UTF-8');
            $currency_position = get_option('woocommerce_currency_pos', 'left');
            
            // Format based on currency position setting (supports all currencies)
            $formatted_order_total = '';
            switch ($currency_position) {
                case 'right':
                    $formatted_order_total = $formatted_number . $symbol; // e.g., "100.00 kr"
                    break;
                case 'left_space':
                    $formatted_order_total = $symbol . ' ' . $formatted_number; // e.g., "$ 100.00"
                    break;
                case 'right_space':
                    $formatted_order_total = $formatted_number . ' ' . $symbol; // e.g., "100.00 kr"
                    break;
                case 'left':
                default:
                    // Default: symbol before amount (most common: $, ₹, €, £, etc.)
                    $formatted_order_total = $symbol . $formatted_number; // e.g., "₹4,000.00" or "$100.00"
                    break;
            }
        }
        
        // Force set orderTotal in variables array (overwrite any existing value to ensure consistency)
        $variables['orderTotal'] = $formatted_order_total;
        
        // Build notification payload
        $payload = $this->build_notification_payload($order, $method, $template, $variables);
        
        // CRITICAL: Force set orderTotal in payload variables (double-check to ensure it's there)
        // This ensures orderTotal is ALWAYS present, even if something went wrong in build_notification_payload
        $payload['variables']['orderTotal'] = $formatted_order_total;
        
        // Debug log: Payload built with detailed info
        // Special attention for pending orders to debug orderTotal issue
        $is_pending = ($event === 'pending');
        authyo_debug_log(
            sprintf('Built notification payload for order #%d, method: %s, template: %s, event: %s', $order->get_id(), $method, $template, $event),
            array(
                'order_id' => $order->get_id(),
                'method' => $method,
                'template' => $template,
                'event' => $event,
                'is_pending_event' => $is_pending,
                'recipient' => $payload['to'],
                'authWay' => $payload['authWay'],
                'templateType' => $payload['templateType'],
                'variables_count' => count($payload['variables']),
                'variables_keys' => array_keys($payload['variables']),
                'orderTotal_in_variables_before_payload' => isset($variables['orderTotal']) ? $variables['orderTotal'] : 'MISSING_IN_VARIABLES',
                'orderTotal_in_payload' => isset($payload['variables']['orderTotal']) ? $payload['variables']['orderTotal'] : 'MISSING_IN_PAYLOAD',
                'orderTotal_type' => isset($payload['variables']['orderTotal']) ? gettype($payload['variables']['orderTotal']) : 'NOT_SET',
                'orderTotal_length' => isset($payload['variables']['orderTotal']) ? strlen($payload['variables']['orderTotal']) : 0,
                'orderTotal_formatted_value' => $formatted_order_total,
                'order_total_raw' => $total,
                'currency' => $currency,
                'currency_symbol' => $symbol,
                'full_payload' => $payload,
                'full_variables' => $variables
            ),
            'api'
        );
        
        // Extra validation for pending orders - ensure orderTotal is definitely present
        // NOTE: If emails show "{orderTotal}}" (with double closing brace), the template on Authyo API side
        // has a typo and needs to be fixed from "wp_pending_order" template in Authyo dashboard.
        // The variable is being sent correctly from this plugin - the issue is in the API template.
        if ($is_pending) {
            // Triple-check that orderTotal is present and is a valid string
            if (!isset($payload['variables']['orderTotal']) || 
                !is_string($payload['variables']['orderTotal']) || 
                trim($payload['variables']['orderTotal']) === '') {
                // Force set it one more time if missing or invalid (shouldn't happen, but safety check)
                $payload['variables']['orderTotal'] = $formatted_order_total;
                authyo_debug_log(
                    sprintf('WARNING: orderTotal was missing or invalid for pending order #%d, force-set to: %s', $order->get_id(), $formatted_order_total),
                    array(
                        'order_id' => $order->get_id(),
                        'event' => $event,
                        'formatted_order_total' => $formatted_order_total,
                        'payload_variables' => $payload['variables'],
                        'orderTotal_was_set' => isset($payload['variables']['orderTotal']),
                        'orderTotal_type' => isset($payload['variables']['orderTotal']) ? gettype($payload['variables']['orderTotal']) : 'NOT_SET',
                        'orderTotal_value' => isset($payload['variables']['orderTotal']) ? $payload['variables']['orderTotal'] : 'MISSING'
                    ),
                    'api'
                );
            } else {
                // Even if orderTotal is present, ensure it's the correctly formatted value
                // This helps catch any edge cases where the value might have been modified
                $payload['variables']['orderTotal'] = $formatted_order_total;
            }
        }
        
        // Validate recipient
        if (empty($payload['to'])) {
            $warning_message = __('Recipient address is missing. Cannot send notification.', 'authyo-otp-authentication-for-woocommerce');
            Authyo_Notification_Logger::log_warning($order->get_id(), $event, $method, $warning_message);
            
            // Debug log: Missing recipient
            authyo_debug_log(
                sprintf('Missing recipient for order #%d, method: %s', $order->get_id(), $method),
                array(
                    'order_id' => $order->get_id(),
                    'method' => $method,
                    'billing_email' => $order->get_billing_email(),
                    'billing_phone' => $order->get_billing_phone()
                ),
                'api'
            );
            
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log("Authyo Notification: Missing recipient for order #{$order->get_id()}, method: {$method}");
                if ($method === 'sms' || $method === 'whatsapp') {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                    error_log("Authyo Notification: Billing phone from order: " . $order->get_billing_phone());
                }
            }
            return;
        }
        
        // Debug log: Recipient validation passed
        authyo_debug_log(
            sprintf('Recipient validated for order #%d, method: %s, recipient: %s', $order->get_id(), $method, $payload['to']),
            array(
                'order_id' => $order->get_id(),
                'method' => $method,
                'recipient' => $payload['to'],
                'recipient_length' => strlen($payload['to'])
            ),
            'api'
        );
        
        // Additional validation for SMS/WhatsApp - ensure phone number is valid
        if (($method === 'sms' || $method === 'whatsapp') && !empty($payload['to'])) {
            // Basic phone validation - should not be empty and should contain digits
            $phone = trim($payload['to']);
            if (empty($phone) || !preg_match('/[0-9]/', $phone)) {
                $warning_message = __('Invalid phone number format. Cannot send SMS/WhatsApp notification.', 'authyo-otp-authentication-for-woocommerce');
                Authyo_Notification_Logger::log_warning($order->get_id(), $event, $method, $warning_message);
                
                // Debug log: Invalid phone format
                authyo_debug_log(
                    sprintf('Invalid phone number format for order #%d, method: %s, phone: %s', $order->get_id(), $method, $phone),
                    array(
                        'order_id' => $order->get_id(),
                        'method' => $method,
                        'phone' => $phone,
                        'billing_phone' => $order->get_billing_phone()
                    ),
                    'api'
                );
                
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                    error_log("Authyo Notification: Invalid phone number format for order #{$order->get_id()}, phone: {$phone}");
                }
                return;
            }
            
            // Debug log: Phone validation passed
            authyo_debug_log(
                sprintf('Phone number validated for order #%d, method: %s, phone: %s', $order->get_id(), $method, $phone),
                array(
                    'order_id' => $order->get_id(),
                    'method' => $method,
                    'phone' => $phone,
                    'phone_length' => strlen($phone)
                ),
                'api'
            );
        }
        
        // Log payload before sending (for debugging)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log("Authyo Notification: Sending notification for order #{$order->get_id()}, method: {$method}, template: {$template}");
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
            error_log("Authyo Notification: Payload: " . print_r($payload, true));
        }
        
        // Send notification via API
        $api = Authyo_Notification_API::get_instance();
        
        // Debug log: API call starting
        authyo_debug_log(
            sprintf('Sending API request for order #%d, method: %s', $order->get_id(), $method),
            array('order_id' => $order->get_id(), 'method' => $method, 'template' => $template, 'payload' => $payload),
            'api'
        );
        
        $response = $api->send_notification($payload);
        
        // Debug log: API response with detailed analysis
        $is_error = is_wp_error($response);
        $response_success = false;
        $response_message = '';
        
        if ($is_error) {
            $response_success = false;
            $response_message = $response->get_error_message();
        } elseif (is_array($response)) {
            $response_success = isset($response['success']) ? $response['success'] : true;
            $response_message = isset($response['message']) ? $response['message'] : 'No message in response';
            
            // Check response data for additional success indicators
            if (isset($response['data']) && is_array($response['data'])) {
                if (isset($response['data']['success']) && $response['data']['success'] === false) {
                    $response_success = false;
                }
                if (isset($response['data']['message'])) {
                    $response_message = $response['data']['message'];
                }
            }
        }
        
        authyo_debug_log(
            sprintf('API response received for order #%d, method: %s, HTTP success: %s, API success: %s', 
                $order->get_id(), 
                $method, 
                $is_error ? 'No' : 'Yes',
                $response_success ? 'Yes' : 'No'
            ),
            array(
                'order_id' => $order->get_id(),
                'method' => $method,
                'template' => $template,
                'is_wp_error' => $is_error,
                'response_success' => $response_success,
                'response_message' => $response_message,
                'full_response' => $is_error ? array(
                    'error_code' => $response->get_error_code(),
                    'error_message' => $response->get_error_message()
                ) : $response
            ),
            'api'
        );
        
        // Log the notification attempt
        Authyo_Notification_Logger::log($order->get_id(), $event, $method, $payload, $response);
        
        // Log response for debugging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r
            error_log("Authyo Notification: API Response: " . print_r($response, true));
        }
        
        // Handle API errors (4xx, 5xx)
        if (is_wp_error($response)) {
            // Error already logged by logger
            return;
        }
        
        // Check for error status codes in response
        if (isset($response['status_code']) && $response['status_code'] >= 400) {
            // Error already logged by logger
            return;
        }
    }
    
    /**
     * Build notification payload
     *
     * @param WC_Order $order Order object
     * @param string $method Notification method
     * @param string $template Template name
     * @param array $variables Template variables
     * @return array Notification payload
     */
    private function build_notification_payload($order, $method, $template, $variables) {
        // Determine recipient based on method
        $to = '';
        if ($method === 'email') {
            $to = $order->get_billing_email();
        } elseif ($method === 'sms' || $method === 'whatsapp') {
            $to = $order->get_billing_phone();
        }
        
        // Map method to authWay (API parameter)
        $auth_way_map = array(
            'email' => 'email',
            'sms' => 'sms',
            'whatsapp' => 'whatsapp',
        );
        
        $auth_way = isset($auth_way_map[$method]) ? $auth_way_map[$method] : $method;
        
        // Build payload
        $payload = array(
            'to' => $to,
            'authWay' => $auth_way,
            'templateType' => $template,
            'variables' => $variables,
        );
        
        return $payload;
    }
    
}
