<?php
/**
 * Notification API Class
 *
 * This class handles API calls for sending notifications via Authyo.io API.
 *
 * @package Authyo_OTP_Auth
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Notification API class
 */
class Authyo_Notification_API {
    
    /**
     * The single instance of the class.
     *
     * @var Authyo_Notification_API
     */
    private static $instance = null;
    
    /**
     * API base URL
     *
     * @var string
     */
    private $base_url = 'https://app.authyo.io/api/v1/';
    
    /**
     * Client ID
     *
     * @var string
     */
    private $client_id;
    
    /**
     * Client Secret
     *
     * @var string
     */
    private $client_secret;
    
    /**
     * Main Instance.
     *
     * @return Authyo_Notification_API
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor.
     */
    private function __construct() {
        $this->client_id = get_option('authyo_otp_auth_client_id', '');
        $this->client_secret = get_option('authyo_otp_auth_client_secret', '');
    }
    
    /**
     * Send notification
     *
     * @param array $payload Notification payload with to, authWay, templateType, variables
     * @return array|WP_Error Response data or error
     */
    public function send_notification($payload) {
        // Debug log: API call initiated
        authyo_debug_log(
            'API send_notification called',
            array('payload' => $payload),
            'api'
        );
        
        // Validate credentials
        if (empty($this->client_id) || empty($this->client_secret)) {
            authyo_debug_log(
                'API credentials not configured',
                array('has_client_id' => !empty($this->client_id), 'has_client_secret' => !empty($this->client_secret)),
                'api'
            );
            return new WP_Error('config_error', __('Authyo.io credentials are not configured.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        // Validate payload
        if (empty($payload['to']) || empty($payload['authWay']) || empty($payload['templateType'])) {
            authyo_debug_log(
                'Invalid payload - missing required fields',
                array('payload' => $payload),
                'api'
            );
            return new WP_Error('invalid_payload', __('Invalid notification payload. Missing required fields.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        // Build API URL
        $api_url = $this->base_url . 'notifications/order';
        
        // Prepare request body
        $body = array(
            'to' => sanitize_text_field($payload['to']),
            'authWay' => sanitize_text_field($payload['authWay']),
            'templateType' => sanitize_text_field($payload['templateType']),
        );
        
        // Add variables if provided
        if (!empty($payload['variables']) && is_array($payload['variables'])) {
            $body['variables'] = $payload['variables'];
            
            // For SMS, currency variables are now formatted with ISO currency codes (e.g., "INR 2,000.00")
            // instead of currency symbols to avoid encoding issues with SMS providers
            // No HTML entity decoding needed since we're using ISO codes
            
            // CRITICAL: For email/WhatsApp, ensure orderTotal is ALWAYS present and properly formatted
            // This is especially important for pending orders
            // NOTE: WhatsApp uses ISO currency codes (e.g., "INR 2,000.00") instead of symbols
            // Email uses currency symbols (e.g., "₹2,000.00") as they work fine in HTML emails
            // NOTE: If emails show "{orderTotal}}" (with double closing brace), check the template
            // on Authyo API side - the "wp_pending_order" template may have a typo.
            if ($payload['authWay'] === 'email' || $payload['authWay'] === 'whatsapp') {
                // Check if orderTotal is missing or invalid
                // Note: Don't use empty() as it treats "0.00" as empty - use explicit checks
                $order_total_missing = !isset($body['variables']['orderTotal']) || 
                                       !is_string($body['variables']['orderTotal']) ||
                                       trim($body['variables']['orderTotal']) === '' ||
                                       $body['variables']['orderTotal'] === null;
                
                if ($order_total_missing) {
                    // Try to get orderTotal from payload variables (fallback)
                    if (isset($payload['variables']['orderTotal']) && 
                        !empty($payload['variables']['orderTotal']) && 
                        is_string($payload['variables']['orderTotal']) &&
                        trim($payload['variables']['orderTotal']) !== '') {
                        $body['variables']['orderTotal'] = trim($payload['variables']['orderTotal']);
                    } else {
                        // Log critical error - orderTotal is completely missing
                        // This should NEVER happen if our code is working correctly
                        authyo_debug_log(
                            'CRITICAL ERROR: orderTotal missing in API payload - this should not happen!',
                            array(
                                'authWay' => $payload['authWay'],
                                'templateType' => $payload['templateType'],
                                'variables_keys' => array_keys($body['variables']),
                                'payload_variables_keys' => isset($payload['variables']) ? array_keys($payload['variables']) : 'NO_VARIABLES',
                                'payload_variables' => isset($payload['variables']) ? $payload['variables'] : 'NO_VARIABLES',
                                'body_variables' => $body['variables'],
                                'full_payload' => $payload
                            ),
                            'api'
                        );
                    }
                } else {
                    // Ensure orderTotal is trimmed and not just whitespace
                    $body['variables']['orderTotal'] = trim($body['variables']['orderTotal']);
                }
                
                // Special check for pending order template - ensure orderTotal is definitely present
                // This helps catch any edge cases where the variable might not be properly set
                if ($payload['templateType'] === 'wp_pending_order') {
                    if (!isset($body['variables']['orderTotal']) || 
                        !is_string($body['variables']['orderTotal']) || 
                        trim($body['variables']['orderTotal']) === '') {
                        // Try to get from payload as fallback
                        if (isset($payload['variables']['orderTotal']) && 
                            is_string($payload['variables']['orderTotal']) &&
                            trim($payload['variables']['orderTotal']) !== '') {
                            $body['variables']['orderTotal'] = trim($payload['variables']['orderTotal']);
                        } else {
                            // Log warning for pending order template
                            authyo_debug_log(
                                'WARNING: orderTotal missing for wp_pending_order template - this may cause template variable replacement to fail',
                                array(
                                    'templateType' => $payload['templateType'],
                                    'authWay' => $payload['authWay'],
                                    'variables_keys' => array_keys($body['variables']),
                                    'payload_variables' => $payload['variables']
                                ),
                                'api'
                            );
                        }
                    }
                }
                
                // Final verification: log orderTotal value that will be sent
                authyo_debug_log(
                    sprintf('Final orderTotal check before API call - authWay: %s, template: %s', $payload['authWay'], $payload['templateType']),
                    array(
                        'authWay' => $payload['authWay'],
                        'templateType' => $payload['templateType'],
                        'orderTotal_present' => isset($body['variables']['orderTotal']),
                        'orderTotal_value' => isset($body['variables']['orderTotal']) ? $body['variables']['orderTotal'] : 'MISSING',
                        'orderTotal_type' => isset($body['variables']['orderTotal']) ? gettype($body['variables']['orderTotal']) : 'NOT_SET',
                        'orderTotal_length' => isset($body['variables']['orderTotal']) ? strlen($body['variables']['orderTotal']) : 0,
                        'orderTotal_is_empty' => isset($body['variables']['orderTotal']) ? empty($body['variables']['orderTotal']) : true,
                        'all_variables_keys' => array_keys($body['variables']),
                        'all_variables' => $body['variables']
                    ),
                    'api'
                );
            }
        }
        
        // For email and WhatsApp notifications, use JSON_UNESCAPED_UNICODE to send actual currency symbols
        // This ensures currency symbols like ₹, $, €, £, etc. are sent directly instead of Unicode escapes
        $is_email_or_whatsapp = (isset($payload['authWay']) && ($payload['authWay'] === 'email' || $payload['authWay'] === 'whatsapp'));
        
        if ($is_email_or_whatsapp) {
            // Use json_encode with JSON_UNESCAPED_UNICODE flag for email/WhatsApp to preserve currency symbols
            $json_body = json_encode($body, JSON_UNESCAPED_UNICODE);
            
            // Check for JSON encoding errors
            if ($json_body === false && json_last_error() !== JSON_ERROR_NONE) {
                // Fallback to wp_json_encode if json_encode fails
                $json_body = wp_json_encode($body);
            }
        } else {
            // For SMS, use default wp_json_encode (escapes Unicode)
            $json_body = wp_json_encode($body);
        }
        
        // Prepare request arguments
        $args = array(
            'method' => 'POST',
            'headers' => array(
                'Content-Type' => 'application/json',
                'clientId' => $this->client_id,
                'clientSecret' => $this->client_secret,
            ),
            'body' => $json_body,
            'timeout' => 30,
            'sslverify' => true,
        );
        
        // Debug log: HTTP request details with full body
        authyo_debug_log(
            sprintf('Making HTTP POST request to: %s', $api_url),
            array(
                'url' => $api_url,
                'method' => $payload['authWay'],
                'template' => $payload['templateType'],
                'recipient' => $payload['to'],
                'request_body' => $body,
                'json_body' => $json_body
            ),
            'api'
        );
        
        // Send request
        $response = wp_remote_post($api_url, $args);
        
        // Handle errors
        if (is_wp_error($response)) {
            authyo_debug_log(
                sprintf('HTTP request failed: %s', $response->get_error_message()),
                array('error_code' => $response->get_error_code(), 'error_message' => $response->get_error_message()),
                'api'
            );
            return $response;
        }
        
        // Get response code and body
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        $response_data = json_decode($response_body, true);
        
        // Debug log: HTTP response received with full details
        authyo_debug_log(
            sprintf('HTTP response received: Status %d', $response_code),
            array(
                'status_code' => $response_code,
                'response_body' => $response_body,
                'response_data' => $response_data,
                'request_recipient' => isset($payload['to']) ? $payload['to'] : 'unknown',
                'request_method' => isset($payload['authWay']) ? $payload['authWay'] : 'unknown',
                'request_template' => isset($payload['templateType']) ? $payload['templateType'] : 'unknown'
            ),
            'api'
        );
        
        // Check for JSON decode errors
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('json_error', __('Invalid JSON response from Authyo.io API.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        // Handle error responses (4xx, 5xx)
        if ($response_code >= 400) {
            $error_message = __('API request failed.', 'authyo-otp-authentication-for-woocommerce');
            
            // Try to extract error message from response
            if (isset($response_data['message'])) {
                $error_message = $response_data['message'];
            } elseif (isset($response_data['error'])) {
                $error_message = $response_data['error'];
            } elseif (isset($response_data['data']['message'])) {
                $error_message = $response_data['data']['message'];
            } elseif (isset($response_data['data']['error'])) {
                $error_message = $response_data['data']['error'];
            } else {
                /* translators: %d: HTTP status code */
                $error_message = sprintf(__('HTTP %d error occurred', 'authyo-otp-authentication-for-woocommerce'), $response_code);
            }
            
            return new WP_Error('api_error', $error_message, array(
                'status_code' => $response_code,
                'response_data' => $response_data,
            ));
        }
        
        // Check if response indicates failure even with 200 status
        $api_success = true;
        $api_message = __('Notification sent successfully.', 'authyo-otp-authentication-for-woocommerce');
        
        // Check for success/error indicators in response
        if (isset($response_data['success']) && $response_data['success'] === false) {
            $api_success = false;
            $api_message = isset($response_data['message']) ? $response_data['message'] : __('API returned failure status.', 'authyo-otp-authentication-for-woocommerce');
        } elseif (isset($response_data['status']) && strtolower($response_data['status']) === 'error') {
            $api_success = false;
            $api_message = isset($response_data['message']) ? $response_data['message'] : __('API returned error status.', 'authyo-otp-authentication-for-woocommerce');
        } elseif (isset($response_data['error'])) {
            $api_success = false;
            $api_message = is_string($response_data['error']) ? $response_data['error'] : __('API returned an error.', 'authyo-otp-authentication-for-woocommerce');
        } elseif (isset($response_data['message'])) {
            $api_message = $response_data['message'];
        }
        
        // Debug log: API success status check - ALWAYS log this for debugging
        authyo_debug_log(
            sprintf('API response analysis: HTTP %d, API Success: %s, Message: %s', $response_code, $api_success ? 'Yes' : 'No', $api_message),
            array(
                'status_code' => $response_code,
                'api_success' => $api_success,
                'api_message' => $api_message,
                'response_data_keys' => is_array($response_data) ? array_keys($response_data) : 'not_array',
                'response_data' => $response_data,
                'has_success_field' => isset($response_data['success']),
                'has_status_field' => isset($response_data['status']),
                'has_error_field' => isset($response_data['error']),
                'has_message_field' => isset($response_data['message']),
            ),
            'api'
        );
        
        // If API indicates failure, return as error
        if (!$api_success) {
            return new WP_Error('api_response_error', $api_message, array(
                'status_code' => $response_code,
                'response_data' => $response_data,
            ));
        }
        
        // Success response
        return array(
            'success' => $api_success,
            'status_code' => $response_code,
            'data' => $response_data,
            'message' => $api_message,
        );
    }
}
