/**
 * Notification Flow Admin JavaScript
 *
 * @package Authyo_OTP_Auth
 * @since 1.0.0
 */

(function($) {
    'use strict';

    $(document).ready(function() {
        const $addFlowBtn = $('#authyo-add-flow-btn');
        const $popup = $('#authyo-add-flow-popup');
        const $closePopup = $('#authyo-close-popup');
        const $cancelBtn = $('#authyo-cancel-flow-btn');
        const $form = $('#authyo-add-flow-form');
        const $saveBtn = $('#authyo-save-flow-btn');
        
        // Open popup for adding new flow
        $addFlowBtn.on('click', function() {
            resetPopupToAddMode();
            $popup.fadeIn(200);
        });
        
        // Open popup for editing existing flow
        $(document).on('click', '.authyo-edit-flow-btn', function(e) {
            e.preventDefault();
            const event = $(this).data('event');
            openEditPopup(event);
        });
        
        // Function to reset popup to add mode
        function resetPopupToAddMode() {
            $form[0].reset();
            $('#authyo-flow-mode').val('add');
            $('#authyo-flow-edit-event').val('');
            $('#authyo-popup-title').text('Add Notification Flow');
            $('#authyo-save-btn-text').text('Save Flow');
            $('.authyo-event-checkbox').prop('disabled', false);
            
            // Reset accordion state - hide all method sections
            $('.authyo-event-accordion-content').hide();
            $('.authyo-event-accordion-item').removeClass('active');
            
            // Show all event items
            $('.authyo-event-accordion-item').show();
            
            // Reset select all buttons
            $('#authyo-select-all-events-btn').text('Select All Events');
            $('.authyo-select-all-methods-container').hide();
            $('#authyo-select-all-methods-btn').text('Select All Notification Methods');
        }
        
        // Handle event checkbox change - expand/collapse methods
        $(document).on('change', '.authyo-event-checkbox', function() {
            const $checkbox = $(this);
            const event = $checkbox.data('event');
            const isChecked = $checkbox.is(':checked');
            const $accordionItem = $checkbox.closest('.authyo-event-accordion-item');
            const $accordionContent = $accordionItem.find('.authyo-event-accordion-content');
            
            if (isChecked) {
                // Expand the methods section
                $accordionItem.addClass('active');
                $accordionContent.slideDown(200);
            } else {
                // Collapse the methods section
                $accordionItem.removeClass('active');
                $accordionContent.slideUp(200);
                // Uncheck all methods for this event
                $accordionContent.find('.authyo-method-checkbox').prop('checked', false);
            }
            
            // Update "Select All Methods" button visibility and "Select All Events" button text
            updateSelectAllMethodsVisibility();
            updateSelectAllEventsButtonText();
        });
        
        // Function to update "Select All Events" button text
        function updateSelectAllEventsButtonText() {
            const totalEvents = $('.authyo-event-checkbox:not(:disabled)').length;
            const checkedEvents = $('.authyo-event-checkbox:not(:disabled):checked').length;
            
            if (checkedEvents === totalEvents && totalEvents > 0) {
                $('#authyo-select-all-events-btn').text('Deselect All Events');
            } else {
                $('#authyo-select-all-events-btn').text('Select All Events');
            }
        }
        
        // Select All Events button handler
        $('#authyo-select-all-events-btn').on('click', function() {
            const $btn = $(this);
            const allChecked = $('.authyo-event-checkbox:not(:disabled)').length === $('.authyo-event-checkbox:not(:disabled):checked').length;
            
            if (allChecked) {
                // Uncheck all events
                $('.authyo-event-checkbox:not(:disabled)').prop('checked', false).trigger('change');
                $btn.text('Select All Events');
            } else {
                // Check all events
                $('.authyo-event-checkbox:not(:disabled)').prop('checked', true).trigger('change');
                $btn.text('Deselect All Events');
            }
        });
        
        // Select All Methods button handler
        $('#authyo-select-all-methods-btn').on('click', function() {
            const $btn = $(this);
            const $checkedEvents = $('.authyo-event-checkbox:checked:not(:disabled)');
            
            if ($checkedEvents.length === 0) {
                return;
            }
            
            // Check if all methods are already selected for all checked events
            let allMethodsSelected = true;
            $checkedEvents.each(function() {
                const event = $(this).data('event');
                const $methods = $('.authyo-method-checkbox[data-event="' + event + '"]');
                const checkedCount = $methods.filter(':checked').length;
                if (checkedCount !== $methods.length) {
                    allMethodsSelected = false;
                    return false;
                }
            });
            
            if (allMethodsSelected) {
                // Uncheck all methods for all checked events
                $checkedEvents.each(function() {
                    const event = $(this).data('event');
                    $('.authyo-method-checkbox[data-event="' + event + '"]').prop('checked', false);
                });
                $btn.text('Select All Notification Methods');
            } else {
                // Check all methods for all checked events
                $checkedEvents.each(function() {
                    const event = $(this).data('event');
                    $('.authyo-method-checkbox[data-event="' + event + '"]').prop('checked', true);
                });
                $btn.text('Deselect All Notification Methods');
            }
            
            // Update button text after state change
            updateSelectAllMethodsVisibility();
        });
        
        // Update button text when methods are manually changed
        $(document).on('change', '.authyo-method-checkbox', function() {
            updateSelectAllMethodsVisibility();
        });
        
        // Initialize button states after all functions are defined
        updateSelectAllEventsButtonText();
        
        // Function to update "Select All Methods" button visibility
        function updateSelectAllMethodsVisibility() {
            const $checkedEvents = $('.authyo-event-checkbox:checked:not(:disabled)');
            const $selectAllMethodsContainer = $('.authyo-select-all-methods-container');
            
            if ($checkedEvents.length > 0) {
                $selectAllMethodsContainer.slideDown(200);
            } else {
                $selectAllMethodsContainer.slideUp(200);
            }
            
            // Update button text based on current state
            if ($checkedEvents.length > 0) {
                let allMethodsSelected = true;
                $checkedEvents.each(function() {
                    const event = $(this).data('event');
                    const $methods = $('.authyo-method-checkbox[data-event="' + event + '"]');
                    const checkedCount = $methods.filter(':checked').length;
                    if (checkedCount !== $methods.length) {
                        allMethodsSelected = false;
                        return false;
                    }
                });
                
                if (allMethodsSelected) {
                    $('#authyo-select-all-methods-btn').text('Deselect All Notification Methods');
                } else {
                    $('#authyo-select-all-methods-btn').text('Select All Notification Methods');
                }
            }
        }
        
        // Function to open popup in edit mode
        function openEditPopup(event) {
            // Get flow data via AJAX
            $.ajax({
                url: authyoNotificationFlow.ajaxurl,
                type: 'POST',
                data: {
                    action: 'authyo_get_notification_flows',
                    nonce: authyoNotificationFlow.nonce
                },
                success: function(response) {
                    if (response.success && response.data.flows && response.data.flows[event]) {
                        const flow = response.data.flows[event];
                        
                        // Set form to edit mode
                        $('#authyo-flow-mode').val('edit');
                        $('#authyo-flow-edit-event').val(event);
                        $('#authyo-popup-title').text('Edit Notification Flow');
                        $('#authyo-save-btn-text').text('Update Flow');
                        
                        // Hide all event items except the one being edited
                        $('.authyo-event-accordion-item').hide();
                        $('.authyo-event-accordion-item[data-event="' + event + '"]').show();
                        
                        // Clear and set event checkbox (only the one being edited, disabled in edit mode)
                        $('.authyo-event-checkbox').prop('checked', false).prop('disabled', false);
                        const $eventCheckbox = $('.authyo-event-checkbox[value="' + event + '"]');
                        $eventCheckbox.prop('checked', true).prop('disabled', true);
                        
                        // Expand the accordion for the event being edited
                        const $accordionItem = $eventCheckbox.closest('.authyo-event-accordion-item');
                        const $accordionContent = $accordionItem.find('.authyo-event-accordion-content');
                        $accordionItem.addClass('active');
                        $accordionContent.show();
                        
                        // Clear and set method checkboxes for this specific event
                        $accordionContent.find('.authyo-method-checkbox').prop('checked', false);
                        if (flow.methods && Array.isArray(flow.methods)) {
                            flow.methods.forEach(function(method) {
                                $accordionContent.find('.authyo-method-checkbox[value="' + method + '"]').prop('checked', true);
                            });
                        }
                        
                        $popup.fadeIn(200);
                    }
                },
                error: function() {
                    showNotice('Failed to load flow data.', 'error');
                }
            });
        }
        
        // Close popup
        function closePopup() {
            $popup.fadeOut(200);
            resetPopupToAddMode();
        }
        
        $closePopup.on('click', closePopup);
        $cancelBtn.on('click', closePopup);
        
        // Close on overlay click
        $popup.on('click', function(e) {
            if ($(e.target).is('.authyo-popup-overlay')) {
                closePopup();
            }
        });
        
        // Close on ESC key
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape' && $popup.is(':visible')) {
                closePopup();
            }
        });
        
        // Form submission
        $form.on('submit', function(e) {
            e.preventDefault();
            
            const mode = $('#authyo-flow-mode').val();
            const events = [];
            const eventMethodsMap = {};
            
            // Collect all checked events and their methods
            $('.authyo-event-checkbox:checked').each(function() {
                const event = $(this).val();
                events.push(event);
                
                // Collect methods for this specific event
                const methods = [];
                $(this).closest('.authyo-event-accordion-item').find('.authyo-method-checkbox:checked').each(function() {
                    methods.push($(this).val());
                });
                eventMethodsMap[event] = methods;
            });
            
            // Validate
            if (events.length === 0) {
                alert(authyoNotificationFlow.messages.save_error || 'Please select at least one event.');
                return;
            }
            
            // Validate that each selected event has at least one method
            let hasInvalidEvent = false;
            events.forEach(function(event) {
                if (!eventMethodsMap[event] || eventMethodsMap[event].length === 0) {
                    hasInvalidEvent = true;
                }
            });
            
            if (hasInvalidEvent) {
                alert(authyoNotificationFlow.messages.save_error || 'Please select at least one notification method for each selected event.');
                return;
            }
            
            // Disable submit button
            const originalBtnText = $('#authyo-save-btn-text').text();
            $saveBtn.prop('disabled', true);
            $('#authyo-save-btn-text').text(mode === 'edit' ? 'Updating...' : 'Saving...');
            
            // In edit mode, we only update the single event being edited
            if (mode === 'edit') {
                const editEvent = $('#authyo-flow-edit-event').val();
                if (!editEvent) {
                    showNotice('Invalid edit mode.', 'error');
                    $saveBtn.prop('disabled', false);
                    $('#authyo-save-btn-text').text(originalBtnText);
                    return;
                }
                
                const methods = eventMethodsMap[editEvent] || [];
                
                // Update single flow
                $.ajax({
                    url: authyoNotificationFlow.ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'authyo_update_notification_flow',
                        nonce: authyoNotificationFlow.nonce,
                        event: editEvent,
                        methods: methods
                    },
                    success: function(response) {
                        if (response.success) {
                            showNotice(response.data.message, 'success');
                            closePopup();
                            location.reload();
                        } else {
                            showNotice(response.data.message || authyoNotificationFlow.messages.save_error, 'error');
                            $saveBtn.prop('disabled', false);
                            $('#authyo-save-btn-text').text(originalBtnText);
                        }
                    },
                    error: function() {
                        showNotice(authyoNotificationFlow.messages.save_error, 'error');
                        $saveBtn.prop('disabled', false);
                        $('#authyo-save-btn-text').text(originalBtnText);
                    }
                });
            } else {
                // Add mode: create flows for each selected event with its specific methods
                let completed = 0;
                let hasError = false;
                const totalEvents = events.length;
                
                // Save flows for each event
                events.forEach(function(event) {
                    const methods = eventMethodsMap[event] || [];
                    
                    $.ajax({
                        url: authyoNotificationFlow.ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'authyo_save_notification_flow',
                            nonce: authyoNotificationFlow.nonce,
                            event: event,
                            methods: methods
                        },
                        success: function(response) {
                            completed++;
                            if (!response.success) {
                                hasError = true;
                            }
                            
                            // When all requests complete
                            if (completed === totalEvents) {
                                if (hasError) {
                                    showNotice('Some flows could not be saved. Please try again.', 'error');
                                } else {
                                    showNotice('Notification flows saved successfully!', 'success');
                                    closePopup();
                                    location.reload();
                                }
                                $saveBtn.prop('disabled', false);
                                $('#authyo-save-btn-text').text(originalBtnText);
                            }
                        },
                        error: function() {
                            completed++;
                            hasError = true;
                            
                            if (completed === totalEvents) {
                                showNotice('Error saving flows. Please try again.', 'error');
                                $saveBtn.prop('disabled', false);
                                $('#authyo-save-btn-text').text(originalBtnText);
                            }
                        }
                    });
                });
            }
        });
        
        // Delete flow
        $(document).on('click', '.authyo-delete-flow-btn', function(e) {
            e.preventDefault();
            
            if (!confirm(authyoNotificationFlow.messages.confirm_delete)) {
                return;
            }
            
            const $btn = $(this);
            const event = $btn.data('event');
            
            // Disable button
            $btn.prop('disabled', true);
            
            // Send AJAX request
            $.ajax({
                url: authyoNotificationFlow.ajaxurl,
                type: 'POST',
                data: {
                    action: 'authyo_delete_notification_flow',
                    nonce: authyoNotificationFlow.nonce,
                    flow_id: event,
                    event: event
                },
                success: function(response) {
                    if (response.success) {
                        // Show success message
                        showNotice(response.data.message, 'success');
                        
                        // Remove flow item from DOM
                        $btn.closest('.authyo-flow-item').fadeOut(300, function() {
                            $(this).remove();
                            
                            // Update event indicator
                            $('.authyo-event-item[data-event="' + event + '"]').removeClass('has-flow').find('.authyo-flow-indicator').remove();
                            
                            // Reload if no flows left
                            if ($('.authyo-flow-item').length === 0) {
                                location.reload();
                            }
                        });
                    } else {
                        showNotice(response.data.message || authyoNotificationFlow.messages.delete_error, 'error');
                        $btn.prop('disabled', false);
                    }
                },
                error: function() {
                    showNotice(authyoNotificationFlow.messages.delete_error, 'error');
                    $btn.prop('disabled', false);
                }
            });
        });
        
        // Show notice
        function showNotice(message, type) {
            const noticeClass = type === 'success' ? 'notice-success' : 'notice-error';
            const $notice = $('<div class="notice ' + noticeClass + ' is-dismissible"><p>' + message + '</p></div>');
            
            $('.authyo-notification-flow-wrap > h1').after($notice);
            
            // Auto-dismiss after 3 seconds
            setTimeout(function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 3000);
            
            // Dismiss on click
            $notice.on('click', '.notice-dismiss', function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            });
        }
        
        // Event item click - filter flows by event
        $(document).on('click', '.authyo-event-item', function() {
            const event = $(this).data('event');
            
            // Update active state
            $('.authyo-event-item').removeClass('active');
            $(this).addClass('active');
            
            // Update panel title
            const eventName = $(this).find('.authyo-event-name').text();
            $('#authyo-panel-title').text(eventName + ' - Notification Templates');
            
            // Show/hide flows based on event
            $('.authyo-flow-item').each(function() {
                const flowEvent = $(this).data('event');
                if (flowEvent === event) {
                    $(this).fadeIn(200);
                } else {
                    $(this).fadeOut(200);
                }
            });
        });
        
        // Show all flows by default (no filter)
        $('.authyo-flow-item').show();
        
        // Delete All Flows button handler
        $('#authyo-delete-all-flows-btn').on('click', function(e) {
            e.preventDefault();
            
            if (!confirm(authyoNotificationFlow.messages.delete_all_confirm || 'Are you sure you want to delete ALL notification flows? This action cannot be undone.')) {
                return;
            }
            
            const $btn = $(this);
            const originalText = $btn.html();
            $btn.prop('disabled', true).html('<span class="dashicons dashicons-update"></span> ' + (authyoNotificationFlow.messages.deleting || 'Deleting...'));
            
            $.ajax({
                url: authyoNotificationFlow.ajaxurl,
                type: 'POST',
                data: {
                    action: 'authyo_delete_all_notification_flows',
                    nonce: authyoNotificationFlow.nonce
                },
                success: function(response) {
                    if (response.success) {
                        showNotice(response.data.message || 'All flows deleted successfully!', 'success');
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        showNotice(response.data.message || 'Failed to delete all flows.', 'error');
                        $btn.prop('disabled', false).html(originalText);
                    }
                },
                error: function() {
                    showNotice('Failed to delete all flows.', 'error');
                    $btn.prop('disabled', false).html(originalText);
                }
            });
        });
        
        // Edit All Flows button handler
        $('#authyo-edit-all-flows-btn').on('click', function(e) {
            e.preventDefault();
            const $editAllPopup = $('#authyo-edit-all-flows-popup');
            $editAllPopup.fadeIn(200);
            
            // Initialize accordion state based on existing flows
            $('.authyo-event-checkbox-edit-all').each(function() {
                const $checkbox = $(this);
                const event = $checkbox.data('event');
                const isChecked = $checkbox.is(':checked');
                const $accordionItem = $checkbox.closest('.authyo-event-accordion-item');
                const $accordionContent = $accordionItem.find('.authyo-event-accordion-content');
                
                if (isChecked) {
                    $accordionItem.addClass('active');
                    $accordionContent.show();
                } else {
                    $accordionItem.removeClass('active');
                    $accordionContent.hide();
                }
            });
            
            // Update select all buttons
            updateSelectAllEventsEditAll();
            updateSelectAllMethodsEditAllVisibility();
        });
        
        // Close Edit All Flows popup
        $('#authyo-close-edit-all-popup, #authyo-cancel-edit-all-btn').on('click', function() {
            $('#authyo-edit-all-flows-popup').fadeOut(200);
        });
        
        // Close Edit All popup on overlay click
        $('#authyo-edit-all-flows-popup').on('click', function(e) {
            if ($(e.target).is('.authyo-popup-overlay')) {
                $(this).fadeOut(200);
            }
        });
        
        // Close Edit All popup on ESC key
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape' && $('#authyo-edit-all-flows-popup').is(':visible')) {
                $('#authyo-edit-all-flows-popup').fadeOut(200);
            }
        });
        
        // Handle event checkbox change in Edit All popup
        $(document).on('change', '.authyo-event-checkbox-edit-all', function() {
            const $checkbox = $(this);
            const isChecked = $checkbox.is(':checked');
            const $accordionItem = $checkbox.closest('.authyo-event-accordion-item');
            const $accordionContent = $accordionItem.find('.authyo-event-accordion-content');
            
            if (isChecked) {
                $accordionItem.addClass('active');
                $accordionContent.slideDown(200);
            } else {
                $accordionItem.removeClass('active');
                $accordionContent.slideUp(200);
                $accordionContent.find('.authyo-method-checkbox-edit-all').prop('checked', false);
            }
            
            updateSelectAllEventsEditAll();
            updateSelectAllMethodsEditAllVisibility();
        });
        
        // Select All Events button in Edit All popup
        $('#authyo-select-all-events-edit-all-btn').on('click', function() {
            const allChecked = $('.authyo-event-checkbox-edit-all:not(:disabled)').length === $('.authyo-event-checkbox-edit-all:not(:disabled):checked').length;
            
            if (allChecked) {
                $('.authyo-event-checkbox-edit-all:not(:disabled)').prop('checked', false).trigger('change');
            } else {
                $('.authyo-event-checkbox-edit-all:not(:disabled)').prop('checked', true).trigger('change');
            }
        });
        
        // Select All Methods button in Edit All popup
        $('#authyo-select-all-methods-edit-all-btn').on('click', function() {
            const allChecked = $('.authyo-method-checkbox-edit-all:not(:disabled)').length === $('.authyo-method-checkbox-edit-all:not(:disabled):checked').length;
            
            if (allChecked) {
                $('.authyo-method-checkbox-edit-all:not(:disabled)').prop('checked', false);
            } else {
                $('.authyo-method-checkbox-edit-all:not(:disabled)').prop('checked', true);
            }
            
            updateSelectAllMethodsEditAllButtonText();
        });
        
        // Update Select All Events button text in Edit All popup
        function updateSelectAllEventsEditAll() {
            const totalEvents = $('.authyo-event-checkbox-edit-all:not(:disabled)').length;
            const checkedEvents = $('.authyo-event-checkbox-edit-all:not(:disabled):checked').length;
            
            if (checkedEvents === totalEvents && totalEvents > 0) {
                $('#authyo-select-all-events-edit-all-btn').text('Deselect All Events');
            } else {
                $('#authyo-select-all-events-edit-all-btn').text('Select All Events');
            }
        }
        
        // Update Select All Methods visibility and button text in Edit All popup
        function updateSelectAllMethodsEditAllVisibility() {
            const checkedEvents = $('.authyo-event-checkbox-edit-all:checked').length;
            const $container = $('.authyo-select-all-methods-container');
            
            if (checkedEvents > 0) {
                $container.show();
            } else {
                $container.hide();
            }
            
            updateSelectAllMethodsEditAllButtonText();
        }
        
        function updateSelectAllMethodsEditAllButtonText() {
            const totalMethods = $('.authyo-method-checkbox-edit-all:not(:disabled)').length;
            const checkedMethods = $('.authyo-method-checkbox-edit-all:not(:disabled):checked').length;
            
            if (checkedMethods === totalMethods && totalMethods > 0) {
                $('#authyo-select-all-methods-edit-all-btn').text('Deselect All Notification Methods');
            } else {
                $('#authyo-select-all-methods-edit-all-btn').text('Select All Notification Methods');
            }
        }
        
        // Form submission for Edit All Flows
        $('#authyo-edit-all-flows-form').on('submit', function(e) {
            e.preventDefault();
            
            const events = [];
            const eventMethodsMap = {};
            
            // Collect all checked events and their methods
            $('.authyo-event-checkbox-edit-all:checked').each(function() {
                const event = $(this).val();
                events.push(event);
                
                // Collect methods for this specific event
                const methods = [];
                $(this).closest('.authyo-event-accordion-item').find('.authyo-method-checkbox-edit-all:checked').each(function() {
                    methods.push($(this).val());
                });
                eventMethodsMap[event] = methods;
            });
            
            // Validate that each selected event has at least one method
            let hasInvalidEvent = false;
            events.forEach(function(event) {
                if (!eventMethodsMap[event] || eventMethodsMap[event].length === 0) {
                    hasInvalidEvent = true;
                }
            });
            
            if (hasInvalidEvent) {
                alert('Please select at least one notification method for each selected event.');
                return;
            }
            
            // Disable submit button
            const $saveBtn = $('#authyo-save-all-flows-btn');
            const originalBtnText = $saveBtn.text();
            $saveBtn.prop('disabled', true).text('Saving...');
            
            // Prepare form data
            const formData = {
                action: 'authyo_update_all_notification_flows',
                nonce: authyoNotificationFlow.nonce,
                events: events,
                methods: eventMethodsMap
            };
            
            // Send AJAX request
            $.ajax({
                url: authyoNotificationFlow.ajaxurl,
                type: 'POST',
                data: formData,
                success: function(response) {
                    if (response.success) {
                        showNotice(response.data.message || 'All flows updated successfully!', 'success');
                        $('#authyo-edit-all-flows-popup').fadeOut(200);
                        setTimeout(function() {
                            location.reload();
                        }, 1000);
                    } else {
                        showNotice(response.data.message || 'Failed to update flows.', 'error');
                        $saveBtn.prop('disabled', false).text(originalBtnText);
                    }
                },
                error: function() {
                    showNotice('Failed to update flows.', 'error');
                    $saveBtn.prop('disabled', false).text(originalBtnText);
                }
            });
        });
        
        // Toggle switch handler
        $(document).on('change', '.authyo-flow-toggle', function() {
            const $toggle = $(this);
            const event = $toggle.data('event');
            const isEnabled = $toggle.is(':checked');
            const $flowItem = $toggle.closest('.authyo-flow-item');
            
            // Disable toggle during request
            $toggle.prop('disabled', true);
            
            // Send AJAX request
            $.ajax({
                url: authyoNotificationFlow.ajaxurl,
                type: 'POST',
                data: {
                    action: 'authyo_toggle_notification_flow',
                    nonce: authyoNotificationFlow.nonce,
                    event: event,
                    enabled: isEnabled ? 1 : 0
                },
                success: function(response) {
                    if (response.success) {
                        // Update UI
                        if (isEnabled) {
                            $flowItem.removeClass('authyo-flow-disabled');
                        } else {
                            $flowItem.addClass('authyo-flow-disabled');
                        }
                        
                        // Show brief success message
                        const message = isEnabled 
                            ? (authyoNotificationFlow.messages.flow_enabled || 'Notification flow enabled')
                            : (authyoNotificationFlow.messages.flow_disabled || 'Notification flow disabled');
                        showNotice(message, 'success');
                    } else {
                        // Revert toggle on error
                        $toggle.prop('checked', !isEnabled);
                        showNotice(response.data.message || (authyoNotificationFlow.messages.toggle_error || 'Failed to update flow status.'), 'error');
                    }
                },
                error: function() {
                    // Revert toggle on error
                    $toggle.prop('checked', !isEnabled);
                    showNotice(authyoNotificationFlow.messages.toggle_error || 'Failed to update flow status.', 'error');
                },
                complete: function() {
                    // Re-enable toggle
                    $toggle.prop('disabled', false);
                }
            });
        });
    });
    
})(jQuery);