<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * The core plugin class
 */
class Authyo_OTP_Auth {
    protected $loader;
    protected $plugin_name;
    protected $version;
    protected $debug;

    public function __construct() {
        $this->plugin_name = 'authyo-otp-authentication-for-woocommerce';
        $this->version = AUTHYO_OTP_AUTH_VERSION;
        
        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    private function load_dependencies() {
        // Development logs removed for production
        require plugin_dir_path(__FILE__) . 'class-authyo-otp-auth-loader.php';

        require plugin_dir_path(__FILE__) . 'class-authyo-otp-auth-i18n.php';

        require plugin_dir_path(__FILE__) . 'class-authyo-otp-auth-admin.php';

        require plugin_dir_path(__FILE__) . 'class-authyo-otp-auth-public.php';

        require plugin_dir_path(__FILE__) . 'class-authyo-otp-auth-docs.php';

        require plugin_dir_path(__FILE__) . 'class-authyo-otp-auth-deactivation-feedback.php';

        require plugin_dir_path(__FILE__) . 'class-authyo-debug-logger.php';

        // Initialize loader after including its class
        if ( class_exists( 'Authyo_OTP_Auth_Loader' ) ) {
            $this->loader = new Authyo_OTP_Auth_Loader();
        }

        // Ensure public-facing hooks (shortcodes, AJAX) are registered
        if ( class_exists( 'Authyo_OTP_Auth_Public' ) ) {
            Authyo_OTP_Auth_Public::get_instance();
        }

        // Initialize checkout OTP functionality if WooCommerce is active
        if ( class_exists( 'WooCommerce' ) ) {
            require plugin_dir_path(__FILE__) . 'class-authyo-otp-auth-checkout.php';
            if ( class_exists( 'Authyo_OTP_Auth_Checkout' ) ) {
                Authyo_OTP_Auth_Checkout::get_instance();
            }
        }

        // Initialize Notification Flow module
        require_once AUTHYO_OTP_AUTH_PLUGIN_DIR . 'modules/notification-flow/class-notification-flow.php';
        if ( class_exists( 'Authyo_Notification_Flow' ) ) {
            Authyo_Notification_Flow::get_instance();
        }
    }

    private function set_locale() {
        // WordPress 4.6+ automatically loads translations for plugins
        // hosted on WordPress.org. Manual loading is no longer required.
        // 
        // For development or custom installations, you can manually load:
        // add_action('plugins_loaded', array($this, 'load_plugin_textdomain'));
    }

    public function load_plugin_textdomain() {
        // WordPress 4.6+ automatically loads translations for plugins
        // hosted on WordPress.org. No manual loading is required.
        // 
        // For development or custom installations, uncomment the following:
        /*
        load_plugin_textdomain(
            'authyo-otp-authentication-for-woocommerce',
            false,
            dirname(dirname(plugin_basename(__FILE__))) . '/languages/'
        );
        */
    }

    private function define_admin_hooks() {
        $plugin_admin = Authyo_OTP_Auth_Admin::get_instance();

        // Add settings link to plugin page
        $this->loader->add_filter(
            'plugin_action_links_' . AUTHYO_OTP_AUTH_PLUGIN_BASENAME,
            $plugin_admin,
            'add_action_links'
        );

        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_admin_scripts');
        $this->loader->add_action('admin_menu', $plugin_admin, 'add_plugin_menu');
        $this->loader->add_action('admin_init', $plugin_admin, 'register_settings');

        // Initialize docs class
        if (class_exists('Authyo_OTP_Auth_Docs')) {
            Authyo_OTP_Auth_Docs::get_instance();
        }

        // Initialize deactivation feedback
        if (class_exists('Authyo_OTP_Auth_Deactivation_Feedback')) {
            new Authyo_OTP_Auth_Deactivation_Feedback();
        }

        // AJAX handlers for admin
        $this->loader->add_action('wp_ajax_get_otp_logs', $plugin_admin, 'get_logs');
        $this->loader->add_action('wp_ajax_delete_otp_logs', $plugin_admin, 'delete_logs');
        $this->loader->add_action('wp_ajax_clear_country_codes_cache', $plugin_admin, 'clear_country_codes_cache');

        // Legacy AJAX handlers
        $this->loader->add_action('wp_ajax_nopriv_authyo_otp_auth_send_otp', $this, 'send_login_otp');
        $this->loader->add_action('wp_ajax_authyo_otp_auth_send_otp', $this, 'send_login_otp');
    }

    private function define_public_hooks() {
        // Removed automatic WooCommerce form hooks - OTP functionality now only works via shortcodes
        // This ensures OTP only appears on pages where shortcodes are explicitly placed
        // Login form hooks removed - use [authyo_otp_auth_login] shortcode instead
        // Registration form hooks removed - use [authyo_otp_auth_register] shortcode instead
    }

    public function add_otp_field_to_login() {
        echo wp_kses_post($this->render_auth_form());
    }

    public function add_otp_field_to_register() {
        echo wp_kses_post($this->render_auth_form());
    }

    public function validate_login_otp($validation_error, $username, $password) {
        $settings = get_option('authyo_otp_auth_settings', array());

        if (!isset($settings['enable_login']) || $settings['enable_login'] !== 'yes') {
            return $validation_error;
        }

        // Verify nonce
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'woocommerce-login')) {
            return new WP_Error('nonce_failed', esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce'));
        }

        // Sanitize and validate OTP
        $otp = isset($_POST['authyo_otp']) ? sanitize_text_field(wp_unslash($_POST['authyo_otp'])) : '';

        if (empty($otp)) {
            return new WP_Error('otp_required', esc_html__('Please enter the OTP sent to your email.', 'authyo-otp-authentication-for-woocommerce'));
        }

        // Validate OTP format (numeric, 6 digits)
        if (!preg_match('/^\d{6}$/', $otp)) {
            return new WP_Error('invalid_otp_format', esc_html__('Invalid OTP format. Please enter a 6-digit code.', 'authyo-otp-authentication-for-woocommerce'));
        }

        // Initialize Authyo SDK
        $authyo = new Authyo_OTP_Auth_SDK();

        // Verify OTP using Authyo
        $result = $authyo->verify_otp($username, $otp);

        if (is_wp_error($result)) {
            return new WP_Error('invalid_otp', $result->get_error_message());
        }

        return $validation_error;
    }

    public function validate_register_otp($validation_error, $username, $email) {
        $settings = get_option('authyo_otp_auth_settings', array());

        if (!isset($settings['enable_register']) || $settings['enable_register'] !== 'yes') {
            return $validation_error;
        }

        // Verify nonce
        if (!isset($_POST['woocommerce-register-nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['woocommerce-register-nonce'])), 'woocommerce-register')) {
             return new WP_Error('nonce_failed', esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce'));
        }

        // Sanitize and validate OTP
        $otp = isset($_POST['authyo_otp']) ? sanitize_text_field(wp_unslash($_POST['authyo_otp'])) : '';

        if (empty($otp)) {
            return new WP_Error('otp_required', esc_html__('Please enter the OTP sent to your email.', 'authyo-otp-authentication-for-woocommerce'));
        }

        // Validate OTP format (numeric, 6 digits)
        if (!preg_match('/^\d{6}$/', $otp)) {
            return new WP_Error('invalid_otp_format', esc_html__('Invalid OTP format. Please enter a 6-digit code.', 'authyo-otp-authentication-for-woocommerce'));
        }

        // Initialize Authyo SDK
        $authyo = new Authyo_OTP_Auth_SDK();

        // Verify OTP using Authyo
        $result = $authyo->verify_otp($email, $otp);

        if (is_wp_error($result)) {
            return new WP_Error('invalid_otp', $result->get_error_message());
        }

        return $validation_error;
    }

    public function render_auth_form() {
        ob_start();
        include AUTHYO_OTP_AUTH_PLUGIN_DIR . 'templates/auth-form.php';
        return ob_get_clean();
    }

    public function render_auth_widget() {
        include AUTHYO_OTP_AUTH_PLUGIN_DIR . 'templates/auth-form.php';
    }

    public function send_login_otp() {
        // Verify nonce first before accessing any $_POST data
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'authyo_otp_auth_nonce')) {
            wp_send_json_error(array('message' => esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }

        // Get and sanitize required data after nonce verification
        $email = isset($_POST['email']) ? sanitize_text_field(wp_unslash($_POST['email'])) : '';
        $otp = isset($_POST['otp']) ? sanitize_text_field(wp_unslash($_POST['otp'])) : '';

        // Sanitize and validate email or mobile
        if (!$this->validate_email_or_mobile($email)) {
            wp_send_json_error(array('message' => esc_html__('Please enter a valid email address or 10-digit mobile number.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }

        // Initialize Authyo SDK
        $authyo = new Authyo_OTP_Auth_SDK();

        // Verify OTP using Authyo
        $result = $authyo->verify_otp($email, $otp);

        if (is_wp_error($result)) {
            $this->log_activity($email, $otp, 'login_failed');
            wp_send_json_error(array('message' => esc_html($result->get_error_message())));
            return;
        }

        // Store in transient for later verification (instead of session for better WordPress compatibility)
        set_transient(
            'authyo_otp_auth_verified_' . md5($email),
            array(
                'email' => $email,
                'time' => time()
            ),
            300 // 5 minutes expiry
        );

        $this->log_activity($email, $otp, 'login_success');
        wp_send_json_success(array('message' => esc_html__('OTP verified successfully.', 'authyo-otp-authentication-for-woocommerce')));
    }

    private function validate_email_or_mobile($input) {
        // Return false if input is empty or null
        if (empty($input)) {
            return false;
        }
        
        // Check if it's a valid email
        if (filter_var($input, FILTER_VALIDATE_EMAIL)) {
            return true;
        }

        // Check if it's a valid mobile number (10-15 digits)
        $mobile = preg_replace('/\D/', '', $input);
        
        // Validate mobile number format - accept 10-15 digits for country codes
        if (strlen($mobile) >= 10 && strlen($mobile) <= 15 && ctype_digit($mobile)) {
            return true;
        }
        
        return false;
    }

    private function is_valid_mobile($mobile) {
        // Return false if mobile is empty or null
        if (empty($mobile)) {
            return false;
        }
        
        // Remove any non-digit characters except +
        $mobile = preg_replace('/[^0-9+]/', '', $mobile);
        
        // Check if mobile number has at least 1 digit after +
        if (strpos($mobile, '+') === 0) {
            // Must have at least 10 digits after + for international format
            return strlen($mobile) >= 11 && ctype_digit(substr($mobile, 1));
        }
        
        // If no + prefix, check if it has 10-15 digits (standard mobile format or country code + mobile)
        return strlen($mobile) >= 10 && strlen($mobile) <= 15 && ctype_digit($mobile);
    }

    public function send_otp_email($email) {
        if (!is_email($email)) {
            return new WP_Error('invalid_email', __('Invalid email address provided.', 'authyo-otp-authentication-for-woocommerce'));
        }

        $otp = $this->generate_otp();
        $result = $this->store_otp($email, $otp);

        if (is_wp_error($result)) {
            return $result;
        }

        return $this->send_email($email, $otp);
    }

    public function send_otp_sms($mobile) {
        if (!$this->is_valid_mobile($mobile)) {
            return new WP_Error('invalid_mobile', __('Invalid mobile number provided.', 'authyo-otp-authentication-for-woocommerce'));
        }

        $otp = $this->generate_otp();
        $result = $this->store_otp($mobile, $otp);

        if (is_wp_error($result)) {
            return $result;
        }

        return $this->send_sms($mobile, $otp);
    }

    private function get_client_ip() {
        $ipaddress = '';
        
        // Check for IP in various server variables
        $ip_sources = array(
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        );
        
        foreach ($ip_sources as $source) {
            if (isset($_SERVER[$source])) {
                // Sanitize the IP address
                $ip = sanitize_text_field(wp_unslash($_SERVER[$source]));
                
                // Validate IP address format
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    $ipaddress = $ip;
                    break;
                }
            }
        }
        
        // If no valid IP found, return a default value
        if (empty($ipaddress)) {
            $ipaddress = '0.0.0.0';
        }
            
        return $ipaddress;
    }

    private function generate_otp() {
        // Generate a cryptographically secure random number
        $bytes = random_bytes(3); // 3 bytes = 6 digits
        $number = hexdec(bin2hex($bytes));
        
        // Ensure it's a 6-digit number
        $otp = str_pad($number % 1000000, 6, '0', STR_PAD_LEFT);
        
        return $otp;
    }

    public function run() {
        $this->loader->run();
    }

    public function get_plugin_name() {
        return $this->plugin_name;
    }

    public function get_loader() {
        return $this->loader;
    }

    public function get_version() {
        return $this->version;
    }

    public function activate() {
        // Set default options
        add_option('authyo_otp_auth_version', AUTHYO_OTP_AUTH_VERSION);
    }

    private function log_activity($email, $otp, $status, $user_id = 0) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'authyo_otp_auth_logs';
        
        // Prepare data for insertion
        $data = array(
            'email' => sanitize_email($email),
            'otp' => sanitize_text_field($otp),
            'status' => sanitize_text_field($status),
            'ip_address' => $this->get_client_ip(),
            'created_at' => current_time('mysql'),
            'user_id' => absint($user_id)
        );
        
        $format = array('%s', '%s', '%s', '%s', '%s', '%d');
        
        // Use wpdb::insert() with error handling
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin needs to log OTP activities to custom table
        $result = $wpdb->insert($table_name, $data, $format);
        
        // Log error if insertion fails
        if (false === $result) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Error logging for database failures
            error_log('Authyo OTP: Failed to log activity - ' . $wpdb->last_error);
        }
        
        return $result;
    }
} 