<?php
/**
 * SDK for Authyo.io API integration
 */

if (!defined('ABSPATH')) {
    exit;
}

class Authyo_OTP_Auth_SDK {
    private $base_url = 'https://app.authyo.io/api/v1/';
    private $client_id;
    private $client_secret;

    public function __construct() {
        $this->client_id = get_option('authyo_otp_auth_client_id', '');
        $this->client_secret = get_option('authyo_otp_auth_client_secret', '');
    }

    public function send_otp($email_or_mobile, $method = null, $type = 'login') {
        // Check if bypass is enabled and this is a test phone number or email
        if ($this->should_bypass_otp($email_or_mobile)) {
            // Determine if it's email or phone for the message
            $is_email = filter_var($email_or_mobile, FILTER_VALIDATE_EMAIL);
            $bypass_type = $is_email ? 'test email address' : 'test phone number';
            
            // Return a mock success response for test numbers/emails
            return array(
                'success' => true,
                'message' => 'OTP bypassed for testing (' . $bypass_type . ')',
                'maskId' => 'test_' . md5($email_or_mobile . time()),
                'method' => $method ?: ($is_email ? 'email' : 'sms')
            );
        }

        if (empty($this->client_id) || empty($this->client_secret)) {
            return new WP_Error('config_error', 'Authyo.io credentials are not configured');
        }

        // Get OTP configuration from settings
        $otp_length = get_option('authyo_otp_auth_length', 6);
        $otp_expiry = get_option('authyo_otp_auth_expiry', 300);

        // Determine OTP method based on input
        // If method is explicitly provided (for fallback), use it
        // Otherwise, auto-detect based on input type
        $is_mobile_number = !filter_var($email_or_mobile, FILTER_VALIDATE_EMAIL);
        $method_explicitly_provided = !empty($method);
        
        if (empty($method)) {
            $method = $this->determine_otp_method($email_or_mobile);
        }

        // Process phone number to ensure proper format for SMS delivery
        $processed_phone = $this->process_phone_number($email_or_mobile);
        
        // Track if we need to try fallback methods for SMS issues
        $tried_methods = array();
        $max_attempts = 3; // Try up to 3 different methods before giving up
        
        // If method was explicitly provided (user-selected fallback), don't auto-retry with other methods
        // Only retry if the explicitly provided method fails
        if ($method_explicitly_provided) {
            $max_attempts = 1; // Only try once when method is explicitly provided
        }
        
        // Retry loop with automatic fallback for SMS issues
        $attempt = 0;
        $last_error = null;
        $current_method = $method;
        
        while ($attempt < $max_attempts) {
            $attempt++;
            
            // Skip methods we've already tried
            if (in_array($current_method, $tried_methods)) {
                // Get next available method
                $current_method = $this->get_next_available_method($email_or_mobile, $tried_methods);
                if (!$current_method) {
                    break; // No more methods to try
                }
            }
            
            // Track this attempt
            $tried_methods[] = $current_method;
            
            // Build the URL with query parameters
            $url_params = array(
                'to' => sanitize_text_field($processed_phone),
                'expiry' => absint($otp_expiry),
                'otpLength' => absint($otp_length)
            );

            // Add authWay parameter if specified (Authyo API expects authWay, not method)
            if (!empty($current_method)) {
                // Convert 'voice' to 'voicecall' for Authyo.io API
                $api_method = ($current_method === 'voice') ? 'voicecall' : $current_method;
                $url_params['authWay'] = sanitize_text_field($api_method);
            }

            $url = add_query_arg($url_params, $this->base_url . 'auth/sendotp');

            $args = array(
                'method' => 'GET',
                'headers' => array(
                    'clientId' => $this->client_id,
                    'clientSecret' => $this->client_secret
                ),
                'timeout' => 30,
                'sslverify' => true
            );
            
            $response = wp_remote_get($url, $args);

            if (is_wp_error($response)) {
                $last_error = new WP_Error('network_error', 'Network error: ' . $response->get_error_message());
                // Continue to try next method if this is a mobile number
                if ($is_mobile_number && $attempt < $max_attempts) {
                    $current_method = $this->get_next_available_method($email_or_mobile, $tried_methods);
                    if (!$current_method) {
                        return $last_error;
                    }
                    continue;
                }
                return $last_error;
            }

            $response_code = wp_remote_retrieve_response_code($response);
            $response_body = wp_remote_retrieve_body($response);
            $response_data = json_decode($response_body, true);

            // Check for JSON decode errors
            if (json_last_error() !== JSON_ERROR_NONE) {
                $last_error = new WP_Error('json_error', 'Invalid JSON response from Authyo.io API');
                // Continue to try next method if this is a mobile number
                if ($is_mobile_number && $attempt < $max_attempts) {
                    $current_method = $this->get_next_available_method($email_or_mobile, $tried_methods);
                    if (!$current_method) {
                        return $last_error;
                    }
                    continue;
                }
                return $last_error;
            }

            // Check for service availability issues even with 200 status (country/channel not supported)
            if (isset($response_data['data']['results'][0]['message']) && 
                strpos($response_data['data']['results'][0]['message'], 'not supported') !== false) {
                $error_message = $response_data['data']['results'][0]['message'];
                $last_error = new WP_Error('service_unavailable', $error_message);
                
                // For mobile numbers, try next available method automatically
                if ($is_mobile_number && $attempt < $max_attempts) {
                    $current_method = $this->get_next_available_method($email_or_mobile, $tried_methods);
                    if (!$current_method) {
                        // No more methods to try, return error with suggestions
                        $available_methods = $this->get_available_fallback_methods($email_or_mobile);
                        if (!empty($available_methods)) {
                            $method_list = implode(', ', array_values($available_methods));
                            /* translators: %s: Comma-separated list of available OTP delivery methods (e.g., "SMS, Voice Call") */
                            $error_message = $error_message . ' ' . sprintf(esc_html__('Please try using: %s', 'authyo-otp-authentication-for-woocommerce'), $method_list);
                        }
                        return new WP_Error('service_unavailable', $error_message);
                    }
                    // Continue with next method
                    continue;
                }
                return $last_error;
            }

            if ($response_code !== 200) {
                $error_message = 'Unknown error occurred';
                
                // Try to extract error message from various possible locations
                if (isset($response_data['message'])) {
                    $error_message = $response_data['message'];
                } elseif (isset($response_data['error'])) {
                    $error_message = $response_data['error'];
                } elseif (isset($response_data['data']['message'])) {
                    $error_message = $response_data['data']['message'];
                } elseif (isset($response_data['data']['error'])) {
                    $error_message = $response_data['data']['error'];
                } else {
                    $error_message = 'HTTP ' . $response_code . ' error occurred';
                }
                
                $last_error = new WP_Error('api_error', $error_message);
                
                // For mobile numbers, try next available method
                if ($is_mobile_number && $attempt < $max_attempts) {
                    $current_method = $this->get_next_available_method($email_or_mobile, $tried_methods);
                    if (!$current_method) {
                        return $last_error;
                    }
                    continue;
                }
                return $last_error;
            }

            // Success! Check if the response has the expected structure
            // Try multiple possible response structures from Authyo.io API
            $maskId = null;
            
            // Structure 1: data.results[0].maskId (most common)
            if (isset($response_data['data']['results'][0]['maskId'])) {
                $maskId = $response_data['data']['results'][0]['maskId'];
            }
            // Structure 2: data.maskId (alternative structure)
            elseif (isset($response_data['data']['maskId'])) {
                $maskId = $response_data['data']['maskId'];
            }
            // Structure 3: maskId (direct structure)
            elseif (isset($response_data['maskId'])) {
                $maskId = $response_data['maskId'];
            }
            // Structure 4: results[0].maskId (without data wrapper)
            elseif (isset($response_data['results'][0]['maskId'])) {
                $maskId = $response_data['results'][0]['maskId'];
            }
            // Structure 5: data.results.maskId (single result object)
            elseif (isset($response_data['data']['results']['maskId'])) {
                $maskId = $response_data['data']['results']['maskId'];
            }
            // Structure 6: data.result.maskId (singular result)
            elseif (isset($response_data['data']['result']['maskId'])) {
                $maskId = $response_data['data']['result']['maskId'];
            }
            // Structure 7: result.maskId (direct result)
            elseif (isset($response_data['result']['maskId'])) {
                $maskId = $response_data['result']['maskId'];
            }
            // Structure 8: Check for any nested array with maskId
            elseif (isset($response_data['data']) && is_array($response_data['data'])) {
                foreach ($response_data['data'] as $key => $value) {
                    if (is_array($value) && isset($value['maskId'])) {
                        $maskId = $value['maskId'];
                        break;
                    }
                }
            }
            
            if (empty($maskId)) {
                // Try to extract any available ID from the response as a last resort
                $fallback_id = null;
                
                // Try various ID field names
                $id_fields = ['id', 'maskId', 'mask_id', 'transactionId', 'transaction_id', 'otpId', 'otp_id', 'requestId', 'request_id', 'sessionId', 'session_id'];
                
                foreach ($id_fields as $field) {
                    if (isset($response_data[$field])) {
                        $fallback_id = $response_data[$field];
                        break;
                    } elseif (isset($response_data['data'][$field])) {
                        $fallback_id = $response_data['data'][$field];
                        break;
                    } elseif (isset($response_data['result'][$field])) {
                        $fallback_id = $response_data['result'][$field];
                        break;
                    } elseif (isset($response_data['data']['result'][$field])) {
                        $fallback_id = $response_data['data']['result'][$field];
                        break;
                    }
                }
                
                // If still no ID found, try to find any string value that looks like an ID
                if (!$fallback_id) {
                    $fallback_id = $this->find_id_like_value($response_data);
                }
                
                if ($fallback_id) {
                    $maskId = $fallback_id;
                } else {
                    $last_error = new WP_Error('invalid_response', 'Invalid response structure from Authyo.io API - maskId not found in expected locations');
                    // Try next method if available
                    if ($is_mobile_number && $attempt < $max_attempts) {
                        $current_method = $this->get_next_available_method($email_or_mobile, $tried_methods);
                        if (!$current_method) {
                            return $last_error;
                        }
                        continue;
                    }
                    return $last_error;
                }
            }
            
            // Determine the method to return in response
            // If method was explicitly provided (user-selected fallback), always use that
            // Otherwise, try to get the actual method from API response
            $actual_method = $current_method; // Default to the method we used
            
            // Only check API response for method if method wasn't explicitly provided
            // When user explicitly selects a fallback method, we should trust that choice
            if (!$method_explicitly_provided) {
                // Try multiple possible locations for method information
                if (isset($response_data['data']['results'][0]['authType'])) {
                    // API returns authType in the results array
                    $auth_type = strtolower(sanitize_text_field($response_data['data']['results'][0]['authType']));
                    $actual_method = ($auth_type === 'voicecall') ? 'voice' : $auth_type;
                } elseif (isset($response_data['data']['results'][0]['method'])) {
                    $method_from_api = sanitize_text_field($response_data['data']['results'][0]['method']);
                    $actual_method = ($method_from_api === 'voicecall') ? 'voice' : $method_from_api;
                } elseif (isset($response_data['data']['authType'])) {
                    // Alternative structure: data.authType
                    $auth_type = strtolower(sanitize_text_field($response_data['data']['authType']));
                    $actual_method = ($auth_type === 'voicecall') ? 'voice' : $auth_type;
                } elseif (isset($response_data['data']['method'])) {
                    // Alternative structure: data.method
                    $method_from_api = sanitize_text_field($response_data['data']['method']);
                    $actual_method = ($method_from_api === 'voicecall') ? 'voice' : $method_from_api;
                } elseif (isset($response_data['results'][0]['authType'])) {
                    // Structure without data wrapper: results[0].authType
                    $auth_type = strtolower(sanitize_text_field($response_data['results'][0]['authType']));
                    $actual_method = ($auth_type === 'voicecall') ? 'voice' : $auth_type;
                } elseif (isset($response_data['results'][0]['method'])) {
                    // Structure without data wrapper: results[0].method
                    $method_from_api = sanitize_text_field($response_data['results'][0]['method']);
                    $actual_method = ($method_from_api === 'voicecall') ? 'voice' : $method_from_api;
                } elseif (isset($response_data['method'])) {
                    // Direct structure: method
                    $method_from_api = sanitize_text_field($response_data['method']);
                    $actual_method = ($method_from_api === 'voicecall') ? 'voice' : $method_from_api;
                } elseif (isset($response_data['authType'])) {
                    // Direct structure: authType
                    $auth_type = strtolower(sanitize_text_field($response_data['authType']));
                    $actual_method = ($auth_type === 'voicecall') ? 'voice' : $auth_type;
                }
            }
            // If method_explicitly_provided is true, we keep $actual_method = $current_method (the explicitly requested method)

            // Success! Return the result
            return array(
                'maskId' => $maskId,
                'message' => $response_data['message'] ?? 'OTP sent successfully',
                'method' => $actual_method
            );
        }
        
        // If we've exhausted all attempts, return the last error
        if ($last_error) {
            return $last_error;
        }
        
        // Fallback error if something unexpected happened
        return new WP_Error('unknown_error', 'Failed to send OTP after multiple attempts');
    }

    public function verify_otp($email, $otp, $mask_id) {
        // Check if bypass is enabled and this is a test phone number or email
        if ($this->should_bypass_otp($email)) {
            return true; // Bypass OTP verification for test numbers/emails
        }

        if (empty($this->client_id) || empty($this->client_secret)) {
            return new WP_Error('auth_config', __('Authyo.io credentials are not configured.', 'authyo-otp-authentication-for-woocommerce'));
        }

        // Construct API URL with query parameters
        $api_url = $this->base_url . 'auth/verifyotp?' . http_build_query(array(
            'maskId' => $mask_id,
            'otp' => $otp
        ));

        $args = array(
            'method' => 'GET',
            'headers' => array(
                'clientId' => $this->client_id,
                'clientSecret' => $this->client_secret
            ),
            'sslverify' => true,
            'httpversion' => '1.1',
            'timeout' => 30
        );

        $response = wp_remote_get($api_url, $args);

        if (is_wp_error($response)) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        if ($response_code !== 200) {
            return new WP_Error('api_error', __('Failed to verify OTP. Please try again.', 'authyo-otp-authentication-for-woocommerce'));
        }

        $data = json_decode($response_body, true);

        // Log response for debugging
        // Debug code removed for production - uncomment if needed during development
        // if (defined('WP_DEBUG') && WP_DEBUG) {
        //     error_log('=== AUTHYO VERIFY OTP RESPONSE ===');
        //     error_log('Status Code: ' . $response_code);
        //     error_log('Response Body: ' . $response_body);
        //     error_log('Decoded Data: ' . print_r($data, true));
        //     error_log('===============================');
        // }

        if (!is_array($data)) {
            return new WP_Error('invalid_response', __('Invalid response from OTP service.', 'authyo-otp-authentication-for-woocommerce'));
        }

        // Check multiple possible success indicators
        $is_success = false;
        
        // Check various success patterns
        if (isset($data['success']) && $data['success'] === true) {
            $is_success = true;
        } elseif (isset($data['status']) && $data['status'] === 'verified') {
            $is_success = true;
        } elseif (isset($data['verified']) && $data['verified'] === true) {
            $is_success = true;
        } elseif (isset($data['data']['success']) && $data['data']['success'] === true) {
            $is_success = true;
        } elseif (isset($data['data']['status']) && $data['data']['status'] === 'verified') {
            $is_success = true;
        }

        if ($is_success) {
            return true;
        }

        // If OTP is not matched or not verified, return appropriate error
        if (isset($data['message'])) {
            return new WP_Error('otp_mismatch', $data['message']);
        } elseif (isset($data['data']['message'])) {
            return new WP_Error('otp_mismatch', $data['data']['message']);
        } elseif (isset($data['error'])) {
            return new WP_Error('otp_mismatch', $data['error']);
        }

        return new WP_Error('verification_failed', __('OTP verification failed. Please try again.', 'authyo-otp-authentication-for-woocommerce'));
    }

    public function get_otp_status($mask_id) {
        if (empty($this->client_id) || empty($this->client_secret)) {
            return new WP_Error('missing_credentials', __('Authyo credentials are not configured.', 'authyo-otp-authentication-for-woocommerce'));
        }

        $status_url = add_query_arg(
            array(
                'maskId' => $mask_id
            ),
            $this->base_url . 'auth/status'
        );

        $response = wp_remote_get($status_url, array(
            'headers' => array(
                'clientId' => $this->client_id,
                'clientSecret' => $this->client_secret
            )
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);

        if ($status_code !== 200 || (isset($body['success']) && $body['success'] === false)) {
            $error_message = isset($body['message']) ? $body['message'] : __('Failed to get OTP status.', 'authyo-otp-authentication-for-woocommerce');
            return new WP_Error('authyo_error', $error_message);
        }

        return $body;
    }

    /**
     * Check if OTP should be bypassed for testing
     */
    private function should_bypass_otp($email) {
        // Check if bypass is enabled
        $bypass_enabled = get_option('authyo_otp_auth_bypass_enabled', 'no');
        if ($bypass_enabled !== 'yes') {
            return false;
        }

        // Check if input is email or phone
        $is_email = filter_var($email, FILTER_VALIDATE_EMAIL);
        
        if ($is_email) {
            // Check test emails
            $test_emails = get_option('authyo_otp_auth_test_emails', '');
            if (empty($test_emails)) {
                return false;
            }

            // Convert to array and clean up
            $test_email_list = array_map('trim', explode("\n", $test_emails));
            $test_email_list = array_filter($test_email_list); // Remove empty lines

            if (empty($test_email_list)) {
                return false;
            }

            // Normalize the input email
            $normalized_input = strtolower(trim($email));
            
            // Check if the normalized input matches any test email
            foreach ($test_email_list as $test_email) {
                $normalized_test = strtolower(trim($test_email));
                if ($normalized_input === $normalized_test) {
                    return true;
                }
            }
        } else {
            // Check test phone numbers
            $test_phones = get_option('authyo_otp_auth_test_phones', '');
            if (empty($test_phones)) {
                return false;
            }

            // Convert to array and clean up
            $test_numbers = array_map('trim', explode("\n", $test_phones));
            $test_numbers = array_filter($test_numbers); // Remove empty lines

            if (empty($test_numbers)) {
                return false;
            }

            // Normalize the input phone
            $normalized_input = $this->normalize_phone_number($email);
            
            // Check if the normalized input matches any test number
            foreach ($test_numbers as $test_number) {
                $normalized_test = $this->normalize_phone_number($test_number);
                if ($normalized_input === $normalized_test) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Normalize phone number for comparison
     */
    private function normalize_phone_number($phone) {
        // Remove all non-digit characters except +
        $phone = preg_replace('/[^\d+]/', '', $phone);
        
        // If it doesn't start with +, add it
        if (substr($phone, 0, 1) !== '+') {
            $phone = '+' . $phone;
        }
        
        return $phone;
    }

    /**
     * Dynamic country code detection using comprehensive global patterns
     */
    private function detect_country_code_dynamic($phone) {
        $phone_length = strlen($phone);
        
        // Get user's preferred default country from settings
        $default_country = get_option('authyo_otp_auth_default_country', '91');
        
        // Rule 1: Check if number already has a country code (11+ digits)
        if ($phone_length >= 11) {
            // Check for all possible country codes at the beginning
            $country_codes = $this->get_global_country_codes();
            foreach ($country_codes as $code => $expected_length) {
                if (strpos($phone, $code) === 0 && $phone_length === $expected_length) {
                    return '+' . $phone;
                }
            }
            
            // If it's 11 digits and starts with 1, likely US/Canada
            if ($phone_length === 11 && substr($phone, 0, 1) === '1') {
                return '+' . $phone;
            }
        }
        
        // Rule 2: 10-digit numbers - use global pattern recognition
        if ($phone_length === 10) {
            return $this->detect_10_digit_country_global($phone, $default_country);
        }
        
        // Rule 3: 11-digit numbers - likely US/Canada
        if ($phone_length === 11) {
            return '+1' . $phone;
        }
        
        // Rule 4: 12+ digit numbers - check for country codes
        if ($phone_length >= 12) {
            return $this->detect_long_number_country_global($phone, $default_country);
        }
        
        // Rule 5: Fallback - use default country
        return '+' . $default_country . $phone;
    }

    /**
     * Detect country for 10-digit numbers using global pattern recognition
     */
    private function detect_10_digit_country_global($phone, $default_country) {
        // Get global phone patterns
        $patterns = $this->get_global_phone_patterns();
        
        // Check each pattern
        foreach ($patterns as $pattern) {
            if (preg_match($pattern['regex'], $phone)) {
                return '+' . $pattern['country_code'] . $phone;
            }
        }
        
        // Default to configured country
        return '+' . $default_country . $phone;
    }

    /**
     * Detect country for longer numbers (12+ digits) using global patterns
     */
    private function detect_long_number_country_global($phone, $default_country) {
        $country_codes = $this->get_global_country_codes();
        
        // Check for known country codes
        foreach ($country_codes as $code => $expected_length) {
            if (strpos($phone, $code) === 0 && strlen($phone) === $expected_length) {
                return '+' . $phone;
            }
        }
        
        // If no match found, assume it's already formatted correctly
        return '+' . $phone;
    }

    /**
     * Get comprehensive global country codes with expected lengths
     * Now uses dynamic data from Authyo API with fallback to static list
     */
    private function get_global_country_codes() {
        // Try to get from cache first
        $cached_codes = get_transient('authyo_country_codes_cache');
        if ($cached_codes !== false) {
            return $cached_codes;
        }

        // Try to get from Authyo API
        $api_url = 'https://app.authyo.io/api/v1/user/getcountrylist';
        
        $response = wp_remote_get($api_url, array(
            'timeout' => 10,
            'sslverify' => true
        ));

        if (!is_wp_error($response)) {
            $body = wp_remote_retrieve_body($response);
            $countries = json_decode($body, true);

            if ($countries && is_array($countries)) {
                $country_codes = array();
                foreach ($countries as $country) {
                    if (isset($country['code']) && isset($country['phoneLength'])) {
                        $country_codes[$country['code']] = (int)$country['phoneLength'];
                    }
                }
                
                if (!empty($country_codes)) {
                    // Cache for 24 hours
                    set_transient('authyo_country_codes_cache', $country_codes, 24 * HOUR_IN_SECONDS);
                    return $country_codes;
                }
            }
        }

        // Fallback to static list if API fails
        $static_codes = $this->get_static_country_codes_fallback();
        
        // Cache fallback for 1 hour
        set_transient('authyo_country_codes_cache', $static_codes, HOUR_IN_SECONDS);
        
        return $static_codes;
    }

    /**
     * Clear country codes cache (useful for forcing refresh)
     */
    public function clear_country_codes_cache() {
        delete_transient('authyo_country_codes_cache');
    }

    /**
     * Static fallback country codes (used when API is unavailable)
     */
    private function get_static_country_codes_fallback() {
        return array(
            // Major countries by population and economy
            '1' => 11,      // US/Canada
            '7' => 12,      // Russia/Kazakhstan
            '20' => 13,     // Egypt
            '27' => 12,     // South Africa
            '30' => 12,     // Greece
            '31' => 12,     // Netherlands
            '32' => 12,     // Belgium
            '33' => 12,     // France
            '34' => 12,     // Spain
            '36' => 12,     // Hungary
            '39' => 12,     // Italy
            '40' => 12,     // Romania
            '41' => 12,     // Switzerland
            '43' => 12,     // Austria
            '44' => 13,     // UK
            '45' => 11,     // Denmark
            '46' => 12,     // Sweden
            '47' => 11,     // Norway
            '48' => 12,     // Poland
            '49' => 13,     // Germany
            '51' => 12,     // Peru
            '52' => 13,     // Mexico
            '53' => 12,     // Cuba
            '54' => 13,     // Argentina
            '55' => 13,     // Brazil
            '56' => 12,     // Chile
            '57' => 12,     // Colombia
            '58' => 12,     // Venezuela
            '60' => 12,     // Malaysia
            '61' => 12,     // Australia
            '62' => 13,     // Indonesia
            '63' => 12,     // Philippines
            '64' => 11,     // New Zealand
            '65' => 11,     // Singapore
            '66' => 12,     // Thailand
            '81' => 12,     // Japan
            '82' => 12,     // South Korea
            '84' => 12,     // Vietnam
            '86' => 13,     // China
            '90' => 12,     // Turkey
            '91' => 12,     // India
            '92' => 12,     // Pakistan
            '93' => 12,     // Afghanistan
            '94' => 12,     // Sri Lanka
            '95' => 12,     // Myanmar
            '98' => 12,     // Iran
            '212' => 13,    // Morocco
            '213' => 13,    // Algeria
            '216' => 12,    // Tunisia
            '218' => 13,    // Libya
            '220' => 12,    // Gambia
            '221' => 12,    // Senegal
            '222' => 12,    // Mauritania
            '223' => 12,    // Mali
            '224' => 12,    // Guinea
            '225' => 12,    // Ivory Coast
            '226' => 12,    // Burkina Faso
            '227' => 12,    // Niger
            '228' => 12,    // Togo
            '229' => 12,    // Benin
            '230' => 12,    // Mauritius
            '231' => 12,    // Liberia
            '232' => 12,    // Sierra Leone
            '233' => 12,    // Ghana
            '234' => 13,    // Nigeria
            '235' => 12,    // Chad
            '236' => 12,    // Central African Republic
            '237' => 12,    // Cameroon
            '238' => 12,    // Cape Verde
            '239' => 12,    // São Tomé and Príncipe
            '240' => 12,    // Equatorial Guinea
            '241' => 12,    // Gabon
            '242' => 12,    // Republic of the Congo
            '243' => 12,    // Democratic Republic of the Congo
            '244' => 12,    // Angola
            '245' => 12,    // Guinea-Bissau
            '246' => 12,    // British Indian Ocean Territory
            '248' => 12,    // Seychelles
            '249' => 12,    // Sudan
            '250' => 12,    // Rwanda
            '251' => 12,    // Ethiopia
            '252' => 12,    // Somalia
            '253' => 12,    // Djibouti
            '254' => 12,    // Kenya
            '255' => 12,    // Tanzania
            '256' => 12,    // Uganda
            '257' => 12,    // Burundi
            '258' => 12,    // Mozambique
            '260' => 12,    // Zambia
            '261' => 12,    // Madagascar
            '262' => 12,    // Mayotte
            '263' => 12,    // Zimbabwe
            '264' => 12,    // Namibia
            '265' => 12,    // Malawi
            '266' => 12,    // Lesotho
            '267' => 12,    // Botswana
            '268' => 12,    // Swaziland
            '269' => 12,    // Comoros
            '290' => 12,    // Saint Helena
            '291' => 12,    // Eritrea
            '297' => 12,    // Aruba
            '298' => 12,    // Faroe Islands
            '299' => 12,    // Greenland
            '350' => 12,    // Gibraltar
            '351' => 12,    // Portugal
            '352' => 12,    // Luxembourg
            '353' => 12,    // Ireland
            '354' => 12,    // Iceland
            '355' => 12,    // Albania
            '356' => 12,    // Malta
            '357' => 12,    // Cyprus
            '358' => 12,    // Finland
            '359' => 12,    // Bulgaria
            '370' => 12,    // Lithuania
            '371' => 12,    // Latvia
            '372' => 12,    // Estonia
            '373' => 12,    // Moldova
            '374' => 12,    // Armenia
            '375' => 12,    // Belarus
            '376' => 12,    // Andorra
            '377' => 12,    // Monaco
            '378' => 12,    // San Marino
            '380' => 12,    // Ukraine
            '381' => 12,    // Serbia
            '382' => 12,    // Montenegro
            '383' => 12,    // Kosovo
            '385' => 12,    // Croatia
            '386' => 12,    // Slovenia
            '387' => 12,    // Bosnia and Herzegovina
            '389' => 12,    // North Macedonia
            '420' => 12,    // Czech Republic
            '421' => 12,    // Slovakia
            '423' => 12,    // Liechtenstein
            '500' => 12,    // Falkland Islands
            '501' => 12,    // Belize
            '502' => 12,    // Guatemala
            '503' => 12,    // El Salvador
            '504' => 12,    // Honduras
            '505' => 12,    // Nicaragua
            '506' => 12,    // Costa Rica
            '507' => 12,    // Panama
            '508' => 12,    // Saint Pierre and Miquelon
            '509' => 12,    // Haiti
            '590' => 12,    // Guadeloupe
            '591' => 12,    // Bolivia
            '592' => 12,    // Guyana
            '593' => 12,    // Ecuador
            '594' => 12,    // French Guiana
            '595' => 12,    // Paraguay
            '596' => 12,    // Martinique
            '597' => 12,    // Suriname
            '598' => 12,    // Uruguay
            '599' => 12,    // Netherlands Antilles
            '670' => 12,    // East Timor
            '672' => 12,    // Australian External Territories
            '673' => 12,    // Brunei
            '674' => 12,    // Nauru
            '675' => 12,    // Papua New Guinea
            '676' => 12,    // Tonga
            '677' => 12,    // Solomon Islands
            '678' => 12,    // Vanuatu
            '679' => 12,    // Fiji
            '680' => 12,    // Palau
            '681' => 12,    // Wallis and Futuna
            '682' => 12,    // Cook Islands
            '683' => 12,    // Niue
            '684' => 12,    // American Samoa
            '685' => 12,    // Samoa
            '686' => 12,    // Kiribati
            '687' => 12,    // New Caledonia
            '688' => 12,    // Tuvalu
            '689' => 12,    // French Polynesia
            '690' => 12,    // Tokelau
            '691' => 12,    // Micronesia
            '692' => 12,    // Marshall Islands
            '850' => 12,    // North Korea
            '852' => 12,    // Hong Kong
            '853' => 12,    // Macau
            '855' => 12,    // Cambodia
            '856' => 12,    // Laos
            '880' => 13,    // Bangladesh
            '886' => 12,    // Taiwan
            '960' => 12,    // Maldives
            '961' => 12,    // Lebanon
            '962' => 12,    // Jordan
            '963' => 12,    // Syria
            '964' => 12,    // Iraq
            '965' => 12,    // Kuwait
            '966' => 12,    // Saudi Arabia
            '967' => 12,    // Yemen
            '968' => 12,    // Oman
            '970' => 12,    // Palestine
            '971' => 12,    // UAE
            '972' => 12,    // Israel
            '973' => 12,    // Bahrain
            '974' => 12,    // Qatar
            '975' => 12,    // Bhutan
            '976' => 12,    // Mongolia
            '977' => 12,    // Nepal
            '992' => 12,    // Tajikistan
            '993' => 12,    // Turkmenistan
            '994' => 12,    // Azerbaijan
            '995' => 12,    // Georgia
            '996' => 12,    // Kyrgyzstan
            '998' => 12,    // Uzbekistan
        );
    }

    /**
     * Get comprehensive global phone number patterns for 10-digit numbers
     */
    private function get_global_phone_patterns() {
        return array(
            // North America
            array('regex' => '/^[2-9]\d{9}$/', 'country_code' => '1', 'country' => 'US/Canada'),
            
            // India - Mobile numbers start with 6-9
            array('regex' => '/^[6-9]\d{9}$/', 'country_code' => '91', 'country' => 'India'),
            
            // China - Mobile numbers start with 1
            array('regex' => '/^1\d{9}$/', 'country_code' => '86', 'country' => 'China'),
            
            // UK - Mobile numbers start with 7
            array('regex' => '/^7\d{9}$/', 'country_code' => '44', 'country' => 'UK'),
            
            // Germany - Mobile numbers start with 15-17
            array('regex' => '/^1[5-7]\d{8}$/', 'country_code' => '49', 'country' => 'Germany'),
            
            // France - Mobile numbers start with 6-7
            array('regex' => '/^[67]\d{8}$/', 'country_code' => '33', 'country' => 'France'),
            
            // Spain - Mobile numbers start with 6-7
            array('regex' => '/^[67]\d{8}$/', 'country_code' => '34', 'country' => 'Spain'),
            
            // Italy - Mobile numbers start with 3
            array('regex' => '/^3\d{9}$/', 'country_code' => '39', 'country' => 'Italy'),
            
            // Australia - Mobile numbers start with 4
            array('regex' => '/^4\d{8}$/', 'country_code' => '61', 'country' => 'Australia'),
            
            // Japan - Mobile numbers start with 7-9
            array('regex' => '/^[789]\d{9}$/', 'country_code' => '81', 'country' => 'Japan'),
            
            // South Korea - Mobile numbers start with 1
            array('regex' => '/^1\d{9}$/', 'country_code' => '82', 'country' => 'South Korea'),
            
            // Brazil - Mobile numbers start with 9
            array('regex' => '/^9\d{9}$/', 'country_code' => '55', 'country' => 'Brazil'),
            
            // Russia - Mobile numbers start with 9
            array('regex' => '/^9\d{9}$/', 'country_code' => '7', 'country' => 'Russia'),
            
            // Mexico - Mobile numbers start with 1
            array('regex' => '/^1\d{9}$/', 'country_code' => '52', 'country' => 'Mexico'),
            
            // Indonesia - Mobile numbers start with 8
            array('regex' => '/^8\d{9}$/', 'country_code' => '62', 'country' => 'Indonesia'),
            
            // Philippines - Mobile numbers start with 9
            array('regex' => '/^9\d{9}$/', 'country_code' => '63', 'country' => 'Philippines'),
            
            // Thailand - Mobile numbers start with 6-9
            array('regex' => '/^[6-9]\d{8}$/', 'country_code' => '66', 'country' => 'Thailand'),
            
            // Vietnam - Mobile numbers start with 3, 5, 7, 8, 9
            array('regex' => '/^[35789]\d{8}$/', 'country_code' => '84', 'country' => 'Vietnam'),
            
            // Turkey - Mobile numbers start with 5
            array('regex' => '/^5\d{9}$/', 'country_code' => '90', 'country' => 'Turkey'),
            
            // Pakistan - Mobile numbers start with 3
            array('regex' => '/^3\d{9}$/', 'country_code' => '92', 'country' => 'Pakistan'),
            
            // Bangladesh - Mobile numbers start with 1
            array('regex' => '/^1\d{9}$/', 'country_code' => '880', 'country' => 'Bangladesh'),
            
            // Nigeria - Mobile numbers start with 7-9
            array('regex' => '/^[789]\d{9}$/', 'country_code' => '234', 'country' => 'Nigeria'),
            
            // Egypt - Mobile numbers start with 1
            array('regex' => '/^1\d{9}$/', 'country_code' => '20', 'country' => 'Egypt'),
            
            // South Africa - Mobile numbers start with 6-8
            array('regex' => '/^[6-8]\d{8}$/', 'country_code' => '27', 'country' => 'South Africa'),
            
            // Argentina - Mobile numbers start with 9
            array('regex' => '/^9\d{9}$/', 'country_code' => '54', 'country' => 'Argentina'),
            
            // Chile - Mobile numbers start with 9
            array('regex' => '/^9\d{8}$/', 'country_code' => '56', 'country' => 'Chile'),
            
            // Colombia - Mobile numbers start with 3
            array('regex' => '/^3\d{9}$/', 'country_code' => '57', 'country' => 'Colombia'),
            
            // Peru - Mobile numbers start with 9
            array('regex' => '/^9\d{8}$/', 'country_code' => '51', 'country' => 'Peru'),
            
            // Venezuela - Mobile numbers start with 4
            array('regex' => '/^4\d{9}$/', 'country_code' => '58', 'country' => 'Venezuela'),
            
            // Malaysia - Mobile numbers start with 1
            array('regex' => '/^1\d{9}$/', 'country_code' => '60', 'country' => 'Malaysia'),
            
            // Singapore - Mobile numbers start with 8-9
            array('regex' => '/^[89]\d{7}$/', 'country_code' => '65', 'country' => 'Singapore'),
            
            // New Zealand - Mobile numbers start with 2
            array('regex' => '/^2\d{8}$/', 'country_code' => '64', 'country' => 'New Zealand'),
            
            // Netherlands - Mobile numbers start with 6
            array('regex' => '/^6\d{8}$/', 'country_code' => '31', 'country' => 'Netherlands'),
            
            // Belgium - Mobile numbers start with 4
            array('regex' => '/^4\d{8}$/', 'country_code' => '32', 'country' => 'Belgium'),
            
            // Switzerland - Mobile numbers start with 7
            array('regex' => '/^7\d{8}$/', 'country_code' => '41', 'country' => 'Switzerland'),
            
            // Austria - Mobile numbers start with 6
            array('regex' => '/^6\d{9}$/', 'country_code' => '43', 'country' => 'Austria'),
            
            // Sweden - Mobile numbers start with 7
            array('regex' => '/^7\d{8}$/', 'country_code' => '46', 'country' => 'Sweden'),
            
            // Norway - Mobile numbers start with 4-9
            array('regex' => '/^[4-9]\d{7}$/', 'country_code' => '47', 'country' => 'Norway'),
            
            // Denmark - Mobile numbers start with 2
            array('regex' => '/^2\d{7}$/', 'country_code' => '45', 'country' => 'Denmark'),
            
            // Finland - Mobile numbers start with 4-5
            array('regex' => '/^[45]\d{8}$/', 'country_code' => '358', 'country' => 'Finland'),
            
            // Poland - Mobile numbers start with 5-9
            array('regex' => '/^[5-9]\d{8}$/', 'country_code' => '48', 'country' => 'Poland'),
            
            // Czech Republic - Mobile numbers start with 6-7
            array('regex' => '/^[67]\d{8}$/', 'country_code' => '420', 'country' => 'Czech Republic'),
            
            // Hungary - Mobile numbers start with 2-3
            array('regex' => '/^[23]\d{8}$/', 'country_code' => '36', 'country' => 'Hungary'),
            
            // Romania - Mobile numbers start with 7
            array('regex' => '/^7\d{8}$/', 'country_code' => '40', 'country' => 'Romania'),
            
            // Bulgaria - Mobile numbers start with 8-9
            array('regex' => '/^[89]\d{8}$/', 'country_code' => '359', 'country' => 'Bulgaria'),
            
            // Croatia - Mobile numbers start with 9
            array('regex' => '/^9\d{8}$/', 'country_code' => '385', 'country' => 'Croatia'),
            
            // Serbia - Mobile numbers start with 6
            array('regex' => '/^6\d{8}$/', 'country_code' => '381', 'country' => 'Serbia'),
            
            // Ukraine - Mobile numbers start with 5-9
            array('regex' => '/^[5-9]\d{8}$/', 'country_code' => '380', 'country' => 'Ukraine'),
            
            // Belarus - Mobile numbers start with 2-5
            array('regex' => '/^[2-5]\d{8}$/', 'country_code' => '375', 'country' => 'Belarus'),
            
            // Lithuania - Mobile numbers start with 6
            array('regex' => '/^6\d{7}$/', 'country_code' => '370', 'country' => 'Lithuania'),
            
            // Latvia - Mobile numbers start with 2
            array('regex' => '/^2\d{7}$/', 'country_code' => '371', 'country' => 'Latvia'),
            
            // Estonia - Mobile numbers start with 5
            array('regex' => '/^5\d{7}$/', 'country_code' => '372', 'country' => 'Estonia'),
            
            // Greece - Mobile numbers start with 6
            array('regex' => '/^6\d{8}$/', 'country_code' => '30', 'country' => 'Greece'),
            
            // Portugal - Mobile numbers start with 9
            array('regex' => '/^9\d{8}$/', 'country_code' => '351', 'country' => 'Portugal'),
            
            // Ireland - Mobile numbers start with 8
            array('regex' => '/^8\d{8}$/', 'country_code' => '353', 'country' => 'Ireland'),
            
            // Iceland - Mobile numbers start with 6-9
            array('regex' => '/^[6-9]\d{6}$/', 'country_code' => '354', 'country' => 'Iceland'),
            
            // Malta - Mobile numbers start with 7-9
            array('regex' => '/^[7-9]\d{7}$/', 'country_code' => '356', 'country' => 'Malta'),
            
            // Cyprus - Mobile numbers start with 9
            array('regex' => '/^9\d{7}$/', 'country_code' => '357', 'country' => 'Cyprus'),
            
            // Luxembourg - Mobile numbers start with 6
            array('regex' => '/^6\d{8}$/', 'country_code' => '352', 'country' => 'Luxembourg'),
            
            // Monaco - Mobile numbers start with 6
            array('regex' => '/^6\d{8}$/', 'country_code' => '377', 'country' => 'Monaco'),
            
            // Andorra - Mobile numbers start with 3-6
            array('regex' => '/^[3-6]\d{7}$/', 'country_code' => '376', 'country' => 'Andorra'),
            
            // San Marino - Mobile numbers start with 3-6
            array('regex' => '/^[3-6]\d{7}$/', 'country_code' => '378', 'country' => 'San Marino'),
            
            // Liechtenstein - Mobile numbers start with 7
            array('regex' => '/^7\d{8}$/', 'country_code' => '423', 'country' => 'Liechtenstein'),
            
            // Vatican City - Mobile numbers start with 3-6
            array('regex' => '/^[3-6]\d{7}$/', 'country_code' => '379', 'country' => 'Vatican City'),
        );
    }

    private function get_client_ip() {
        $ipaddress = '';
        
        $ip_sources = array(
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        );
        
        foreach ($ip_sources as $source) {
            if ( isset( $_SERVER[ $source ] ) ) {
                $candidate = sanitize_text_field( wp_unslash( $_SERVER[ $source ] ) );
                if ( filter_var( $candidate, FILTER_VALIDATE_IP ) ) {
                    $ipaddress = $candidate;
                    break;
                }
            }
        }
        
        if (empty($ipaddress)) {
            $ipaddress = 'UNKNOWN';
        }
            
        return $ipaddress;
    }

    private function get_api_url($endpoint) {
        return $this->base_url . $endpoint;
    }

    /**
     * Recursively search for any value that looks like an ID
     */
    private function find_id_like_value($data, $max_depth = 3, $current_depth = 0) {
        if ($current_depth >= $max_depth || !is_array($data)) {
            return null;
        }
        
        foreach ($data as $key => $value) {
            if (is_string($value)) {
                // Check if the value looks like an ID (alphanumeric, 8+ characters)
                if (preg_match('/^[a-zA-Z0-9_-]{8,}$/', $value)) {
                    return $value;
                }
            } elseif (is_array($value)) {
                $found = $this->find_id_like_value($value, $max_depth, $current_depth + 1);
                if ($found) {
                    return $found;
                }
            }
        }
        
        return null;
    }

    /**
     * Determine OTP method based on input type.
     *
     * @param string $input Email or mobile number.
     * @return string Method type: 'email', 'whatsapp', 'sms', 'voice'.
     */
    private function determine_otp_method($input) {
        // Return email if input is empty or null
        if (empty($input)) {
            return 'email';
        }
        
        // Remove all non-alphanumeric characters except +, @ and .
        $clean_input = preg_replace('/[^0-9a-zA-Z+@.]/', '', $input);

        // Check if it's an email
        if (filter_var($clean_input, FILTER_VALIDATE_EMAIL)) {
            return 'email';
        }

        // Check if it's a mobile number
        if (preg_match('/^\+?[0-9]{10,15}$/', $clean_input)) {
            // Check which methods are enabled
            $whatsapp_enabled = get_option('authyo_otp_auth_whatsapp_enabled', 'no') === 'yes';
            $sms_enabled = get_option('authyo_otp_auth_sms_enabled', 'no') === 'yes';
            $voice_enabled = get_option('authyo_otp_auth_voice_enabled', 'no') === 'yes';
            
            // Get the admin-selected priority method
            $priority_method = get_option('authyo_otp_auth_priority_method', 'whatsapp');
            
            // If priority method is enabled, use it
            $method_enabled = false;
            switch ($priority_method) {
                case 'whatsapp':
                    $method_enabled = $whatsapp_enabled;
                    break;
                case 'sms':
                    $method_enabled = $sms_enabled;
                    break;
                case 'voice':
                    $method_enabled = $voice_enabled;
                    break;
            }
            
            // Log for debugging
            // Debug code removed for production - uncomment if needed during development
            // if (defined('WP_DEBUG') && WP_DEBUG) {
            //     error_log('=== METHOD DETERMINATION ===');
            //     error_log('Priority Method Setting: ' . $priority_method);
            //     error_log('WhatsApp Enabled: ' . ($whatsapp_enabled ? 'yes' : 'no'));
            //     error_log('SMS Enabled: ' . ($sms_enabled ? 'yes' : 'no'));
            //     error_log('Voice Enabled: ' . ($voice_enabled ? 'yes' : 'no'));
            //     error_log('Priority Method Enabled? ' . ($method_enabled ? 'yes' : 'no'));
            // }
            
            if ($method_enabled) {
                // Debug code removed for production - uncomment if needed during development
                // if (defined('WP_DEBUG') && WP_DEBUG) {
                //     error_log('Using Priority Method: ' . $priority_method);
                //     error_log('=========================');
                // }
                return $priority_method;
            }
            
            // If priority method is not enabled, fall back to first available enabled method
            if ($whatsapp_enabled) {
                // Debug code removed for production - uncomment if needed during development
                // if (defined('WP_DEBUG') && WP_DEBUG) {
                //     error_log('Priority method not enabled, using WhatsApp as fallback');
                //     error_log('=========================');
                // }
                return 'whatsapp';
            } elseif ($sms_enabled) {
                // Debug code removed for production - uncomment if needed during development
                // if (defined('WP_DEBUG') && WP_DEBUG) {
                //     error_log('Priority method not enabled, using SMS as fallback');
                //     error_log('=========================');
                // }
                return 'sms';
            } elseif ($voice_enabled) {
                // Debug code removed for production - uncomment if needed during development
                // if (defined('WP_DEBUG') && WP_DEBUG) {
                //     error_log('Priority method not enabled, using Voice as fallback');
                //     error_log('=========================');
                // }
                return 'voice';
            } else {
                // Debug code removed for production - uncomment if needed during development
                // if (defined('WP_DEBUG') && WP_DEBUG) {
                //     error_log('No methods enabled, defaulting to SMS');
                //     error_log('=========================');
                // }
                // Default to SMS if no preference
                return 'sms';
            }
        }

        // Default to email
        return 'email';
    }

    /**
     * Get the next available OTP method to try when current method fails.
     *
     * @param string $input Email or mobile number.
     * @param array $tried_methods Methods already attempted.
     * @return string|null Next method to try, or null if no methods available.
     */
    private function get_next_available_method($input, $tried_methods = array()) {
        // Check if it's an email - only email method available
        if (filter_var($input, FILTER_VALIDATE_EMAIL)) {
            return in_array('email', $tried_methods) ? null : 'email';
        }
        
        // For mobile numbers, check enabled methods
        $whatsapp_enabled = get_option('authyo_otp_auth_whatsapp_enabled', 'no') === 'yes';
        $sms_enabled = get_option('authyo_otp_auth_sms_enabled', 'no') === 'yes';
        $voice_enabled = get_option('authyo_otp_auth_voice_enabled', 'no') === 'yes';
        
        // Get priority method setting
        $priority_method = get_option('authyo_otp_auth_priority_method', 'whatsapp');
        
        // Build priority order: priority method first, then others
        $priority_order = array();
        
        // Add priority method first if enabled and not tried
        if (!in_array($priority_method, $tried_methods)) {
            $priority_enabled = false;
            switch ($priority_method) {
                case 'whatsapp':
                    $priority_enabled = $whatsapp_enabled;
                    break;
                case 'sms':
                    $priority_enabled = $sms_enabled;
                    break;
                case 'voice':
                    $priority_enabled = $voice_enabled;
                    break;
            }
            if ($priority_enabled) {
                $priority_order[] = $priority_method;
            }
        }
        
        // Add other enabled methods that haven't been tried, in order: whatsapp, sms, voice
        $all_methods = array('whatsapp', 'sms', 'voice');
        foreach ($all_methods as $method) {
            if ($method === $priority_method) {
                continue; // Already added priority method
            }
            if (in_array($method, $tried_methods)) {
                continue; // Already tried
            }
            
            $is_enabled = false;
            switch ($method) {
                case 'whatsapp':
                    $is_enabled = $whatsapp_enabled;
                    break;
                case 'sms':
                    $is_enabled = $sms_enabled;
                    break;
                case 'voice':
                    $is_enabled = $voice_enabled;
                    break;
            }
            
            if ($is_enabled) {
                $priority_order[] = $method;
            }
        }
        
        // Return first method from priority order
        if (!empty($priority_order)) {
            return $priority_order[0];
        }
        
        // No available methods left
        return null;
    }

    /**
     * Get available fallback methods for a given input.
     *
     * @param string $input Email or mobile number.
     * @param string $exclude_method Method to exclude from list.
     * @return array Available fallback methods.
     */
    public function get_available_fallback_methods($input, $exclude_method = null) {
        $available_methods = array();

        // Check if it's an email
        if (filter_var($input, FILTER_VALIDATE_EMAIL)) {
            // For email, only email method is available
            if ($exclude_method !== 'email') {
                $email_enabled = get_option('authyo_otp_auth_email_enabled', 'yes') === 'yes';
                if ($email_enabled) {
                    $available_methods['email'] = esc_html__('Email', 'authyo-otp-authentication-for-woocommerce');
                }
            }
        } else {
            // For mobile, check all mobile-enabled methods
            $whatsapp_enabled = get_option('authyo_otp_auth_whatsapp_enabled', 'no') === 'yes';
            $sms_enabled = get_option('authyo_otp_auth_sms_enabled', 'no') === 'yes';
            $voice_enabled = get_option('authyo_otp_auth_voice_enabled', 'no') === 'yes';

            if ($exclude_method !== 'whatsapp' && $whatsapp_enabled) {
                $available_methods['whatsapp'] = esc_html__('WhatsApp', 'authyo-otp-authentication-for-woocommerce');
            }
            if ($exclude_method !== 'sms' && $sms_enabled) {
                $available_methods['sms'] = esc_html__('SMS', 'authyo-otp-authentication-for-woocommerce');
            }
            if ($exclude_method !== 'voice' && $voice_enabled) {
                $available_methods['voice'] = esc_html__('Voice Call', 'authyo-otp-authentication-for-woocommerce');
            }
        }

        return $available_methods;
    }

    /**
     * Process phone number to ensure proper format for SMS delivery
     * 
     * @param string $phone_number The phone number to process
     * @return string Processed phone number in international format
     */
    private function process_phone_number($phone_number) {
        // If it's an email, return as is
        if (filter_var($phone_number, FILTER_VALIDATE_EMAIL)) {
            return $phone_number;
        }
        
        // Clean the phone number - remove all non-digit characters except +
        $clean_phone = preg_replace('/[^0-9+]/', '', $phone_number);
        
        // If phone number already starts with +, return as is
        if (strpos($clean_phone, '+') === 0) {
            return $clean_phone;
        }
        
        // If phone number doesn't start with +, we need to determine the country code
        // This is a fallback for cases where frontend didn't properly format the number
        
        // Check if it's a valid mobile number (10-15 digits)
        if (preg_match('/^[0-9]{10,15}$/', $clean_phone)) {
            return $this->detect_country_code_dynamic($clean_phone);
        }
        
        // If it doesn't match expected patterns, return as is
        return $phone_number;
    }
} 