<?php
/**
 * The public-facing functionality of the plugin.
 */

if (!defined('ABSPATH')) {
    exit;
}

// Include the SDK class
require_once plugin_dir_path(__FILE__) . 'class-authyo-otp-auth-sdk.php';

class Authyo_OTP_Auth_Public {
    private static $instance = null;
    private $plugin_name;
    private $version;
    private $debug = false;

    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct() {
        $this->plugin_name = 'authyo-otp-authentication-for-woocommerce';
        $this->version = AUTHYO_OTP_AUTH_VERSION;
        $this->debug = false;
        
        // Quick enable fallback via URL parameter (admin only, for debugging)
        add_action('init', array($this, 'handle_quick_enable_fallback'));

        // AJAX actions for login OTP
        add_action('wp_ajax_authyo_otp_auth_send_login_otp', array($this, 'send_login_otp'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_send_login_otp', array($this, 'send_login_otp'));

        add_action('wp_ajax_authyo_otp_auth_send_register_otp', array($this, 'send_register_otp'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_send_register_otp', array($this, 'send_register_otp'));

        add_action('wp_ajax_authyo_otp_auth_resend_fallback', array($this, 'resend_fallback_otp'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_resend_fallback', array($this, 'resend_fallback_otp'));

        add_action('wp_ajax_authyo_otp_auth_verify_register_otp', array($this, 'verify_register_otp'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_verify_register_otp', array($this, 'verify_register_otp'));

        // Legacy AJAX actions for backward compatibility
        add_action('wp_ajax_authyo_otp_auth_login', array($this, 'handle_login_form'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_login', array($this, 'handle_login_form'));

        // Legacy AJAX actions for backward compatibility
        add_action('wp_ajax_authyo_otp_auth', array($this, 'handle_login_form'));
        add_action('wp_ajax_nopriv_authyo_otp_auth', array($this, 'handle_login_form'));
        
        // Legacy AJAX actions for old shortcodes
        add_action('wp_ajax_authyo_otp_login', array($this, 'handle_login_form'));
        add_action('wp_ajax_nopriv_authyo_otp_login', array($this, 'handle_login_form'));

        // Legacy shortcodes for backward compatibility
        add_shortcode('authyo_otp_auth_login', array($this, 'render_login_form'));
        add_shortcode('authyo_otp_auth_register', array($this, 'render_register_form'));

        // Additional legacy shortcodes
        add_shortcode('authyo_otp_auth', array($this, 'render_login_form'));
        add_shortcode('authyo_otp_register', array($this, 'render_register_form'));
        
        // Legacy shortcodes for backward compatibility
        add_shortcode('authyo_otp_login', array($this, 'render_login_form'));
        add_shortcode('authyo_otp_register', array($this, 'render_register_form'));

        // WooCommerce already has billing_phone field in account form, no need to add custom field
        
        
        // Add mobile number to WordPress admin
        add_action('show_user_profile', array($this, 'add_mobile_field_to_admin_profile'));
        add_action('edit_user_profile', array($this, 'add_mobile_field_to_admin_profile'));
        add_action('personal_options_update', array($this, 'save_mobile_field_from_admin_profile'));
        add_action('edit_user_profile_update', array($this, 'save_mobile_field_from_admin_profile'));
        add_filter('manage_users_columns', array($this, 'add_mobile_column_to_users_table'));
        add_filter('manage_users_custom_column', array($this, 'show_mobile_in_users_table'), 10, 3);

        // Enqueue scripts and styles
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));

        // Legacy register OTP aliases
        add_action('wp_ajax_send_register_otp', array($this, 'send_register_otp'));
        add_action('wp_ajax_nopriv_send_register_otp', array($this, 'send_register_otp'));

        add_action('wp_ajax_verify_register_otp', array($this, 'verify_register_otp'));
        add_action('wp_ajax_nopriv_verify_register_otp', array($this, 'verify_register_otp'));

        // AJAX handler for country list
        add_action('wp_ajax_authyo_otp_auth_get_country_list', array($this, 'get_country_list'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_get_country_list', array($this, 'get_country_list'));

        // Note: wp_localize_script calls are now in enqueue_scripts() method
        // to ensure they only run when scripts are actually enqueued
    }

    /**
     * Quick enable fallback via URL parameter (admin only)
     * Usage: Add ?authyo_enable_fallback=1 to any page URL as admin
     */
    public function handle_quick_enable_fallback() {
        if (!isset($_GET['authyo_enable_fallback']) || !current_user_can('manage_options')) {
            return;
        }
        
        // Verify nonce if provided
        if (isset($_GET['_wpnonce']) && !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'authyo_quick_enable')) {
            return;
        }
        
        // Enable all fallback settings
        update_option('authyo_otp_auth_fallback_enabled', 'yes');
        update_option('authyo_otp_auth_sms_enabled', 'yes');
        update_option('authyo_otp_auth_whatsapp_enabled', 'yes');
        update_option('authyo_otp_auth_voice_enabled', 'yes');
        
        // Show admin notice
        add_action('admin_notices', function() {
            ?>
            <div class="notice notice-success is-dismissible">
                <p><strong>Authyo OTP:</strong> Fallback methods have been enabled successfully!</p>
            </div>
            <?php
        });
    }
    
    public function enqueue_scripts() {
        // Get plugin URL
        $plugin_url = defined('AUTHYO_OTP_AUTH_PLUGIN_URL') ? AUTHYO_OTP_AUTH_PLUGIN_URL : plugin_dir_url(dirname(__FILE__) . '/../');
        
        // Check which shortcodes are present on the current page
        global $post;
        $has_login_form = false;
        $has_register_form = false;
        $has_auth_form = false;
        
        if (is_a($post, 'WP_Post')) {
            $has_login_form = has_shortcode($post->post_content, 'authyo_otp_auth_login') || 
                             has_shortcode($post->post_content, 'authyo_otp_login') ||
                             has_shortcode($post->post_content, 'authyo_otp_auth');
            $has_register_form = has_shortcode($post->post_content, 'authyo_otp_auth_register') || 
                                has_shortcode($post->post_content, 'authyo_otp_register');
        }
        
        // Fallback: Check if we're on a page that might use shortcodes via page builders or widgets
        // This ensures scripts load even when shortcode detection fails (e.g., with page builders, cache)
        if (!$has_login_form && !$has_register_form && !$has_auth_form) {
            // Check if the current page content contains our forms (for page builders)
            if (is_a($post, 'WP_Post')) {
                $content_lower = strtolower($post->post_content);
                if (strpos($content_lower, 'authyo') !== false || 
                    strpos($content_lower, 'otp') !== false ||
                    strpos($content_lower, 'login-form') !== false ||
                    strpos($content_lower, 'register-form') !== false) {
                    // Likely contains our forms, load scripts
                    $has_login_form = strpos($content_lower, 'login') !== false;
                    $has_register_form = strpos($content_lower, 'register') !== false;
                }
            }
        }
        
        // Enqueue common styles
        wp_enqueue_style(
            'authyo-otp-authentication-for-woocommerce-public',
            $plugin_url . 'assets/css/public.css',
            array(),
            $this->version
        );

        // Enqueue form-specific styles based on shortcodes present
        if ($has_auth_form) {
            wp_enqueue_style(
                'authyo-otp-authentication-for-woocommerce-form',
                $plugin_url . 'assets/css/auth-form.css',
                array(),
                $this->version
            );
        }

        if ($has_register_form) {
            wp_enqueue_style(
                'authyo-otp-authentication-for-woocommerce-register',
                $plugin_url . 'assets/css/register-form.css',
                array(),
                $this->version
            );
        }

        if ($has_login_form) {
            // Login form specific assets
            wp_enqueue_style(
                'authyo-otp-authentication-for-woocommerce-login',
                $plugin_url . 'assets/css/login-form.css',
                array(),
                $this->version
            );
        }

        // Universal theme compatibility CSS - loaded last to override theme styles
        wp_enqueue_style(
            'authyo-otp-authentication-for-woocommerce-universal-fix',
            $plugin_url . 'assets/css/universal-theme-fix.css',
            array('authyo-otp-authentication-for-woocommerce-public'),
            $this->version
        );

        // Enqueue scripts based on which forms are present
        // Only enqueue public.js if no specialized forms are present
        if (!$has_login_form && !$has_register_form && !$has_auth_form) {
            wp_enqueue_script(
                'authyo-otp-authentication-for-woocommerce-public',
                $plugin_url . 'assets/js/public.js',
                array('jquery'),
                $this->version,
                true
            );
        }

        // Enqueue form-specific scripts only if that form is present
        if ($has_auth_form) {
            wp_enqueue_script(
                'authyo-otp-authentication-for-woocommerce-form',
                $plugin_url . 'assets/js/auth-form.js',
                array('jquery'),
                $this->version,
                true
            );
        }

        if ($has_login_form) {
            wp_enqueue_script(
                'authyo-otp-authentication-for-woocommerce-login',
                $plugin_url . 'assets/js/login-form.js',
                array('jquery'),
                $this->version,
                true
            );
        }

        if ($has_register_form) {
            wp_enqueue_script(
                'authyo-otp-authentication-for-woocommerce-register',
                $plugin_url . 'assets/js/register-form.js',
                array('jquery'),
                $this->version,
                true
            );
        }

        // Get my account page URL
        $my_account_url = authyo_otp_auth_get_page_permalink('myaccount');
        if (!$my_account_url) {
            $my_account_url = home_url('/my-account/');
        }

        // Get OTP configuration from settings
        $otp_length = get_option('authyo_otp_auth_length', 6);
        $otp_expiry = get_option('authyo_otp_auth_expiry', 300);
        
        // Get method enablement status
        $email_enabled = get_option('authyo_otp_auth_email_enabled', 'yes') === 'yes';
        $whatsapp_enabled = get_option('authyo_otp_auth_whatsapp_enabled', 'no') === 'yes';
        $sms_enabled = get_option('authyo_otp_auth_sms_enabled', 'no') === 'yes';
        $voice_enabled = get_option('authyo_otp_auth_voice_enabled', 'no') === 'yes';
        $fallback_enabled = get_option('authyo_otp_auth_fallback_enabled', 'yes') === 'yes';

        // Localize scripts - ALWAYS provide this data to prevent undefined errors
        if (!$has_login_form && !$has_register_form && !$has_auth_form) {
            wp_localize_script('authyo-otp-authentication-for-woocommerce-public', 'authyoOtpAuth', array(
                'ajaxurl' => esc_url(admin_url('admin-ajax.php')),
                'nonce' => wp_create_nonce('authyo_otp_auth_nonce'),
                'myAccountUrl' => esc_url($my_account_url),
                'otpLength' => absint($otp_length),
                'otpExpiry' => absint($otp_expiry),
                'emailEnabled' => $email_enabled,
                'whatsappEnabled' => $whatsapp_enabled,
                'smsEnabled' => $sms_enabled,
                'voiceEnabled' => $voice_enabled,
                'fallbackEnabled' => $fallback_enabled,
                'messages' => array(
                    'login' => esc_html__('Login', 'authyo-otp-authentication-for-woocommerce'),
                    'register' => esc_html__('Register', 'authyo-otp-authentication-for-woocommerce'),
                    'reset' => esc_html__('Reset Password', 'authyo-otp-authentication-for-woocommerce'),
                    'sending' => esc_html__('Sending...', 'authyo-otp-authentication-for-woocommerce'),
                    'verifying' => esc_html__('Verifying...', 'authyo-otp-authentication-for-woocommerce'),
                    'sent' => esc_html__('OTP Sent!', 'authyo-otp-authentication-for-woocommerce'),
                    'error' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce'),
                    'enter_email' => esc_html__('Please enter a valid email or mobile number.', 'authyo-otp-authentication-for-woocommerce'),
                    'send_otp' => esc_html__('Send OTP', 'authyo-otp-authentication-for-woocommerce'),
                    'resend' => esc_html__('Resend OTP', 'authyo-otp-authentication-for-woocommerce'),
                    'mobile_preview' => esc_html__('Mobile number will be sent as: +', 'authyo-otp-authentication-for-woocommerce'),
                    'all_fields_required' => esc_html__('All fields are required.', 'authyo-otp-authentication-for-woocommerce')
                )
            ));
        }

        // Localize script for login form with fallback settings
        if ($has_login_form) {
            wp_localize_script('authyo-otp-authentication-for-woocommerce-login', 'authyoOtpAuth', array(
                'ajaxurl' => esc_url(admin_url('admin-ajax.php')),
                'nonce' => wp_create_nonce('authyo_otp_auth_nonce'),
                'otpLength' => absint($otp_length),
                'otpExpiry' => absint($otp_expiry),
                'emailEnabled' => $email_enabled,
                'whatsappEnabled' => $whatsapp_enabled,
                'smsEnabled' => $sms_enabled,
                'voiceEnabled' => $voice_enabled,
                'fallbackEnabled' => $fallback_enabled,
                'messages' => array(
                    'login' => esc_html__('Login', 'authyo-otp-authentication-for-woocommerce'),
                    'sending' => esc_html__('Sending...', 'authyo-otp-authentication-for-woocommerce'),
                    'verifying' => esc_html__('Verifying...', 'authyo-otp-authentication-for-woocommerce'),
                    'sent' => esc_html__('OTP Sent!', 'authyo-otp-authentication-for-woocommerce'),
                    'error' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce'),
                    'enter_email' => esc_html__('Please enter a valid email or mobile number.', 'authyo-otp-authentication-for-woocommerce'),
                    'send_otp' => esc_html__('Send OTP', 'authyo-otp-authentication-for-woocommerce'),
                    'resend' => esc_html__('Resend OTP', 'authyo-otp-authentication-for-woocommerce'),
                    'mobile_preview' => esc_html__('Mobile number will be sent as: +', 'authyo-otp-authentication-for-woocommerce'),
                    'all_fields_required' => esc_html__('All fields are required.', 'authyo-otp-authentication-for-woocommerce')
                )
            ));
        }

        // Localize script for register form with updated nonce
        if ($has_register_form) {
            wp_localize_script('authyo-otp-authentication-for-woocommerce-register', 'authyoOtpRegister', array(
                'ajaxurl' => esc_url(admin_url('admin-ajax.php')),
                'nonce' => wp_create_nonce('authyo_otp_auth_nonce'),
                'otpLength' => absint($otp_length),
                'otpExpiry' => absint($otp_expiry),
                'emailEnabled' => $email_enabled,
                'whatsappEnabled' => $whatsapp_enabled,
                'smsEnabled' => $sms_enabled,
                'voiceEnabled' => $voice_enabled,
                'fallbackEnabled' => $fallback_enabled,
                'messages' => array(
                    'sending' => esc_html__('Sending OTP...', 'authyo-otp-authentication-for-woocommerce'),
                    'verifying' => esc_html__('Verifying OTP...', 'authyo-otp-authentication-for-woocommerce'),
                    'sent' => esc_html__('OTP Sent!', 'authyo-otp-authentication-for-woocommerce'),
                    'error' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce'),
                    'enter_email' => esc_html__('Please enter your email or mobile number.', 'authyo-otp-authentication-for-woocommerce'),
                    'send_otp' => esc_html__('Send OTP', 'authyo-otp-authentication-for-woocommerce'),
                    'resend' => esc_html__('Resend OTP', 'authyo-otp-authentication-for-woocommerce'),
                    'mobile_preview' => esc_html__('Mobile number will be sent as: ', 'authyo-otp-authentication-for-woocommerce'),
                    'all_fields_required' => esc_html__('All fields are required.', 'authyo-otp-authentication-for-woocommerce')
                )
            ));
        }
    }

    public function render_login_form() {
        ob_start();
        include AUTHYO_OTP_AUTH_PLUGIN_DIR . 'templates/login-form.php';
        return ob_get_clean();
    }

    public function render_register_form() {
        ob_start();
        include AUTHYO_OTP_AUTH_PLUGIN_DIR . 'templates/register-form.php';
        return ob_get_clean();
    }

    private function log_activity($email, $otp, $status, $user_id = 0) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'authyo_otp_auth_logs';
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Plugin needs to log OTP activities to custom table
        $wpdb->insert(
            $table_name,
            array(
                'email' => sanitize_email($email),
                'otp' => sanitize_text_field($otp),
                'status' => sanitize_text_field($status),
                'ip_address' => $this->get_client_ip(),
                'created_at' => current_time('mysql'),
                'user_id' => absint($user_id)
            ),
            array('%s', '%s', '%s', '%s', '%s', '%d')
        );
    }

    private function is_valid_mobile($mobile) {
        // Return false if mobile is empty or null
        if (empty($mobile)) {
            return false;
        }
        
        // Remove any non-digit characters except +
        $mobile = preg_replace('/[^0-9+]/', '', $mobile);
        
        // Check if mobile number has at least 1 digit after +
        if (strpos($mobile, '+') === 0) {
            // Must have at least 7 digits after + for international format (global support)
            $digits_after_plus = substr($mobile, 1);
            return strlen($digits_after_plus) >= 7 && strlen($digits_after_plus) <= 15 && ctype_digit($digits_after_plus);
        }
        
        // If no + prefix, check if it has 7-15 digits (global mobile format support)
        return strlen($mobile) >= 7 && strlen($mobile) <= 15 && ctype_digit($mobile);
    }

    private function is_valid_email($email) {
        return filter_var($email, FILTER_VALIDATE_EMAIL);
    }

    /**
     * Find user by phone number - handles both formats (+91xxxxxxxxxx and 91xxxxxxxxxx)
     */
    private function find_user_by_phone($phone) {
        global $wpdb;
        
        // Return null if phone is empty
        if (empty($phone)) {
            return null;
        }
        
        // Clean the input phone number
        $clean_phone = preg_replace('/[^0-9+]/', '', $phone);
        
        // Try different variations of the phone number
        $phone_variations = array();
        
        // If phone starts with +, try both with and without +
        if (strpos($clean_phone, '+') === 0) {
            $phone_variations[] = $clean_phone; // +91xxxxxxxxxx
            $phone_variations[] = ltrim($clean_phone, '+'); // 91xxxxxxxxxx
        } else {
            $phone_variations[] = $clean_phone; // 91xxxxxxxxxx
            $phone_variations[] = '+' . $clean_phone; // +91xxxxxxxxxx
        }
        
        // Remove duplicates
        $phone_variations = array_unique($phone_variations);
        
        // Search for user with any of these phone number variations
        foreach ($phone_variations as $phone_var) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin needs to query user meta for phone authentication
            $user_id = $wpdb->get_var($wpdb->prepare(
                "SELECT user_id FROM {$wpdb->usermeta} WHERE meta_key = 'billing_phone' AND meta_value = %s",
                sanitize_text_field($phone_var)
            ));
            
            if ($user_id) {
                return $user_id;
            }
        }
        
        return null;
    }

    public function send_login_otp() {
        // Add nonce security check
        if ( ! isset( $_POST['authyo_otp_auth_login_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['authyo_otp_auth_login_nonce'] ) ), 'authyo_otp_auth_login' ) ) {
            wp_send_json_error( array( 'message' => 'Invalid nonce.' ) );
            return;
        }

        // Verify nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
        if (!check_ajax_referer('authyo_otp_auth_nonce', 'nonce', false) || empty($nonce)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }

        // Get and sanitize required data
        $email = isset($_POST['email']) ? sanitize_text_field(wp_unslash($_POST['email'])) : '';
        $otp = isset($_POST['otp']) ? sanitize_text_field(wp_unslash($_POST['otp'])) : '';

        // Rate limiting: Check if too many requests from this email/IP
        $rate_limit_key = 'authyo_otp_rate_limit_' . md5($email . $this->get_client_ip());
        $rate_limit_data = get_transient($rate_limit_key);
        
        if ($rate_limit_data) {
            $attempts = isset($rate_limit_data['attempts']) ? (int)$rate_limit_data['attempts'] : 0;
            $last_attempt = isset($rate_limit_data['time']) ? (int)$rate_limit_data['time'] : 0;
            
            // Allow max 5 attempts per 10 minutes
            if ($attempts >= 5 && (time() - $last_attempt) < 600) {
                $this->log_activity($email, '', 'rate_limit_exceeded');
                wp_send_json_error(array(
                    'message' => esc_html__('Too many OTP requests. Please try again in a few minutes.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }
        }
        
        // Update rate limit counter
        set_transient($rate_limit_key, array(
            'attempts' => isset($rate_limit_data['attempts']) ? $rate_limit_data['attempts'] + 1 : 1,
            'time' => time()
        ), 600); // 10 minutes

        try {
            // Check if input is email or mobile
            $is_email = $this->is_valid_email($email);
            $is_mobile = $this->is_valid_mobile($email);

            if (!$is_email && !$is_mobile) {
                $this->log_activity($email, '', 'invalid_input');
                wp_send_json_error(array(
                    'message' => esc_html__('Please enter a valid email or mobile number.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            global $wpdb;

            // Check if user exists based on input type
            if ($is_mobile) {
                // Use the new helper function to find user by phone (handles both formats)
                $user_id = $this->find_user_by_phone($email);

                // Handle user not found
                if (!$user_id) {
                    $this->log_activity($email, '', 'mobile_not_found');
                    wp_send_json_error(array(
                        'message' => esc_html__('No account found with this mobile number.', 'authyo-otp-authentication-for-woocommerce')
                    ));
                    return;
                }
                
                // Get user object
                $user = get_user_by('ID', $user_id);
            } else {
                // Check if email exists
                // get_user_by internally uses $wpdb and is a standard WP function.
                $user = get_user_by('email', $email);
                if (!$user) {
                    $this->log_activity($email, '', 'email_not_found');
                    wp_send_json_error(array(
                        'message' => esc_html__('No account found with this email address.', 'authyo-otp-authentication-for-woocommerce')
                    ));
                    return;
                }
            }

            // Send OTP via Authyo
            // This calls the SDK, which should handle its own sanitization/preparation for its API calls.
            $authyo = new Authyo_OTP_Auth_SDK();
            
            // Debug log the email/mobile being sent
            // Debug code removed for production - uncomment if needed during development
            // if (defined('WP_DEBUG') && WP_DEBUG) {
            //     error_log('=== AUTHYO PUBLIC SEND LOGIN OTP ===');
            //     error_log('Email/Mobile being sent: ' . $email);
            //     error_log('Is Mobile: ' . ($is_mobile ? 'yes' : 'no'));
            //     error_log('Is Email: ' . ($is_email ? 'yes' : 'no'));
            //     error_log('=====================================');
            // }
            
            $result = $authyo->send_otp($email, null, 'login');

            if (is_wp_error($result)) {
                $this->log_activity($email, '', 'otp_send_failed', isset($user->ID) ? $user->ID : 0);
                
                $error_message = $result->get_error_message();
                
                // Handle service unavailable errors with helpful suggestions
                if ($result->get_error_code() === 'service_unavailable') {
                    // Get available fallback methods
                    $authyo = new Authyo_OTP_Auth_SDK();
                    $available_methods = $authyo->get_available_fallback_methods($email);
                    
                    if (!empty($available_methods)) {
                        $method_list = implode(', ', array_values($available_methods));
                        // translators: %s is the list of available authentication methods
                        $error_message = $error_message . ' ' . sprintf(esc_html__('Please try using: %s', 'authyo-otp-authentication-for-woocommerce'), $method_list);
                    }
                }
                
                wp_send_json_error(array(
                    'message' => esc_html($error_message)
                ));
                return;
            }

            if (!isset($result['maskId'])) {
                $this->log_activity($email, '', 'invalid_response', isset($user->ID) ? $user->ID : 0);
                wp_send_json_error(array(
                    'message' => esc_html__('Invalid response from OTP service.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Get OTP expiry time from settings
            // get_option does not require sanitization for the key, and the return value is handled.
            $otp_expiry = get_option('authyo_otp_auth_expiry', 300);

            // Get the method from the response
            $method = isset($result['method']) ? sanitize_text_field($result['method']) : 'unknown';
            
            // Store OTP data in transient
            // set_transient is a standard WP function and handles serialization/storage.
            set_transient(
                'authyo_otp_auth_login_' . md5($email),
                array(
                    'mask_id' => sanitize_text_field($result['maskId']),
                    'time' => time(),
                    'user_id' => isset($user->ID) ? absint($user->ID) : 0,
                    'is_mobile' => $is_mobile,
                    'method' => $method
                ),
                absint($otp_expiry)
            );

            $this->log_activity($email, '', 'otp_sent', isset($user->ID) ? $user->ID : 0);

            wp_send_json_success(array(
                'message' => esc_html__('OTP sent successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'maskId' => sanitize_text_field($result['maskId']),
                'method' => $method
            ));

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
        }
        
        wp_die();
    }

    public function verify_register_otp() {
        if ($this->debug) {
            // Debug log removed for production; uncomment if needed during development
            // error_log('Authyo OTP Auth Public Debug: verify_register_otp called');
            // error_log('Authyo OTP Auth Public Debug: POST data: ' . print_r(wp_unslash($_POST), true));
        }

        // Add nonce security check
        if ( ! isset( $_POST['authyo_otp_auth_register_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['authyo_otp_auth_register_nonce'] ) ), 'authyo_otp_auth_register' ) ) {
            wp_send_json_error( array( 'message' => 'Invalid nonce.' ) );
            return;
        }

        // Verify nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
        if (!check_ajax_referer('authyo_otp_auth_nonce', 'nonce', false) || empty($nonce)) {
            $this->log_activity(
                isset($_POST['email']) ? sanitize_text_field(wp_unslash($_POST['email'])) : '',
                isset($_POST['otp']) ? sanitize_text_field(wp_unslash($_POST['otp'])) : '',
                'security_check_failed'
            );
            wp_send_json_error(array(
                'message' => esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        $email = isset($_POST['email']) ? sanitize_text_field(wp_unslash($_POST['email'])) : '';
        $otp = isset($_POST['otp']) ? sanitize_text_field(wp_unslash($_POST['otp'])) : '';
        $maskId = isset($_POST['maskId']) ? sanitize_text_field(wp_unslash($_POST['maskId'])) : '';
        
        // Provide specific error messages for each missing field
        if (empty($email)) {
            $this->log_activity($email, $otp, 'missing_email');
            wp_send_json_error(array(
                'message' => esc_html__('Email or mobile number is required.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }
        
        if (empty($otp)) {
            $this->log_activity($email, $otp, 'missing_otp');
            wp_send_json_error(array(
                'message' => esc_html__('OTP is required.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }
        
        if (empty($maskId)) {
            $this->log_activity($email, $otp, 'missing_maskid');
            wp_send_json_error(array(
                'message' => esc_html__('Invalid session. Please request a new OTP.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Check if input is email or mobile
        $is_mobile = $this->is_valid_mobile($email);
        $is_email = $this->is_valid_email($email);

        // Check for error indicators from frontend processing
        if ($email === 'ERROR_NO_COUNTRY' || $email === 'ERROR_INVALID_FORMAT' || strpos($email, 'ERROR_') === 0) {
            $this->log_activity($email, $otp, 'invalid_input_format');
            wp_send_json_error(array(
                'message' => esc_html__('Please enter a valid email or mobile number.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        if (!$is_mobile && !$is_email) {
            $this->log_activity($email, $otp, 'invalid_input');
            wp_send_json_error(array(
                'message' => esc_html__('Please enter a valid email or mobile number.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Check OTP attempt limit before verification
        $attempt_check = $this->check_otp_attempt_limit($email);
        if ($attempt_check && $attempt_check['blocked']) {
            $this->log_activity($email, $otp, 'otp_attempt_limit_exceeded');
            wp_send_json_error(array(
                'message' => $attempt_check['message']
            ));
            return;
        }

        try {

            // Verify OTP using Authyo with maskId
            // This calls the SDK, which should handle its own sanitization/preparation for its API calls.
            $authyo = new Authyo_OTP_Auth_SDK();
            $result = $authyo->verify_otp($email, $otp, $maskId);

            if (is_wp_error($result)) {
                // Increment failed attempt counter
                $this->increment_otp_attempts($email);
                $this->log_activity($email, $otp, 'otp_verification_failed');
                wp_send_json_error(array(
                    'message' => esc_html($result->get_error_message())
                ));
                return;
            }
            
            // Reset attempt counter on successful verification
            $this->reset_otp_attempts($email);

            global $wpdb;

            // Check if user already exists
            if ($is_mobile) {
                // Use the new helper function to find user by phone (handles both formats)
                $existing_user = $this->find_user_by_phone($email);

                if ($existing_user) {
                    $this->log_activity($email, $otp, 'mobile_exists', $existing_user);
                    wp_send_json_error(array(
                        'message' => esc_html__('An account already exists with this mobile number.', 'authyo-otp-authentication-for-woocommerce')
                    ));
                    return;
                }
            } else {
                // email_exists internally uses $wpdb and is a standard WP function.
                if (email_exists($email)) {
                    $existing_user = get_user_by('email', $email);
                    $this->log_activity($email, $otp, 'email_exists', isset($existing_user->ID) ? $existing_user->ID : 0);
                    wp_send_json_error(array(
                        'message' => esc_html__('An account already exists with this email address.', 'authyo-otp-authentication-for-woocommerce')
                    ));
                    return;
                }
            }

            // Generate username
            if ($is_mobile) {
                $clean_mobile = ltrim($email, '+');
                $username = 'user' . $clean_mobile; // Simple username format: user + mobile number
            } else {
                $username = sanitize_user(current(explode('@', $email)), true);
            }
            // username_exists internally uses $wpdb and is a standard WP function.
            $username = $this->generate_unique_username($username);

            // Create user
            // wp_insert_user internally uses $wpdb and is a standard WP function.
            $userdata = array(
                'user_login' => $username,
                'user_pass' => wp_generate_password(),
                'user_registered' => current_time('mysql'),
                'display_name' => $username,
                'role' => 'customer'
            );

            if ($is_mobile) {
                // For mobile registration, store only numbers without + prefix
                $clean_mobile = ltrim($email, '+');
                $userdata['user_email'] = ''; // Set email as empty string for mobile registrations
            } else {
                // For email registration
                $userdata['user_email'] = sanitize_email($email); // Sanitize before using in array
            }

            $user_id = wp_insert_user($userdata);

            if (is_wp_error($user_id)) {
                $this->log_activity($email, $otp, 'user_creation_failed');
                wp_send_json_error(array(
                    'message' => esc_html($user_id->get_error_message())
                ));
                return;
            }

            // Update WooCommerce billing fields after user creation
            if ($is_mobile) {
                $clean_mobile = ltrim($email, '+');
                
                // Store phone number in WooCommerce billing_phone field
                update_user_meta($user_id, 'billing_phone', sanitize_text_field($clean_mobile));
                
                // Clear billing_email for mobile registrations
                update_user_meta($user_id, 'billing_email', '');
            } else {
                // Store email in WooCommerce billing_email field
                update_user_meta($user_id, 'billing_email', sanitize_email($email));
                
                // Clear billing_phone for email registrations
                update_user_meta($user_id, 'billing_phone', '');
            }


            // Log successful registration
            $this->log_activity($email, $otp, 'registration_success', $user_id);

            // Clean up OTP data
            // delete_transient is a standard WP function.
            delete_transient('authyo_otp_auth_reg_' . md5($email));

            // Log the user in
            // wp_set_auth_cookie is a standard WP function.
            wp_set_auth_cookie($user_id, true);

            // Get my account page URL
            // authyo_otp_auth_get_page_permalink is our custom function.
            $my_account_url = authyo_otp_auth_get_page_permalink('myaccount');
            if (!$my_account_url) {
                $my_account_url = home_url('/my-account/');
            }

            wp_send_json_success(array(
                'message' => esc_html__('Registration successful! Redirecting...', 'authyo-otp-authentication-for-woocommerce'),
                'redirect' => esc_url($my_account_url)
            ));

        } catch (Exception $e) {
            if ($this->debug) {
                // Debug log removed for production; uncomment if needed during development
                // error_log('Authyo OTP Auth Public Error: ' . $e->getMessage());
                // error_log('Authyo OTP Auth Public Error trace: ' . $e->getTraceAsString());
            }
            wp_send_json_error(array(
                'message' => esc_html__('An unexpected error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
        }
        
        wp_die();
    }

    private function generate_unique_username($username) {
        $original_username = $username;
        $counter = 1;
        
        if ($this->debug) {
             // Debug log removed for production; uncomment if needed during development
             // error_log('Authyo OTP Auth Public Debug: Generating unique username for: ' . $original_username); 
        }

        while (username_exists($username)) {
            $username = $original_username . $counter;
            $counter++;
        }
        
        if ($this->debug) {
             // Debug log removed for production; uncomment if needed during development
             // error_log('Authyo OTP Auth Public Debug: Generated unique username: ' . $username); 
        }
        
        return $username;
    }

    public function send_register_otp() {
        if ($this->debug) {
            // Debug log removed for production; uncomment if needed during development
            // error_log('Authyo OTP Auth Public Debug: send_register_otp called');
            // error_log('Authyo OTP Auth Public Debug: POST data: ' . print_r(wp_unslash($_POST), true));
        }

        // Add nonce security check
        if ( ! isset( $_POST['authyo_otp_auth_register_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['authyo_otp_auth_register_nonce'] ) ), 'authyo_otp_auth_register' ) ) {
            wp_send_json_error( array( 'message' => 'Invalid nonce.' ) );
            return;
        }

        // Verify nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
        if ( empty( $nonce ) || ! check_ajax_referer( 'authyo_otp_auth_nonce', 'nonce', false ) ) {
            wp_send_json_error(array(
                'message' => esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        $email = isset($_POST['email']) ? sanitize_text_field(wp_unslash($_POST['email'])) : '';
        
        if (empty($email)) {
            wp_send_json_error(array(
                'message' => esc_html__('Please enter a valid email or mobile number.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Rate limiting: Check if too many requests from this email/IP
        $rate_limit_key = 'authyo_otp_rate_limit_register_' . md5($email . $this->get_client_ip());
        $rate_limit_data = get_transient($rate_limit_key);
        
        if ($rate_limit_data) {
            $attempts = isset($rate_limit_data['attempts']) ? (int)$rate_limit_data['attempts'] : 0;
            $last_attempt = isset($rate_limit_data['time']) ? (int)$rate_limit_data['time'] : 0;
            
            // Allow max 5 attempts per 10 minutes for registration
            if ($attempts >= 5 && (time() - $last_attempt) < 600) {
                wp_send_json_error(array(
                    'message' => esc_html__('Too many registration attempts. Please try again in a few minutes.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }
        }
        
        // Update rate limit counter
        set_transient($rate_limit_key, array(
            'attempts' => isset($rate_limit_data['attempts']) ? $rate_limit_data['attempts'] + 1 : 1,
            'time' => time()
        ), 600); // 10 minutes

        // Check if input is email or mobile
        $is_mobile = $this->is_valid_mobile($email);
        $is_email = $this->is_valid_email($email);

        if (!$is_mobile && !$is_email) {
            wp_send_json_error(array(
                'message' => esc_html__('Please enter a valid email or mobile number.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        global $wpdb;

        // Check if user exists
        if ($is_mobile) {
            // Use the new helper function to find user by phone (handles both formats)
            $existing_user = $this->find_user_by_phone($email);

            if ($existing_user) {
                wp_send_json_error(array(
                    'message' => esc_html__('An account already exists with this mobile number.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }
        } else {
            if (email_exists($email)) {
                wp_send_json_error(array(
                    'message' => esc_html__('An account already exists with this email address.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }
        }

        try {
            // Send OTP
            $authyo = new Authyo_OTP_Auth_SDK();
            
            // Debug log the email/mobile being sent
            // Debug code removed for production - uncomment if needed during development
            // if (defined('WP_DEBUG') && WP_DEBUG) {
            //     error_log('=== AUTHYO PUBLIC SEND REGISTER OTP ===');
            //     error_log('Email/Mobile being sent: ' . $email);
            //     error_log('Is Mobile: ' . ($is_mobile ? 'yes' : 'no'));
            //     error_log('Is Email: ' . ($is_email ? 'yes' : 'no'));
            //     error_log('=======================================');
            // }
            
            $result = $authyo->send_otp($email, null, 'register');

            if (is_wp_error($result)) {
                if ($this->debug) {
                     // Debug log removed for production; uncomment if needed during development
                     // error_log('Authyo OTP Auth Public Debug: Failed to send registration OTP: ' . $result->get_error_message());
                }
                
                $error_message = $result->get_error_message();
                
                // Handle service unavailable errors with helpful suggestions
                if ($result->get_error_code() === 'service_unavailable') {
                    // Get available fallback methods
                    $authyo = new Authyo_OTP_Auth_SDK();
                    $available_methods = $authyo->get_available_fallback_methods($email);
                    
                    if (!empty($available_methods)) {
                        $method_list = implode(', ', array_values($available_methods));
                        // translators: %s is the list of available authentication methods
                        $error_message = $error_message . ' ' . sprintf(esc_html__('Please try using: %s', 'authyo-otp-authentication-for-woocommerce'), $method_list);
                    }
                }
                
                wp_send_json_error(array(
                    'message' => esc_html($error_message)
                ));
                return;
            }

            if ($this->debug) {
                 // Debug log removed for production; uncomment if needed during development
                 // error_log('Authyo OTP Auth Public Debug: Registration OTP sent successfully. Mask ID: ' . $result['maskId']);
            }

            // Store OTP data in transient for registration
            $otp_expiry = get_option('authyo_otp_auth_expiry', 300);
            set_transient(
                'authyo_otp_auth_reg_' . md5($email),
                array(
                    'mask_id' => sanitize_text_field($result['maskId']),
                    'time' => time(),
                    'method' => isset($result['method']) ? sanitize_text_field($result['method']) : 'unknown',
                    'is_mobile' => $is_mobile
                ),
                absint($otp_expiry)
            );

            wp_send_json_success(array(
                'message' => esc_html__('OTP sent successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'maskId' => sanitize_text_field($result['maskId']),
                'method' => isset($result['method']) ? sanitize_text_field($result['method']) : 'unknown'
            ));

        } catch (Exception $e) {
            if ($this->debug) {
                // Debug log removed for production; uncomment if needed during development
                // error_log('Authyo OTP Auth Public Error: ' . $e->getMessage());
                // error_log('Authyo OTP Auth Public Error trace: ' . $e->getTraceAsString());
            }
            wp_send_json_error(array(
                'message' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
        }
        
        wp_die();
    }

    /**
     * Check if OTP attempts have exceeded the limit
     *
     * @param string $email Email or mobile number
     * @return array|false Returns array with 'blocked' => true and 'message' if blocked, false otherwise
     */
    private function check_otp_attempt_limit($email) {
        $max_attempts = absint(get_option('authyo_otp_auth_max_attempts', 5));
        $lockout_period = absint(get_option('authyo_otp_auth_lockout_period', 15));
        
        // If max attempts is 0 or not set, no limit
        if ($max_attempts <= 0) {
            return false;
        }
        
        $attempt_key = 'authyo_otp_attempts_' . md5($email);
        $attempt_data = get_transient($attempt_key);
        
        if (!$attempt_data) {
            return false; // No previous attempts
        }
        
        $attempts = isset($attempt_data['count']) ? absint($attempt_data['count']) : 0;
        $lockout_until = isset($attempt_data['lockout_until']) ? absint($attempt_data['lockout_until']) : 0;
        $current_time = time();
        
        // Check if still in lockout period
        if ($lockout_until > $current_time) {
            $remaining_minutes = ceil(($lockout_until - $current_time) / 60);
            return array(
                'blocked' => true,
                'message' => sprintf(
                    /* translators: %d: Number of minutes remaining in lockout period */
                    esc_html__('Too many failed OTP attempts. Please try again after %d minute(s).', 'authyo-otp-authentication-for-woocommerce'),
                    $remaining_minutes
                )
            );
        }
        
        // If lockout period expired, reset attempts
        if ($lockout_until > 0 && $lockout_until <= $current_time) {
            delete_transient($attempt_key);
            return false;
        }
        
        // Check if attempts exceeded limit
        if ($attempts >= $max_attempts) {
            // Set lockout period
            $lockout_until = $current_time + ($lockout_period * 60);
            set_transient($attempt_key, array(
                'count' => $attempts,
                'lockout_until' => $lockout_until
            ), $lockout_period * 60);
            
            return array(
                'blocked' => true,
                'message' => sprintf(
                    /* translators: %d: Lockout period duration in minutes */
                    esc_html__('Too many failed OTP attempts. Please try again after %d minute(s).', 'authyo-otp-authentication-for-woocommerce'),
                    $lockout_period
                )
            );
        }
        
        return false;
    }
    
    /**
     * Increment failed OTP attempt counter
     *
     * @param string $email Email or mobile number
     * @return void
     */
    private function increment_otp_attempts($email) {
        $max_attempts = absint(get_option('authyo_otp_auth_max_attempts', 5));
        $lockout_period = absint(get_option('authyo_otp_auth_lockout_period', 15));
        
        // If max attempts is 0 or not set, don't track
        if ($max_attempts <= 0) {
            return;
        }
        
        $attempt_key = 'authyo_otp_attempts_' . md5($email);
        $attempt_data = get_transient($attempt_key);
        
        if (!$attempt_data) {
            $attempt_data = array(
                'count' => 1,
                'lockout_until' => 0
            );
        } else {
            $attempt_data['count'] = isset($attempt_data['count']) ? absint($attempt_data['count']) + 1 : 1;
        }
        
        // Store for lockout period duration (or longer if already locked)
        $expiration = $lockout_period * 60;
        if (isset($attempt_data['lockout_until']) && $attempt_data['lockout_until'] > time()) {
            $expiration = $attempt_data['lockout_until'] - time();
        }
        
        set_transient($attempt_key, $attempt_data, $expiration);
    }
    
    /**
     * Reset OTP attempt counter (on successful verification)
     *
     * @param string $email Email or mobile number
     * @return void
     */
    private function reset_otp_attempts($email) {
        $attempt_key = 'authyo_otp_attempts_' . md5($email);
        delete_transient($attempt_key);
    }

    private function get_client_ip() {
        $ip = '';
        
        // Check for IP in various server variables
        $ip_sources = array(
            'HTTP_CLIENT_IP',
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_FORWARDED',
            'HTTP_FORWARDED_FOR',
            'HTTP_FORWARDED',
            'REMOTE_ADDR'
        );
        
        foreach ($ip_sources as $source) {
            if (isset($_SERVER[$source])) {
                // Sanitize and validate the IP address in one step
                $candidate = sanitize_text_field( wp_unslash( $_SERVER[ $source ] ) );
                if ( filter_var( $candidate, FILTER_VALIDATE_IP ) ) {
                    $ip = $candidate;
                    break;
                }
            }
        }
        
        // If no valid IP found, return a default value
        if (empty($ip)) {
             if ($this->debug) {
                 // Debug log removed for production; uncomment if needed during development
                 // error_log('Authyo OTP Auth Public Debug: No valid client IP found, using 0.0.0.0'); 
             }
            $ip = '0.0.0.0';
        }

        if ($this->debug) {
             // Debug log removed for production; uncomment if needed during development
             // error_log('Authyo OTP Auth Public Debug: Client IP: ' . $ip); 
        }
            
        return $ip;
    }

    public function handle_login_form() {
        if ($this->debug) {
            // Debug log removed for production; uncomment if needed during development
            // error_log('Authyo OTP Auth Public Debug: handle_login_form called');
            // error_log('Authyo OTP Auth Public Debug: POST data: ' . print_r(wp_unslash($_POST), true));
        }

        try {
            // Add nonce security check
            if ( ! isset( $_POST['authyo_otp_auth_login_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['authyo_otp_auth_login_nonce'] ) ), 'authyo_otp_auth_login' ) ) {
                wp_send_json_error( array( 'message' => 'Invalid request. Security check failed.' ) );
                return;
            }

            // Verify nonce
            $nonce = isset($_POST['_wpnonce']) ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : '';
            $nonce_valid = false;
            if (wp_verify_nonce($nonce, 'authyo_otp_auth_login_nonce')) {
                $nonce_valid = true;
            } elseif (wp_verify_nonce($nonce, 'authyo_otp_auth_nonce')) { // Legacy value sent from authyoOtpAuth.nonce
                $nonce_valid = true;
            } elseif (wp_verify_nonce($nonce, 'authyo_otp_auth_login_nonce')) { // Legacy value generated in template
                $nonce_valid = true;
            }

            if (!$nonce_valid) {
                $this->log_activity(
                    isset($_POST['email']) ? sanitize_text_field(wp_unslash($_POST['email'])) : '',
                    isset($_POST['otp']) ? sanitize_text_field(wp_unslash($_POST['otp'])) : '',
                    'security_check_failed'
                );
                wp_send_json_error(array(
                    'message' => esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            $email = isset($_POST['email']) ? sanitize_text_field(wp_unslash($_POST['email'])) : '';
            $otp = isset($_POST['otp']) ? sanitize_text_field(wp_unslash($_POST['otp'])) : '';
            $maskId = isset($_POST['maskId']) ? sanitize_text_field(wp_unslash($_POST['maskId'])) : '';
            
            if (empty($email) || empty($otp) || empty($maskId)) {
                $this->log_activity($email, $otp, 'missing_fields');
                wp_send_json_error(array(
                    'message' => esc_html__('All fields are required.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            global $wpdb;

            // Check if input is email or mobile
            $is_mobile = $this->is_valid_mobile($email);
            $is_email = $this->is_valid_email($email);

            if (!$is_mobile && !$is_email) {
                $this->log_activity($email, $otp, 'invalid_input');
                wp_send_json_error(array(
                    'message' => esc_html__('Please enter a valid email or mobile number.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Get user based on input type
            $user = null;
            if ($is_mobile) {
                // Use the new helper function to find user by phone (handles both formats)
                $user_id = $this->find_user_by_phone($email);
                
                if ($user_id) {
                    $user = get_user_by('ID', $user_id);
                }
            } else {
                // get_user_by internally uses $wpdb and is a standard WP function.
                $user = get_user_by('email', $email);
            }

            if (!$user) {
                $this->log_activity($email, $otp, 'user_not_found');
                wp_send_json_error(array(
                    'message' => esc_html__('No account found with this email/mobile number.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Check OTP attempt limit before verification
            $attempt_check = $this->check_otp_attempt_limit($email);
            if ($attempt_check && $attempt_check['blocked']) {
                $this->log_activity($email, $otp, 'otp_attempt_limit_exceeded', $user->ID);
                wp_send_json_error(array(
                    'message' => $attempt_check['message']
                ));
                return;
            }

            // Verify OTP
            // This calls the SDK, which should handle its own sanitization/preparation for its API calls.
            $authyo = new Authyo_OTP_Auth_SDK();
            $result = $authyo->verify_otp($email, $otp, $maskId);

            if (is_wp_error($result)) {
                // Increment failed attempt counter
                $this->increment_otp_attempts($email);
                $this->log_activity($email, $otp, 'otp_verification_failed', $user->ID);
                wp_send_json_error(array(
                    'message' => esc_html($result->get_error_message())
                ));
                return;
            }
            
            // Reset attempt counter on successful verification
            $this->reset_otp_attempts($email);

            // Log successful login
            $this->log_activity($email, $otp, 'login_success', $user->ID);

            // Clean up OTP data
            // delete_transient is a standard WP function.
            delete_transient('authyo_otp_auth_login_' . md5($email));

            // Log the user in
            // wp_set_current_user and wp_set_auth_cookie are standard WP functions.
            wp_set_current_user(absint($user->ID)); // Ensure user_id is absolute integer
            wp_set_auth_cookie(absint($user->ID), true); // Ensure user_id is absolute integer

            // Get my account page URL
            // authyo_otp_auth_get_page_permalink is our custom function.
            $my_account_url = authyo_otp_auth_get_page_permalink('myaccount');
            if (!$my_account_url) {
                $my_account_url = home_url('/my-account/');
            }

            // Return success with redirect URL
            wp_send_json_success(array(
                'message' => esc_html__('Login successful!', 'authyo-otp-authentication-for-woocommerce'),
                'redirect' => esc_url($my_account_url)
            ));

        } catch (Exception $e) {
            if ($this->debug) {
                // Debug log removed for production; uncomment if needed during development
                // error_log('Authyo OTP Auth Public Error: ' . $e->getMessage());
                // error_log('Authyo OTP Auth Public Error trace: ' . $e->getTraceAsString());
            }
            wp_send_json_error(array(
                'message' => esc_html__('An unexpected error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
        }
        
        wp_die();
    }

    
    /**
     * Add mobile number field to WordPress admin user profile
     */
    public function add_mobile_field_to_admin_profile($user) {
        $mobile = get_user_meta($user->ID, 'billing_phone', true);
        ?>
        <h3><?php esc_html_e('Contact Information', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
        <table class="form-table">
            <tr>
                <th><label for="billing_phone"><?php esc_html_e('Phone Number', 'authyo-otp-authentication-for-woocommerce'); ?></label></th>
                <td>
                    <input type="tel" name="billing_phone" id="billing_phone" value="<?php echo esc_attr($mobile); ?>" class="regular-text" />
                    <p class="description"><?php esc_html_e('User\'s phone number for OTP authentication and billing.', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }
    
    /**
     * Save mobile number field from WordPress admin user profile
     */
    public function save_mobile_field_from_admin_profile($user_id) {
        if (!current_user_can('edit_user', $user_id)) {
            return false;
        }
        
        // Verify nonce for admin profile save
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'update-user_' . $user_id)) {
            return false;
        }
        
        if (isset($_POST['billing_phone'])) {
            $mobile = sanitize_text_field(wp_unslash($_POST['billing_phone']));
            
            // Validate mobile number if not empty
            if (!empty($mobile) && !$this->is_valid_mobile($mobile)) {
                add_action('user_profile_update_errors', function($errors) {
                    $errors->add('mobile_error', esc_html__('Please enter a valid mobile number.', 'authyo-otp-authentication-for-woocommerce'));
                });
                return false;
            }
            
            // Check if mobile number is already in use by another user
            if (!empty($mobile)) {
                // Use the helper function to find if phone exists, but exclude current user
                $found_user_id = $this->find_user_by_phone($mobile);
                
                if ($found_user_id && $found_user_id != $user_id) {
                    add_action('user_profile_update_errors', function($errors) {
                        $errors->add('mobile_error', esc_html__('This mobile number is already registered with another account.', 'authyo-otp-authentication-for-woocommerce'));
                    });
                    return false;
                }
            }
            
            // Update WooCommerce billing_phone field
            update_user_meta($user_id, 'billing_phone', sanitize_text_field($mobile));
        }
    }
    
    /**
     * Add phone number column to WordPress users table
     */
    public function add_mobile_column_to_users_table($columns) {
        $columns['billing_phone'] = esc_html__('Phone Number', 'authyo-otp-authentication-for-woocommerce');
        return $columns;
    }
    
    /**
     * Show phone number in WordPress users table
     */
    public function show_mobile_in_users_table($value, $column_name, $user_id) {
        if ($column_name === 'billing_phone') {
            $mobile = get_user_meta($user_id, 'billing_phone', true);
            return $mobile ? esc_html($mobile) : '—';
        }
        return $value;
    }

    /**
     * Handle fallback OTP resend request
     */
    public function resend_fallback_otp() {
        // Enable debug logging only when WP_DEBUG is enabled
        $debug = defined('WP_DEBUG') && WP_DEBUG;
        
        if ($debug) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('=== AUTHYO FALLBACK OTP REQUEST START ===');
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('Timestamp: ' . current_time('mysql'));
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log,WordPress.PHP.DevelopmentFunctions.error_log_print_r,WordPress.Security.NonceVerification.Missing -- Debug logging only runs when WP_DEBUG is enabled, logging POST data for debugging before nonce verification (actual processing happens after verification)
            error_log('POST data: ' . print_r($_POST, true));
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('Server: ' . (isset($_SERVER['HTTP_HOST']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_HOST'])) : 'unknown'));
        }
        
        // Get action type first to determine which nonce to check
        $action_type = isset($_POST['action_type']) ? sanitize_text_field(wp_unslash($_POST['action_type'])) : 'login'; // 'login' or 'register'
        
        if ($debug) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('Action Type: ' . $action_type);
        }
        
        // Verify register-specific nonce if action type is register
        if ($action_type === 'register') {
            if (!isset($_POST['authyo_otp_auth_register_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['authyo_otp_auth_register_nonce'])), 'authyo_otp_auth_register')) {
                if ($debug) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
                    error_log('Register nonce verification failed');
                }
                wp_send_json_error(array(
                    'message' => esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }
        }
        
        // Verify general nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!check_ajax_referer('authyo_otp_auth_nonce', 'nonce', false) || empty($nonce)) {
            if ($debug) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
                error_log('General nonce verification failed');
            }
            wp_send_json_error(array(
                'message' => esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Check if fallback is enabled
        $fallback_enabled = get_option('authyo_otp_auth_fallback_enabled', 'yes') === 'yes';
        
        if ($debug) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('Fallback setting check:');
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('  - Raw option value: ' . get_option('authyo_otp_auth_fallback_enabled', 'not_set'));
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('  - Evaluated as enabled: ' . ($fallback_enabled ? 'YES' : 'NO'));
        }
        
        if (!$fallback_enabled) {
            if ($debug) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
                error_log('❌ Fallback method is DISABLED in settings');
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
                error_log('TIP: Enable it in WordPress Admin → Authyo OTP → Settings → Fallback Methods');
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
                error_log('You can also enable it via URL: Add ?authyo_enable_fallback=1 to any page as admin');
            }
            wp_send_json_error(array(
                'message' => esc_html__('Fallback method is disabled. Please contact administrator or enable it in plugin settings.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }
        
        if ($debug) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('✓ Fallback method is enabled');
        }

        // Get and sanitize input data
        $email = isset($_POST['email']) ? sanitize_text_field(wp_unslash($_POST['email'])) : '';
        $method = isset($_POST['method']) ? sanitize_text_field(wp_unslash($_POST['method'])) : '';

        if ($debug) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('Email/Mobile: ' . $email);
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
            error_log('Method: ' . $method);
        }

        if (empty($email) || empty($method)) {
            if ($debug) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
                error_log('Missing required data - email or method is empty');
            }
            wp_send_json_error(array(
                'message' => esc_html__('Missing required data.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Validate input type
        $is_email = $this->is_valid_email($email);
        $is_mobile = $this->is_valid_mobile($email);

        if (!$is_email && !$is_mobile) {
            wp_send_json_error(array(
                'message' => esc_html__('Please enter a valid email or mobile number.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Fallback methods (SMS, WhatsApp, Voice) are only available for mobile numbers
        if ($is_email) {
            wp_send_json_error(array(
                'message' => esc_html__('Fallback methods (SMS, WhatsApp, Voice Call) are only available for mobile numbers, not for email addresses.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        try {
            // Log fallback request details
            // Debug code removed for production - uncomment if needed during development
            // if (defined('WP_DEBUG') && WP_DEBUG) {
            //     error_log('=== FALLBACK OTP REQUEST ===');
            //     error_log('Email/Mobile: ' . $email);
            //     error_log('Method: ' . $method);
            //     error_log('Action Type: ' . $action_type);
            //     error_log('==============================');
            // }
            
            // Send OTP with specified method
            $authyo = new Authyo_OTP_Auth_SDK();
            $result = $authyo->send_otp($email, $method, $action_type);

            if (is_wp_error($result)) {
                $error_message = $result->get_error_message();
                
                // Handle service unavailable errors with helpful suggestions
                if ($result->get_error_code() === 'service_unavailable') {
                    // Get available fallback methods
                    $authyo = new Authyo_OTP_Auth_SDK();
                    $available_methods = $authyo->get_available_fallback_methods($email, $method);
                    
                    if (!empty($available_methods)) {
                        $method_list = implode(', ', array_values($available_methods));
                        // translators: %s is the list of available authentication methods
                        $error_message = $error_message . ' ' . sprintf(esc_html__('Please try using: %s', 'authyo-otp-authentication-for-woocommerce'), $method_list);
                    }
                }
                
                wp_send_json_error(array(
                    'message' => esc_html($error_message)
                ));
                return;
            }

            if (!isset($result['maskId'])) {
                wp_send_json_error(array(
                    'message' => esc_html__('Invalid response from OTP service.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Get OTP expiry time from settings
            $otp_expiry = get_option('authyo_otp_auth_expiry', 300);

            // Store OTP data in transient with correct key format
            // Use 'reg' for register to match send_register_otp and verify_register_otp
            $transient_key_prefix = ($action_type === 'register') ? 'authyo_otp_auth_reg_' : 'authyo_otp_auth_login_';
            
            set_transient(
                $transient_key_prefix . md5($email),
                array(
                    'mask_id' => sanitize_text_field($result['maskId']),
                    'time' => time(),
                    'method' => sanitize_text_field($method),
                    'is_mobile' => $is_mobile
                ),
                absint($otp_expiry)
            );

            // Get user ID if login
            $user_id = 0;
            if ($action_type === 'login') {
                if ($is_mobile) {
                    $user_id = $this->find_user_by_phone($email);
                } else {
                    $user = get_user_by('email', $email);
                    $user_id = $user ? $user->ID : 0;
                }
            }

            $this->log_activity($email, '', 'fallback_otp_sent', $user_id);
            
            if ($debug) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
                error_log('✅ Fallback OTP sent successfully');
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
                error_log('Method used: ' . $method);
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only runs when WP_DEBUG is enabled
                error_log('MaskId: ' . $result['maskId']);
            }

            $method_labels = array(
                'email' => esc_html__('Email', 'authyo-otp-authentication-for-woocommerce'),
                'whatsapp' => esc_html__('WhatsApp', 'authyo-otp-authentication-for-woocommerce'),
                'sms' => esc_html__('SMS', 'authyo-otp-authentication-for-woocommerce'),
                'voice' => esc_html__('Voice Call', 'authyo-otp-authentication-for-woocommerce'),
            );

            $method_label = isset($method_labels[$method]) ? $method_labels[$method] : ucfirst($method);

            // Log the actual method returned from SDK
            // Debug code removed for production - uncomment if needed during development
            // if (defined('WP_DEBUG') && WP_DEBUG) {
            //     error_log('=== FALLBACK RESPONSE ===');
            //     error_log('Requested Method: ' . $method);
            //     error_log('Returned Method: ' . (isset($result['method']) ? $result['method'] : 'not set'));
            //     error_log('=======================');
            // }
            
            // Use the method from SDK response if available, otherwise use requested method
            $actual_method = isset($result['method']) ? $result['method'] : $method;
            $method_label = isset($method_labels[$actual_method]) ? $method_labels[$actual_method] : ucfirst($actual_method);
            
            wp_send_json_success(array(
                'message' => sprintf(
                    // translators: %s is the method name (e.g., SMS, WhatsApp, Voice Call)
                    esc_html__('OTP sent successfully via %s!', 'authyo-otp-authentication-for-woocommerce'),
                    $method_label
                ),
                'maskId' => sanitize_text_field($result['maskId']),
                'method' => sanitize_text_field($actual_method)
            ));

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
        }

        wp_die();
    }

    /**
     * AJAX handler to fetch country list
     */
    public function get_country_list() {
        // Verify nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!check_ajax_referer('authyo_otp_auth_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }

        // Try to get from cache first
        $cached_countries = get_transient('authyo_country_list_cache');
        if ($cached_countries !== false) {
            wp_send_json_success(array('countries' => $cached_countries));
            return;
        }

        // Try to get from Authyo API
        $api_url = 'https://app.authyo.io/api/v1/user/getcountrylist';
        
        $response = wp_remote_get($api_url, array(
            'timeout' => 30,
            'sslverify' => true
        ));

        if (!is_wp_error($response)) {
            $body = wp_remote_retrieve_body($response);
            $countries = json_decode($body, true);

            if ($countries && is_array($countries)) {
                // Cache successful API response for 24 hours
                set_transient('authyo_country_list_cache', $countries, 24 * HOUR_IN_SECONDS);
                wp_send_json_success(array('countries' => $countries));
                return;
            }
        }

        // Fallback to minimal essential country list
        $countries = $this->get_fallback_country_list();
        
        // Cache fallback for 1 hour (shorter duration to retry API sooner)
        set_transient('authyo_country_list_cache', $countries, HOUR_IN_SECONDS);
        
        wp_send_json_success(array('countries' => $countries));
    }

    /**
     * Get minimal essential fallback country list (used when API is unavailable)
     * This is a reduced list of the most commonly used countries.
     * The full list will be loaded from the API and cached.
     */
    private function get_fallback_country_list() {
        return array(
            // North America
            array('name' => 'United States', 'countryCode' => '+1', 'code' => '1', 'phoneLength' => 11),
            array('name' => 'Canada', 'countryCode' => '+1', 'code' => '1', 'phoneLength' => 11),
            array('name' => 'Mexico', 'countryCode' => '+52', 'code' => '52', 'phoneLength' => 13),
            
            // Europe (Major)
            array('name' => 'United Kingdom', 'countryCode' => '+44', 'code' => '44', 'phoneLength' => 13),
            array('name' => 'Germany', 'countryCode' => '+49', 'code' => '49', 'phoneLength' => 13),
            array('name' => 'France', 'countryCode' => '+33', 'code' => '33', 'phoneLength' => 12),
            array('name' => 'Italy', 'countryCode' => '+39', 'code' => '39', 'phoneLength' => 12),
            array('name' => 'Spain', 'countryCode' => '+34', 'code' => '34', 'phoneLength' => 12),
            array('name' => 'Netherlands', 'countryCode' => '+31', 'code' => '31', 'phoneLength' => 12),
            array('name' => 'Switzerland', 'countryCode' => '+41', 'code' => '41', 'phoneLength' => 12),
            array('name' => 'Poland', 'countryCode' => '+48', 'code' => '48', 'phoneLength' => 12),
            array('name' => 'Sweden', 'countryCode' => '+46', 'code' => '46', 'phoneLength' => 12),
            array('name' => 'Belgium', 'countryCode' => '+32', 'code' => '32', 'phoneLength' => 12),
            array('name' => 'Austria', 'countryCode' => '+43', 'code' => '43', 'phoneLength' => 12),
            array('name' => 'Norway', 'countryCode' => '+47', 'code' => '47', 'phoneLength' => 11),
            array('name' => 'Denmark', 'countryCode' => '+45', 'code' => '45', 'phoneLength' => 11),
            array('name' => 'Russia', 'countryCode' => '+7', 'code' => '7', 'phoneLength' => 12),
            
            // Asia Pacific
            array('name' => 'India', 'countryCode' => '+91', 'code' => '91', 'phoneLength' => 12),
            array('name' => 'China', 'countryCode' => '+86', 'code' => '86', 'phoneLength' => 13),
            array('name' => 'Japan', 'countryCode' => '+81', 'code' => '81', 'phoneLength' => 12),
            array('name' => 'South Korea', 'countryCode' => '+82', 'code' => '82', 'phoneLength' => 12),
            array('name' => 'Australia', 'countryCode' => '+61', 'code' => '61', 'phoneLength' => 12),
            array('name' => 'New Zealand', 'countryCode' => '+64', 'code' => '64', 'phoneLength' => 11),
            array('name' => 'Singapore', 'countryCode' => '+65', 'code' => '65', 'phoneLength' => 11),
            array('name' => 'Malaysia', 'countryCode' => '+60', 'code' => '60', 'phoneLength' => 12),
            array('name' => 'Indonesia', 'countryCode' => '+62', 'code' => '62', 'phoneLength' => 13),
            array('name' => 'Philippines', 'countryCode' => '+63', 'code' => '63', 'phoneLength' => 12),
            array('name' => 'Thailand', 'countryCode' => '+66', 'code' => '66', 'phoneLength' => 12),
            array('name' => 'Vietnam', 'countryCode' => '+84', 'code' => '84', 'phoneLength' => 12),
            array('name' => 'Pakistan', 'countryCode' => '+92', 'code' => '92', 'phoneLength' => 12),
            array('name' => 'Bangladesh', 'countryCode' => '+880', 'code' => '880', 'phoneLength' => 13),
            
            // Middle East
            array('name' => 'UAE', 'countryCode' => '+971', 'code' => '971', 'phoneLength' => 12),
            array('name' => 'Saudi Arabia', 'countryCode' => '+966', 'code' => '966', 'phoneLength' => 12),
            array('name' => 'Turkey', 'countryCode' => '+90', 'code' => '90', 'phoneLength' => 12),
            array('name' => 'Israel', 'countryCode' => '+972', 'code' => '972', 'phoneLength' => 12),
            
            // Africa
            array('name' => 'South Africa', 'countryCode' => '+27', 'code' => '27', 'phoneLength' => 12),
            array('name' => 'Nigeria', 'countryCode' => '+234', 'code' => '234', 'phoneLength' => 13),
            array('name' => 'Egypt', 'countryCode' => '+20', 'code' => '20', 'phoneLength' => 13),
            array('name' => 'Kenya', 'countryCode' => '+254', 'code' => '254', 'phoneLength' => 12),
            
            // South America
            array('name' => 'Brazil', 'countryCode' => '+55', 'code' => '55', 'phoneLength' => 13),
            array('name' => 'Argentina', 'countryCode' => '+54', 'code' => '54', 'phoneLength' => 13),
            array('name' => 'Colombia', 'countryCode' => '+57', 'code' => '57', 'phoneLength' => 12),
            array('name' => 'Chile', 'countryCode' => '+56', 'code' => '56', 'phoneLength' => 12),
        );
    }
}

// Helper functions to replace WooCommerce functions
if (!function_exists('authyo_otp_auth_get_page_permalink')) {
    function authyo_otp_auth_get_page_permalink($page) {
        if (function_exists('wc_get_page_permalink')) {
            return wc_get_page_permalink($page);
        }
        return home_url('/my-account/');
    }
}

if (!function_exists('authyo_otp_auth_add_notice')) {
    function authyo_otp_auth_add_notice($message, $type = 'notice') {
        if (function_exists('wc_add_notice')) {
            wc_add_notice($message, $type);
        } else {
            // Fallback to WordPress admin notice
            add_action('admin_notices', function() use ($message, $type) {
                echo '<div class="notice notice-' . esc_attr($type) . '"><p>' . esc_html($message) . '</p></div>';
            });
        }
    }
}