<?php
/**
 * Handles deactivation feedback collection and API submission
 */

if (!defined('ABSPATH')) {
    exit;
}

class Authyo_OTP_Auth_Deactivation_Feedback {
    
    /**
     * Feedback API endpoint
     */
    private $feedback_api_url = 'https://app.authyo.io/api/v1/user/WordpressWebhook';
    
    /**
     * Initialize the feedback system
     */
    public function __construct() {
        add_action('admin_footer', array($this, 'render_feedback_modal'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_authyo_otp_auth_submit_deactivation_feedback', array($this, 'handle_feedback_submission'));
    }
    
    /**
     * Enqueue scripts and styles for the feedback popup
     */
    public function enqueue_scripts($hook) {
        // Only load on plugins page
        if ($hook !== 'plugins.php') {
            return;
        }
        
        wp_enqueue_style(
            'authyo-otp-auth-deactivation-feedback',
            AUTHYO_OTP_AUTH_PLUGIN_URL . 'assets/css/deactivation-feedback.css',
            array(),
            AUTHYO_OTP_AUTH_VERSION
        );
        
        wp_enqueue_script(
            'authyo-otp-auth-deactivation-feedback',
            AUTHYO_OTP_AUTH_PLUGIN_URL . 'assets/js/deactivation-feedback.js',
            array('jquery'),
            AUTHYO_OTP_AUTH_VERSION,
            true
        );
        
        wp_localize_script('authyo-otp-auth-deactivation-feedback', 'authyoDeactivationFeedback', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('authyo_otp_auth_deactivation_feedback'),
            'pluginSlug' => AUTHYO_OTP_AUTH_PLUGIN_BASENAME,
            'pluginName' => 'Authyo OTP Authentication for WooCommerce'
        ));
    }
    
    /**
     * Render the feedback modal HTML
     */
    public function render_feedback_modal() {
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'plugins') {
            return;
        }
        ?>
        <div id="authyo-deactivation-feedback-modal" class="authyo-feedback-modal" style="display: none;">
            <div class="authyo-feedback-modal-overlay"></div>
            <div class="authyo-feedback-modal-content">
                <div class="authyo-feedback-modal-header">
                    <h2><?php esc_html_e('Quick Feedback', 'authyo-otp-authentication-for-woocommerce'); ?></h2>
                    <button type="button" class="authyo-feedback-modal-close" aria-label="<?php esc_attr_e('Close', 'authyo-otp-authentication-for-woocommerce'); ?>">
                        <span class="dashicons dashicons-no-alt"></span>
                    </button>
                </div>
                <div class="authyo-feedback-modal-body">
                    <p class="authyo-feedback-question">
                        <?php esc_html_e('Would you mind sharing why you are deactivating the Authyo OTP Authentication for WooCommerce plugin?', 'authyo-otp-authentication-for-woocommerce'); ?>
                    </p>
                    <form id="authyo-deactivation-feedback-form">
                        <div class="authyo-feedback-options">
                            <label class="authyo-feedback-option">
                                <input type="radio" name="reason" value="temporary" />
                                <span><?php esc_html_e('This is temporary — I will re-enable it later.', 'authyo-otp-authentication-for-woocommerce'); ?></span>
                            </label>
                            <label class="authyo-feedback-option">
                                <input type="radio" name="reason" value="complicated" />
                                <span><?php esc_html_e('It is too complicated to use.', 'authyo-otp-authentication-for-woocommerce'); ?></span>
                            </label>
                            <label class="authyo-feedback-option">
                                <input type="radio" name="reason" value="better_alternative" />
                                <span><?php esc_html_e('I found a better alternative.', 'authyo-otp-authentication-for-woocommerce'); ?></span>
                            </label>
                            <label class="authyo-feedback-option">
                                <input type="radio" name="reason" value="other" />
                                <span><?php esc_html_e('Other (please share below)', 'authyo-otp-authentication-for-woocommerce'); ?></span>
                            </label>
                        </div>
                        <div id="authyo-feedback-other-container" class="authyo-feedback-other-container" style="display: none;">
                            <textarea 
                                id="authyo-feedback-details" 
                                name="details" 
                                rows="4" 
                                placeholder="<?php esc_attr_e('Please share your feedback...', 'authyo-otp-authentication-for-woocommerce'); ?>"
                            ></textarea>
                        </div>
                        <div class="authyo-feedback-actions">
                            <button type="button" class="button button-secondary authyo-feedback-skip">
                                <?php esc_html_e('Skip', 'authyo-otp-authentication-for-woocommerce'); ?>
                            </button>
                            <button type="submit" class="button button-primary authyo-feedback-submit">
                                <?php esc_html_e('Submit & Deactivate', 'authyo-otp-authentication-for-woocommerce'); ?>
                            </button>
                        </div>
                    </form>
                    <div id="authyo-feedback-thank-you" class="authyo-feedback-thank-you" style="display: none;">
                        <p><?php esc_html_e('Thank you for your feedback!', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Handle AJAX feedback submission
     */
    public function handle_feedback_submission() {
        // Verify nonce
        if (!check_ajax_referer('authyo_otp_auth_deactivation_feedback', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        // Check permissions
        if (!current_user_can('deactivate_plugins')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }
        
        $reason = isset($_POST['reason']) ? sanitize_text_field(wp_unslash($_POST['reason'])) : 'skipped';
        $details = isset($_POST['details']) ? sanitize_textarea_field(wp_unslash($_POST['details'])) : '';
        $pageurl = isset($_POST['pageurl']) ? esc_url_raw(wp_unslash($_POST['pageurl'])) : '';
        
        // If reason is "other" but no details provided, use empty string
        if ($reason === 'other' && empty($details)) {
            $details = '';
        }
        
        // Send feedback to API
        $api_response = $this->send_feedback_to_api($reason, $details, $pageurl);
        
        if (is_wp_error($api_response)) {
            // Even if API call fails, allow deactivation to proceed
            // Log error for debugging
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Error logging for debugging API failures, only runs when WP_DEBUG is enabled
                error_log('Authyo OTP: Failed to send deactivation feedback: ' . $api_response->get_error_message());
            }
        }
        
        wp_send_json_success(array(
            'message' => esc_html__('Feedback received. Thank you!', 'authyo-otp-authentication-for-woocommerce')
        ));
    }
    
    /**
     * Send feedback data to the API
     *
     * @param string $reason The selected reason
     * @param string $details Custom feedback details
     * @param string $pageurl The current page URL
     * @return array|WP_Error API response or error
     */
    private function send_feedback_to_api($reason, $details = '', $pageurl = '') {
        // Get App ID for API key
        $api_key = get_option('authyo_otp_auth_app_id', '');
        
        // Prepare payload
        $payload = array(
            'subject' => 'Authyo_Woocommerce_Plugin_Deactivation_Popup',
            'api_key' => $api_key,
            'data' => array(
                'reason'    => $reason,
                'details'   => $details,
                'plugin'    => AUTHYO_OTP_AUTH_PLUGIN_BASENAME,
                'version'   => AUTHYO_OTP_AUTH_VERSION,
                'timestamp' => gmdate('c'),
                'pageurl'   => $pageurl,
            ),
        );
        
        // Send to API
        $response = wp_remote_post($this->feedback_api_url, array(
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json',
            ),
            'body' => wp_json_encode($payload),
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code >= 200 && $response_code < 300) {
            return json_decode($response_body, true);
        }
        
        return new WP_Error('api_error', sprintf(
            /* translators: %d: HTTP status code (e.g., 404, 500) */
            esc_html__('API returned status code %d', 'authyo-otp-authentication-for-woocommerce'),
            $response_code
        ));
    }
}