<?php
/**
 * Country configuration helper for Authyo OTP.
 *
 * Centralizes all logic related to limiting which countries
 * are available in the phone-number country dropdowns.
 *
 * @package Authyo_OTP_Auth
 */

if (!defined('ABSPATH')) {
    exit;
}

class Authyo_OTP_Auth_Country_Config {

    /**
     * Option name for country mode.
     *
     * @var string
     */
    const OPTION_MODE = 'authyo_otp_auth_country_mode';

    /**
     * Option name for allowed countries (array of numeric country codes).
     *
     * @var string
     */
    const OPTION_ALLOWED = 'authyo_otp_auth_allowed_countries';

    /**
     * Get the current country mode.
     *
     * Modes:
     * - all     : all countries are available (default / backwards compatible)
     * - limited : only the configured list of countries is available
     *
     * @return string
     */
    public static function get_mode()
    {
        $mode = get_option(self::OPTION_MODE, 'all');

        if (!in_array($mode, array('all', 'limited'), true)) {
            $mode = 'all';
        }

        return $mode;
    }

    /**
     * Get the list of allowed country codes as an array of numeric strings.
     *
     * Country codes are stored as an array, e.g. ['91', '1', '44'].
     *
     * @return array
     */
    public static function get_allowed_country_codes()
    {
        $raw = get_option(self::OPTION_ALLOWED, array());

        if (!is_array($raw)) {
            $raw = array();
        }

        $allowed = array();

        foreach ($raw as $code) {
            if (!is_string($code) && !is_int($code)) {
                continue;
            }

            $digits = preg_replace('/\D+/', '', (string) $code);

            if ($digits === '') {
                continue;
            }

            $allowed[] = $digits;
        }

        return array_values(array_unique($allowed));
    }

    /**
     * Sanitize allowed countries option.
     *
     * @param mixed $value Raw value from the settings form.
     *
     * @return array
     */
    public static function sanitize_allowed_countries($value)
    {
        if (!is_array($value)) {
            return array();
        }

        $sanitized = array();

        foreach ($value as $code) {
            if (!is_string($code) && !is_int($code)) {
                continue;
            }

            $digits = preg_replace('/\D+/', '', (string) $code);

            if ($digits === '') {
                continue;
            }

            $sanitized[] = $digits;
        }

        return array_values(array_unique($sanitized));
    }

    /**
     * Filter a country list based on the current configuration.
     *
     * Expects an array of countries as returned by the Authyo API or
     * the fallback lists, with keys like:
     * - 'code'        => '91'
     * - 'countryCode' => '+91'
     *
     * @param array $countries Raw countries array.
     *
     * @return array Filtered countries.
     */
    public static function filter_countries($countries)
    {
        if (!is_array($countries) || empty($countries)) {
            return array();
        }

        if (self::get_mode() === 'all') {
            // Backwards-compatible: do not filter.
            return $countries;
        }

        $allowed = self::get_allowed_country_codes();

        if (empty($allowed)) {
            // No explicit allowed list configured – keep original behaviour.
            return $countries;
        }

        $filtered = array();

        foreach ($countries as $country) {
            if (!is_array($country)) {
                continue;
            }

            $code_digits = '';

            if (isset($country['code'])) {
                $code_digits = preg_replace('/\D+/', '', (string) $country['code']);
            }

            if ($code_digits === '' && isset($country['countryCode'])) {
                $code_digits = preg_replace('/\D+/', '', (string) $country['countryCode']);
            }

            if ($code_digits === '') {
                continue;
            }

            if (in_array($code_digits, $allowed, true)) {
                $filtered[] = $country;
            }
        }

        // If filtering removed everything (e.g., misconfiguration), fallback to original list
        // to avoid breaking the UI completely.
        if (empty($filtered)) {
            return $countries;
        }

        return array_values($filtered);
    }

    /**
     * Register settings related to country configuration.
     *
     * This is called from the admin settings registration.
     *
     * @return void
     */
    public static function register_settings()
    {
        // Country mode (all / limited)
        register_setting('authyo_otp_auth_settings', self::OPTION_MODE, array(
            'type'              => 'string',
            'default'           => 'all',
            'sanitize_callback' => function ($value) {
                $value = is_string($value) ? $value : 'all';
                return in_array($value, array('all', 'limited'), true) ? $value : 'all';
            },
        ));

        // Allowed country codes (array of numeric strings)
        register_setting('authyo_otp_auth_settings', self::OPTION_ALLOWED, array(
            'type'              => 'array',
            'default'           => array(),
            'sanitize_callback' => array(__CLASS__, 'sanitize_allowed_countries'),
        ));
    }

    /**
     * Get a basic list of commonly used countries for the admin UI.
     *
     * NOTE: This is only used for rendering the settings multi-select.
     * The actual runtime lists come from the Authyo API / fallback lists.
     *
     * @return array[]
     */
    public static function get_admin_country_list()
    {
        return array(
            array('name' => 'United States', 'code' => '1'),
            array('name' => 'Canada', 'code' => '1'),
            array('name' => 'United Kingdom', 'code' => '44'),
            array('name' => 'Germany', 'code' => '49'),
            array('name' => 'France', 'code' => '33'),
            array('name' => 'Italy', 'code' => '39'),
            array('name' => 'Spain', 'code' => '34'),
            array('name' => 'Netherlands', 'code' => '31'),
            array('name' => 'Switzerland', 'code' => '41'),
            array('name' => 'Poland', 'code' => '48'),
            array('name' => 'Sweden', 'code' => '46'),
            array('name' => 'Austria', 'code' => '43'),
            array('name' => 'Norway', 'code' => '47'),
            array('name' => 'Denmark', 'code' => '45'),
            array('name' => 'India', 'code' => '91'),
            array('name' => 'China', 'code' => '86'),
            array('name' => 'Japan', 'code' => '81'),
            array('name' => 'South Korea', 'code' => '82'),
            array('name' => 'Australia', 'code' => '61'),
            array('name' => 'New Zealand', 'code' => '64'),
            array('name' => 'Singapore', 'code' => '65'),
            array('name' => 'Malaysia', 'code' => '60'),
            array('name' => 'Indonesia', 'code' => '62'),
            array('name' => 'Philippines', 'code' => '63'),
            array('name' => 'Thailand', 'code' => '66'),
            array('name' => 'Vietnam', 'code' => '84'),
            array('name' => 'Pakistan', 'code' => '92'),
            array('name' => 'Bangladesh', 'code' => '880'),
            array('name' => 'UAE', 'code' => '971'),
            array('name' => 'Saudi Arabia', 'code' => '966'),
            array('name' => 'Turkey', 'code' => '90'),
            array('name' => 'South Africa', 'code' => '27'),
            array('name' => 'Nigeria', 'code' => '234'),
            array('name' => 'Egypt', 'code' => '20'),
            array('name' => 'Brazil', 'code' => '55'),
            array('name' => 'Argentina', 'code' => '54'),
            array('name' => 'Mexico', 'code' => '52'),
            array('name' => 'Chile', 'code' => '56'),
            array('name' => 'Colombia', 'code' => '57'),
        );
    }

    /**
     * Render the admin fields for country configuration.
     *
     * Called from the settings page so all UI stays in one place.
     *
     * @return void
     */
    public static function render_admin_fields()
    {
        $mode          = self::get_mode();
        $allowed       = self::get_allowed_country_codes();
        $admin_countries = self::get_admin_country_list();
        if (!defined('AUTHYO_COUNTRY_CONFIG_STYLES_PRINTED')) {
            define('AUTHYO_COUNTRY_CONFIG_STYLES_PRINTED', true);
            ?>
            <style>
                .authyo-country-config-card {
                    border: 1px solid #dcdcde;
                    border-radius: 10px;
                    padding: 20px;
                    background: #fff;
                    box-shadow: 0 1px 2px rgba(0,0,0,0.05);
                }
                .authyo-country-config {
                    display: grid;
                    gap: 20px;
                }
                @media (min-width: 768px) {
                    .authyo-country-config {
                        grid-template-columns: 280px 1fr;
                    }
                }
                .authyo-country-config__heading {
                    margin: 0 0 12px;
                    font-size: 16px;
                    font-weight: 600;
                }
                .authyo-chip-group {
                    display: grid;
                    gap: 10px;
                }
                .authyo-chip {
                    display: flex;
                    align-items: center;
                    gap: 10px;
                    padding: 12px 14px;
                    border: 1px solid #dcdcde;
                    border-radius: 8px;
                    cursor: pointer;
                    transition: border-color 0.2s, box-shadow 0.2s;
                    background: #f6f7f7;
                }
                .authyo-chip input[type="radio"] {
                    margin: 0;
                }
                .authyo-chip--active {
                    border-color: #3858e9;
                    box-shadow: 0 0 0 1px rgba(56,88,233,0.2);
                    background: #fff;
                }
                .authyo-country-select-wrapper {
                    border: 1px solid #dcdcde;
                    border-radius: 8px;
                    overflow: hidden;
                    background: #fefefe;
                }
                .authyo-country-select-wrapper select {
                    width: 100%;
                    border: none;
                    box-shadow: none;
                    padding: 10px;
                    min-height: 240px;
                }
                .authyo-country-search {
                    display: flex;
                    align-items: center;
                    gap: 8px;
                    padding: 10px 12px;
                    border: 1px solid #dcdcde;
                    border-radius: 8px;
                    background: #fff;
                }
                .authyo-country-search input {
                    width: 100%;
                    border: none;
                    box-shadow: none;
                    outline: none;
                }
                .authyo-country-search .dashicons {
                    color: #50575e;
                }
                .authyo-country-config__meta {
                    margin-top: 10px;
                    font-size: 12px;
                    color: #646970;
                }
            </style>
            <script>
                document.addEventListener('input', function (event) {
                    if (!event.target.matches('.authyo-country-search-input')) {
                        return;
                    }
                    var targetSelector = event.target.getAttribute('data-target');
                    if (!targetSelector) {
                        return;
                    }
                    var select = document.querySelector(targetSelector);
                    if (!select) {
                        return;
                    }
                    var query = event.target.value.toLowerCase();
                    Array.prototype.forEach.call(select.options, function(option) {
                        if (!option.value) {
                            option.hidden = false;
                            return;
                        }
                        option.hidden = option.text.toLowerCase().indexOf(query) === -1;
                    });
                });

                document.addEventListener('change', function(event) {
                    if (!event.target.matches('.authyo-chip input[type="radio"]')) {
                        return;
                    }
                    document.querySelectorAll('.authyo-chip').forEach(function(chip) {
                        chip.classList.remove('authyo-chip--active');
                    });
                    var parent = event.target.closest('.authyo-chip');
                    if (parent) {
                        parent.classList.add('authyo-chip--active');
                    }
                });
            </script>
            <?php
        }
        ?>
        <tr class="authyo-country-config-row">
            <th scope="row" class="authyo-country-config-row__title">
                <span class="authyo-country-config-row__label">
                    <?php esc_html_e('Country Availability Settings', 'authyo-otp-authentication-for-woocommerce'); ?>
                </span>
            </th>
            <td>
                <div class="authyo-country-config-card">
                    <div class="authyo-country-config">
                        <div>
                            <p class="authyo-country-config__heading"><?php esc_html_e('Country Availability', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                            <div class="authyo-chip-group">
                                <label class="authyo-chip <?php echo $mode === 'all' ? 'authyo-chip--active' : ''; ?>">
                                    <input type="radio"
                                           name="<?php echo esc_attr(self::OPTION_MODE); ?>"
                                           value="all"
                                           <?php checked($mode, 'all'); ?> />
                                    <div>
                                        <strong><?php esc_html_e('All countries (default)', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        <p class="description" style="margin: 4px 0 0;">
                                            <?php esc_html_e('Show every available country in the dropdown.', 'authyo-otp-authentication-for-woocommerce'); ?>
                                        </p>
                                    </div>
                                </label>
                                <label class="authyo-chip <?php echo $mode === 'limited' ? 'authyo-chip--active' : ''; ?>">
                                    <input type="radio"
                                           name="<?php echo esc_attr(self::OPTION_MODE); ?>"
                                           value="limited"
                                           <?php checked($mode, 'limited'); ?> />
                                    <div>
                                        <strong><?php esc_html_e('Limit to selected countries only', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        <p class="description" style="margin: 4px 0 0;">
                                            <?php esc_html_e('Perfect for single-region stores. Pick one or more countries below.', 'authyo-otp-authentication-for-woocommerce'); ?>
                                        </p>
                                    </div>
                                </label>
                            </div>
                        </div>
                        <div>
                            <div style="display: flex; flex-direction: column; gap: 10px;">
                                <div style="display: flex; justify-content: space-between; align-items: center; flex-wrap: wrap; gap: 10px;">
                                    <div>
                                        <p class="authyo-country-config__heading" style="margin-bottom: 4px;">
                                            <?php esc_html_e('Allowed countries (when limited)', 'authyo-otp-authentication-for-woocommerce'); ?>
                                        </p>
                                        <p class="description" style="margin: 0;">
                                            <?php esc_html_e('These options sync across login, registration, and checkout phone fields.', 'authyo-otp-authentication-for-woocommerce'); ?>
                                        </p>
                                    </div>
                                    <div class="authyo-country-search" style="min-width: 220px;">
                                        <span class="dashicons dashicons-search"></span>
                                        <input type="search"
                                               class="authyo-country-search-input"
                                               placeholder="<?php esc_attr_e('Search country…', 'authyo-otp-authentication-for-woocommerce'); ?>"
                                               data-target="#<?php echo esc_attr(self::OPTION_ALLOWED); ?>" />
                                    </div>
                                </div>
                                <div class="authyo-country-select-wrapper">
                                    <select id="<?php echo esc_attr(self::OPTION_ALLOWED); ?>"
                                            name="<?php echo esc_attr(self::OPTION_ALLOWED); ?>[]"
                                            multiple="multiple"
                                            size="12">
                                        <?php
                                        foreach ($admin_countries as $country) {
                                            $code_digits = preg_replace('/\D+/', '', (string) $country['code']);
                                            if ($code_digits === '') {
                                                continue;
                                            }
                                            $label = sprintf(
                                                '%s (+%s)',
                                                $country['name'],
                                                $code_digits
                                            );
                                            ?>
                                            <option value="<?php echo esc_attr($code_digits); ?>" <?php selected(in_array($code_digits, $allowed, true)); ?>>
                                                <?php echo esc_html($label); ?>
                                            </option>
                                            <?php
                                        }
                                        ?>
                                    </select>
                                </div>
                                <p class="authyo-country-config__meta">
                                    <?php esc_html_e('Tip: Hold Ctrl (Windows) or Command (Mac) to select multiple countries. Choosing just one country will lock the dropdown for customers.', 'authyo-otp-authentication-for-woocommerce'); ?>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
            </td>
        </tr>
        <?php
    }
}


