<?php
/**
 * Checkout OTP functionality for WooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

// Include the SDK class
require_once plugin_dir_path(__FILE__) . 'class-authyo-otp-auth-sdk.php';

class Authyo_OTP_Auth_Checkout
{
    private static $instance = null;
    private $version;

    public static function get_instance()
    {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    public function __construct()
    {
        $this->version = defined('AUTHYO_OTP_AUTH_VERSION') ? AUTHYO_OTP_AUTH_VERSION : '1.0.3';

        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: WooCommerce not active, hooks not registered');
            }
            return;
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
            error_log('Authyo Checkout OTP: Constructor called, registering hooks');
        }

        // Hook into checkout process - this validates and blocks if OTP not verified
        // Use priority 1 to run early, before other validations
        add_action('woocommerce_checkout_process', array($this, 'validate_checkout_otp'), 1);

        // Additional hook to validate before order is created (safety check)
        add_action('woocommerce_after_checkout_validation', array($this, 'validate_checkout_otp'), 1);

        // Block order creation BEFORE it's created - most important security check
        // Use priority 1 to run before other hooks on this action
        add_action('woocommerce_checkout_create_order', array($this, 'validate_before_order_created'), 1, 2);

        // Block order creation directly - most reliable method (backup check)
        add_action('woocommerce_checkout_order_processed', array($this, 'validate_before_order_creation'), 1, 1);

        // Also hook into before order creation for block-based checkout
        add_filter('woocommerce_store_api_checkout_order_processed', array($this, 'validate_before_order_creation'), 1, 1);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
            error_log('Authyo Checkout OTP: All hooks registered successfully');
        }

        // For block-based checkout, ensure our fields are included in checkout data
        add_filter('woocommerce_store_api_checkout_update_order_from_request', array($this, 'add_otp_fields_to_checkout_data'), 10, 2);

        // Also hook into the checkout update to ensure fields are submitted
        add_filter('woocommerce_checkout_posted_data', array($this, 'ensure_otp_fields_in_posted_data'), 10, 1);
        add_filter('woocommerce_checkout_posted_data', array($this, 'maybe_format_checkout_phone'), 5, 1); // Run early, priority 5
        add_action('woocommerce_checkout_create_order', array($this, 'apply_formatted_phone_to_order'), 5, 2); // Run early, priority 5
        add_action('woocommerce_checkout_update_user_meta', array($this, 'apply_formatted_phone_to_user_meta'), 5, 2); // Run early, priority 5
        add_action('woocommerce_checkout_order_processed', array($this, 'update_order_phone_after_creation'), 10, 1); // Update after order is saved
        add_action('woocommerce_store_api_checkout_order_processed', array($this, 'update_order_phone_after_creation'), 10, 1); // For block checkout

        // Add OTP fields directly after email and phone fields
        // Use multiple hooks to ensure fields are added
        add_action('woocommerce_after_checkout_billing_form', array($this, 'add_otp_fields_to_checkout'), 10);
        add_action('woocommerce_checkout_after_customer_details', array($this, 'add_otp_fields_to_checkout'), 10);
        add_action('woocommerce_after_order_notes', array($this, 'add_otp_fields_to_checkout'), 10);
        add_action('woocommerce_checkout_before_terms_and_conditions', array($this, 'add_otp_fields_to_checkout'), 10);
        add_action('woocommerce_review_order_before_payment', array($this, 'add_otp_fields_to_checkout'), 20);

        // Also add via checkout update to ensure fields are present
        add_action('woocommerce_checkout_update_order_review', array($this, 'maybe_inject_otp_fields'), 10);

        // Enqueue checkout scripts
        add_action('wp_enqueue_scripts', array($this, 'enqueue_checkout_scripts'), 20);

        // AJAX handlers for checkout OTP
        add_action('wp_ajax_authyo_otp_auth_send_checkout_otp', array($this, 'send_checkout_otp'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_send_checkout_otp', array($this, 'send_checkout_otp'));

        add_action('wp_ajax_authyo_otp_auth_verify_checkout_otp', array($this, 'verify_checkout_otp'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_verify_checkout_otp', array($this, 'verify_checkout_otp'));

        add_action('wp_ajax_authyo_otp_auth_resend_checkout_fallback', array($this, 'resend_checkout_fallback'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_resend_checkout_fallback', array($this, 'resend_checkout_fallback'));

        add_action('wp_ajax_authyo_otp_auth_get_country_list', array($this, 'get_country_list'));
        add_action('wp_ajax_nopriv_authyo_otp_auth_get_country_list', array($this, 'get_country_list'));
    }

    /**
     * Check if checkout OTP is enabled
     */
    private function is_checkout_otp_enabled()
    {
        $enabled = get_option('authyo_otp_auth_checkout_enabled', 'no');
        $result = $enabled === 'yes';

        // Log this check for debugging (only when WP_DEBUG is enabled)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
            error_log('Authyo Checkout OTP: is_checkout_otp_enabled() called. Option value: ' . $enabled . ', Result: ' . ($result ? 'true' : 'false'));
        }

        return $result;
    }

    /**
     * Add OTP fields to checkout form
     * Fields will be injected via JavaScript directly after email and phone fields
     */
    public function add_otp_fields_to_checkout()
    {
        // Prevent duplicate rendering - check if section already exists in DOM
        static $already_rendered = false;
        if ($already_rendered) {
            return;
        }

        // Only show if checkout OTP is enabled
        $checkout_enabled = get_option('authyo_otp_auth_checkout_enabled', 'no');
        if ($checkout_enabled !== 'yes') {
            return;
        }

        $email_enabled = get_option('authyo_otp_auth_checkout_email_enabled', 'yes') === 'yes';
        $whatsapp_enabled = get_option('authyo_otp_auth_checkout_whatsapp_enabled', 'no') === 'yes';
        $sms_enabled = get_option('authyo_otp_auth_checkout_sms_enabled', 'no') === 'yes';
        $voice_enabled = get_option('authyo_otp_auth_checkout_voice_enabled', 'no') === 'yes';
        $phone_methods_enabled = $whatsapp_enabled || $sms_enabled || $voice_enabled;

        // Only show section if at least one method is enabled
        if (!$email_enabled && !$phone_methods_enabled) {
            return;
        }

        // Mark as rendered BEFORE output to prevent issues
        $already_rendered = true;

        // Output container - JavaScript will move fields to correct locations
        // Use visibility hidden instead of display none so elements can be found
        ?>
        <div id="authyo-checkout-otp-container"
            style="position: absolute !important; left: -9999px !important; visibility: hidden !important; height: 0 !important; overflow: hidden !important; width: 1px !important;">
            <?php if ($email_enabled): ?>
                <div id="authyo-checkout-otp-email-wrapper" class="authyo-checkout-otp-email-section form-row"
                    style="display: block !important; visibility: visible !important; margin-top: 10px;">
                    <div class="authyo-checkout-otp-input-group"
                        style="display: flex; gap: 10px; align-items: center; margin-top: 5px;">
                        <input type="text" id="authyo_checkout_email_otp" name="authyo_checkout_email_otp" class="input-text"
                            placeholder="<?php esc_attr_e('Enter OTP sent to your email', 'authyo-otp-authentication-for-woocommerce'); ?>"
                            autocomplete="off" maxlength="<?php echo esc_attr(get_option('authyo_otp_auth_length', 6)); ?>"
                            pattern="[0-9]*" inputmode="numeric" style="display: none; flex: 1;" />
                        <button type="button" id="authyo_send_email_otp" class="button authyo-send-otp-btn"
                            style="display: inline-block !important; visibility: visible !important; white-space: nowrap;">
                            <?php esc_html_e('Send Email OTP', 'authyo-otp-authentication-for-woocommerce'); ?>
                        </button>
                    </div>
                    <button type="button" id="authyo_resend_email_otp" class="button button-secondary resend-otp-btn"
                        style="display: none; margin-top: 10px;">
                        <span class="timer"><?php esc_html_e('Resend OTP', 'authyo-otp-authentication-for-woocommerce'); ?></span>
                    </button>
                    <span id="authyo_email_otp_status" class="authyo-otp-status" style="display: block; margin-top: 5px;"></span>
                    <input type="hidden" id="authyo_checkout_email_maskid" name="authyo_checkout_email_maskid" value="" />
                </div>
            <?php endif; ?>

            <?php if ($phone_methods_enabled): ?>
                <div id="authyo-checkout-otp-phone-wrapper" class="authyo-checkout-otp-phone-section form-row"
                    style="display: block !important; visibility: visible !important; margin-top: 10px;">
                    <div class="phone-input-wrapper" style="display: none; margin-bottom: 5px;">
                        <select id="checkout-country-code" name="checkout_country_code" class="authyo-country-select"
                            style="display: block !important; visibility: visible !important; opacity: 1 !important;"
                            aria-label="<?php esc_attr_e('Select country code for mobile number', 'authyo-otp-authentication-for-woocommerce'); ?>">
                            <option value=""><?php esc_html_e('Select Country', 'authyo-otp-authentication-for-woocommerce'); ?>
                            </option>
                        </select>
                    </div>
                    <span id="checkout-number-preview" class="number-preview" role="status" aria-live="polite"
                        style="display:none; font-size: 12px; color: #666; margin-bottom: 5px;"></span>
                    <div class="authyo-checkout-otp-input-group"
                        style="display: flex; gap: 10px; align-items: center; margin-top: 5px;">
                        <input type="text" id="authyo_checkout_phone_otp" name="authyo_checkout_phone_otp" class="input-text"
                            placeholder="<?php esc_attr_e('Enter OTP sent to your phone', 'authyo-otp-authentication-for-woocommerce'); ?>"
                            autocomplete="off" maxlength="<?php echo esc_attr(get_option('authyo_otp_auth_length', 6)); ?>"
                            pattern="[0-9]*" inputmode="numeric" style="display: none; flex: 1;" />
                        <button type="button" id="authyo_send_phone_otp" class="button authyo-send-otp-btn"
                            style="display: inline-block !important; visibility: visible !important; white-space: nowrap;">
                            <?php esc_html_e('Send Phone OTP', 'authyo-otp-authentication-for-woocommerce'); ?>
                        </button>
                    </div>
                    <button type="button" id="authyo_resend_phone_otp" class="button button-secondary resend-otp-btn"
                        style="display: none; margin-top: 10px;">
                        <span class="timer"><?php esc_html_e('Resend OTP', 'authyo-otp-authentication-for-woocommerce'); ?></span>
                    </button>

                    <div class="resend-otp-section authyo-hidden" style="display: none; margin-top: 10px;">
                        <p class="resend-text" style="margin: 0 0 10px 0; font-size: 13px; color: #666;">
                            <?php esc_html_e('Didn\'t receive OTP? Try Resend OTP via:', 'authyo-otp-authentication-for-woocommerce'); ?>
                        </p>
                        <div class="resend-methods-grid" style="display: flex; gap: 8px; flex-wrap: wrap;">
                            <?php if ($whatsapp_enabled): ?>
                                <button type="button" class="resend-method-btn" data-method="whatsapp">WhatsApp</button>
                            <?php endif; ?>
                            <?php if ($sms_enabled): ?>
                                <button type="button" class="resend-method-btn" data-method="sms">SMS</button>
                            <?php endif; ?>
                            <?php if ($voice_enabled): ?>
                                <button type="button" class="resend-method-btn" data-method="voice">Voice Call</button>
                            <?php endif; ?>
                        </div>
                    </div>

                    <span id="authyo_phone_otp_status" class="authyo-otp-status" style="display: block; margin-top: 5px;"></span>
                    <input type="hidden" id="authyo_checkout_phone_maskid" name="authyo_checkout_phone_maskid" value="" />
                </div>
            <?php endif; ?>

            <input type="hidden" id="authyo_checkout_email_verified" name="authyo_checkout_email_verified" value="0" />
            <input type="hidden" id="authyo_checkout_phone_verified" name="authyo_checkout_phone_verified" value="0" />
        </div>
        <?php
    }

    /**
     * Maybe inject OTP fields on checkout update
     */
    public function maybe_inject_otp_fields()
    {
        // This is called during AJAX updates, ensure fields are available
        // The JavaScript will handle the actual injection
    }

    /**
     * Validate OTP during checkout process
     */
    public function validate_checkout_otp()
    {
        // Log that validation function was called (debug only)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
            error_log('Authyo Checkout OTP: validate_checkout_otp() function called');
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.Security.NonceVerification.Missing -- Debug code only, WooCommerce hook already verifies nonce
            error_log('Authyo Checkout OTP: POST data available: ' . (isset($_POST) ? 'yes' : 'no'));
        }

        // Only validate if checkout OTP is enabled
        if (!$this->is_checkout_otp_enabled()) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Validation skipped - Checkout OTP not enabled');
            }
            return;
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
            error_log('Authyo Checkout OTP: Validation started - Checkout OTP is enabled');
        }

        // Get email - works for both logged-in and guest users, classic and block-based checkout
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook, already verified
        $email = '';
        // Priority 1: Get from POST data - try both 'email' (block-based) and 'billing_email' (classic)
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook verifies nonce
        if (isset($_POST['email']) && !empty($_POST['email'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook
            $email = sanitize_email(wp_unslash($_POST['email']));
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook verifies nonce
        } elseif (isset($_POST['billing_email']) && !empty($_POST['billing_email'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook
            $email = sanitize_email(wp_unslash($_POST['billing_email']));
        }

        // Priority 2: For logged-in users, also check checkout object if POST is empty
        if (empty($email) && is_user_logged_in() && WC()->checkout) {
            // Try billing_email first (classic checkout)
            $checkout_email = WC()->checkout->get_value('billing_email');
            if (empty($checkout_email)) {
                // Try email (block-based checkout)
                $checkout_email = WC()->checkout->get_value('email');
            }
            if (!empty($checkout_email)) {
                $email = sanitize_email($checkout_email);
            }
        }

        // Priority 3: For logged-in users, get from user meta as last resort
        if (empty($email) && is_user_logged_in()) {
            $user = wp_get_current_user();
            if ($user && !empty($user->user_email)) {
                $email = sanitize_email($user->user_email);
            }
        }

        // Get phone - works for both logged-in and guest users
        $phone = '';
        // Priority 1: Get from POST data (works for both guest and logged-in users)
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook
        if (isset($_POST['billing_phone']) && !empty($_POST['billing_phone'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook
            $phone = sanitize_text_field(wp_unslash($_POST['billing_phone']));
        }

        // Priority 2: For logged-in users, also check checkout object if POST is empty
        if (empty($phone) && is_user_logged_in() && WC()->checkout) {
            $checkout_phone = WC()->checkout->get_value('billing_phone');
            if (!empty($checkout_phone)) {
                $phone = sanitize_text_field($checkout_phone);
            }
        }

        // SECURITY FIX: Session data is the source of truth, not POST data
        // POST data can be manipulated by users, so we must rely on session data
        // Ensure session is initialized (critical for guest checkout)
        if (!WC()->session) {
            // Initialize WooCommerce session if not already initialized
            WC()->initialize_session();
        }

        if (WC()->session && !WC()->session->has_session()) {
            WC()->session->set_customer_session_cookie(true);
        }

        // Initialize verification status from session (primary source of truth)
        $email_verified = '0';
        $phone_verified = '0';
        $email_maskid = '';
        $phone_maskid = '';

        if (WC()->session) {
            // Get verification status from session (PRIMARY CHECK - cannot be manipulated)
            $session_email_verified = WC()->session->get('authyo_checkout_email_verified');
            if ($session_email_verified === true || $session_email_verified === '1') {
                $email_verified = '1';
                // Also verify that the email in session matches the current email
                $session_email = WC()->session->get('authyo_checkout_email_contact');
                if (!empty($session_email) && !empty($email) && $session_email !== $email) {
                    // Email changed, reset verification
                    $email_verified = '0';
                    WC()->session->set('authyo_checkout_email_verified', false);
                } else {
                    // Get maskid from session (required for verification)
                    $email_maskid = WC()->session->get('authyo_checkout_email_maskid');
                }
            } else {
                // Session explicitly says NOT verified - always use '0' regardless of POST
                $email_verified = '0';
            }

            $session_phone_verified = WC()->session->get('authyo_checkout_phone_verified');
            if ($session_phone_verified === true || $session_phone_verified === '1') {
                $phone_verified = '1';
                // Also verify that the phone in session matches the current phone
                $session_phone = WC()->session->get('authyo_checkout_phone_contact');
                
                // Debug logging
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                    error_log('Authyo Debug: Session phone = ' . ($session_phone ? $session_phone : 'EMPTY'));
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                    error_log('Authyo Debug: Current phone = ' . ($phone ? $phone : 'EMPTY'));
                }
                
                if (!empty($session_phone) && !empty($phone)) {
                    // Use smart matching (handles country code suffix issues)
                    $is_match = $this->is_phone_match($session_phone, $phone);
                    
                    // Debug logging
                    if (defined('WP_DEBUG') && WP_DEBUG) {
                        // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                        error_log('Authyo Debug: Phones match result = ' . ($is_match ? 'YES' : 'NO'));
                    }
                    
                    if (!$is_match) {
                        // Phone changed significantly - reset verification
                        $phone_verified = '0';
                        WC()->session->set('authyo_checkout_phone_verified', false);
                    } else {
                        // Phone matches (either exact or suffix)
                        // CRITICAL FIX: Ensure the verified session phone (with country code) is what gets saved
                        // This fixes the issue where billing_phone doesn't have country code but verification does
                        if ($session_phone !== $phone) {
                            $_POST['billing_phone'] = $session_phone;
                            $_POST['phone'] = $session_phone;
                            
                            // Also update checkout object if available
                            if (WC()->checkout) {
                                // We can't easily set value on checkout object directly as it reads from POST usually
                                // But updating POST is the standard way to hook into WooCommerce flow here
                            }
                            
                            if (defined('WP_DEBUG') && WP_DEBUG) {
                                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                                error_log('Authyo Debug: Updated POST billing_phone to session phone: ' . $session_phone);
                            }
                        }
                        
                        // Get maskid from session (required for verification)
                        $phone_maskid = WC()->session->get('authyo_checkout_phone_maskid');
                    }
                } else {
                    // Get maskid from session (required for verification)
                    $phone_maskid = WC()->session->get('authyo_checkout_phone_maskid');
                }
            } else {
                // Session explicitly says NOT verified - always use '0' regardless of POST
                $phone_verified = '0';
                
                // Debug logging
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.PHP.DevelopmentFunctions.error_log_var_export
                    error_log('Authyo Debug: Session phone verified = ' . ($session_phone_verified ? var_export($session_phone_verified, true) : 'NULL/FALSE'));
                }
            }
        }

        // Fallback: Only use POST data if session doesn't exist (shouldn't happen, but safety check)
        // This is a last resort - session should always be available
        if (!WC()->session || !WC()->session->has_session()) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook
            $email_verified = isset($_POST['authyo_checkout_email_verified']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_email_verified'])) : '0';
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook
            $phone_verified = isset($_POST['authyo_checkout_phone_verified']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_phone_verified'])) : '0';
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook
            $email_maskid = isset($_POST['authyo_checkout_email_maskid']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_email_maskid'])) : '';
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout hook
            $phone_maskid = isset($_POST['authyo_checkout_phone_maskid']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_phone_maskid'])) : '';
        }

        // Get enabled methods
        $email_enabled = get_option('authyo_otp_auth_checkout_email_enabled', 'yes') === 'yes';
        $whatsapp_enabled = get_option('authyo_otp_auth_checkout_whatsapp_enabled', 'no') === 'yes';
        $sms_enabled = get_option('authyo_otp_auth_checkout_sms_enabled', 'no') === 'yes';
        $voice_enabled = get_option('authyo_otp_auth_checkout_voice_enabled', 'no') === 'yes';
        $phone_method_enabled = $whatsapp_enabled || $sms_enabled || $voice_enabled;

        // Track validation errors
        $has_error = false;
        $error_messages = array();

        // Check if email verification is required
        // Email is ALWAYS required in checkout, so if email OTP is enabled, verification is mandatory
        if ($email_enabled) {
            if (empty($email)) {
                // Email is required but missing - this should be caught by WooCommerce, but add our message
                $error_messages[] = esc_html__('Email address is required.', 'authyo-otp-authentication-for-woocommerce');
                $has_error = true;
            } else {
                // Determine if email is verified (check session first, then POST)
                $is_email_verified = ($email_verified === '1');

                // Strict validation: must be verified AND have maskid
                if (!$is_email_verified) {
                    $error_messages[] = esc_html__('Please verify your email address with the OTP sent to your email before placing the order.', 'authyo-otp-authentication-for-woocommerce');
                    $has_error = true;
                } elseif (empty($email_maskid)) {
                    // If verified but no maskid, still block (security check)
                    $error_messages[] = esc_html__('Email OTP session expired. Please request a new OTP.', 'authyo-otp-authentication-for-woocommerce');
                    $has_error = true;
                }
            }
        }

        // Check if phone verification is required
        // Phone is optional in WooCommerce, so only verify if phone field has a value
        if ($phone_method_enabled && !empty($phone)) {
            // Determine if phone is verified (check session first, then POST)
            $is_phone_verified = ($phone_verified === '1');
            
            // Debug logging
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('Authyo Debug: Final phone_verified value = ' . $phone_verified);
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('Authyo Debug: Phone maskid = ' . ($phone_maskid ? $phone_maskid : 'EMPTY'));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('Authyo Debug: is_phone_verified = ' . ($is_phone_verified ? 'TRUE' : 'FALSE'));
            }

            // Strict validation: must be verified AND have maskid
            if (!$is_phone_verified) {
                $error_messages[] = esc_html__('Please verify your phone number with the OTP sent to your phone before placing the order.', 'authyo-otp-authentication-for-woocommerce');
                $has_error = true;
            } elseif (empty($phone_maskid)) {
                // If verified but no maskid, still block (security check)
                $error_messages[] = esc_html__('Phone OTP session expired. Please request a new OTP.', 'authyo-otp-authentication-for-woocommerce');
                $has_error = true;
            }
        }

        // If there are errors, add notices to block checkout
        if ($has_error) {
            // Log for debugging (debug mode only)
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: ========== VALIDATION FAILED ==========');
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Email enabled: ' . ($email_enabled ? 'yes' : 'no'));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Phone methods enabled: ' . ($phone_method_enabled ? 'yes' : 'no'));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Email verified: ' . ($email_verified === '1' ? 'yes' : 'no'));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Phone verified: ' . ($phone_verified === '1' ? 'yes' : 'no'));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Email: ' . ($email ? $email : 'EMPTY'));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Phone: ' . ($phone ? $phone : 'EMPTY'));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Error messages: ' . implode(', ', $error_messages));
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Debug code only, WooCommerce hook verifies nonce
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.PHP.DevelopmentFunctions.error_log_print_r -- Debug code only, data is sanitized
                error_log('Authyo Checkout OTP: POST data: ' . print_r(array(
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Debug code only, data is sanitized
                    'authyo_checkout_email_verified' => isset($_POST['authyo_checkout_email_verified']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_email_verified'])) : 'NOT SET',
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Debug code only, data is sanitized
                    'authyo_checkout_phone_verified' => isset($_POST['authyo_checkout_phone_verified']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_phone_verified'])) : 'NOT SET',
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Debug code only, data is sanitized
                    'email' => isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : (isset($_POST['billing_email']) ? sanitize_email(wp_unslash($_POST['billing_email'])) : 'NOT SET'),
                    // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Debug code only, data is sanitized
                    'phone' => isset($_POST['billing_phone']) ? sanitize_text_field(wp_unslash($_POST['billing_phone'])) : 'NOT SET'
                ), true));
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: ======================================');
            }

            // Add all error messages
            foreach ($error_messages as $error_msg) {
                wc_add_notice($error_msg, 'error');
            }

            // Force validation failure by throwing an exception
            // This will definitely prevent order creation
            // Escape error messages for security (WordPress coding standards)
            $escaped_messages = array_map('esc_html', $error_messages);
            $exception_message = esc_html(implode(' ', $escaped_messages));
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- $exception_message is already escaped via esc_html
            throw new Exception($exception_message);
        } else {
            // Log successful validation (debug mode only)
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Validation passed - All OTPs verified');
            }
        }
    }

    /**
     * Validate before order is created - prevents order creation if OTP not verified
     * This runs on woocommerce_checkout_create_order hook
     *
     * @param WC_Order $order Order object being created.
     * @param array    $data  Posted checkout data.
     *
     * @throws Exception If OTP verification fails.
     */
    public function validate_before_order_created($order, $data)
    {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
            error_log('Authyo Checkout OTP: validate_before_order_created() called');
        }

        // Only validate if checkout OTP is enabled
        if (!$this->is_checkout_otp_enabled()) {
            return;
        }

        // Re-run the main validation function which will throw exception if validation fails
        // This prevents the order from being created in the first place
        try {
            $this->validate_checkout_otp();
        } catch (Exception $e) {
            // Re-throw the exception to prevent order creation
            throw $e;
        }
    }

    /**
     * Validate before order creation - final safety check
     * This will delete the order if validation fails
     *
     * @param int|WC_Order $order_id Order ID or order object.
     */
    public function validate_before_order_creation($order_id)
    {
        // phpcs:disable WordPress.Security.NonceVerification.Missing
        // This function is called via woocommerce_checkout_order_processed hook which runs after WooCommerce has already verified the nonce.
        // We primarily use session data (which cannot be manipulated) and only fall back to POST data if session is unavailable.
        // All POST data is sanitized before use.
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
            error_log('Authyo Checkout OTP: validate_before_order_creation() called with order_id: ' . $order_id);
        }

        // Only validate if checkout OTP is enabled
        if (!$this->is_checkout_otp_enabled()) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Order validation skipped - Checkout OTP not enabled');
            }
            return $order_id;
        }

        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
            error_log('Authyo Checkout OTP: Order validation started - Checkout OTP is enabled');
        }

        // Re-run validation to check current state
        $email_enabled = get_option('authyo_otp_auth_checkout_email_enabled', 'yes') === 'yes';
        $whatsapp_enabled = get_option('authyo_otp_auth_checkout_whatsapp_enabled', 'no') === 'yes';
        $sms_enabled = get_option('authyo_otp_auth_checkout_sms_enabled', 'no') === 'yes';
        $voice_enabled = get_option('authyo_otp_auth_checkout_voice_enabled', 'no') === 'yes';
        $phone_method_enabled = $whatsapp_enabled || $sms_enabled || $voice_enabled;

        // Get email and phone from order
        $order = is_numeric($order_id) ? wc_get_order($order_id) : $order_id;
        if (!$order instanceof WC_Order) {
            return $order_id;
        }

        $email = $order->get_billing_email();
        $phone = $order->get_billing_phone();

        // SECURITY FIX: Session data is the source of truth, not POST data
        // Initialize verification status from session (primary source of truth)
        $email_verified = '0';
        $phone_verified = '0';

        // Ensure session is initialized
        if (!WC()->session) {
            WC()->initialize_session();
        }

        if (WC()->session && !WC()->session->has_session()) {
            WC()->session->set_customer_session_cookie(true);
        }

        // Check session (PRIMARY CHECK - cannot be manipulated)
        if (WC()->session) {
            $session_email_verified = WC()->session->get('authyo_checkout_email_verified');
            if ($session_email_verified === true || $session_email_verified === '1') {
                // Verify that the email in session matches the order email
                $session_email = WC()->session->get('authyo_checkout_email_contact');
                if (!empty($session_email) && !empty($email) && $session_email === $email) {
                    $email_verified = '1';
                } else {
                    // Email doesn't match - reset verification
                    $email_verified = '0';
                    WC()->session->set('authyo_checkout_email_verified', false);
                }
            } else {
                // Session explicitly says NOT verified - always use '0' regardless of POST
                $email_verified = '0';
            }

            $session_phone_verified = WC()->session->get('authyo_checkout_phone_verified');
            if ($session_phone_verified === true || $session_phone_verified === '1') {
                // Verify that the phone in session matches the order phone
                $session_phone = WC()->session->get('authyo_checkout_phone_contact');
                if (!empty($session_phone) && !empty($phone)) {
                    // Use smart matching (handles country code suffix issues)
                    $is_match = $this->is_phone_match($session_phone, $phone);
                    
                    if ($is_match) {
                        $phone_verified = '1';
                        // Note: We can't easily update order phone here as order is already processed/processing
                        // But the primary validation hook should have already handled the update via POST data
                    } else {
                        // Phone doesn't match - reset verification
                        $phone_verified = '0';
                        WC()->session->set('authyo_checkout_phone_verified', false);
                    }
                } else {
                    // Phone doesn't match - reset verification
                    $phone_verified = '0';
                    WC()->session->set('authyo_checkout_phone_verified', false);
                }
            } else {
                // Session explicitly says NOT verified - always use '0' regardless of POST
                $phone_verified = '0';
            }
        }

        // Fallback: Only use POST data if session doesn't exist (shouldn't happen, but safety check)
        if (!WC()->session || !WC()->session->has_session()) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- This function is called via woocommerce_checkout_order_processed hook which runs after WooCommerce has already verified the nonce. This is a fallback only used when session is unavailable, and data is sanitized.
            $email_verified = isset($_POST['authyo_checkout_email_verified']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_email_verified'])) : '0';
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- This function is called via woocommerce_checkout_order_processed hook which runs after WooCommerce has already verified the nonce. This is a fallback only used when session is unavailable, and data is sanitized.
            $phone_verified = isset($_POST['authyo_checkout_phone_verified']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_phone_verified'])) : '0';
        }

        $has_error = false;
        $error_messages = array();

        // Check email verification
        if ($email_enabled && !empty($email)) {
            if ($email_verified !== '1') {
                $has_error = true;
                $error_messages[] = __('Please verify your email address with the OTP sent to your email before placing the order.', 'authyo-otp-authentication-for-woocommerce');
            }
        }

        // Check phone verification
        if ($phone_method_enabled && !empty($phone)) {
            if ($phone_verified !== '1') {
                $has_error = true;
                $error_messages[] = __('Please verify your phone number with the OTP sent to your phone before placing the order.', 'authyo-otp-authentication-for-woocommerce');
            }
        }

        // If validation fails, delete the order and prevent processing
        if ($has_error) {
            // Delete the order
            wp_delete_post(is_numeric($order_id) ? $order_id : $order->get_id(), true);

            // Add error notices
            foreach ($error_messages as $error_msg) {
                wc_add_notice($error_msg, 'error');
            }

            // Log for debugging (debug mode only)
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Order ' . (is_numeric($order_id) ? $order_id : $order->get_id()) . ' deleted due to missing OTP verification');
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Email verified: ' . $email_verified);
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                error_log('Authyo Checkout OTP: Phone verified: ' . $phone_verified);
            }

            // Prevent further processing by throwing exception
            // Escape error messages for security (WordPress coding standards)
            $escaped_messages = array_map('esc_html', $error_messages);
            $exception_message = esc_html(implode(' ', $escaped_messages));
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- $exception_message is already escaped via esc_html
            throw new Exception($exception_message);
        }
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        return $order_id;
    }

    /**
     * Add OTP fields to checkout data for block-based checkout
     *
     * @param WC_Order $order   Order object.
     * @param array    $request Request data.
     */
    public function add_otp_fields_to_checkout_data($order, $request)
    {
        // Ensure our verification fields are in the request
        if (isset($request['extensions']['authyo_otp'])) {
            $otp_data = $request['extensions']['authyo_otp'];
            if (isset($otp_data['email_verified'])) {
                $_POST['authyo_checkout_email_verified'] = sanitize_text_field($otp_data['email_verified']); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Block checkout verifies nonce separately.
            }
            if (isset($otp_data['phone_verified'])) {
                $_POST['authyo_checkout_phone_verified'] = sanitize_text_field($otp_data['phone_verified']); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Block checkout verifies nonce separately.
            }
            if (isset($otp_data['email_maskid'])) {
                $_POST['authyo_checkout_email_maskid'] = sanitize_text_field($otp_data['email_maskid']); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Block checkout verifies nonce separately.
            }
            if (isset($otp_data['phone_maskid'])) {
                $_POST['authyo_checkout_phone_maskid'] = sanitize_text_field($otp_data['phone_maskid']); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Block checkout verifies nonce separately.
            }
        }

        // When the Store API is placing the final order (POST /checkout), validate OTP before payment.
        if ($request instanceof \WP_REST_Request && $request->get_method() === \WP_REST_Server::CREATABLE) {
            $this->validate_store_api_checkout_request($request);
        }

        return $order;
    }

    /**
     * Validate Store API checkout submission to ensure OTP requirements are met.
     *
     * @param \WP_REST_Request $request Request object.
     *
     * @throws \Automattic\WooCommerce\StoreApi\Exceptions\RouteException When validation fails.
     */
    private function validate_store_api_checkout_request(\WP_REST_Request $request)
    {
        if (!$this->is_checkout_otp_enabled()) {
            return;
        }

        $simulated_post = array();
        $billing_address = (array)$request->get_param('billing_address');

        if (!empty($billing_address)) {
            if (!empty($billing_address['email'])) {
                $email = sanitize_email(wp_unslash($billing_address['email']));
                $simulated_post['email'] = $email;
                $simulated_post['billing_email'] = $email;
            }
            if (!empty($billing_address['phone'])) {
                $simulated_post['billing_phone'] = sanitize_text_field(wp_unslash($billing_address['phone']));
            }
        }

        $extensions = (array)$request->get_param('extensions');
        if (isset($extensions['authyo_otp']) && is_array($extensions['authyo_otp'])) {
            $otp_data = $extensions['authyo_otp'];
            $mapping = array(
                'authyo_checkout_email_verified' => 'email_verified',
                'authyo_checkout_phone_verified' => 'phone_verified',
                'authyo_checkout_email_maskid' => 'email_maskid',
                'authyo_checkout_phone_maskid' => 'phone_maskid',
            );

            foreach ($mapping as $post_key => $extension_key) {
                if (isset($otp_data[$extension_key])) {
                    $simulated_post[$post_key] = sanitize_text_field(wp_unslash($otp_data[$extension_key]));
                }
            }
        }

        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Store API handles nonce verification separately, and we're only reading $_POST to preserve existing data before merging sanitized data.
        $original_post = $_POST ?? array();
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Store API handles nonce verification separately. Data is sanitized above and we're merging into $_POST temporarily for validation function compatibility.
        $_POST = array_merge($original_post, $simulated_post);

        try {
            $this->validate_checkout_otp();
        } catch (\Exception $exception) {
            $_POST = $original_post;
            throw new \Automattic\WooCommerce\StoreApi\Exceptions\RouteException(
                'authyo_otp_verification_required',
                esc_html($exception->getMessage()),
                400
            );
        }

        $_POST = $original_post;
    }

    /**
     * Ensure OTP fields are always in posted data
     *
     * @param array $data Posted checkout data.
     */
    public function ensure_otp_fields_in_posted_data($data)
    {
        // If fields are missing, add them with default values
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout filter
        if (!isset($data['authyo_checkout_email_verified'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout filter
            $data['authyo_checkout_email_verified'] = isset($_POST['authyo_checkout_email_verified']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_email_verified'])) : '0';
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout filter
        if (!isset($data['authyo_checkout_phone_verified'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout filter
            $data['authyo_checkout_phone_verified'] = isset($_POST['authyo_checkout_phone_verified']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_phone_verified'])) : '0';
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout filter
        if (!isset($data['authyo_checkout_email_maskid'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout filter
            $data['authyo_checkout_email_maskid'] = isset($_POST['authyo_checkout_email_maskid']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_email_maskid'])) : '';
        }
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout filter
        if (!isset($data['authyo_checkout_phone_maskid'])) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce checkout filter
            $data['authyo_checkout_phone_maskid'] = isset($_POST['authyo_checkout_phone_maskid']) ? sanitize_text_field(wp_unslash($_POST['authyo_checkout_phone_maskid'])) : '';
        }
        return $data;
    }

    /**
     * Normalize checkout phone number so orders store E.164 format.
     *
     * @param array $data Posted checkout data.
     *
     * @return array
     */
    public function maybe_format_checkout_phone($data)
    {
        if (empty($data) || !is_array($data) || empty($data['billing_phone'])) {
            return $data;
        }

        $raw_phone = $data['billing_phone'];

        // Skip if phone already has country code
        if (strpos($raw_phone, '+') === 0) {
            return $data;
        }

        // Priority 1: Use session phone contact (from OTP verification) - most reliable
        $formatted = $this->get_session_phone_contact();

        // Priority 2: Try to format with country code from form submission
        if (empty($formatted)) {
            $country_code = '';

            if (!empty($data['checkout_country_code'])) {
                $country_code = $data['checkout_country_code'];
            } elseif (isset($_POST['checkout_country_code'])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce already verified nonce for checkout submission.
                $country_code = sanitize_text_field(wp_unslash($_POST['checkout_country_code'])); // phpcs:ignore WordPress.Security.NonceVerification.Missing
            }

            $formatted = $this->format_phone_with_country($raw_phone, $country_code);
        }

        // Update data if we have a formatted phone
        if (!empty($formatted) && $formatted !== $raw_phone) {
            $data['billing_phone'] = $formatted;
            // Keep the global POST value consistent for downstream hooks.
            $_POST['billing_phone'] = $formatted; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Value sanitized above and maintains consistency.
        }

        return $data;
    }

    /**
     * Apply formatted phone to order object before saving.
     *
     * @param WC_Order $order Order being created.
     * @param array    $data  Posted checkout data.
     */
    public function apply_formatted_phone_to_order($order, $data)
    {
        if (!$order instanceof WC_Order) {
            return;
        }

        // Get phone from order first, but also check data array and POST as fallbacks
        $raw_phone = $order->get_billing_phone();

        if (empty($raw_phone) && !empty($data['billing_phone'])) {
            $raw_phone = $data['billing_phone'];
        }

        if (empty($raw_phone)) {
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce already verified nonce for checkout submission.
            $raw_phone = isset($_POST['billing_phone']) ? sanitize_text_field(wp_unslash($_POST['billing_phone'])) : '';
        }

        if (empty($raw_phone)) {
            return;
        }

        // Skip if phone already has country code
        if (strpos($raw_phone, '+') === 0) {
            return;
        }

        // Priority 1: Use session phone contact (from OTP verification) - most reliable
        $formatted = $this->get_session_phone_contact();

        // Priority 2: Try to format with country code from form submission
        if (empty($formatted)) {
            $country_code = '';

            if (!empty($data['checkout_country_code'])) {
                $country_code = $data['checkout_country_code'];
            } elseif (isset($_POST['checkout_country_code'])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce already verified nonce for checkout submission.
                $country_code = sanitize_text_field(wp_unslash($_POST['checkout_country_code'])); // phpcs:ignore WordPress.Security.NonceVerification.Missing
            }

            $formatted = $this->format_phone_with_country($raw_phone, $country_code);
        }

        if (!empty($formatted) && $formatted !== $raw_phone) {
            $order->set_billing_phone($formatted);
        }
    }

    /**
     * Ensure user meta (billing phone) is stored with country code for future orders.
     *
     * @param int   $customer_id Customer ID being updated.
     * @param array $data        Checkout data.
     */
    public function apply_formatted_phone_to_user_meta($customer_id, $data)
    {
        if (empty($customer_id) || empty($data['billing_phone'])) {
            return;
        }

        $raw_phone = $data['billing_phone'];

        // Skip if phone already has country code
        if (strpos($raw_phone, '+') === 0) {
            return;
        }

        // Priority 1: Use session phone contact (from OTP verification) - most reliable
        $formatted = $this->get_session_phone_contact();

        // Priority 2: Try to format with country code from form submission
        if (empty($formatted)) {
            $country_code = '';

            if (!empty($data['checkout_country_code'])) {
                $country_code = $data['checkout_country_code'];
            } elseif (isset($_POST['checkout_country_code'])) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce already verified nonce for checkout submission.
                $country_code = sanitize_text_field(wp_unslash($_POST['checkout_country_code'])); // phpcs:ignore WordPress.Security.NonceVerification.Missing
            }

            $formatted = $this->format_phone_with_country($raw_phone, $country_code);
        }

        if (!empty($formatted) && $formatted !== $raw_phone) {
            update_user_meta($customer_id, 'billing_phone', $formatted);
        }
    }

    /**
     * Convert phone number into +E.164 format using selected or default country code.
     *
     * @param string $phone        Raw phone input.
     * @param string $country_code Country code selected at checkout (may include +).
     *
     * @return string
     */
    private function format_phone_with_country($phone, $country_code = '')
    {
        if (empty($phone) || !is_string($phone)) {
            return '';
        }

        $phone = trim($phone);

        // If the phone already includes a + prefix, validate and normalize digits.
        if (strpos($phone, '+') === 0) {
            $digits = preg_replace('/[^\d]/', '', substr($phone, 1));
            if (strlen($digits) >= 7 && strlen($digits) <= 15) {
                return '+' . $digits;
            }
        }

        $clean_phone = preg_replace('/\D+/', '', $phone);

        if (strlen($clean_phone) < 7 || strlen($clean_phone) > 15) {
            return '';
        }

        if (empty($country_code)) {
            $country_code = get_option('authyo_otp_auth_default_country', '91');
        }

        $country_code = preg_replace('/\D+/', '', (string) $country_code);

        if (empty($country_code)) {
            return '';
        }

        $full_number = '+' . $country_code . $clean_phone;

        // Final length check (E.164 allows up to 15 digits after +).
        $digits_after_plus = preg_replace('/\D+/', '', substr($full_number, 1));
        if (strlen($digits_after_plus) < 7 || strlen($digits_after_plus) > 15) {
            return '';
        }

        return $full_number;
    }

    /**
     * Update order phone number after order creation (fallback to ensure phone is formatted).
     *
     * @param int|WC_Order $order_id Order ID or order object.
     */
    public function update_order_phone_after_creation($order_id)
    {
        $order = is_numeric($order_id) ? wc_get_order($order_id) : $order_id;

        if (!$order instanceof WC_Order) {
            return;
        }

        $raw_phone = $order->get_billing_phone();

        // If phone already has country code (+ prefix), skip
        if (!empty($raw_phone) && strpos($raw_phone, '+') === 0) {
            return;
        }

        // Priority 1: Use session phone contact (from OTP verification) - most reliable
        $formatted = $this->get_session_phone_contact();

        // Priority 2: Try to format with country code from form submission
        if (empty($formatted)) {
            $country_code = '';
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce already verified nonce for checkout submission.
            if (isset($_POST['checkout_country_code']) && !empty($_POST['checkout_country_code'])) {
                // phpcs:ignore WordPress.Security.NonceVerification.Missing -- WooCommerce already verified nonce.
                $country_code = sanitize_text_field(wp_unslash($_POST['checkout_country_code']));
            }

            // Try to format phone with country code
            $formatted = $this->format_phone_with_country($raw_phone, $country_code);
        }

        // If we have a formatted phone, update the order
        if (!empty($formatted) && $formatted !== $raw_phone) {
            $order->set_billing_phone($formatted);
            $order->save(); // Save the order with formatted phone
        }
    }

    /**
     * Retrieve the phone contact stored during OTP verification (if available).
     *
     * @return string
     */
    private function get_session_phone_contact()
    {
        if (function_exists('WC') && WC()->session) {
            $session = WC()->session;

            if (!$session->has_session()) {
                $session->set_customer_session_cookie(true);
            }

            $stored = $session->get('authyo_checkout_phone_contact');
            if (!empty($stored) && is_string($stored)) {
                return preg_replace('/[^+\d]/', '', $stored);
            }
        }

        return '';
    }

    /**
     * Normalize phone number for comparison by removing all non-numeric characters.
     * This allows comparison between different formats like +919876543210, 9876543210, (919) 876-543210, etc.
     *
     * @param string $phone Phone number to normalize.
     *
     * @return string Normalized phone number containing only digits.
     */
    private function normalize_phone_number($phone)
    {
        if (empty($phone)) {
            return '';
        }
        // Remove all non-numeric characters (spaces, dashes, parentheses, plus signs, etc.)
        return preg_replace('/[^0-9]/', '', $phone);
    }

    /**
     * Check if two phone numbers match, accounting for country codes.
     * Returns true if they are identical or if one is a suffix of the other (likely valid match).
     *
     * @param string $session_phone Phone from session (usually has country code).
     * @param string $input_phone   Phone from input (might lack country code).
     *
     * @return boolean
     */
    private function is_phone_match($session_phone, $input_phone)
    {
        $normalized_session = $this->normalize_phone_number($session_phone);
        $normalized_input = $this->normalize_phone_number($input_phone);

        // Exact match
        if ($normalized_session === $normalized_input) {
            return true;
        }

        // Check if input is suffix of session (e.g. input 9876543210 is in session 919876543210)
        // Only do this if input is at least 7 digits to avoid false positives with short numbers
        if (strlen($normalized_input) >= 7 && substr($normalized_session, -strlen($normalized_input)) === $normalized_input) {
            return true;
        }

        return false;
    }


    /**
     * Send OTP for checkout email
     */
    public function send_checkout_otp()
    {
        // Prevent any output before headers
        if (ob_get_level()) {
            ob_clean();
        }

        // Log for debugging (remove in production)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
            error_log('Authyo Checkout OTP: send_checkout_otp called');
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.PHP.DevelopmentFunctions.error_log_print_r, WordPress.Security.NonceVerification.Missing -- Debug code only, nonce is verified separately in this function
            error_log('Authyo Checkout OTP: POST data: ' . print_r($_POST, true));
        }

        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            wp_send_json_error(array(
                'message' => esc_html__('WooCommerce is not active.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Verify nonce - use die=false to handle errors gracefully
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';

        if (empty($nonce)) {
            wp_send_json_error(array(
                'message' => esc_html__('Security token is missing. Please refresh the page and try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Verify nonce
        $nonce_check = check_ajax_referer('authyo_otp_auth_nonce', 'nonce', false);
        if (!$nonce_check) {
            wp_send_json_error(array(
                'message' => esc_html__('Security check failed. Please refresh the page and try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        $type = isset($_POST['type']) ? sanitize_text_field(wp_unslash($_POST['type'])) : ''; // 'email' or 'phone'

        if ($type === 'email') {
            // Get email from POST data - works for both guest and logged-in users
            // Handle both classic checkout (billing_email) and block-based checkout (email)
            $email = '';

            // Priority 1: Try 'email' field (block-based checkout)
            if (isset($_POST['email']) && !empty($_POST['email'])) {
                $email = sanitize_email(wp_unslash($_POST['email']));
            }

            // Priority 2: Try 'billing_email' field (classic checkout)
            if (empty($email) && isset($_POST['billing_email']) && !empty($_POST['billing_email'])) {
                $email = sanitize_email(wp_unslash($_POST['billing_email']));
            }

            // Priority 3: For logged-in users, also try checkout object if POST is empty
            if (empty($email) && is_user_logged_in() && WC()->checkout) {
                // Try billing_email first (classic)
                $checkout_email = WC()->checkout->get_value('billing_email');
                if (empty($checkout_email)) {
                    // Try email (block-based)
                    $checkout_email = WC()->checkout->get_value('email');
                }
                if (!empty($checkout_email)) {
                    $email = sanitize_email($checkout_email);
                }
            }

            if (empty($email) || !is_email($email)) {
                // Log for debugging
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug code only runs when WP_DEBUG is enabled
                    error_log('Authyo Checkout OTP: Invalid email - ' . $email);
                }
                wp_send_json_error(array(
                    'message' => esc_html__('Please enter a valid email address.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Set contact for email type
            $contact = $email;
        } elseif ($type === 'phone') {
            // Get phone from POST data - works for both guest and logged-in users
            $phone = isset($_POST['phone']) ? sanitize_text_field(wp_unslash($_POST['phone'])) : '';

            // Fallback: try to get from billing_phone field (works for guest checkout)
            if (empty($phone) && isset($_POST['billing_phone'])) {
                $phone = sanitize_text_field(wp_unslash($_POST['billing_phone']));
            }

            // For logged-in users, also try checkout object if POST is empty
            if (empty($phone) && is_user_logged_in() && WC()->checkout) {
                $checkout_phone = WC()->checkout->get_value('billing_phone');
                if (!empty($checkout_phone)) {
                    $phone = sanitize_text_field($checkout_phone);
                }
            }

            if (empty($phone)) {
                wp_send_json_error(array(
                    'message' => esc_html__('Please enter a valid phone number.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Check if any phone method is enabled
            $whatsapp_enabled = get_option('authyo_otp_auth_checkout_whatsapp_enabled', 'no') === 'yes';
            $sms_enabled = get_option('authyo_otp_auth_checkout_sms_enabled', 'no') === 'yes';
            $voice_enabled = get_option('authyo_otp_auth_checkout_voice_enabled', 'no') === 'yes';

            if (!$whatsapp_enabled && !$sms_enabled && !$voice_enabled) {
                wp_send_json_error(array(
                    'message' => esc_html__('Phone OTP verification is disabled.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Phone number should already be processed with country code from frontend
            // But ensure it's in proper format
            $phone = preg_replace('/[^\d+]/', '', $phone);
            if (strpos($phone, '+') !== 0) {
                // If no + prefix, this should have been handled by frontend, but add default
                $default_country = get_option('authyo_otp_auth_default_country', '91');
                $phone = '+' . $default_country . $phone;
            }

            $contact = $phone;
        } else {
            wp_send_json_error(array(
                'message' => esc_html__('Invalid request type.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Check if SDK class exists
        if (!class_exists('Authyo_OTP_Auth_SDK')) {
            wp_send_json_error(array(
                'message' => esc_html__('OTP service is not available. Please contact support.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        try {
            // Determine the best method based on checkout settings
            $method = null;

            if ($type === 'email') {
                $method = 'email';
            } elseif ($type === 'phone') {
                // Re-fetch enabled status here to be safe
                $whatsapp_enabled = get_option('authyo_otp_auth_checkout_whatsapp_enabled', 'no') === 'yes';
                $sms_enabled = get_option('authyo_otp_auth_checkout_sms_enabled', 'no') === 'yes';
                $voice_enabled = get_option('authyo_otp_auth_checkout_voice_enabled', 'no') === 'yes';

                // Get global priority to respect user preference order
                $priority_method = get_option('authyo_otp_auth_priority_method', 'whatsapp');

                // Check if priority method is enabled for checkout
                if ($priority_method === 'whatsapp' && $whatsapp_enabled) {
                    $method = 'whatsapp';
                } elseif ($priority_method === 'sms' && $sms_enabled) {
                    $method = 'sms';
                } elseif ($priority_method === 'voice' && $voice_enabled) {
                    $method = 'voice';
                }

                // If priority method not enabled for checkout, pick first available
                if (!$method) {
                    if ($whatsapp_enabled)
                        $method = 'whatsapp';
                    elseif ($sms_enabled)
                        $method = 'sms';
                    elseif ($voice_enabled)
                        $method = 'voice';
                }
            }

            // Send OTP via Authyo
            $authyo = new Authyo_OTP_Auth_SDK();
            // Pass the determined method explicitly
            $result = $authyo->send_otp($contact, $method, 'checkout');

            if (is_wp_error($result)) {
                wp_send_json_error(array(
                    'message' => esc_html($result->get_error_message())
                ));
                return;
            }

            if (!isset($result['maskId'])) {
                wp_send_json_error(array(
                    'message' => esc_html__('Invalid response from OTP service.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Store OTP data in transient
            $otp_expiry = get_option('authyo_otp_auth_expiry', 300);
            $transient_key = 'authyo_checkout_otp_' . $type . '_' . md5($contact);
            set_transient(
                $transient_key,
                array(
                    'mask_id' => sanitize_text_field($result['maskId']),
                    'contact' => sanitize_text_field($contact),
                    'time' => time(),
                    'type' => $type
                ),
                absint($otp_expiry)
            );

            wp_send_json_success(array(
                'message' => esc_html__('OTP sent successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'maskId' => sanitize_text_field($result['maskId']),
                'method' => isset($result['method']) ? sanitize_text_field($result['method']) : ($type === 'email' ? 'email' : 'sms')
            ));

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
        }

        wp_die();
    }
    /**
     * Verify OTP for checkout
     */
    public function verify_checkout_otp()
    {
        // Verify nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!check_ajax_referer('authyo_otp_auth_nonce', 'nonce', false) || empty($nonce)) {
            wp_send_json_error(array(
                'message' => esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        $type = isset($_POST['type']) ? sanitize_text_field(wp_unslash($_POST['type'])) : ''; // 'email' or 'phone'
        $contact = isset($_POST['contact']) ? sanitize_text_field(wp_unslash($_POST['contact'])) : '';
        $otp = isset($_POST['otp']) ? sanitize_text_field(wp_unslash($_POST['otp'])) : '';
        $maskId = isset($_POST['maskId']) ? sanitize_text_field(wp_unslash($_POST['maskId'])) : '';

        if (empty($type) || empty($contact) || empty($otp) || empty($maskId)) {
            wp_send_json_error(array(
                'message' => esc_html__('All fields are required.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        try {
            // Verify OTP using Authyo
            $authyo = new Authyo_OTP_Auth_SDK();
            $result = $authyo->verify_otp($contact, $otp, $maskId);

            if (is_wp_error($result)) {
                wp_send_json_error(array(
                    'message' => esc_html($result->get_error_message())
                ));
                return;
            }

            // Verify against stored transient
            $transient_key = 'authyo_checkout_otp_' . $type . '_' . md5($contact);
            $stored_data = get_transient($transient_key);

            if (!$stored_data || !isset($stored_data['mask_id']) || $stored_data['mask_id'] !== $maskId) {
                wp_send_json_error(array(
                    'message' => esc_html__('Invalid OTP session. Please request a new OTP.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Mark as verified - store in session if WooCommerce session is available
            // Ensure session is initialized (critical for guest checkout)
            if (!WC()->session) {
                // Initialize WooCommerce session if not already initialized
                WC()->initialize_session();
            }

            if (WC()->session && !WC()->session->has_session()) {
                WC()->session->set_customer_session_cookie(true);
            }

            if (WC()->session) {
                WC()->session->set('authyo_checkout_' . $type . '_verified', true);
                WC()->session->set('authyo_checkout_' . $type . '_maskid', $maskId);
                // Also store the contact (email/phone) for validation
                WC()->session->set('authyo_checkout_' . $type . '_contact', $contact);
            }

            wp_send_json_success(array(
                'message' => esc_html__('OTP verified successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'verified' => true
            ));

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
        }

        wp_die();
    }

    /**
     * Resend OTP via fallback method for checkout
     */
    public function resend_checkout_fallback()
    {
        // Verify nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!check_ajax_referer('authyo_otp_auth_nonce', 'nonce', false) || empty($nonce)) {
            wp_send_json_error(array(
                'message' => esc_html__('Security check failed. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        $type = isset($_POST['type']) ? sanitize_text_field(wp_unslash($_POST['type'])) : ''; // 'email' or 'phone'
        $contact = isset($_POST['contact']) ? sanitize_text_field(wp_unslash($_POST['contact'])) : '';
        $method = isset($_POST['method']) ? sanitize_text_field(wp_unslash($_POST['method'])) : '';

        if (empty($type) || empty($contact) || empty($method)) {
            wp_send_json_error(array(
                'message' => esc_html__('Missing required data.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        // Validate input type
        $is_email = $type === 'email';
        $is_mobile = $type === 'phone';

        // Fallback methods (SMS, WhatsApp, Voice) are only available for mobile numbers
        if ($is_email) {
            wp_send_json_error(array(
                'message' => esc_html__('Fallback methods (SMS, WhatsApp, Voice Call) are only available for mobile numbers, not for email addresses.', 'authyo-otp-authentication-for-woocommerce')
            ));
            return;
        }

        try {
            // Send OTP with specified method
            $authyo = new Authyo_OTP_Auth_SDK();
            $result = $authyo->send_otp($contact, $method, 'checkout');

            if (is_wp_error($result)) {
                wp_send_json_error(array(
                    'message' => esc_html($result->get_error_message())
                ));
                return;
            }

            if (!isset($result['maskId'])) {
                wp_send_json_error(array(
                    'message' => esc_html__('Invalid response from OTP service.', 'authyo-otp-authentication-for-woocommerce')
                ));
                return;
            }

            // Store OTP data in transient
            $otp_expiry = get_option('authyo_otp_auth_expiry', 300);
            $transient_key = 'authyo_checkout_otp_' . $type . '_' . md5($contact);
            set_transient(
                $transient_key,
                array(
                    'mask_id' => sanitize_text_field($result['maskId']),
                    'contact' => sanitize_text_field($contact),
                    'time' => time(),
                    'type' => $type,
                    'method' => sanitize_text_field($method)
                ),
                absint($otp_expiry)
            );

            $method_labels = array(
                'whatsapp' => esc_html__('WhatsApp', 'authyo-otp-authentication-for-woocommerce'),
                'sms' => esc_html__('SMS', 'authyo-otp-authentication-for-woocommerce'),
                'voice' => esc_html__('Voice Call', 'authyo-otp-authentication-for-woocommerce'),
            );

            $method_label = isset($method_labels[$method]) ? $method_labels[$method] : ucfirst($method);
            $actual_method = isset($result['method']) ? $result['method'] : $method;
            $method_label = isset($method_labels[$actual_method]) ? $method_labels[$actual_method] : ucfirst($actual_method);

            wp_send_json_success(array(
                'message' => sprintf(
                    /* translators: %s is the method name (e.g., SMS, WhatsApp, Voice Call) */
                    esc_html__('OTP sent successfully via %s!', 'authyo-otp-authentication-for-woocommerce'),
                    $method_label
                ),
                'maskId' => sanitize_text_field($result['maskId']),
                'method' => sanitize_text_field($actual_method)
            ));

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            ));
        }

        wp_die();
    }

    /**
     * Get country list for checkout
     */
    public function get_country_list()
    {
        // Verify nonce
        $nonce = isset($_POST['nonce']) ? sanitize_text_field(wp_unslash($_POST['nonce'])) : '';
        if (!check_ajax_referer('authyo_otp_auth_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }

        // Try to get from cache first
        $cached_countries = get_transient('authyo_country_list_cache');
        if ($cached_countries !== false) {
            wp_send_json_success(array('countries' => $cached_countries));
            return;
        }

        // Try to get from Authyo API
        $api_url = 'https://app.authyo.io/api/v1/user/getcountrylist';

        $response = wp_remote_get($api_url, array(
            'timeout' => 30,
            'sslverify' => true
        ));

        if (!is_wp_error($response)) {
            $body = wp_remote_retrieve_body($response);
            $countries = json_decode($body, true);

            if ($countries && is_array($countries)) {
                // Cache successful API response for 24 hours
                set_transient('authyo_country_list_cache', $countries, 24 * HOUR_IN_SECONDS);
                wp_send_json_success(array('countries' => $countries));
                return;
            }
        }

        // Fallback to minimal essential country list
        $countries = $this->get_fallback_country_list();

        // Cache fallback for 1 hour
        set_transient('authyo_country_list_cache', $countries, HOUR_IN_SECONDS);

        wp_send_json_success(array('countries' => $countries));
    }

    /**
     * Get fallback country list
     */
    private function get_fallback_country_list()
    {
        // Minimal essential fallback country list (used when API is unavailable)
        return array(
            // North America
            array('name' => 'United States', 'countryCode' => '+1', 'code' => '1', 'phoneLength' => 11),
            array('name' => 'Canada', 'countryCode' => '+1', 'code' => '1', 'phoneLength' => 11),
            array('name' => 'Mexico', 'countryCode' => '+52', 'code' => '52', 'phoneLength' => 13),

            // Europe (Major)
            array('name' => 'United Kingdom', 'countryCode' => '+44', 'code' => '44', 'phoneLength' => 13),
            array('name' => 'Germany', 'countryCode' => '+49', 'code' => '49', 'phoneLength' => 13),
            array('name' => 'France', 'countryCode' => '+33', 'code' => '33', 'phoneLength' => 12),
            array('name' => 'Italy', 'countryCode' => '+39', 'code' => '39', 'phoneLength' => 12),
            array('name' => 'Spain', 'countryCode' => '+34', 'code' => '34', 'phoneLength' => 12),
            array('name' => 'Netherlands', 'countryCode' => '+31', 'code' => '31', 'phoneLength' => 12),

            // Asia Pacific
            array('name' => 'India', 'countryCode' => '+91', 'code' => '91', 'phoneLength' => 12),
            array('name' => 'China', 'countryCode' => '+86', 'code' => '86', 'phoneLength' => 13),
            array('name' => 'Japan', 'countryCode' => '+81', 'code' => '81', 'phoneLength' => 12),
            array('name' => 'South Korea', 'countryCode' => '+82', 'code' => '82', 'phoneLength' => 12),
            array('name' => 'Australia', 'countryCode' => '+61', 'code' => '61', 'phoneLength' => 12),
            array('name' => 'Singapore', 'countryCode' => '+65', 'code' => '65', 'phoneLength' => 11),
            array('name' => 'Malaysia', 'countryCode' => '+60', 'code' => '60', 'phoneLength' => 12),
            array('name' => 'Indonesia', 'countryCode' => '+62', 'code' => '62', 'phoneLength' => 13),
            array('name' => 'Philippines', 'countryCode' => '+63', 'code' => '63', 'phoneLength' => 12),
            array('name' => 'Thailand', 'countryCode' => '+66', 'code' => '66', 'phoneLength' => 12),
            array('name' => 'Vietnam', 'countryCode' => '+84', 'code' => '84', 'phoneLength' => 12),
            array('name' => 'Pakistan', 'countryCode' => '+92', 'code' => '92', 'phoneLength' => 12),
            array('name' => 'Bangladesh', 'countryCode' => '+880', 'code' => '880', 'phoneLength' => 13),

            // Middle East
            array('name' => 'UAE', 'countryCode' => '+971', 'code' => '971', 'phoneLength' => 12),
            array('name' => 'Saudi Arabia', 'countryCode' => '+966', 'code' => '966', 'phoneLength' => 12),
            array('name' => 'Turkey', 'countryCode' => '+90', 'code' => '90', 'phoneLength' => 12),

            // Africa
            array('name' => 'South Africa', 'countryCode' => '+27', 'code' => '27', 'phoneLength' => 12),
            array('name' => 'Nigeria', 'countryCode' => '+234', 'code' => '234', 'phoneLength' => 13),
            array('name' => 'Egypt', 'countryCode' => '+20', 'code' => '20', 'phoneLength' => 13),

            // South America
            array('name' => 'Brazil', 'countryCode' => '+55', 'code' => '55', 'phoneLength' => 13),
            array('name' => 'Argentina', 'countryCode' => '+54', 'code' => '54', 'phoneLength' => 13),
            array('name' => 'Colombia', 'countryCode' => '+57', 'code' => '57', 'phoneLength' => 12),
            array('name' => 'Chile', 'countryCode' => '+56', 'code' => '56', 'phoneLength' => 12),
        );
    }

    /**
     * Enqueue checkout scripts and styles
     */
    public function enqueue_checkout_scripts()
    {
        // Only load on checkout page
        if (!is_checkout()) {
            return;
        }

        // Only load if checkout OTP is enabled
        if (!$this->is_checkout_otp_enabled()) {
            return;
        }

        // Enqueue styles
        wp_enqueue_style(
            'authyo-otp-checkout',
            AUTHYO_OTP_AUTH_PLUGIN_URL . 'assets/css/checkout.css',
            array(),
            $this->version
        );

        // Enqueue scripts - make wc-checkout dependency optional for compatibility
        $dependencies = array('jquery');
        if (wp_script_is('wc-checkout', 'registered')) {
            $dependencies[] = 'wc-checkout';
        }

        wp_enqueue_script(
            'authyo-otp-checkout',
            AUTHYO_OTP_AUTH_PLUGIN_URL . 'assets/js/checkout.js',
            $dependencies,
            $this->version,
            true
        );

        // Get OTP configuration from settings
        $otp_length = get_option('authyo_otp_auth_length', 6);
        $otp_expiry = get_option('authyo_otp_auth_expiry', 300);

        // Get method enablement status
        $email_enabled = get_option('authyo_otp_auth_checkout_email_enabled', 'yes') === 'yes';
        $whatsapp_enabled = get_option('authyo_otp_auth_checkout_whatsapp_enabled', 'no') === 'yes';
        $sms_enabled = get_option('authyo_otp_auth_checkout_sms_enabled', 'no') === 'yes';
        $voice_enabled = get_option('authyo_otp_auth_checkout_voice_enabled', 'no') === 'yes';
        $fallback_enabled = get_option('authyo_otp_auth_fallback_enabled', 'yes') === 'yes';

        // Ensure AJAX URL is properly formatted (handle both HTTP and HTTPS)
        $ajax_url = admin_url('admin-ajax.php');
        // Force HTTPS if site is using HTTPS
        if (is_ssl()) {
            $ajax_url = str_replace('http://', 'https://', $ajax_url);
        }

        // Localize script
        wp_localize_script('authyo-otp-checkout', 'authyoOtpCheckout', array(
            'ajaxurl' => esc_url($ajax_url),
            'nonce' => wp_create_nonce('authyo_otp_auth_nonce'),
            'otpLength' => absint($otp_length),
            'otpExpiry' => absint($otp_expiry),
            'emailEnabled' => $email_enabled,
            'whatsappEnabled' => $whatsapp_enabled,
            'smsEnabled' => $sms_enabled,
            'voiceEnabled' => $voice_enabled,
            'fallbackEnabled' => $fallback_enabled,
            'messages' => array(
                'sending' => esc_html__('Sending OTP...', 'authyo-otp-authentication-for-woocommerce'),
                'sent' => esc_html__('OTP sent successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'verifying' => esc_html__('Verifying OTP...', 'authyo-otp-authentication-for-woocommerce'),
                'verified' => esc_html__('OTP verified successfully!', 'authyo-otp-authentication-for-woocommerce'),
                'email_verified' => esc_html__('This email is verified', 'authyo-otp-authentication-for-woocommerce'),
                'phone_verified' => esc_html__('This phone number is verified', 'authyo-otp-authentication-for-woocommerce'),
                'error' => esc_html__('An error occurred. Please try again.', 'authyo-otp-authentication-for-woocommerce'),
                'required' => esc_html__('OTP verification is required to complete your order.', 'authyo-otp-authentication-for-woocommerce'),
                'enter_email' => esc_html__('Please enter a valid email address.', 'authyo-otp-authentication-for-woocommerce'),
                'enter_phone' => esc_html__('Please enter a valid phone number.', 'authyo-otp-authentication-for-woocommerce'),
                'resend' => esc_html__('Resend OTP', 'authyo-otp-authentication-for-woocommerce'),
                'mobile_preview' => esc_html__('Mobile number will be sent as: ', 'authyo-otp-authentication-for-woocommerce'),
                'all_fields_required' => esc_html__('All fields are required.', 'authyo-otp-authentication-for-woocommerce')
            )
        ));
    }
}