<?php
/**
 * The admin-specific functionality of the plugin.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Authyo_OTP_Auth_Admin {
    /**
     * The single instance of the class.
     */
    private static $instance = null;
    
    private $version;
    private $activation_ping_sent = false;
    private $webhook_url = 'https://app.authyo.io/api/v1/user/WordpressWebhook';
    
    /**
     * Main Instance.
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
            self::$instance->version = defined('AUTHYO_OTP_AUTH_VERSION') ? AUTHYO_OTP_AUTH_VERSION : '1.0.3';
        }
        return self::$instance;
    }
    
    /**
     * Constructor.
     */
    private function __construct() {
        add_action('admin_menu', array($this, 'add_plugin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_action('admin_notices', array($this, 'check_credentials_notice'));
        add_action('updated_option', array($this, 'maybe_send_activation_ping'), 10, 3);
    }
    
    /**
     * Display admin notice if API credentials are not configured.
     *
     * @since 1.0.5
     */
    public function check_credentials_notice() {
        // Only show on admin pages
        if (!current_user_can('manage_options')) {
            return;
        }

        $client_id = get_option('authyo_otp_auth_client_id', '');
        $client_secret = get_option('authyo_otp_auth_client_secret', '');
        
        // Don't show on the settings page itself
        $screen = get_current_screen();
        if ($screen && strpos($screen->id, 'authyo-otp-authentication-for-woocommerce') !== false) {
            return;
        }
        
        if (empty($client_id) || empty($client_secret)) {
            ?>
            <div class="notice notice-warning is-dismissible">
                <p>
                    <strong><?php esc_html_e('Authyo OTP Authentication:', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                    <?php
                    printf(
                        /* translators: %s: Link to plugin settings page */
                        esc_html__('Please configure your API credentials in %s to enable OTP authentication.', 'authyo-otp-authentication-for-woocommerce'),
                        '<a href="' . esc_url(admin_url('admin.php?page=authyo-otp-authentication-for-woocommerce')) . '">' . esc_html__('plugin settings', 'authyo-otp-authentication-for-woocommerce') . '</a>'
                    );
                    ?>
                </p>
            </div>
            <?php
        }
        
        // Check if fallback is disabled (important for user experience)
        $fallback_enabled = get_option('authyo_otp_auth_fallback_enabled', 'yes');
        if ($fallback_enabled !== 'yes') {
            ?>
            <div class="notice notice-info is-dismissible">
                <p>
                    <strong><?php esc_html_e('Authyo OTP Authentication:', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                    <?php
                    printf(
                        /* translators: %s: Link to plugin settings page */
                        esc_html__('Fallback methods (SMS, WhatsApp, Voice) are currently disabled. Enable them in %s for better user experience.', 'authyo-otp-authentication-for-woocommerce'),
                        '<a href="' . esc_url(admin_url('admin.php?page=authyo-otp-authentication-for-woocommerce')) . '">' . esc_html__('plugin settings', 'authyo-otp-authentication-for-woocommerce') . '</a>'
                    );
                    ?>
                </p>
            </div>
            <?php
        }
    }
    
    /**
     * Add menu items.
     */
    public function add_plugin_menu() {
        add_menu_page(
            __('WC OTP Auth', 'authyo-otp-authentication-for-woocommerce'),
            __('WC OTP Auth', 'authyo-otp-authentication-for-woocommerce'),
            'manage_options',
            'authyo-otp-authentication-for-woocommerce',
            array($this, 'render_settings_page'),
            'dashicons-lock',
            56
        );
        
        add_submenu_page(
            'authyo-otp-authentication-for-woocommerce',
            __('Settings', 'authyo-otp-authentication-for-woocommerce'),
            __('Settings', 'authyo-otp-authentication-for-woocommerce'),
            'manage_options',
            'authyo-otp-authentication-for-woocommerce',
            array($this, 'render_settings_page')
        );
        
        add_submenu_page(
            'authyo-otp-authentication-for-woocommerce',
            __('OTP Logs', 'authyo-otp-authentication-for-woocommerce'),
            __('OTP Logs', 'authyo-otp-authentication-for-woocommerce'),
            'manage_options',
            'authyo-otp-authentication-for-woocommerce-logs',
            array($this, 'render_logs_page')
        );

        add_submenu_page(
            'authyo-otp-authentication-for-woocommerce',
            __('Shortcodes', 'authyo-otp-authentication-for-woocommerce'),
            __('Shortcodes', 'authyo-otp-authentication-for-woocommerce'),
            'manage_options',
            'authyo-otp-authentication-for-woocommerce-shortcodes',
            array($this, 'render_shortcodes_page')
        );
    }

    /**
     * Register plugin settings.
     */
    public function register_settings() {
        // Login settings
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_enable_login', array(
            'type' => 'string',
            'default' => 'yes',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Registration settings
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_enable_registration', array(
            'type' => 'string',
            'default' => 'yes',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // App ID
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_app_id', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Client ID
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_client_id', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Client Secret
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_client_secret', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Email enabled
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_email_enabled', array(
            'type' => 'string',
            'default' => 'yes',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // WhatsApp enabled
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_whatsapp_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // SMS enabled
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_sms_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Voice enabled
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_voice_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Fallback method enabled
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_fallback_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Priority method for mobile OTP
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_priority_method', array(
            'type' => 'string',
            'default' => 'whatsapp',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // OTP Length
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_length', array(
            'type' => 'integer',
            'default' => 6,
            'sanitize_callback' => 'absint'
        ));

        // OTP Expiry
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_expiry', array(
            'type' => 'integer',
            'default' => 300,
            'sanitize_callback' => array($this, 'sanitize_otp_expiry')
        ));

        // Maximum OTP Attempts
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_max_attempts', array(
            'type' => 'integer',
            'default' => 5,
            'sanitize_callback' => 'absint'
        ));

        // OTP Attempt Lockout Period (in minutes)
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_lockout_period', array(
            'type' => 'integer',
            'default' => 15,
            'sanitize_callback' => 'absint'
        ));

        // OTP Bypass for Testing
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_bypass_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Test Phone Numbers for Bypass
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_test_phones', array(
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'sanitize_textarea_field'
        ));

        // Test Emails for Bypass
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_test_emails', array(
            'type' => 'string',
            'default' => '',
            'sanitize_callback' => 'sanitize_textarea_field'
        ));

        // Default Country Code
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_default_country', array(
            'type' => 'string',
            'default' => '91',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Checkout Security Settings
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_checkout_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_checkout_email_enabled', array(
            'type' => 'string',
            'default' => 'yes',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_checkout_whatsapp_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_checkout_sms_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_checkout_voice_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // WooCommerce Order Notifications
        register_setting('authyo_otp_auth_settings', 'authyo_otp_auth_order_notifications_enabled', array(
            'type' => 'string',
            'default' => 'no',
            'sanitize_callback' => 'sanitize_text_field'
        ));

        // Set default values if they don't exist
        if (get_option('authyo_otp_auth_email_enabled') === false) {
            update_option('authyo_otp_auth_email_enabled', 'yes'); // Enable email by default
        }
        if (get_option('authyo_otp_auth_whatsapp_enabled') === false) {
            update_option('authyo_otp_auth_whatsapp_enabled', 'no');
        }
        if (get_option('authyo_otp_auth_sms_enabled') === false) {
            update_option('authyo_otp_auth_sms_enabled', 'no');
        }
        if (get_option('authyo_otp_auth_voice_enabled') === false) {
            update_option('authyo_otp_auth_voice_enabled', 'no');
        }
        if (get_option('authyo_otp_auth_fallback_enabled') === false) {
            update_option('authyo_otp_auth_fallback_enabled', 'yes');
        }

        // Set default OTP settings if they don't exist
        if (get_option('authyo_otp_auth_length') === false) {
            update_option('authyo_otp_auth_length', 6);
        }
        if (get_option('authyo_otp_auth_expiry') === false) {
            update_option('authyo_otp_auth_expiry', 300);
        }
        if (get_option('authyo_otp_auth_bypass_enabled') === false) {
            update_option('authyo_otp_auth_bypass_enabled', 'no');
        }
        if (get_option('authyo_otp_auth_test_phones') === false) {
            update_option('authyo_otp_auth_test_phones', '');
        }
        if (get_option('authyo_otp_auth_test_emails') === false) {
            update_option('authyo_otp_auth_test_emails', '');
        }
        if (get_option('authyo_otp_auth_default_country') === false) {
            update_option('authyo_otp_auth_default_country', '91');
        }
    }

    /**
     * Render checkbox field.
     */
    private function render_checkbox_field($args) {
        $option = get_option($args['name']);
        $checked = $option === 'yes' ? 'checked' : '';
        ?>
        <label>
            <input type="checkbox" 
                   name="<?php echo esc_attr($args['name']); ?>" 
                   value="yes" 
                   <?php echo esc_attr($checked); ?> />
            <?php echo esc_html($args['label']); ?>
        </label>
        <?php
    }
    
    /**
     * Render text field.
     */
    private function render_text_field($args) {
        $option = get_option($args['name']);
        $name = isset($option) ? $option : '';
        ?>
        <input type="text" 
               name="<?php echo esc_attr($args['name']); ?>" 
               value="<?php echo esc_attr($name); ?>" 
               class="regular-text" />
        <?php
    }

    /**
     * Render settings page.
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'authyo-otp-authentication-for-woocommerce'));
        }
        
        ?>
        <div class="wrap authyo-modern-wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <form action="options.php" method="post" class="authyo-modern-form">
                <?php
                settings_fields('authyo_otp_auth_settings');
                do_settings_sections('authyo_otp_auth_settings');
                ?>
                
                <div class="authyo-settings-grid">
                    <!-- API Credentials Section (3-Column Grid) -->
                    <div class="authyo-settings-card authyo-grid-item-full">
                        <h3 style="margin-bottom: 20px;"><?php esc_html_e('API Credentials', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
                        <div class="authyo-api-credentials-grid">
                            <div class="authyo-api-credential-item">
                                <label for="authyo_otp_auth_app_id"><?php esc_html_e('Authyo App ID', 'authyo-otp-authentication-for-woocommerce'); ?></label>
                                <?php $this->render_text_field(array(
                                    'name' => 'authyo_otp_auth_app_id'
                                )); ?>
                            </div>
                            <div class="authyo-api-credential-item">
                                <label for="authyo_otp_auth_client_id"><?php esc_html_e('Authyo Client ID', 'authyo-otp-authentication-for-woocommerce'); ?></label>
                                <?php
                                $client_id = get_option('authyo_otp_auth_client_id', '');
                                $client_id_display = !empty($client_id) ? str_repeat('*', min(strlen($client_id), 32)) : '';
                                ?>
                                <input type="password" 
                                       id="authyo_otp_auth_client_id" 
                                       name="authyo_otp_auth_client_id" 
                                       value="<?php echo esc_attr($client_id); ?>" 
                                       class="regular-text authyo-masked-field"
                                       autocomplete="off"
                                       placeholder="<?php esc_attr_e('Enter Client ID', 'authyo-otp-authentication-for-woocommerce'); ?>" />
                                <?php if (!empty($client_id)): ?>
                                    <p class="description" style="margin-top: 5px; color: #646970;">
                                        <?php echo esc_html($client_id_display); ?> (<?php esc_html_e('Masked', 'authyo-otp-authentication-for-woocommerce'); ?>)
                                    </p>
                                <?php endif; ?>
                            </div>
                            <div class="authyo-api-credential-item">
                                <label for="authyo_otp_auth_client_secret"><?php esc_html_e('Authyo Client Secret', 'authyo-otp-authentication-for-woocommerce'); ?></label>
                                <?php
                                $client_secret = get_option('authyo_otp_auth_client_secret', '');
                                $client_secret_display = !empty($client_secret) ? str_repeat('*', min(strlen($client_secret), 32)) : '';
                                ?>
                                <input type="password" 
                                       id="authyo_otp_auth_client_secret" 
                                       name="authyo_otp_auth_client_secret" 
                                       value="<?php echo esc_attr($client_secret); ?>" 
                                       class="regular-text authyo-masked-field"
                                       autocomplete="off"
                                       placeholder="<?php esc_attr_e('Enter Client Secret', 'authyo-otp-authentication-for-woocommerce'); ?>" />
                                <?php if (!empty($client_secret)): ?>
                                    <p class="description" style="margin-top: 5px; color: #646970;">
                                        <?php echo esc_html($client_secret_display); ?> (<?php esc_html_e('Masked', 'authyo-otp-authentication-for-woocommerce'); ?>)
                                    </p>
                                <?php endif; ?>
                            </div>
                        </div>
                        <div class="authyo-help-text">
                            <span class="dashicons dashicons-info"></span>
                            <?php esc_html_e('Get these credentials from your Authyo dashboard at', 'authyo-otp-authentication-for-woocommerce'); ?>
                            <a href="https://app.authyo.io/account/welcome?utm_source=plugin-authyo-otp-authentication-for-woocommerce" target="_blank" rel="noopener">authyo.io</a>
                        </div>
                    </div>

                    <!-- Basic Settings Card -->
                    <div class="authyo-settings-card authyo-grid-item-full">
                        <h3><?php esc_html_e('Basic Settings', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
                        <div class="authyo-basic-settings-grid" style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px;">
                            <div class="authyo-setting-option">
                                <label class="authyo-checkbox-label">
                                    <?php
                                    $enable_login = get_option('authyo_otp_auth_enable_login', 'yes');
                                    ?>
                                    <input type="checkbox" 
                                           name="authyo_otp_auth_enable_login" 
                                           value="yes" 
                                           <?php checked($enable_login, 'yes'); ?> />
                                    <span class="authyo-checkbox-text">
                                        <strong><?php esc_html_e('Enable OTP on Login', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        <small><?php esc_html_e('Enable OTP verification during login', 'authyo-otp-authentication-for-woocommerce'); ?></small>
                                    </span>
                                </label>
                            </div>
                            <div class="authyo-setting-option">
                                <label class="authyo-checkbox-label">
                                    <?php
                                    $enable_registration = get_option('authyo_otp_auth_enable_registration', 'yes');
                                    ?>
                                    <input type="checkbox" 
                                           name="authyo_otp_auth_enable_registration" 
                                           value="yes" 
                                           <?php checked($enable_registration, 'yes'); ?> />
                                    <span class="authyo-checkbox-text">
                                        <strong><?php esc_html_e('Enable OTP on Registration', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        <small><?php esc_html_e('Enable OTP verification during registration', 'authyo-otp-authentication-for-woocommerce'); ?></small>
                                    </span>
                                </label>
                            </div>
                            
                            <?php if (class_exists('WooCommerce')): ?>
                            <div class="authyo-setting-option">
                                <label class="authyo-checkbox-label">
                                    <?php
                                    $checkout_enabled = get_option('authyo_otp_auth_checkout_enabled', 'no');
                                    ?>
                                    <input type="checkbox" 
                                           name="authyo_otp_auth_checkout_enabled" 
                                           value="yes" 
                                           id="authyo_checkout_enabled_toggle"
                                           <?php checked($checkout_enabled, 'yes'); ?> />
                                    <span class="authyo-checkbox-text">
                                        <strong><?php esc_html_e('Enable OTP Verification at Checkout', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        <small><?php esc_html_e('Require customers to verify their email and phone number before placing an order.', 'authyo-otp-authentication-for-woocommerce'); ?></small>
                                    </span>
                                </label>
                            </div>
                            <div class="authyo-setting-option">
                                <label class="authyo-checkbox-label">
                                    <?php
                                    $order_notifications_enabled = get_option('authyo_otp_auth_order_notifications_enabled', 'no');
                                    ?>
                                    <input type="checkbox" 
                                           name="authyo_otp_auth_order_notifications_enabled" 
                                           value="yes" 
                                           id="authyo_order_notifications_enabled_toggle"
                                           <?php checked($order_notifications_enabled, 'yes'); ?> />
                                    <span class="authyo-checkbox-text">
                                        <strong><?php esc_html_e('Enable WooCommerce Order Notifications', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        <small><?php esc_html_e('Send order status notifications to customers via Email, SMS, and WhatsApp.', 'authyo-otp-authentication-for-woocommerce'); ?></small>
                                    </span>
                                </label>
                            </div>
                            <?php endif; ?>
                        </div>
                        
                        <!-- OTP Delivery Methods Section -->
                        <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd;">
                            <h4 style="margin-top: 0; margin-bottom: 10px; font-size: 14px; font-weight: 600;"><?php esc_html_e('OTP Delivery Methods', 'authyo-otp-authentication-for-woocommerce'); ?></h4>
                            <p class="description" style="margin-bottom: 20px;">
                                <?php esc_html_e('These OTP delivery methods are used for login and registration.', 'authyo-otp-authentication-for-woocommerce'); ?>
                            </p>
                            <div class="authyo-delivery-methods-grid" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px;">
                                <div class="authyo-setting-option">
                                    <label class="authyo-checkbox-label">
                                        <?php
                                        $email_enabled = get_option('authyo_otp_auth_email_enabled', 'yes');
                                        ?>
                                        <input type="checkbox" 
                                               name="authyo_otp_auth_email_enabled" 
                                               value="yes" 
                                               <?php checked($email_enabled, 'yes'); ?> />
                                        <span class="authyo-checkbox-text">
                                            <strong><?php esc_html_e('Send OTP via Email', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        </span>
                                    </label>
                                </div>
                                <div class="authyo-setting-option">
                                    <label class="authyo-checkbox-label">
                                        <?php
                                        $whatsapp_enabled = get_option('authyo_otp_auth_whatsapp_enabled', 'no');
                                        ?>
                                        <input type="checkbox" 
                                               name="authyo_otp_auth_whatsapp_enabled" 
                                               value="yes" 
                                               <?php checked($whatsapp_enabled, 'yes'); ?> />
                                        <span class="authyo-checkbox-text">
                                            <strong><?php esc_html_e('Send OTP via WhatsApp', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        </span>
                                    </label>
                                </div>
                                <div class="authyo-setting-option">
                                    <label class="authyo-checkbox-label">
                                        <?php
                                        $sms_enabled = get_option('authyo_otp_auth_sms_enabled', 'no');
                                        ?>
                                        <input type="checkbox" 
                                               name="authyo_otp_auth_sms_enabled" 
                                               value="yes" 
                                               <?php checked($sms_enabled, 'yes'); ?> />
                                        <span class="authyo-checkbox-text">
                                            <strong><?php esc_html_e('Send OTP via SMS', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        </span>
                                    </label>
                                </div>
                                <div class="authyo-setting-option">
                                    <label class="authyo-checkbox-label">
                                        <?php
                                        $voice_enabled = get_option('authyo_otp_auth_voice_enabled', 'no');
                                        ?>
                                        <input type="checkbox" 
                                               name="authyo_otp_auth_voice_enabled" 
                                               value="yes" 
                                               <?php checked($voice_enabled, 'yes'); ?> />
                                        <span class="authyo-checkbox-text">
                                            <strong><?php esc_html_e('Send OTP via Voice Call', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        </span>
                                    </label>
                                    <p class="description" style="margin-top: 5px; margin-bottom: 0; font-size: 12px; color: red; font-style: italic;">
                                        <?php esc_html_e('Note: Voice Call is currently available only for India.', 'authyo-otp-authentication-for-woocommerce'); ?>
                                    </p>
                                </div>
                            </div>
                        </div>
                        
                        <?php if (class_exists('WooCommerce')): ?>
                        <!-- Checkout OTP Methods (shown when checkout is enabled) -->
                        <div class="authyo-checkout-methods-grid" style="display: <?php echo $checkout_enabled === 'yes' ? 'block' : 'none'; ?>; margin-top: 20px; padding: 15px; background: #f9f9f9; border-radius: 4px;">
                            <h4 style="margin-top: 0; margin-bottom: 15px; font-size: 14px; font-weight: 600;"><?php esc_html_e('OTP Methods for Checkout', 'authyo-otp-authentication-for-woocommerce'); ?></h4>
                            <p class="description" style="margin-bottom: 15px; font-size: 13px;">
                                <?php esc_html_e('Select which OTP delivery methods should be available during checkout.', 'authyo-otp-authentication-for-woocommerce'); ?>
                            </p>
                            
                            <div class="authyo-delivery-methods-grid" style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px;">
                                <div class="authyo-setting-option">
                                    <label class="authyo-checkbox-label">
                                        <?php
                                        $checkout_email_enabled = get_option('authyo_otp_auth_checkout_email_enabled', 'yes');
                                        ?>
                                        <input type="checkbox" 
                                               name="authyo_otp_auth_checkout_email_enabled" 
                                               value="yes" 
                                               <?php checked($checkout_email_enabled, 'yes'); ?> />
                                        <span class="authyo-checkbox-text">
                                            <strong><?php esc_html_e('Send OTP via Email', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        </span>
                                    </label>
                                </div>
                                <div class="authyo-setting-option">
                                    <label class="authyo-checkbox-label">
                                        <?php
                                        $checkout_whatsapp_enabled = get_option('authyo_otp_auth_checkout_whatsapp_enabled', 'no');
                                        ?>
                                        <input type="checkbox" 
                                               name="authyo_otp_auth_checkout_whatsapp_enabled" 
                                               value="yes" 
                                               <?php checked($checkout_whatsapp_enabled, 'yes'); ?> />
                                        <span class="authyo-checkbox-text">
                                            <strong><?php esc_html_e('Send OTP via WhatsApp', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        </span>
                                    </label>
                                </div>
                                <div class="authyo-setting-option">
                                    <label class="authyo-checkbox-label">
                                        <?php
                                        $checkout_sms_enabled = get_option('authyo_otp_auth_checkout_sms_enabled', 'no');
                                        ?>
                                        <input type="checkbox" 
                                               name="authyo_otp_auth_checkout_sms_enabled" 
                                               value="yes" 
                                               <?php checked($checkout_sms_enabled, 'yes'); ?> />
                                        <span class="authyo-checkbox-text">
                                            <strong><?php esc_html_e('Send OTP via SMS', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        </span>
                                    </label>
                                </div>
                                <div class="authyo-setting-option">
                                    <label class="authyo-checkbox-label">
                                        <?php
                                        $checkout_voice_enabled = get_option('authyo_otp_auth_checkout_voice_enabled', 'no');
                                        ?>
                                        <input type="checkbox" 
                                               name="authyo_otp_auth_checkout_voice_enabled" 
                                               value="yes" 
                                               <?php checked($checkout_voice_enabled, 'yes'); ?> />
                                        <span class="authyo-checkbox-text">
                                            <strong><?php esc_html_e('Send OTP via Voice Call', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        </span>
                                    </label>
                                    <p class="description" style="margin-top: 5px; margin-bottom: 0; font-size: 12px; color: red; font-style: italic;">
                                        <?php esc_html_e('Note: Voice Call is currently available only for India.', 'authyo-otp-authentication-for-woocommerce'); ?>
                                    </p>
                                </div>
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>

                    <!-- OTP Configuration Card -->
                    <div class="authyo-settings-card">
                        <h3><?php esc_html_e('OTP Configuration', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
                        <div class="authyo-otp-config-grid">
                            <div class="authyo-config-option">
                                <label for="authyo_otp_auth_length"><?php esc_html_e('OTP Length', 'authyo-otp-authentication-for-woocommerce'); ?></label>
                                <input type="number" 
                                       id="authyo_otp_auth_length" 
                                       name="authyo_otp_auth_length" 
                                       value="<?php echo esc_attr(get_option('authyo_otp_auth_length', 6)); ?>" 
                                       min="4" 
                                       max="8" 
                                       class="regular-text" />
                                <p class="description"><?php esc_html_e('Number of digits (4-8)', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                            </div>
                            <div class="authyo-config-option">
                                <label for="authyo_otp_auth_expiry"><?php esc_html_e('OTP Expiry Time', 'authyo-otp-authentication-for-woocommerce'); ?></label>
                                <input type="number" 
                                       id="authyo_otp_auth_expiry" 
                                       name="authyo_otp_auth_expiry" 
                                       value="<?php echo esc_attr(get_option('authyo_otp_auth_expiry', 300)); ?>" 
                                       min="10" 
                                       max="600" 
                                       class="regular-text" />
                                <p class="description"><?php esc_html_e('Validity in seconds (10-600)', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                            </div>
                            <div class="authyo-config-option">
                                <label for="authyo_otp_auth_max_attempts"><?php esc_html_e('Maximum OTP Attempts', 'authyo-otp-authentication-for-woocommerce'); ?></label>
                                <input type="number" 
                                       id="authyo_otp_auth_max_attempts" 
                                       name="authyo_otp_auth_max_attempts" 
                                       value="<?php echo esc_attr(get_option('authyo_otp_auth_max_attempts', 5)); ?>" 
                                       min="1" 
                                       max="20" 
                                       class="regular-text" />
                                <p class="description"><?php esc_html_e('Maximum number of failed OTP verification attempts allowed (1-20)', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                            </div>
                            <div class="authyo-config-option">
                                <label for="authyo_otp_auth_lockout_period"><?php esc_html_e('Lockout Period', 'authyo-otp-authentication-for-woocommerce'); ?></label>
                                <input type="number" 
                                       id="authyo_otp_auth_lockout_period" 
                                       name="authyo_otp_auth_lockout_period" 
                                       value="<?php echo esc_attr(get_option('authyo_otp_auth_lockout_period', 15)); ?>" 
                                       min="1" 
                                       max="1440" 
                                       class="regular-text" />
                                <p class="description"><?php esc_html_e('Lockout duration in minutes after max attempts reached (1-1440)', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                            </div>
                        </div>
                    </div>

                    <!-- Priority & Fallback Card -->
                    <div class="authyo-settings-card">
                        <h3><?php esc_html_e('Priority & Fallback', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
                        <div class="authyo-priority-fallback-grid">
                            <div class="authyo-config-option">
                                <label for="authyo_otp_auth_priority_method"><?php esc_html_e('Priority Method', 'authyo-otp-authentication-for-woocommerce'); ?></label>
                                <select id="authyo_otp_auth_priority_method" 
                                        name="authyo_otp_auth_priority_method"
                                        class="regular-text">
                                    <option value="whatsapp" <?php selected(get_option('authyo_otp_auth_priority_method', 'whatsapp'), 'whatsapp'); ?>><?php esc_html_e('WhatsApp', 'authyo-otp-authentication-for-woocommerce'); ?></option>
                                    <option value="sms" <?php selected(get_option('authyo_otp_auth_priority_method', 'whatsapp'), 'sms'); ?>><?php esc_html_e('SMS', 'authyo-otp-authentication-for-woocommerce'); ?></option>
                                    <option value="voice" <?php selected(get_option('authyo_otp_auth_priority_method', 'whatsapp'), 'voice'); ?>><?php esc_html_e('Voice Call', 'authyo-otp-authentication-for-woocommerce'); ?></option>
                                </select>
                                <p class="description"><?php esc_html_e('Primary method for mobile OTP delivery.', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                            </div>
                            <div class="authyo-setting-option">
                                <label class="authyo-checkbox-label">
                                    <?php
                                    $fallback_enabled = get_option('authyo_otp_auth_fallback_enabled', 'yes');
                                    ?>
                                    <input type="checkbox" 
                                           name="authyo_otp_auth_fallback_enabled" 
                                           value="yes" 
                                           <?php checked($fallback_enabled, 'yes'); ?> />
                                    <span class="authyo-checkbox-text">
                                        <strong><?php esc_html_e('Enable Fallback Method', 'authyo-otp-authentication-for-woocommerce'); ?></strong>
                                        <small><?php esc_html_e('Allow alternative delivery methods if primary fails.', 'authyo-otp-authentication-for-woocommerce'); ?></small>
                                    </span>
                                </label>
                            </div>
                        </div>
                    </div>

                    <!-- Testing Mode Card (2-Column Grid) -->
                    <div class="authyo-settings-card">
                        <h3><?php esc_html_e('Testing Mode', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Enable Bypass', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                                <td>
                                    <?php $this->render_checkbox_field(array(
                                        'name' => 'authyo_otp_auth_bypass_enabled',
                                        'label' => esc_html__('Enable OTP bypass for testing', 'authyo-otp-authentication-for-woocommerce')
                                    )); ?>
                                    <p class="description"><?php esc_html_e('Skip OTP verification for test numbers and emails.', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php esc_html_e('Test Phone Numbers', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                                <td>
                                    <textarea id="authyo_otp_auth_test_phones" 
                                              name="authyo_otp_auth_test_phones" 
                                              rows="3" 
                                              class="large-text"><?php echo esc_textarea(get_option('authyo_otp_auth_test_phones', '')); ?></textarea>
                                    <p class="description"><?php esc_html_e('Enter phone numbers (one per line) with country code (e.g., +1234567890).', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php esc_html_e('Test Email Addresses', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                                <td>
                                    <textarea id="authyo_otp_auth_test_emails" 
                                              name="authyo_otp_auth_test_emails" 
                                              rows="3" 
                                              class="large-text"><?php echo esc_textarea(get_option('authyo_otp_auth_test_emails', '')); ?></textarea>
                                    <p class="description"><?php esc_html_e('Enter email addresses (one per line) e.g., test@example.com.', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row"><?php esc_html_e('Default Country Code', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                                <td>
                                    <input type="text" 
                                           id="authyo_otp_auth_default_country" 
                                           name="authyo_otp_auth_default_country" 
                                           value="<?php echo esc_attr(get_option('authyo_otp_auth_default_country', '91')); ?>" 
                                           class="regular-text" 
                                           placeholder="91" />
                                    <p class="description"><?php esc_html_e('Default country code (e.g., 91 for India, 1 for US/Canada).', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                                </td>
                            </tr>
                        </table>
                    </div>

                    <!-- Country Codes Management Card (2-Column Grid) -->
                    <div class="authyo-settings-card">
                        <h3><?php esc_html_e('Country Codes Management', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
                        <table class="form-table">
                            <tr>
                                <th scope="row"><?php esc_html_e('Country Codes Cache', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                                <td>
                                    <p><?php esc_html_e('Country codes are automatically fetched from Authyo API and cached for 24 hours. Clear cache to force refresh.', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                                    <p>
                                        <button type="button" 
                                                id="clear-country-codes-cache" 
                                                class="button button-secondary">
                                            <?php esc_html_e('Clear Country Codes Cache', 'authyo-otp-authentication-for-woocommerce'); ?>
                                        </button>
                                        <span id="cache-clear-message" style="margin-left: 10px;"></span>
                                    </p>
                                </td>
                            </tr>
                        </table>
                    </div>

                </div>
                
                <div style="margin: 0 24px 24px 24px; border-top: 2px solid #f6f7f7;">
                    <?php submit_button(); ?>
                </div>
            </form>
        </div>
        <?php
    }
    
    /**
     * AJAX handler to clear country codes cache
     */
    public function clear_country_codes_cache() {
        // Verify nonce
        if (!check_ajax_referer('authyo_otp_auth_admin_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => esc_html__('Insufficient permissions.', 'authyo-otp-authentication-for-woocommerce')));
            return;
        }

        // Clear both country caches
        // 1. SDK country codes cache (for validation)
        $sdk = new Authyo_OTP_Auth_SDK();
        $sdk->clear_country_codes_cache();
        
        // 2. Public country list cache (for frontend dropdown)
        delete_transient('authyo_country_list_cache');

        wp_send_json_success(array('message' => esc_html__('Country codes cache cleared successfully!', 'authyo-otp-authentication-for-woocommerce')));
    }

    /**
     * Render logs page.
     */
    public function render_logs_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'authyo-otp-authentication-for-woocommerce'));
        }

        global $wpdb;
        $table_name = $wpdb->prefix . 'authyo_otp_auth_logs';
        $escaped_table_name = esc_sql($table_name);

        // Check if table exists
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin needs to check custom table existence
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;
        
        if (!$table_exists) {
            ?>
            <div class="wrap">
                <h1><?php esc_html_e('OTP Logs', 'authyo-otp-authentication-for-woocommerce'); ?></h1>
                <div class="notice notice-error">
                    <p><?php esc_html_e('Error: Logs table does not exist.', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                </div>
            </div>
            <?php
            return;
        }
        
        // Get total count of logs
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is escaped with esc_sql(), plugin needs custom table access
        $total_logs = $wpdb->get_var("SELECT COUNT(*) FROM `{$escaped_table_name}`");
        
        // Get sample logs (for display/info, not critical path)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is escaped with esc_sql(), plugin needs custom table access
        $sample_logs = $wpdb->get_results("SELECT * FROM `{$escaped_table_name}` ORDER BY created_at DESC LIMIT 5");
        
        // Handle bulk actions
        if (isset($_POST['action']) && sanitize_text_field(wp_unslash($_POST['action'])) === 'delete') {
            // Nonce and capability check are already good
            if (check_admin_referer('authyo_otp_auth_logs') && current_user_can('manage_options')) {
                if (isset($_POST['log_ids']) && is_array($_POST['log_ids'])) {
                    // Sanitize log_ids before use in deletion
                    $log_ids = array_map('intval', wp_unslash($_POST['log_ids']));
                    
                    // Remove any non-positive integer IDs
                    $log_ids = array_filter($log_ids, function($id) { return $id > 0; });

                    if (!empty($log_ids)) {
                        // Use $wpdb->delete with IN clause and prepare for safety
                        $format = implode(', ', array_fill(0, count($log_ids), '%d'));
                        $query = "DELETE FROM `{$escaped_table_name}` WHERE id IN ({$format})";
                        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is escaped with esc_sql(), ids are prepared, plugin needs custom table access
                        $deleted_count = $wpdb->query($wpdb->prepare($query, $log_ids));
                        
                        if ($deleted_count !== false) {
                            echo '<div class="updated"><p>' . sprintf(
                                /* translators: %d: number of logs deleted */
                                esc_html__('Successfully deleted %d logs.', 'authyo-otp-authentication-for-woocommerce'),
                                absint($deleted_count)
                            ) . '</p></div>';
                        } else {
                             echo '<div class="error"><p>' . esc_html__('Error deleting logs. Please try again.', 'authyo-otp-authentication-for-woocommerce') . '</p></div>';
                        }
                    } else {
                         echo '<div class="notice notice-warning"><p>' . esc_html__('No valid logs selected for deletion.', 'authyo-otp-authentication-for-woocommerce') . '</p></div>';
                    }
                } else {
                     echo '<div class="error"><p>' . esc_html__('No logs selected for deletion.', 'authyo-otp-authentication-for-woocommerce') . '</p></div>';
                }
            } else {
                 echo '<div class="error"><p>' . esc_html__('Security check failed or insufficient permissions.', 'authyo-otp-authentication-for-woocommerce') . '</p></div>';
            }
        }

        // Get logs with pagination
        $per_page = 20;
        // Ensure page number is a positive integer
        $current_page = isset($_GET['paged']) ? max(1, absint(wp_unslash($_GET['paged']))) : 1;
        $offset = ($current_page - 1) * $per_page;

        // Use $wpdb->get_results with prepare
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin needs custom table access
        $logs = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM `{$escaped_table_name}` ORDER BY created_at DESC LIMIT %d OFFSET %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is escaped with esc_sql()
            $per_page,
            $offset
        ));

        // Use get_var for total logs count, no user input
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is escaped with esc_sql(), plugin needs custom table access
        $total_logs = (int) $wpdb->get_var("SELECT COUNT(*) FROM `{$escaped_table_name}`");
        $total_pages = $total_logs > 0 ? (int) ceil($total_logs / $per_page) : 1;

        ?>
        <div class="wrap authyo-modern-wrap">
            <h1><?php esc_html_e('OTP Logs', 'authyo-otp-authentication-for-woocommerce'); ?></h1>

            <?php // Removed the conditional check here as it's handled above ?>

            <div class="authyo-modern-card">
                <form method="post">
                <input type="hidden" name="action" value="delete" />
                <?php wp_nonce_field('authyo_otp_auth_logs'); ?>

                <div class="tablenav top">
                    <div class="alignleft actions">
                        <select name="action2">
                            <option value="-1"><?php esc_html_e('Bulk Actions', 'authyo-otp-authentication-for-woocommerce'); ?></option>
                            <option value="delete"><?php esc_html_e('Delete', 'authyo-otp-authentication-for-woocommerce'); ?></option>
                        </select>
                        <input type="submit" class="button action" value="<?php esc_html_e('Apply', 'authyo-otp-authentication-for-woocommerce'); ?>" />
                    </div>

                    <div class="tablenav-pages">
                        <?php
                        // translators: %s: number of items in the list
                        $item_text = _n(
                            '%s item',
                            '%s items',
                            absint($total_logs), // Ensure total_logs is treated as absolute integer
                            'authyo-otp-authentication-for-woocommerce'
                        );
                        /* translators: %s: number of items */
                        $item_text = sprintf($item_text, number_format_i18n($total_logs));
                        ?>
                        <span class="displaying-num"><?php echo esc_html($item_text); ?></span>

                        <span class="pagination-links">
                            <?php
                            $pagination = paginate_links(array(
                                'base' => add_query_arg('paged', '%#%'),
                                'format' => '',
                                'prev_text' => esc_html__('&laquo;', 'authyo-otp-authentication-for-woocommerce'),
                                'next_text' => esc_html__('&raquo;', 'authyo-otp-authentication-for-woocommerce'),
                                'total' => $total_pages,
                                'current' => $current_page
                            ));
                            
                            if (!empty($pagination)) {
                                echo wp_kses(
                                    $pagination,
                                    array(
                                        'a' => array(
                                            'class' => array(),
                                            'href' => array(),
                                            'aria-current' => array()
                                        ),
                                        'span' => array(
                                            'class' => array(),
                                            'aria-current' => array()
                                        )
                                    )
                                );
                            }
                            ?>
                        </span>
                    </div>
                </div>

                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <td class="manage-column column-cb check-column">
                                <input type="checkbox" />
                            </td>
                            <th><?php esc_html_e('Email', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                            <th><?php esc_html_e('OTP', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                            <th><?php esc_html_e('Status', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                            <th><?php esc_html_e('IP Address', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                            <th><?php esc_html_e('Created', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                            <th><?php esc_html_e('User ID', 'authyo-otp-authentication-for-woocommerce'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if ($logs): ?>
                        <?php foreach ($logs as $log): ?>
                            <tr>
                                <th scope="row" class="check-column">
                                    <input type="checkbox" name="log_ids[]" value="<?php echo esc_attr($log->id); ?>" />
                                </th>
                                <td><?php echo esc_html($log->email ?? ''); ?></td>
                                <td><?php echo esc_html($log->otp ?? ''); ?></td>
                                <td>
                                    <span class="status-badge status-<?php echo esc_attr(strtolower($log->status ?? 'pending')); ?>">
                                        <?php echo esc_html($log->status ?? ''); ?>
                                    </span>
                                </td>
                                <td><?php echo esc_html($log->ip_address ?? ''); ?></td>
                                <td><?php echo esc_html($log->created_at ?? ''); ?></td>
                                <td><?php echo esc_html($log->user_id ?? ''); ?></td>
                            </tr>
                        <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="7"><?php esc_html_e('No logs found.', 'authyo-otp-authentication-for-woocommerce'); ?></td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
                </form>
            </div>
        </div>
        <?php
    }

    public function render_shortcodes_page() {
        ?>
        <div class="wrap authyo-modern-wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <div class="authyo-shortcodes-grid">
                <div class="shortcode-section">
                    <h3><?php esc_html_e('Login Form', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
                    <p><?php esc_html_e('Display the login form with OTP functionality:', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                    <code>[authyo_otp_login]</code>
                    <p class="description"><?php esc_html_e('Use this shortcode on your login page.', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                    
                    <div class="shortcode-preview">
                        <h4><?php esc_html_e('Features:', 'authyo-otp-authentication-for-woocommerce'); ?></h4>
                        <ul>
                            <li><?php esc_html_e('Username/Email field', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                            <li><?php esc_html_e('Password field', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                            <li><?php esc_html_e('OTP field (if enabled)', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                            <li><?php esc_html_e('Remember me checkbox', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                            <li><?php esc_html_e('Links to registration and forgot password', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                        </ul>
                    </div>
                </div>

                <div class="shortcode-section">
                    <h3><?php esc_html_e('Registration Form', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
                    <p><?php esc_html_e('Display the registration form with OTP verification:', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                    <code>[authyo_otp_register]</code>
                    <p class="description"><?php esc_html_e('Use this shortcode on your registration page.', 'authyo-otp-authentication-for-woocommerce'); ?></p>
                    
                    <div class="shortcode-preview">
                        <h4><?php esc_html_e('Features:', 'authyo-otp-authentication-for-woocommerce'); ?></h4>
                        <ul>
                            <li><?php esc_html_e('Username field', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                            <li><?php esc_html_e('Email field', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                            <li><?php esc_html_e('Phone number field', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                            <li><?php esc_html_e('Password field', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                            <li><?php esc_html_e('OTP field (if enabled)', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                            <li><?php esc_html_e('Terms and conditions checkbox', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                        </ul>
                    </div>
                </div>

                <div class="shortcode-instructions">
                    <h3><?php esc_html_e('How to Use', 'authyo-otp-authentication-for-woocommerce'); ?></h3>
                    <ol>
                        <li><?php esc_html_e('Go to Pages → Add New in your WordPress dashboard', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                        <li><?php esc_html_e('Create separate pages for Login, Registration, and Password Reset', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                        <li><?php esc_html_e('Copy the respective shortcode and paste it into the content editor of each page', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                        <li><?php esc_html_e('Publish the pages', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                        <li><?php esc_html_e('Go to WC OTP Auth → Settings to enable/disable OTP verification for login and registration', 'authyo-otp-authentication-for-woocommerce'); ?></li>
                    </ol>
                </div>
            </div>
        </div>
        <?php
    }

    public function enqueue_admin_scripts($hook) {
        // Only load on plugin pages
        if (strpos($hook, 'authyo-otp-authentication-for-woocommerce') === false) {
            return;
        }

        // Enqueue admin styles
        wp_enqueue_style(
            'authyo-otp-authentication-for-woocommerce-admin',
            plugin_dir_url(__FILE__) . '../assets/css/admin.css',
            array(),
            $this->version
        );

        wp_enqueue_style(
            'authyo-otp-authentication-for-woocommerce-dashboard',
            plugin_dir_url(__FILE__) . '../assets/css/admin-dashboard.css',
            array(),
            $this->version
        );

        // Enqueue admin scripts
        wp_enqueue_script(
            'authyo-otp-authentication-for-woocommerce-admin',
            plugin_dir_url(__FILE__) . '../assets/js/admin.js',
            array('jquery'),
            $this->version,
            true
        );

        // Localize script with sanitized data
        wp_localize_script('authyo-otp-authentication-for-woocommerce-admin', 'authyoOtpAuthAdmin', array(
            'ajaxurl' => esc_url(admin_url('admin-ajax.php')),
            'nonce' => wp_create_nonce('authyo_otp_auth_admin_nonce'),
            'messages' => array(
                'confirm_delete' => esc_html__('Are you sure you want to delete the selected logs?', 'authyo-otp-authentication-for-woocommerce'),
                'delete_success' => esc_html__('Selected logs deleted successfully.', 'authyo-otp-authentication-for-woocommerce'),
                'delete_error' => esc_html__('Error deleting logs. Please try again.', 'authyo-otp-authentication-for-woocommerce')
            )
        ));
    }

    public function check_logs_table() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'authyo_otp_auth_logs';
        $escaped_table_name = esc_sql($table_name);
        
        // Check if table exists using prepare
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin needs to check custom table existence
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) === $table_name;
        
        if (!$table_exists) {
            return false;
        }
        
        // Get total count of logs (no user input)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is escaped with esc_sql(), plugin needs custom table access
        $total_count = $wpdb->get_var("SELECT COUNT(*) FROM `{$escaped_table_name}`");
        
        // Get sample logs (no user input)
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared,WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Table name is escaped with esc_sql(), plugin needs custom table access
        $sample_logs = $wpdb->get_results("SELECT * FROM `{$escaped_table_name}` ORDER BY created_at DESC LIMIT 5");
        
        return array(
            'table_exists' => $table_exists,
            'total_count' => $total_count,
            'sample_logs' => $sample_logs
        );
    }

    public function handle_bulk_delete() {
        // Nonce and capability checks are handled in render_logs_page before calling delete action
        // This method is not currently hooked up to handle bulk delete, the logic is in render_logs_page
        // If this method were to be used directly, it would need nonce/capability checks and input sanitization/validation.
        // The existing implementation in render_logs_page is safer.
        
        // Keeping this method here but noting it's not the active bulk delete handler.
        // If it were, it would need significant changes to be secure.

        // For demonstration purposes, if this were the handler:
        /*
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'authyo_otp_auth_logs')) { // Use the correct nonce name
             // Security check failed
             wp_die(esc_html__('Security check failed.', 'authyo-otp-authentication-for-woocommerce'));
         }

         if (!current_user_can('manage_options')) {
             wp_die(esc_html__('You do not have sufficient permissions.', 'authyo-otp-authentication-for-woocommerce'));
         }

         if (isset($_POST['log_ids']) && is_array($_POST['log_ids'])) {
             $log_ids = array_map('intval', wp_unslash($_POST['log_ids'])); // Sanitize
             $log_ids = array_filter($log_ids, function($id) { return $id > 0; }); // Validate

             if (!empty($log_ids)) {
                 global $wpdb;
                 $table_name = $wpdb->prefix . 'authyo_otp_auth_logs';
                 $format = implode(', ', array_fill(0, count($log_ids), '%d'));
                 $query = "DELETE FROM {$table_name} WHERE id IN ({$format})";
                 $wpdb->query($wpdb->prepare($query, $log_ids)); // Use prepare
             }
         }
         // Redirect back to logs page with a success/error message
         $redirect_url = add_query_arg(array('page' => 'authyo-otp-authentication-for-woocommerce-logs', 'deleted' => count($log_ids)), admin_url('admin.php'));
         wp_safe_redirect($redirect_url);
         exit;
        */
    }

    public function get_logs($per_page = 20, $page_number = 1) {
        global $wpdb;
        $table_name = $wpdb->prefix . 'authyo_otp_auth_logs';
        $escaped_table_name = esc_sql($table_name);
        
        // Ensure page number and per page are positive integers
        $per_page = absint($per_page);
        $page_number = absint($page_number);
        
        if ($per_page === 0) { // Avoid division by zero
             $per_page = 20;
        }

        $offset = ($page_number - 1) * $per_page;

        // Use $wpdb->get_results with prepare
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin needs custom table access
        $logs = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM `{$escaped_table_name}` ORDER BY created_at DESC LIMIT %d OFFSET %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is escaped with esc_sql()
                $per_page,
                $offset
            ),
            ARRAY_A
        );
        
        return $logs;
    }

    /**
     * Add Settings link on Plugins list page
     *
     * @param array $links Existing links.
     * @return array Modified links.
     */
    public function add_action_links( $links ) {
        $settings_url = admin_url( 'admin.php?page=authyo-otp-authentication-for-woocommerce' );
        $links[]      = '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'Settings', 'authyo-otp-authentication-for-woocommerce' ) . '</a>';
        return $links;
    }

    /**
     * Sanitize OTP expiry value to ensure it's between 10 and 600 seconds
     *
     * @param mixed $value The value to sanitize.
     * @return int Sanitized value between 10 and 600.
     */
    public function sanitize_otp_expiry( $value ) {
        $value = absint( $value );
        
        // Ensure minimum of 10 seconds
        if ( $value < 10 ) {
            $value = 10;
        }
        
        // Ensure maximum of 600 seconds (10 minutes)
        if ( $value > 600 ) {
            $value = 600;
        }
        
        return $value;
    }

    /**
     * Trigger activation webhook whenever plugin settings are saved.
     *
     * @param string $option_name Updated option name.
     * @param mixed  $old_value   Old value.
     * @param mixed  $value       New value.
     */
    public function maybe_send_activation_ping( $option_name, $old_value, $value ) {
        if ( $this->activation_ping_sent || ! is_admin() ) {
            return;
        }

        if ( ! isset( $_POST['option_page'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified below before sending data.
            return;
        }

        $option_page = sanitize_text_field( wp_unslash( $_POST['option_page'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Sanitizing read-only access.
        if ( 'authyo_otp_auth_settings' !== $option_page ) {
            return;
        }

        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $nonce = isset( $_POST['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Value validated immediately.
        if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'authyo_otp_auth_settings-options' ) ) {
            return;
        }

        $this->activation_ping_sent = true;
        $this->send_activation_ping();
    }

    /**
     * Send activation payload to Authyo webhook.
     */
    private function send_activation_ping() {
        $api_key = get_option( 'authyo_otp_auth_app_id', '' );
        $pageurl = wp_get_referer();

        if ( empty( $pageurl ) ) {
            $pageurl = admin_url( 'admin.php?page=authyo-otp-authentication-for-woocommerce' );
        }

        $parsed = wp_parse_url( $pageurl );
        if ( empty( $parsed['host'] ) ) {
            $pageurl = home_url( $pageurl );
        }
        $pageurl = esc_url_raw( $pageurl );

        $payload = array(
            'subject' => 'Authyo_Woocommerce_Plugin_Activation',
            'api_key' => $api_key,
            'data'    => array(
                'reason'    => 'Plugin_activate',
                'plugin'    => AUTHYO_OTP_AUTH_PLUGIN_BASENAME,
                'version'   => AUTHYO_OTP_AUTH_VERSION,
                'timestamp' => gmdate( 'c' ),
                'pageurl'   => $pageurl,
            ),
        );

        $response = wp_remote_post(
            $this->webhook_url,
            array(
                'timeout' => 15,
                'headers' => array(
                    'Content-Type' => 'application/json',
                ),
                'body'    => wp_json_encode( $payload ),
            )
        );

        if ( is_wp_error( $response ) && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'Authyo OTP: Failed to send activation ping: ' . $response->get_error_message() ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Logged only when WP_DEBUG is enabled.
        }
    }
} 