<?php
/**
 * Debug Logger Helper
 *
 * Temporary logging system for testing and debugging.
 * This will be removed once everything is confirmed working.
 *
 * @package Authyo_OTP_Auth
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Debug Logger class
 */
class Authyo_Debug_Logger {
    
    /**
     * Option name for storing logs
     *
     * @var string
     */
    private static $option_name = 'authyo_debug_logs';
    
    /**
     * Maximum number of log entries to keep
     *
     * @var int
     */
    private static $max_logs = 500;
    
    /**
     * Log a debug message
     *
     * @param string $message Log message
     * @param mixed  $data    Optional data to log
     * @param string $context Optional context (e.g., 'notification', 'template', 'api')
     * @return void
     */
    public static function log($message, $data = null, $context = 'general') {
        $log_entry = array(
            'timestamp' => current_time('mysql'),
            'time' => time(),
            'message' => sanitize_text_field($message),
            'context' => sanitize_text_field($context),
            'data' => $data,
        );
        
        // Get existing logs
        $logs = get_option(self::$option_name, array());
        
        // Ensure logs is an array
        if (!is_array($logs)) {
            $logs = array();
        }
        
        // Add new log entry
        $logs[] = $log_entry;
        
        // Keep only the most recent logs
        if (count($logs) > self::$max_logs) {
            $logs = array_slice($logs, -self::$max_logs);
        }
        
        // Save logs
        update_option(self::$option_name, $logs);
    }
    
    /**
     * Get all logs
     *
     * @param int    $limit  Maximum number of logs to return
     * @param string $context Filter by context (optional)
     * @return array Log entries
     */
    public static function get_logs($limit = 500, $context = null) {
        $logs = get_option(self::$option_name, array());
        
        if (!is_array($logs)) {
            return array();
        }
        
        // Filter by context if provided
        if (!empty($context)) {
            $logs = array_filter($logs, function($log) use ($context) {
                return isset($log['context']) && $log['context'] === $context;
            });
        }
        
        // Sort by timestamp (newest first)
        usort($logs, function($a, $b) {
            $time_a = isset($a['time']) ? $a['time'] : 0;
            $time_b = isset($b['time']) ? $b['time'] : 0;
            return $time_b - $time_a;
        });
        
        // Limit results
        if ($limit > 0) {
            $logs = array_slice($logs, 0, $limit);
        }
        
        return $logs;
    }
    
    /**
     * Clear all logs
     *
     * @return bool True on success, false on failure
     */
    public static function clear_logs() {
        return delete_option(self::$option_name);
    }
    
    /**
     * Get log count
     *
     * @return int Number of log entries
     */
    public static function get_log_count() {
        $logs = get_option(self::$option_name, array());
        return is_array($logs) ? count($logs) : 0;
    }
}

/**
 * Helper function to log debug messages
 *
 * @param string $message Log message
 * @param mixed  $data    Optional data to log
 * @param string $context Optional context
 * @return void
 */
function authyo_debug_log($message, $data = null, $context = 'general') {
    Authyo_Debug_Logger::log($message, $data, $context);
}

