<?php
/**
 * Plugin Name: Authyo Authentication and Notification for WooCommerce
 * Plugin URI: https://wordpress.org/plugins/authyo-otp-authentication-for-woocommerce/
 * Description: Our plugin enables secure OTP verification on the Login, Registration, and Checkout pages via Email, WhatsApp, SMS, and Voice Call. In addition, it provides seamless WooCommerce notifications through Email, WhatsApp, and SMS, ensuring reliable communication across multiple channels.
 * Version: 1.0.14
 * Author: Authyo
 * Author URI: https://authyo.io
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: authyo-otp-authentication-for-woocommerce
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 9.0
 *
 * @package Authyo_OTP_Auth
 * @since 1.0.0
 */

// Exit if accessed directly
if (!defined('WPINC')) {
    die;
}

// Define plugin constants
define('AUTHYO_OTP_AUTH_VERSION', '1.0.14');
define('AUTHYO_OTP_AUTH_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('AUTHYO_OTP_AUTH_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AUTHYO_OTP_AUTH_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Declare WooCommerce HPOS compatibility
add_action('before_woocommerce_init', function () {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('cart_checkout_blocks', __FILE__, true);
    }
});

// Include required classes
require_once AUTHYO_OTP_AUTH_PLUGIN_DIR . 'includes/class-authyo-otp-auth-activator.php';
require_once AUTHYO_OTP_AUTH_PLUGIN_DIR . 'includes/class-authyo-otp-auth-deactivator.php';
require_once AUTHYO_OTP_AUTH_PLUGIN_DIR . 'includes/class-authyo-otp-auth.php';

// Activation hook
register_activation_hook(__FILE__, array('Authyo_OTP_Auth_Activator', 'activate'));

// Deactivation hook
register_deactivation_hook(__FILE__, array('Authyo_OTP_Auth_Deactivator', 'deactivate'));

/**
 * Check and create database table if needed.
 *
 * Verifies that the plugin's custom database table exists and creates it if missing.
 * This function runs on every page load to ensure database integrity.
 *
 * @since 1.0.0
 * @global wpdb $wpdb WordPress database abstraction object.
 * @return void
 */
function authyo_otp_auth_check_database()
{
    global $wpdb;
    $table_name = $wpdb->prefix . 'authyo_otp_auth_logs';

    // Check if table exists using prepared statement
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin needs to check custom table existence
    $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name));

    if ($table_exists != $table_name) {
        // Table doesn't exist, create it
        Authyo_OTP_Auth_Activator::activate();

        // Verify table was created using prepared statement
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.DirectDatabaseQuery.NoCaching -- Plugin needs to verify custom table creation
        $table_exists_after = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name));

        if ($table_exists_after != $table_name) {
            // If still not created, log error using WordPress logging
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG is enabled
                error_log('Authyo OTP: Failed to create database table ' . $table_name);
            }
        }
    }
}

/**
 * Check and run upgrade routines if needed.
 *
 * Runs upgrade checks and migrations for existing installations.
 * This ensures that settings are properly migrated when plugin is updated.
 *
 * @since 1.0.4
 * @return void
 */
function authyo_otp_auth_check_upgrades()
{
    $current_version = get_option('authyo_otp_auth_version', '0');

    // If this is a new installation or upgrade from version < 1.0.4
    if (version_compare($current_version, '1.0.4', '<')) {
        // Ensure fallback is enabled by default (fix for live sites)
        if (get_option('authyo_otp_auth_fallback_enabled') === false) {
            update_option('authyo_otp_auth_fallback_enabled', 'yes');
        }

        // Update version
        update_option('authyo_otp_auth_version', AUTHYO_OTP_AUTH_VERSION);

        // Log the upgrade
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('Authyo OTP: Upgraded from ' . $current_version . ' to ' . AUTHYO_OTP_AUTH_VERSION);
        }
    }
}

/**
 * Initialize the plugin.
 *
 * Main function to start the plugin execution. Ensures database table exists
 * and initializes the core plugin class.
 *
 * @since 1.0.0
 * @return void
 */
function authyo_otp_auth_init()
{
    // Run upgrade checks
    authyo_otp_auth_check_upgrades();

    // Ensure database table exists
    authyo_otp_auth_check_database();

    // Initialize and run the plugin
    $plugin = new Authyo_OTP_Auth();
    $plugin->run();
}
add_action('init', 'authyo_otp_auth_init');