jQuery(document).ready(function($) {
    // Prevent multiple initialization
    if (window.authyoRegisterFormInitialized) {
        return;
    }
    window.authyoRegisterFormInitialized = true;
    
    // Critical dependency check - ensure authyoOtpRegister is defined
    if (typeof authyoOtpRegister === 'undefined') {
        // Create a minimal fallback object to prevent crashes
        window.authyoOtpRegister = {
            ajaxurl: (typeof ajaxurl !== 'undefined') ? ajaxurl : '/wp-admin/admin-ajax.php',
            nonce: '',
            otpExpiry: 300,
            debug: true,
            messages: {
                sending: 'Sending OTP...',
                verifying: 'Verifying OTP...',
                sent: 'OTP Sent!',
                error: 'An error occurred. Please try again.',
                enter_email: 'Please enter your email or mobile number.',
                send_otp: 'Send OTP',
                resend: 'Resend OTP',
                mobile_preview: 'Mobile number will be sent as: ',
                all_fields_required: 'All fields are required.'
            }
        };
    }
    
    var timeLeft = authyoOtpRegister.otpExpiry || 300; // default 5 minutes
    var timer;
    var messageTimeout;
    var $registerBtn = $('#register-with-otp');
    var $resendOtpBtn = $('#resend-register-otp');
    var $otpField = $('.otp-field');
    var $numberPreview = $('#register-number-preview');
    var $fallbackMethods = $('.resend-otp-section');
    var $countrySelect = $('#register-country-code');
    var otpSent = false;
    var initialMethod = null;
    var countriesList = [];
    var selectedCountry = null;
    
    // Debug mode - set to false in production
    var debugMode = typeof authyoOtpRegister !== 'undefined' && authyoOtpRegister.debug === true;
    
    function debugLog(message, data) {
        // Debug logging disabled for production
    }

    // Validation functions
    function validateEmail(email) {
        var re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }

    function validateMobile(mobile) {
        // Remove any non-digit characters except +
        mobile = mobile.replace(/[^\d+]/g, '');
        
        // Check if mobile number has at least 10 digits after +
        if (mobile.startsWith('+')) {
            return mobile.length >= 11 && mobile.length <= 16 && /^\d+$/.test(mobile.substring(1));
        }
        
        // For mobile numbers without +, we need to check if country is selected
        // This will be handled in processMobileNumber function
        // Support for global mobile number lengths (7-15 digits)
        return mobile.length >= 7 && mobile.length <= 15 && /^\d+$/.test(mobile);
    }

    // Helper function to process mobile number with country code
    function processMobileNumber(phoneNumber) {
        var isMobile = validateMobile(phoneNumber);
        if (!isMobile) return phoneNumber;
        
        var cleanMobile = phoneNumber.replace(/[^\d]/g, ''); // Remove all non-digits
        var originalValue = $('#reg_email').val();
        
        // Check if country is selected
        if (!selectedCountry) {
            // If no country selected, check if input already has country code
            if (originalValue.startsWith('+')) {
                // Input already has country code, use it
                return originalValue.replace(/[^\d+]/g, '');
            } else {
                // No country selected and no country code in input - return error indicator
                return 'ERROR_NO_COUNTRY';
            }
        } else {
            // Country is selected, use country code
            var countryCode = selectedCountry.replace(/[^\d]/g, ''); // Extract digits from country code
            return '+' + countryCode + cleanMobile;
        }
    }

    function startTimer() {
        // Hide fallback during timer
        $('.resend-otp-section').addClass('authyo-hidden');
        
        timer = setInterval(function() {
            timeLeft--;
            var minutes = Math.floor(timeLeft / 60);
            var seconds = timeLeft % 60;
            
            if (timeLeft > 0) {
                $resendOtpBtn.prop('disabled', true);
                $resendOtpBtn.find('.timer').text(authyoOtpRegister.messages.resend + ' (' + minutes + ':' + (seconds < 10 ? '0' : '') + seconds + 's)');
            } else {
                clearInterval(timer);
                $resendOtpBtn.prop('disabled', false);
                timeLeft = authyoOtpRegister.otpExpiry || 300;
                $resendOtpBtn.find('.timer').text(authyoOtpRegister.messages.resend);
                // Show resend methods after timer completes
                showResendMethods(initialMethod);
            }
        }, 1000);
    }

    function showMessage(message, isError) {
        var messageClass = isError ? 'error' : 'success';
        var $messageDiv = $('#register-message');
        
        // Clear any pending timeout to prevent duplicate animations
        if (messageTimeout) {
            clearTimeout(messageTimeout);
            messageTimeout = null;
        }
        
        // Stop any ongoing animations, clear the queue, and reset to visible state
        $messageDiv.stop(true, true).hide();
        
        // Update and show the message with a clean slate
        $messageDiv.removeClass('success error').addClass(messageClass);
        $messageDiv.html('<p>' + message + '</p>').show();
        
        // Auto-hide success messages after 5 seconds
        if (!isError) {
            messageTimeout = setTimeout(function() {
                $messageDiv.fadeOut(400, function() {
                    // Clear the message content and remove classes after fade out completes
                    $messageDiv.empty().removeClass('success error').hide();
                });
                messageTimeout = null;
            }, 5000);
        }
    }

    function clearMessage() {
        // Clear any pending timeout
        if (messageTimeout) {
            clearTimeout(messageTimeout);
            messageTimeout = null;
        }
        
        // Stop animations, hide, clear the message, and remove classes
        var $messageDiv = $('#register-message');
        $messageDiv.stop(true, true).hide().empty().removeClass('success error');
    }

    // Show resend methods after initial OTP is sent
    function showResendMethods(usedMethod) {
        initialMethod = usedMethod;
        
        // Get current email/mobile value
        var currentValue = $('#reg_email').val().trim();
        
        // Check email FIRST - if it's an email, never show fallback methods
        var isEmail = validateEmail(currentValue);
        if (isEmail) {
            // Hide fallback methods for email - use both class and inline style
            $('.resend-otp-section').addClass('authyo-hidden').hide();
            return;
        }
        
        // Only check if mobile after confirming it's not an email
        var isMobile = validateMobile(currentValue);
        if (isMobile) {
            // Show the resend section for mobile numbers - remove class and inline style
            $('.resend-otp-section').removeClass('authyo-hidden').show();
            
            // Enable all resend method buttons
            $('.resend-method-btn').prop('disabled', false);
            
            // Disable the method that was just used
            if (usedMethod) {
                $('.resend-method-btn[data-method="' + usedMethod + '"]').prop('disabled', true);
            }
        } else {
            // Hide fallback methods for invalid input
            $('.resend-otp-section').addClass('authyo-hidden').hide();
        }
    }

    // Handle resend method button click
    // Scope the event handler to only work within the registration form
    $(document).on('click', '#authyo-otp-authentication-for-woocommerce-register-form .resend-method-btn', function(e) {
        e.preventDefault(); // Prevent any default button behavior
        e.stopPropagation(); // Stop event from bubbling to other handlers
        
        var $btn = $(this);
        
        // Check if button is disabled
        if ($btn.prop('disabled')) {
            debugLog('Button is disabled, ignoring click');
            return;
        }
        
        var method = $btn.data('method');
        var email = $('#reg_email').val().trim();

        debugLog('Fallback button clicked', {
            method: method,
            email: email,
            buttonDisabled: $btn.prop('disabled'),
            ajaxurl: authyoOtpRegister.ajaxurl,
            hasNonce: !!authyoOtpRegister.nonce
        });

        if (!email || !method) {
            showMessage('Please enter email or mobile number', true);
            return;
        }

        // Validate that fallback methods can only be used with mobile numbers
        var isMobile = validateMobile(email);
        var isEmail = validateEmail(email);
        
        if (isEmail) {
            showMessage('Fallback methods (SMS, WhatsApp, Voice) are only available for mobile numbers.', true);
            return;
        }
        
        if (!isMobile) {
            showMessage('Please enter a valid email or mobile number.', true);
            return;
        }

        // Process mobile number with country code
        email = processMobileNumber(email);

        // Check if processing resulted in an error
        if (email === 'ERROR_NO_COUNTRY' || email === 'ERROR_INVALID_FORMAT' || email.indexOf('ERROR_') === 0) {
            showMessage('Please select a country for your mobile number to enable SMS delivery.', true);
            return;
        }

        $btn.prop('disabled', true).text('Sending...');
        
        // Get nonce from form field as fallback
        var nonceFromForm = $('input[name="authyo_otp_auth_register_nonce"]').val();

        $.post(authyoOtpRegister.ajaxurl, {
            action: 'authyo_otp_auth_resend_fallback',
            email: email,
            method: method,
            action_type: 'register',
            nonce: authyoOtpRegister.nonce,
            authyo_otp_auth_register_nonce: nonceFromForm || authyoOtpRegister.nonce
        }, function(response) {
            debugLog('Fallback OTP response', response);
            $btn.prop('disabled', false).text(method.charAt(0).toUpperCase() + method.slice(1));
            if (response.success) {
                showMessage(response.data.message, false);
                $('#register-mask-id').val(response.data.maskId);
                startTimer();
                // Update to show this method as used
                initialMethod = response.data.method || method;
            } else {
                showMessage(response.data.message, true);
            }
        }).fail(function(xhr, status, error) {
            debugLog('Fallback OTP request failed', {
                status: status,
                error: error,
                response: xhr.responseText
            });
            $btn.prop('disabled', false).text(method.charAt(0).toUpperCase() + method.slice(1));
            showMessage('Connection error. Please check your internet connection and try again.', true);
        });
    });

    // Fetch country list
    function fetchCountryList() {
        $.post(authyoOtpRegister.ajaxurl, {
            action: 'authyo_otp_auth_get_country_list',
            nonce: authyoOtpRegister.nonce
        }, function(response) {
            if (response.success && response.data.countries) {
                countriesList = response.data.countries;
                populateCountrySelect();
            }
        });
    }

    // Populate country select dropdown
    function populateCountrySelect() {
        if (countriesList.length === 0) return;
        
        $countrySelect.empty();
        $countrySelect.append('<option value="">Select Country</option>');
        
        $.each(countriesList, function(index, country) {
            $countrySelect.append('<option value="' + country.countryCode + '" data-code="' + country.countryCode + '" data-name="' + country.name + '">' + country.name + ' (' + country.countryCode + ')</option>');
        });
    }

    // Show/hide country dropdown based on input type
    function toggleCountryDropdown(value) {
        var isEmail = validateEmail(value);
        var isMobile = validateMobile(value);
        
        // Check if input contains letters (likely email or username)
        var hasLetters = /[a-zA-Z]/.test(value);
        
        // Check if input looks like email prefix (letters followed by digits, e.g. "user123")
        var looksLikeEmailPrefix = hasLetters && /\d/.test(value) && !value.includes('@') && /[a-zA-Z]/.test(value);
        
        // Check if input looks like complete email
        var looksLikeEmail = value.includes('@') || 
                             (value.length >= 5 && /\w+\.\w+/i.test(value)) ||
                             /\w+@/i.test(value);
        
        var hasDigits = /\d/.test(value);
        var hasDigitsOnly = /^[\d\s\+\-\(\)]+$/.test(value); // Only digits, spaces, +, -, (, )
        
        // Show dropdown only if:
        // 1. It's a valid mobile number with no letters, OR
        // 2. It has digits only (no letters) and length >= 3, OR
        // 3. It starts with + followed by digits
        // BUT NOT if it looks like an email prefix
        if (looksLikeEmail || looksLikeEmailPrefix) {
            // Don't show for email
            $countrySelect.hide();
        } else if (isMobile || (hasDigits && hasDigitsOnly && value.length >= 3)) {
            // Show for phone number patterns
            $countrySelect.show();
        } else {
            // Hide otherwise
            $countrySelect.hide();
        }
    }

    // Handle country selection
    $countrySelect.on('change', function() {
        selectedCountry = $(this).val();
        
        if (selectedCountry) {
            // Remove any existing country code from input, keep only the phone number
            var currentValue = $('#reg_email').val();
            // Remove country codes like "+91 ", "+1 ", etc.
            var withoutCountry = currentValue.replace(/^\+\d+\s*/, '');
            $('#reg_email').val(withoutCountry);
        }
        
        // Trigger input event to update preview
        $('#reg_email').trigger('input');
    });

    // Show mobile preview and enforce fallback visibility rules on input
    $('#reg_email').on('input', function() {
        var val = $(this).val().trim();
        var isEmail = validateEmail(val);
        var isMobile = validateMobile(val);
        
        // Toggle country dropdown visibility
        toggleCountryDropdown(val);
        
        if (isMobile) {
            if (selectedCountry) {
                var cleanMobile = val.replace(/\D/g, '');
                var countryCode = selectedCountry.replace(/[^\d]/g, '');
                $numberPreview.text('Will send to: +' + countryCode + cleanMobile).show();
            } else {
                $numberPreview.text('Please select a country for SMS delivery').show();
            }
        } else {
            $numberPreview.hide();
        }

        // Ensure fallback methods are hidden for emails and non-mobile inputs
        if (isEmail || !isMobile) {
            $(this).closest('form').find('.resend-otp-section').addClass('authyo-hidden').hide();
        }
    });

    // Fetch country list on page load
    fetchCountryList();

    // Main register button - handles both OTP sending and registration
    $registerBtn.on('click', function(e) {
        e.preventDefault();
        var email = $('#reg_email').val().trim();
        
        if (!email) {
            showMessage(authyoOtpRegister.messages.enter_email, true);
            return;
        }

        // Basic validation
        var isEmail = /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
        var isMobile = validateMobile(email);
            
        if (!isEmail && !isMobile) {
            showMessage(authyoOtpRegister.messages.enter_email, true);
            return;
        }

        // Process mobile number with country code if it's a mobile
        var processedEmail = isMobile ? processMobileNumber(email) : email;
        
        // Check if country selection is required for mobile numbers
        if (isMobile && processedEmail === 'ERROR_NO_COUNTRY') {
            showMessage('Please select a country for your mobile number to enable SMS delivery.', true);
            return;
        }

        var $btn = $(this);

        if (!otpSent) {
            // First click - Send OTP
            $btn.prop('disabled', true).text(authyoOtpRegister.messages.sending);
            
            // Get nonce from form field as fallback
            var nonceFromForm = $('input[name="authyo_otp_auth_register_nonce"]').val();
            
            $.post(authyoOtpRegister.ajaxurl, {
                action: 'authyo_otp_auth_send_register_otp',
                email: processedEmail,
                nonce: authyoOtpRegister.nonce,
                authyo_otp_auth_register_nonce: nonceFromForm || authyoOtpRegister.nonce
            }, function(response) {
                $btn.prop('disabled', false);
                
                if (response.success) {
                    $('#register-mask-id').val(response.data.maskId);
                    $otpField.show();
                    $btn.text('Complete Registration');
                    otpSent = true;
                    startTimer();
                    
                    // CRITICAL: Hide fallback methods immediately for email, keep hidden for mobile during timer
                    if (isEmail) {
                        // Email: Hide fallback methods completely with class AND inline style - they should NEVER show for email
                        $('.resend-otp-section').addClass('authyo-hidden').hide();
                    } else {
                        // Mobile: Keep hidden during timer, will show after timer completes in showResendMethods()
                        $('.resend-otp-section').addClass('authyo-hidden').hide();
                    }
                    
                    // Check if this is a bypass message
                    // Note: showMessage() already handles clearing previous messages
                    if (response.data.message && response.data.message.includes('bypassed for testing')) {
                        showMessage('Testing Mode: OTP verification bypassed for this test number. You can enter any 6-digit code to proceed.', false);
                    } else {
                        showMessage(response.data.message, false);
                    }
                    
                    // Use the actual method from the response
                    var method = response.data.method || (isEmail ? 'email' : 'whatsapp');
                    // Store method for later use when timer completes
                    initialMethod = method;
                } else {
                    // Better error handling
                    var errorMsg = '';
                    if (response.data && response.data.message) {
                        errorMsg = response.data.message;
                    } else if (response.message) {
                        errorMsg = response.message;
                    } else {
                        errorMsg = authyoOtpRegister.messages.error;
                    }
                    showMessage(errorMsg, true);
                    $btn.text('Register with OTP');
                }
            }).fail(function(xhr, status, error) {
                // Handle AJAX failures
                $btn.prop('disabled', false).text('Register with OTP');
                showMessage('Connection error. Please try again.', true);
            });
        } else {
            // Second click - Verify OTP and Register
            var otp = $('#reg_otp').val().trim();
            var maskId = $('#register-mask-id').val();
            
            // Validate OTP field
            if (!otp) {
                showMessage('Please enter the OTP sent to your email/mobile.', true);
                return;
            }
            
            // Validate maskId field - critical for verification
            if (!maskId) {
                showMessage('Invalid session. Please request a new OTP.', true);
                return;
            }

            // Re-process the email to ensure it's valid for verification
            var currentEmail = $('#reg_email').val().trim();
            
            // Validate email field is not empty
            if (!currentEmail) {
                showMessage('Email or mobile number is required.', true);
                return;
            }
            
            var isEmail = validateEmail(currentEmail);
            var isMobile = validateMobile(currentEmail);
            var finalProcessedEmail = isMobile ? processMobileNumber(currentEmail) : currentEmail;
            
            // Check if processing resulted in an error
            if (finalProcessedEmail === 'ERROR_NO_COUNTRY' || finalProcessedEmail === 'ERROR_INVALID_FORMAT' || finalProcessedEmail.indexOf('ERROR_') === 0) {
                showMessage('Please select a country for your mobile number to enable SMS delivery.', true);
                return;
            }
            
            // Final validation that we have a valid email/mobile
            if (!finalProcessedEmail || finalProcessedEmail.trim() === '') {
                showMessage('Please enter a valid email or mobile number.', true);
                return;
            }

            $btn.prop('disabled', true).text(authyoOtpRegister.messages.verifying);
            
            // Get nonce from form field as fallback
            var nonceFromForm = $('input[name="authyo_otp_auth_register_nonce"]').val();
            
            $.post(authyoOtpRegister.ajaxurl, {
                action: 'authyo_otp_auth_verify_register_otp',
                email: finalProcessedEmail,
                otp: otp,
                maskId: maskId,
                nonce: authyoOtpRegister.nonce,
                authyo_otp_auth_register_nonce: nonceFromForm || authyoOtpRegister.nonce
            }, function(response) {
                if (response.success) {
                    showMessage(response.data.message, false);
                    clearInterval(timer);
                    if (response.data.redirect) {
                        setTimeout(function() {
                            window.location.href = response.data.redirect;
                        }, 1500);
                    }
                } else {
                    // Better error handling for verification
                    var errorMsg = '';
                    if (response.data && response.data.message) {
                        errorMsg = response.data.message;
                    } else if (response.message) {
                        errorMsg = response.message;
                    } else {
                        errorMsg = authyoOtpRegister.messages.error;
                    }
                    showMessage(errorMsg, true);
                    $btn.prop('disabled', false).text('Complete Registration');
                }
            }).fail(function(xhr, status, error) {
                // Handle AJAX failures for verification
                $btn.prop('disabled', false).text('Complete Registration');
                showMessage('Connection error. Please try again.', true);
            });
        }
    });

    // Resend OTP
    $resendOtpBtn.on('click', function(e) {
        e.preventDefault();
        var email = $('#reg_email').val().trim();
        
        if (!email) {
            showMessage(authyoOtpRegister.messages.enter_email, true);
            return;
        }

        // Determine if it's a mobile and process it
        var isEmail = /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
        var isMobile = !isEmail;
        
        // Process mobile number with country code if it's a mobile
        var processedEmail = isMobile ? processMobileNumber(email) : email;
        
        // Check if country selection is required
        if (processedEmail === 'ERROR_NO_COUNTRY') {
            showMessage(authyoOtpRegister.messages.select_country || 'Please select a country for your mobile number.', true);
            return;
        }

        var $btn = $(this);
        $btn.prop('disabled', true).find('.timer').text(authyoOtpRegister.messages.sending);
        
        // Get nonce from form field as fallback
        var nonceFromForm = $('input[name="authyo_otp_auth_register_nonce"]').val();
        
        $.post(authyoOtpRegister.ajaxurl, {
            action: 'authyo_otp_auth_send_register_otp',
            email: processedEmail,
            nonce: authyoOtpRegister.nonce,
            authyo_otp_auth_register_nonce: nonceFromForm || authyoOtpRegister.nonce
        }, function(response) {
            if (response.success) {
                $('#register-mask-id').val(response.data.maskId);
                timeLeft = authyoOtpRegister.otpExpiry || 300;
                startTimer();
                // Note: showMessage() already handles clearing previous messages
                showMessage(response.data.message, false);
            } else {
                // Better error handling for resend
                var errorMsg = '';
                if (response.data && response.data.message) {
                    errorMsg = response.data.message;
                } else if (response.message) {
                    errorMsg = response.message;
                } else {
                    errorMsg = authyoOtpRegister.messages.error;
                }
                showMessage(errorMsg, true);
                $btn.find('.timer').text(authyoOtpRegister.messages.resend);
            }
        }).fail(function(xhr, status, error) {
            // Handle AJAX failures for resend
            showMessage('Connection error. Please try again.', true);
            $btn.find('.timer').text(authyoOtpRegister.messages.resend);
        });
    });
}); 