jQuery(document).ready(function($) {
    // Prevent multiple initialization
    if (window.authyoLoginFormInitialized) {
        return;
    }
    window.authyoLoginFormInitialized = true;
    
    // Critical dependency check - ensure authyoOtpAuth is defined
    if (typeof authyoOtpAuth === 'undefined') {
        // Create a minimal fallback object to prevent crashes
        window.authyoOtpAuth = {
            ajaxurl: (typeof ajaxurl !== 'undefined') ? ajaxurl : '/wp-admin/admin-ajax.php',
            nonce: '',
            otpExpiry: 300,
            debug: true,
            messages: {
                login: 'Login',
                sending: 'Sending...',
                verifying: 'Verifying...',
                resend: 'Resend OTP',
                enter_email: 'Please enter a valid email or mobile number.',
                all_fields_required: 'All fields are required.',
                mobile_preview: 'Mobile number will be sent as: +'
            }
        };
    }
    
    // Match the actual login form ID used in the template
    var $form = $('#authyo-otp-authentication-for-woocommerce-login-form');
    var $email = $('#login-email');
    var $countrySelect = $('#login-country-code');
    var $numberPreview = $('#login-number-preview');
    var $otpFields = $('.otp-fields');
    var $sendOtpBtn = $('#send-login-otp');
    var $verifyOtpBtn = $('#verify-login-otp');
    var $resendOtpBtn = $('#resend-login-otp');
    var $message = $('#login-message');
    var $fallbackMethods = $('.resend-otp-section');
    var $alternativeMethodsSection = $('.alternative-methods-section');
    var timerInterval;
    var timeLeft = 300; // 5 minutes in seconds
    var initialMethod = null;
    var countriesList = [];
    var selectedCountry = null;

    var messageTimeout = null;
    var lastMessage = null;
    var lastMessageTime = 0;
    var messageDisplaying = false;
    
    // Debug mode - set to false in production
    var debugMode = typeof authyoOtpAuth !== 'undefined' && authyoOtpAuth.debug === true;
    
    function debugLog(message, data) {
        // Debug logging removed for production
    }
    
    function showMessage(message, isError = false) {
        // Prevent duplicate messages within 500ms
        var currentTime = Date.now();
        if (messageDisplaying && lastMessage === message && (currentTime - lastMessageTime) < 500) {
            return;
        }
        
        // Update tracking variables
        lastMessage = message;
        lastMessageTime = currentTime;
        messageDisplaying = true;
        
        // Clear any existing timeout to prevent duplicate messages
        if (messageTimeout) {
            clearTimeout(messageTimeout);
            messageTimeout = null;
        }
        
        // Ensure we're only targeting one specific message div
        var $targetMessage = $('#login-message');
        
        if ($targetMessage.length === 0) {
            messageDisplaying = false;
            return;
        }
        
        // If multiple divs with same ID exist (invalid HTML), use first one
        if ($targetMessage.length > 1) {
            $targetMessage = $targetMessage.first();
        }
        
        // Stop any ongoing animations, clear the queue, and reset to visible state
        $targetMessage.stop(true, true).hide();
        
        // Update message content and styling with a clean slate
        $targetMessage.html(message).removeClass('success error').addClass(isError ? 'error' : 'success');
        
        // Show the message cleanly
        $targetMessage.show();
        
        // Set new timeout for fade out
        messageTimeout = setTimeout(function() {
            $targetMessage.fadeOut(400, function() {
                // Clear the message content, remove classes, and hide after fade out completes
                $targetMessage.empty().removeClass('success error').hide();
                messageDisplaying = false;
            });
            messageTimeout = null;
        }, 3000);
    }

    function toggleOTPFields(show) {
        $otpFields.toggle(show);
        $sendOtpBtn.toggle(!show);
        $verifyOtpBtn.toggle(show);
        if (show) {
            startTimer();
        }
    }

    function startTimer() {
        timeLeft = authyoOtpAuth.otpExpiry; // Use configured expiry time
        $resendOtpBtn.prop('disabled', true);
        $('.resend-otp-section').addClass('authyo-hidden'); // Hide fallback during timer
        updateTimerDisplay();
        
        clearInterval(timerInterval);
        timerInterval = setInterval(function() {
            timeLeft--;
            updateTimerDisplay();
            
            if (timeLeft <= 0) {
                clearInterval(timerInterval);
                $resendOtpBtn.prop('disabled', false).find('.timer').text(authyoOtpAuth.messages.resend);
                // Show resend methods after timer completes
                showResendMethods(initialMethod);
            }
        }, 1000);
    }

    // Show resend methods after initial OTP is sent
    function showResendMethods(usedMethod) {
        initialMethod = usedMethod;
        
        // Get current email/mobile value
        var currentValue = $email.val().trim();
        
        debugLog('showResendMethods called', {
            usedMethod: usedMethod,
            currentValue: currentValue,
            sectionCount: $('.resend-otp-section').length,
            buttonCount: $('.resend-method-btn').length
        });
        
        // Check email FIRST - if it's an email, never show fallback methods
        var isEmail = validateEmail(currentValue);
        if (isEmail) {
            debugLog('Hiding fallback - input is email');
            // Hide fallback methods for email - use both class and inline style
            $('.resend-otp-section').addClass('authyo-hidden').hide();
            return;
        }
        
        // Only check if mobile after confirming it's not an email
        var isMobile = validateMobile(currentValue);
        if (isMobile) {
            debugLog('Showing fallback - input is mobile', {
                hasHiddenClass: $('.resend-otp-section').hasClass('authyo-hidden')
            });
            
            // Show the resend section for mobile numbers - remove class and inline style
            $('.resend-otp-section').removeClass('authyo-hidden').show();
            
            // Enable all resend method buttons
            $('.resend-method-btn').prop('disabled', false);
            
            // Disable the method that was just used
            if (usedMethod) {
                debugLog('Disabling method that was used:', usedMethod);
                $('.resend-method-btn[data-method="' + usedMethod + '"]').prop('disabled', true);
            }
            
            debugLog('Final button states', {
                smsDisabled: $('.resend-method-btn[data-method="sms"]').prop('disabled'),
                whatsappDisabled: $('.resend-method-btn[data-method="whatsapp"]').prop('disabled'),
                voiceDisabled: $('.resend-method-btn[data-method="voice"]').prop('disabled'),
                allVisible: $('.resend-method-btn:visible').length
            });
        } else {
            debugLog('Hiding fallback - input is not valid mobile');
            // Hide fallback methods for invalid input
            $('.resend-otp-section').addClass('authyo-hidden').hide();
        }
    }

    // Handle alternative method button click (shown when SMS fails)
    $(document).on('click', '#authyo-otp-authentication-for-woocommerce-login-form .alternative-method-btn', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $btn = $(this);
        var method = $btn.data('method');
        var email = $email.val().trim();
        
        debugLog('Alternative method button clicked', {
            method: method,
            email: email
        });

        if (!email || !method) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        // Validate that alternative methods can only be used with mobile numbers
        var isMobile = validateMobile(email);
        var isEmail = validateEmail(email);
        
        if (isEmail) {
            showMessage('Alternative methods (WhatsApp, Voice) are only available for mobile numbers.', true);
            return;
        }
        
        if (!isMobile) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        // Process mobile number
        if (isMobile) {
            var cleanMobile = email.replace(/[^\d+]/g, '');
            var originalValue = $email.val();
            
            // Check if country is selected
            if (!selectedCountry) {
                if (originalValue.startsWith('+')) {
                    email = originalValue.replace(/[^\d+]/g, '');
                } else {
                    showMessage('Please select a country for your mobile number to enable OTP delivery.', true);
                    return;
                }
            } else {
                var countryCode = selectedCountry.replace(/[^\d]/g, '');
                email = '+' + countryCode + cleanMobile;
            }
        }

        $btn.prop('disabled', true).text('Sending...');
        $sendOtpBtn.prop('disabled', true).text(authyoOtpAuth.messages.sending);

        // Get nonce from form field as fallback
        var nonceFromForm = $('input[name="authyo_otp_auth_login_nonce"]').val();

        $.post(authyoOtpAuth.ajaxurl, {
            action: 'authyo_otp_auth_resend_fallback',
            email: email,
            method: method,
            action_type: 'login',
            nonce: authyoOtpAuth.nonce,
            authyo_otp_auth_login_nonce: nonceFromForm || authyoOtpAuth.nonce
        }, function(response) {
            $btn.prop('disabled', false).text(method.charAt(0).toUpperCase() + method.slice(1));
            $sendOtpBtn.prop('disabled', false).text(authyoOtpAuth.messages.login);
            
            if (response.success) {
                showMessage(response.data.message, false);
                $('#login-mask-id').val(response.data.maskId);
                toggleOTPFields(true);
                
                // Hide alternative methods section since OTP was sent successfully
                $alternativeMethodsSection.addClass('authyo-hidden').hide();
                
                // Store the method used
                initialMethod = response.data.method || method;
                
                // Hide fallback methods during timer (will show after timer completes)
                $('.resend-otp-section').addClass('authyo-hidden').hide();
            } else {
                showMessage(response.data.message, true);
            }
        }).fail(function(jqXHR, textStatus, errorThrown) {
            debugLog('AJAX request failed', {
                status: textStatus,
                error: errorThrown,
                response: jqXHR.responseText
            });
            $btn.prop('disabled', false).text(method.charAt(0).toUpperCase() + method.slice(1));
            $sendOtpBtn.prop('disabled', false).text(authyoOtpAuth.messages.login);
            showMessage('Connection error. Please check your internet connection and try again.', true);
        });
    });

    // Handle resend method button click
    // Scope the event handler to only work within the login form
    $(document).on('click', '#authyo-otp-authentication-for-woocommerce-login-form .resend-method-btn', function(e) {
        e.preventDefault(); // Prevent any default button behavior
        e.stopPropagation(); // Stop event from bubbling to other handlers
        
        var $btn = $(this);
        var method = $btn.data('method');
        var email = $email.val().trim();
        
        debugLog('Fallback button clicked', {
            method: method,
            email: email,
            buttonDisabled: $btn.prop('disabled'),
            ajaxurl: authyoOtpAuth.ajaxurl,
            hasNonce: !!authyoOtpAuth.nonce
        });

        if (!email || !method) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        // Validate that fallback methods can only be used with mobile numbers
        var isMobile = validateMobile(email);
        var isEmail = validateEmail(email);
        
        if (isEmail) {
            showMessage('Fallback methods (SMS, WhatsApp, Voice) are only available for mobile numbers.', true);
            return;
        }
        
        if (!isMobile) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        // Process mobile number
        if (isMobile) {
            var cleanMobile = email.replace(/[^\d+]/g, '');
            email = cleanMobile.startsWith('+') ? cleanMobile : '+' + cleanMobile;
        }

        $btn.prop('disabled', true).text('Sending...');
        
        // Get nonce from form field as fallback
        var nonceFromForm = $('input[name="authyo_otp_auth_login_nonce"]').val();

        $.post(authyoOtpAuth.ajaxurl, {
            action: 'authyo_otp_auth_resend_fallback',
            email: email,
            method: method,
            action_type: 'login',
            nonce: authyoOtpAuth.nonce,
            authyo_otp_auth_login_nonce: nonceFromForm || authyoOtpAuth.nonce
        }, function(response) {
            $btn.prop('disabled', false).text(method.charAt(0).toUpperCase() + method.slice(1));
            if (response.success) {
                showMessage(response.data.message, false);
                $('#login-mask-id').val(response.data.maskId);
                startTimer();
                // Update to show this method as used
                initialMethod = response.data.method || method;
            } else {
                showMessage(response.data.message, true);
            }
        }).fail(function(jqXHR, textStatus, errorThrown) {
            debugLog('AJAX request failed', {
                status: textStatus,
                error: errorThrown,
                response: jqXHR.responseText
            });
            $btn.prop('disabled', false).text(method.charAt(0).toUpperCase() + method.slice(1));
            showMessage('Connection error. Please check your internet connection and try again.', true);
        });
    });

    function updateTimerDisplay() {
        var minutes = Math.floor(timeLeft / 60);
        var seconds = timeLeft % 60;
        $resendOtpBtn.find('.timer').text(authyoOtpAuth.messages.resend + ' (' + minutes + ':' + (seconds < 10 ? '0' : '') + seconds + 's)');
    }

    function validateEmail(email) {
        var re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }

    function validateMobile(mobile) {
        // Remove any non-digit characters except +
        mobile = mobile.replace(/[^\d+]/g, '');
        
        // Check if mobile number has at least 10 digits after +
        if (mobile.startsWith('+')) {
            return mobile.length >= 11 && mobile.length <= 16 && /^\d+$/.test(mobile.substring(1));
        }
        
        // If no + prefix, check if it has 7-15 digits (global mobile format support)
        return mobile.length >= 7 && mobile.length <= 15 && /^\d+$/.test(mobile);
    }

    // Fetch country list
    function fetchCountryList() {
        $.post(authyoOtpAuth.ajaxurl, {
            action: 'authyo_otp_auth_get_country_list',
            nonce: authyoOtpAuth.nonce
        }, function(response) {
            if (response.success && response.data.countries) {
                countriesList = response.data.countries;
                populateCountrySelect();
            }
        });
    }

    // Populate country select dropdown
    function populateCountrySelect() {
        if (countriesList.length === 0) return;
        
        $countrySelect.empty();
        $countrySelect.append('<option value="">Select Country</option>');
        
        $.each(countriesList, function(index, country) {
            $countrySelect.append('<option value="' + country.countryCode + '" data-code="' + country.countryCode + '" data-name="' + country.name + '">' + country.name + ' (' + country.countryCode + ')</option>');
        });
    }

    // Show/hide country dropdown based on input type
    function toggleCountryDropdown(value) {
        var isEmail = validateEmail(value);
        var isMobile = validateMobile(value);
        
        // Check if input contains letters (likely email or username)
        var hasLetters = /[a-zA-Z]/.test(value);
        
        // Check if input looks like email prefix (letters followed by digits, e.g. "user123")
        var looksLikeEmailPrefix = hasLetters && /\d/.test(value) && !value.includes('@') && /[a-zA-Z]/.test(value);
        
        // Check if input looks like complete email
        var looksLikeEmail = value.includes('@') || 
                             (value.length >= 5 && /\w+\.\w+/i.test(value)) ||
                             /\w+@/i.test(value);
        
        var hasDigits = /\d/.test(value);
        var hasDigitsOnly = /^[\d\s\+\-\(\)]+$/.test(value); // Only digits, spaces, +, -, (, )
        
        // Show dropdown only if:
        // 1. It's a valid mobile number with no letters, OR
        // 2. It has digits only (no letters) and length >= 3, OR
        // 3. It starts with + followed by digits
        // BUT NOT if it looks like an email prefix
        if (looksLikeEmail || looksLikeEmailPrefix) {
            // Don't show for email
            $countrySelect.hide();
        } else if (isMobile || (hasDigits && hasDigitsOnly && value.length >= 3)) {
            // Show for phone number patterns
            $countrySelect.show();
        } else {
            // Hide otherwise
            $countrySelect.hide();
        }
    }

    // Handle country selection
    $countrySelect.on('change', function() {
        selectedCountry = $(this).val();
        var countryText = $countrySelect.find('option:selected').text();
        
        if (selectedCountry) {
            // Remove any existing country code from input, keep only the phone number
            var currentValue = $email.val();
            // Remove country codes like "+91 ", "+1 ", etc.
            var withoutCountry = currentValue.replace(/^\+\d+\s*/, '');
            $email.val(withoutCountry);
        }
        
        // Trigger input event to update preview
        $email.trigger('input');
    });

    // Show/hide number preview message
    $email.on('input', function() {
        var value = $(this).val().trim();
        var isEmail = validateEmail(value);
        var isMobile = validateMobile(value);
        
        // Toggle country dropdown visibility
        toggleCountryDropdown(value);
        
        if (isMobile) {
            if (selectedCountry) {
                var cleanMobile = value.replace(/\D/g, '');
                var countryCode = selectedCountry.replace(/[^\d]/g, '');
                $numberPreview.show().text('Will send to: +' + countryCode + cleanMobile);
            } else {
                $numberPreview.show().text('Please select a country for SMS delivery');
            }
        } else if (isEmail) {
            $numberPreview.hide();
        } else {
            $numberPreview.hide();
        }

        // Ensure fallback methods are hidden for emails and non-mobile inputs
        if (isEmail || !isMobile) {
            $form.find('.resend-otp-section').addClass('authyo-hidden').hide();
            $alternativeMethodsSection.addClass('authyo-hidden').hide();
        }
    });

    // Fetch country list on page load
    fetchCountryList();

    $sendOtpBtn.on('click', function() {
        var $btn = $(this);
        
        // Prevent multiple clicks while request is in progress
        if ($btn.prop('disabled')) {
            return;
        }
        
        var email = $email.val().trim();
        
        if (!email) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        var isEmail = validateEmail(email);
        var isMobile = validateMobile(email);
        
        if (!isEmail && !isMobile) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        // Process mobile number - add country code from dropdown if selected
        if (isMobile) {
            var cleanMobile = email.replace(/[^\d]/g, ''); // Remove all non-digits
            var originalValue = $email.val();
            
            // Check if country is selected
            if (!selectedCountry) {
                // If no country selected, check if input already has country code
                if (originalValue.startsWith('+')) {
                    // Input already has country code, use it
                    email = originalValue.replace(/[^\d+]/g, '');
                } else {
                    // No country selected and no country code in input - show error
                    showMessage('Please select a country for your mobile number to enable SMS delivery.', true);
                    return;
                }
            } else {
                // Country is selected, use country code
                var countryCode = selectedCountry.replace(/[^\d]/g, ''); // Extract digits from country code
                email = '+' + countryCode + cleanMobile;
            }
        }

        $btn.prop('disabled', true).text(authyoOtpAuth.messages.sending);

        // Get nonce from form field as fallback
        var nonceFromForm = $('input[name="authyo_otp_auth_login_nonce"]').val();

        $.post(authyoOtpAuth.ajaxurl, {
            action: 'authyo_otp_auth_send_login_otp',
            email: email,
            nonce: authyoOtpAuth.nonce,
            authyo_otp_auth_login_nonce: nonceFromForm || authyoOtpAuth.nonce
        }, function(response) {
            $btn.prop('disabled', false).text(authyoOtpAuth.messages.login);
            if (response.success) {
                // Check if this is a bypass message
                if (response.data.message && response.data.message.includes('bypassed for testing')) {
                    showMessage('Testing Mode: OTP verification bypassed for this test number. You can enter any 6-digit code to proceed.', false);
                } else {
                    showMessage(response.data.message, false);
                }
                $('#login-mask-id').val(response.data.maskId);
                toggleOTPFields(true);
                
                // CRITICAL: Hide fallback methods immediately for email, keep hidden for mobile during timer
                if (isEmail) {
                    // Email: Hide fallback methods completely with class AND inline style - they should NEVER show for email
                    $('.resend-otp-section').addClass('authyo-hidden').hide();
                } else {
                    // Mobile: Keep hidden during timer, will show after timer completes in showResendMethods()
                    $('.resend-otp-section').addClass('authyo-hidden').hide();
                }
                
                // Hide alternative methods section if it was shown
                $alternativeMethodsSection.addClass('authyo-hidden').hide();
                
                // Use the actual method from the response
                var method = response.data.method || (isEmail ? 'email' : 'whatsapp');
                // Store the method for later use when timer completes
                initialMethod = method;
            } else {
                var errorMessage = response.data.message || 'Failed to send OTP';
                showMessage(errorMessage, true);
                
                // Check if this is a "not supported" error for SMS
                if (isMobile && errorMessage.toLowerCase().includes('not supported') && 
                    (errorMessage.toLowerCase().includes('sms') || errorMessage.toLowerCase().includes('channel'))) {
                    // Show alternative methods section immediately so user can choose alternative channel
                    debugLog('SMS not supported - showing alternative methods immediately');
                    
                    // Show the alternative methods section
                    $alternativeMethodsSection.removeClass('authyo-hidden').show();
                    
                    // Enable all alternative method buttons
                    $alternativeMethodsSection.find('.alternative-method-btn').prop('disabled', false);
                } else {
                    // Hide alternative methods section for other errors
                    $alternativeMethodsSection.addClass('authyo-hidden').hide();
                }
            }
        });
    });

    $verifyOtpBtn.on('click', function() {
        var email = $email.val().trim();
        var otp = $('#login-otp').val().trim();
        var maskId = $('#login-mask-id').val().trim();

        if (!email || !otp || !maskId) {
            showMessage(authyoOtpAuth.messages.all_fields_required, true);
            return;
        }

        // Process mobile number - add country code from dropdown if selected
        var isMobile = validateMobile(email);
        if (isMobile) {
            var cleanMobile = email.replace(/[^\d]/g, ''); // Remove all non-digits
            var originalValue = $email.val();
            
            // Check if country is selected
            if (!selectedCountry) {
                // If no country selected, check if input already has country code
                if (originalValue.startsWith('+')) {
                    // Input already has country code, use it
                    email = originalValue.replace(/[^\d+]/g, '');
                } else {
                    // No country selected and no country code in input - show error
                    showMessage('Please select a country for your mobile number to enable SMS delivery.', true);
                    return;
                }
            } else {
                // Country is selected, use country code
                var countryCode = selectedCountry.replace(/[^\d]/g, ''); // Extract digits from country code
                email = '+' + countryCode + cleanMobile;
            }
        }

        var $btn = $(this);
        $btn.prop('disabled', true).text(authyoOtpAuth.messages.verifying);

        // Get nonce from form field as fallback
        var nonceFromForm = $('input[name="authyo_otp_auth_login_nonce"]').val();

        $.post(authyoOtpAuth.ajaxurl, {
            action: 'authyo_otp_auth_login',
            email: email,
            otp: otp,
            maskId: maskId,
            _wpnonce: authyoOtpAuth.nonce,
            authyo_otp_auth_login_nonce: nonceFromForm || authyoOtpAuth.nonce
        }, function(response) {
            $btn.prop('disabled', false).text(authyoOtpAuth.messages.login);
            if (response.success) {
                showMessage(response.data.message, false);
                window.location.href = response.data.redirect; // Redirect on success
            } else {
                showMessage(response.data.message, true);
            }
        });
    });

    $resendOtpBtn.on('click', function() {
        var email = $email.val().trim();

        if (!email) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        // Process mobile number - add country code from dropdown if selected
        var isMobile = validateMobile(email);
        if (isMobile) {
            var cleanMobile = email.replace(/[^\d]/g, ''); // Remove all non-digits
            var originalValue = $email.val();
            
            // Check if country is selected
            if (!selectedCountry) {
                // If no country selected, check if input already has country code
                if (originalValue.startsWith('+')) {
                    // Input already has country code, use it
                    email = originalValue.replace(/[^\d+]/g, '');
                } else {
                    // No country selected and no country code in input - show error
                    showMessage('Please select a country for your mobile number to enable SMS delivery.', true);
                    return;
                }
            } else {
                // Country is selected, use country code
                var countryCode = selectedCountry.replace(/[^\d]/g, ''); // Extract digits from country code
                email = '+' + countryCode + cleanMobile;
            }
        }

        var $btn = $(this);
        $btn.prop('disabled', true).find('.timer').text(authyoOtpAuth.messages.sending);

        // Get nonce from form field as fallback
        var nonceFromForm = $('input[name="authyo_otp_auth_login_nonce"]').val();

        $.post(authyoOtpAuth.ajaxurl, {
            action: 'authyo_otp_auth_send_login_otp',
            email: email,
            nonce: authyoOtpAuth.nonce,
            authyo_otp_auth_login_nonce: nonceFromForm || authyoOtpAuth.nonce
        }, function(response) {
            $btn.prop('disabled', false);
            if (response.success) {
                showMessage(response.data.message, false);
                // Start timer again for resend
                startTimer();
                // Update method tracking (fallback will show when timer completes)
                if (response.data.method) {
                    initialMethod = response.data.method;
                }
            } else {
                showMessage(response.data.message, true);
                // Reset resend button text if there's an error
                $btn.find('.timer').text(authyoOtpAuth.messages.resend);
            }
        });
    });
}); 