/**
 * Deactivation Feedback Handler
 * Intercepts plugin deactivation and shows feedback popup
 */
(function($) {
    'use strict';
    
    var $modal, $form, $otherContainer, $detailsField, $deactivateLink, originalHref;
    
    $(document).ready(function() {
        // Initialize modal elements
        $modal = $('#authyo-deactivation-feedback-modal');
        $form = $('#authyo-deactivation-feedback-form');
        $otherContainer = $('#authyo-feedback-other-container');
        $detailsField = $('#authyo-feedback-details');
        
        // Find the deactivate link for our plugin
        var pluginSlug = authyoDeactivationFeedback.pluginSlug;
        
        // Try multiple methods to find the plugin row
        var pluginRow = $('tr[data-plugin="' + pluginSlug + '"]');
        
        if (pluginRow.length === 0) {
            // Try finding by plugin basename in various places
            pluginRow = $('tr').filter(function() {
                var $row = $(this);
                // Check if row contains our plugin slug
                return $row.find('td.plugin-title strong').text().toLowerCase().indexOf('authyo') !== -1 ||
                       $row.find('td').text().indexOf('authyo') !== -1 ||
                       $row.attr('id') && $row.attr('id').indexOf(pluginSlug.replace(/\//g, '-')) !== -1;
            }).first();
        }
        
        if (pluginRow.length > 0) {
            // Find deactivate link - try multiple selectors
            $deactivateLink = pluginRow.find('span.deactivate a, .deactivate a, a[href*="action=deactivate"], td.column-description a[href*="deactivate"]');
            
            if ($deactivateLink.length === 0) {
                // Last resort: find any link in the row that contains "deactivate" in href
                $deactivateLink = pluginRow.find('a').filter(function() {
                    return $(this).attr('href') && $(this).attr('href').indexOf('action=deactivate') !== -1;
                });
            }
            
            if ($deactivateLink.length > 0) {
                originalHref = $deactivateLink.attr('href');
                
                // Intercept click on deactivate link
                $deactivateLink.on('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    showModal();
                });
            }
        }
        
        // Handle radio button changes
        $form.find('input[name="reason"]').on('change', function() {
            var $option = $(this).closest('.authyo-feedback-option');
            
            // Remove checked state from all options
            $form.find('.authyo-feedback-option').removeClass('authyo-feedback-option-checked');
            
            // Add checked state to current option (for browsers without :has() support)
            $option.addClass('authyo-feedback-option-checked');
            
            if ($(this).val() === 'other') {
                $otherContainer.slideDown(200);
                $detailsField.focus();
            } else {
                $otherContainer.slideUp(200);
                $detailsField.val('');
            }
        });
        
        // Handle form submission
        $form.on('submit', function(e) {
            e.preventDefault();
            submitFeedback();
        });
        
        // Handle skip button
        $('.authyo-feedback-skip').on('click', function() {
            submitFeedback('skipped');
        });
        
        // Handle close button
        $('.authyo-feedback-modal-close, .authyo-feedback-modal-overlay').on('click', function() {
            // If clicking overlay, only close if clicking the overlay itself
            if ($(this).hasClass('authyo-feedback-modal-overlay')) {
                closeModal();
            } else if ($(this).hasClass('authyo-feedback-modal-close')) {
                // If skip was not clicked, submit skipped feedback
                submitFeedback('skipped');
            }
        });
        
        // Prevent modal content clicks from closing modal
        $('.authyo-feedback-modal-content').on('click', function(e) {
            e.stopPropagation();
        });
    });
    
    /**
     * Show the feedback modal
     */
    function showModal() {
        $modal.fadeIn(300);
        $form.show();
        $('#authyo-feedback-thank-you').hide();
        $form[0].reset();
        $otherContainer.hide();
        $detailsField.val('');
    }
    
    /**
     * Close the feedback modal
     */
    function closeModal() {
        $modal.fadeOut(300);
    }
    
    /**
     * Submit feedback to server
     */
    function submitFeedback(reasonOverride) {
        var reason = reasonOverride || $form.find('input[name="reason"]:checked').val();
        var details = $detailsField.val().trim();
        
        // Validate if "other" is selected
        if (reason === 'other' && !reasonOverride && !details) {
            alert(authyoDeactivationFeedback.messages?.details_required || 'Please provide your feedback in the text area.');
            $detailsField.focus();
            return;
        }
        
        // Disable form during submission
        $form.find('button').prop('disabled', true);
        
        // Prepare data
        var data = {
            action: 'authyo_otp_auth_submit_deactivation_feedback',
            nonce: authyoDeactivationFeedback.nonce,
            reason: reason || 'skipped',
            details: details || '',
            pageurl: window.location.href
        };
        
        // Send AJAX request
        $.ajax({
            url: authyoDeactivationFeedback.ajaxurl,
            type: 'POST',
            data: data,
            success: function(response) {
                if (response.success) {
                    // Show thank you message
                    $form.hide();
                    $('#authyo-feedback-thank-you').fadeIn(300);
                    
                    // Wait a moment then proceed with deactivation
                    setTimeout(function() {
                        closeModal();
                        // Proceed with original deactivation
                        if (originalHref) {
                            window.location.href = originalHref;
                        }
                    }, 1500);
                } else {
                    // Even on error, proceed with deactivation
                    alert(response.data?.message || 'Error submitting feedback. Proceeding with deactivation...');
                    closeModal();
                    if (originalHref) {
                        window.location.href = originalHref;
                    }
                }
            },
            error: function() {
                // Even on error, proceed with deactivation
                alert('Error submitting feedback. Proceeding with deactivation...');
                closeModal();
                if (originalHref) {
                    window.location.href = originalHref;
                }
            }
        });
    }
    
})(jQuery);

