/**
 * Checkout OTP JavaScript - Same flow as Login/Register forms
 */

jQuery(document).ready(function ($) {
    'use strict';

    // Prevent multiple initialization
    if (window.authyoCheckoutInitialized) {
        return;
    }
    window.authyoCheckoutInitialized = true;

    // Check if checkout OTP object is defined
    if (typeof authyoOtpCheckout === 'undefined') {
        return;
    }

    // Function to remove duplicate OTP buttons
    function removeDuplicateButtons() {
        // Remove duplicate phone OTP buttons - keep only the first one
        var $phoneButtons = $('#authyo_send_phone_otp');
        if ($phoneButtons.length > 1) {
            // Keep the first button, remove all others
            $phoneButtons.slice(1).each(function () {
                var $duplicate = $(this);
                // Try to find and remove the wrapper or input group containing the duplicate
                var $wrapper = $duplicate.closest('#authyo-checkout-otp-phone-wrapper');
                if ($wrapper.length > 0 && $wrapper.siblings('#authyo-checkout-otp-phone-wrapper').length > 0) {
                    // If there are multiple wrappers, remove the duplicate wrapper
                    $wrapper.remove();
                } else {
                    // Otherwise, just remove the duplicate button
                    $duplicate.remove();
                }
            });
        }

        // Remove duplicate email OTP buttons - keep only the first one
        var $emailButtons = $('#authyo_send_email_otp');
        if ($emailButtons.length > 1) {
            $emailButtons.slice(1).each(function () {
                var $duplicate = $(this);
                var $wrapper = $duplicate.closest('#authyo-checkout-otp-email-wrapper');
                if ($wrapper.length > 0 && $wrapper.siblings('#authyo-checkout-otp-email-wrapper').length > 0) {
                    $wrapper.remove();
                } else {
                    $duplicate.remove();
                }
            });
        }
    }

    // Remove any existing duplicates immediately
    removeDuplicateButtons();

    // Set up periodic duplicate cleanup (safety net)
    setInterval(function () {
        var $emailBtns = $('#authyo_send_email_otp');
        var $phoneBtns = $('#authyo_send_phone_otp');
        if ($emailBtns.length > 1 || $phoneBtns.length > 1) {
            removeDuplicateButtons();
        }
    }, 2000); // Check every 2 seconds

    // Function to inject OTP fields
    function injectOtpFields() {
        // First, ensure country dropdown is visible even if still in hidden container
        var $countrySelect = $('#checkout-country-code');
        if ($countrySelect.length > 0) {
            // Make it visible immediately, even if still in hidden container
            $countrySelect.css({
                'display': 'block !important',
                'visibility': 'visible !important',
                'opacity': '1 !important',
                'height': 'auto !important',
                'position': 'relative !important',
                'z-index': '9999 !important'
            }).show().removeClass('authyo-hidden');
        }

        var $container = $('#authyo-checkout-otp-container');
        if ($container.length === 0) {
            return false;
        }

        var injected = false;

        // Inject email OTP field directly after email field
        if (authyoOtpCheckout.emailEnabled) {
            var $emailWrapper = $('#authyo-checkout-otp-email-wrapper');
            if ($emailWrapper.length > 0 && $emailWrapper.closest('#authyo-checkout-otp-container').length > 0) {
                // Try multiple selectors to find email field (classic and block-based checkout, all themes)
                var $emailField = $(
                    'input[name="billing_email"], ' +
                    '#billing_email, ' +
                    'input#billing_email, ' +
                    '#email, ' +
                    'input#email, ' +
                    'input[type="email"][id="email"], ' +
                    'input[type="email"][name="email"], ' +
                    '.woocommerce-billing-fields input[type="email"], ' +
                    '.wc-block-components-text-input input[type="email"], ' +
                    '.wc-block-components-address-form__email input[type="email"], ' +
                    '#contact input[type="email"], ' +
                    'form.checkout input[type="email"], ' +
                    'form.woocommerce-checkout input[type="email"]'
                );
                if ($emailField.length > 0) {
                    // Check if it's block-based checkout (WooCommerce Blocks)
                    var $blockEmailContainer = $emailField.closest('.wc-block-components-text-input, .wc-block-components-address-form__email');
                    if ($blockEmailContainer.length > 0) {
                        // Block-based checkout: inject after the text input container
                        $blockEmailContainer.after($emailWrapper.detach());
                        $emailWrapper.css({
                            'display': 'block',
                            'visibility': 'visible',
                            'position': 'static',
                            'left': 'auto',
                            'height': 'auto',
                            'overflow': 'visible',
                            'margin-top': '10px'
                        });
                        injected = true;
                    } else {
                        // Classic checkout: Try to find the form row
                        var $emailFormRow = $emailField.closest('.form-row, .woocommerce-billing-fields__field-wrapper, p.form-row');
                        if ($emailFormRow.length > 0) {
                            // Move the wrapper after email form row
                            $emailFormRow.after($emailWrapper.detach());
                            $emailWrapper.css({
                                'display': 'block',
                                'visibility': 'visible',
                                'position': 'static',
                                'left': 'auto',
                                'height': 'auto',
                                'overflow': 'visible'
                            });
                            injected = true;
                        } else {
                            // Fallback: try to find parent container
                            var $emailParent = $emailField.closest('div, fieldset, p, .woocommerce-billing-fields__field-wrapper, #contact');
                            if ($emailParent.length > 0) {
                                $emailParent.after($emailWrapper.detach());
                                $emailWrapper.css({
                                    'display': 'block',
                                    'visibility': 'visible',
                                    'position': 'static',
                                    'left': 'auto',
                                    'height': 'auto',
                                    'overflow': 'visible'
                                });
                                injected = true;
                            }
                        }
                    }
                }
            }
        }

        // Inject phone OTP field directly after phone field
        if (authyoOtpCheckout.whatsappEnabled || authyoOtpCheckout.smsEnabled || authyoOtpCheckout.voiceEnabled) {
            var $phoneWrapper = $('#authyo-checkout-otp-phone-wrapper');
            if ($phoneWrapper.length > 0 && $phoneWrapper.closest('#authyo-checkout-otp-container').length > 0) {
                // Try multiple selectors to find phone field - support both classic and block-based checkout, all themes
                // Priority: block-based checkout first (most specific)
                var $phoneField = $(
                    '#billing-phone, ' +
                    'input#billing-phone, ' +
                    'input[type="tel"][id="billing-phone"], ' +
                    'input[type="tel"][name="billing_phone"], ' +
                    '.wc-block-components-address-form__phone input[type="tel"], ' +
                    '.wc-block-components-text-input input[type="tel"], ' +
                    'input[name="billing_phone"], ' +
                    '#billing_phone, ' +
                    'input#billing_phone, ' +
                    '.woocommerce-billing-fields input[type="tel"], ' +
                    'form.checkout input[type="tel"], ' +
                    'form.woocommerce-checkout input[type="tel"], ' +
                    '.billing_phone_field input[type="tel"], ' +
                    '.billing-phone-field input[type="tel"]'
                );

                if ($phoneField.length > 0) {
                    // Check if it's block-based checkout - look for the container div
                    var $blockPhoneContainer = $phoneField.closest('.wc-block-components-text-input.wc-block-components-address-form__phone, .wc-block-components-address-form__phone, .wc-block-components-text-input');
                    if ($blockPhoneContainer.length > 0) {
                        // Block-based checkout - inject country dropdown inside the phone container
                        var $countrySelect = $('#checkout-country-code');

                        // Move country dropdown to be inside the phone field container (like login form)
                        if ($countrySelect.length > 0) {
                            // Check if country select is already in the phone container
                            if ($blockPhoneContainer.find('#checkout-country-code').length === 0) {
                                // Create a phone-input-wrapper div similar to login form
                                var $phoneInputWrapper = $blockPhoneContainer.find('.phone-input-wrapper');
                                if ($phoneInputWrapper.length === 0) {
                                    // Get the label first (keep it in place)
                                    var $label = $blockPhoneContainer.find('label');

                                    // Create wrapper with grid layout (25% country, 75% phone)
                                    $phoneInputWrapper = $('<div class="phone-input-wrapper" style="display: grid; grid-template-columns: 25% 75%; gap: 10px; align-items: stretch; width: 100%; margin-top: 8px; margin-bottom: 10px; clear: both;"></div>');

                                    // Move country select first into wrapper (25% width)
                                    $countrySelect.detach().prependTo($phoneInputWrapper);
                                    // Populate country dropdown after moving
                                    populateCountrySelect();
                                    $countrySelect.css({
                                        'display': 'block !important',
                                        'visibility': 'visible !important',
                                        'opacity': '1 !important',
                                        'width': '100%',
                                        'min-width': '0',
                                        'max-width': '100%',
                                        'height': 'auto'
                                    }).show().removeClass('authyo-hidden');

                                    // Move phone input into wrapper (75% width)
                                    $phoneField.detach().appendTo($phoneInputWrapper);
                                    $phoneField.css({
                                        'width': '100%',
                                        'min-width': '0',
                                        'max-width': '100%'
                                    });

                                    // Add wrapper to container (after label if exists, otherwise at the end)
                                    if ($label.length > 0) {
                                        // Hide the label when phone OTP validation is enabled
                                        if (authyoOtpCheckout.whatsappEnabled || authyoOtpCheckout.smsEnabled || authyoOtpCheckout.voiceEnabled) {
                                            $label.hide();
                                        } else {
                                            // Ensure label has proper spacing to prevent overlap with country dropdown
                                            $label.css({
                                                'display': 'block',
                                                'margin-bottom': '8px',
                                                'clear': 'both',
                                                'width': '100%'
                                            });
                                        }
                                        $label.after($phoneInputWrapper);
                                    } else {
                                        $blockPhoneContainer.append($phoneInputWrapper);
                                    }
                                } else {
                                    // Wrapper exists, just add country select if not already there
                                    if ($phoneInputWrapper.find('#checkout-country-code').length === 0) {
                                        $countrySelect.detach().prependTo($phoneInputWrapper);
                                        // Populate country dropdown after moving
                                        populateCountrySelect();
                                        $countrySelect.css({
                                            'display': 'block',
                                            'visibility': 'visible',
                                            'width': 'auto',
                                            'min-width': '150px',
                                            'flex-shrink': '0',
                                            'height': 'auto'
                                        });
                                    }
                                }
                            }

                            // Always show country dropdown for phone field with strong CSS rules
                            $countrySelect.css({
                                'display': 'block !important',
                                'visibility': 'visible !important',
                                'opacity': '1 !important',
                                'height': 'auto !important'
                            }).show().removeClass('authyo-hidden');

                            // Ensure the phone number preview appears directly below the phone-input-wrapper
                            var $preview = $('#checkout-number-preview');
                            if ($preview.length > 0 && $phoneInputWrapper && $phoneInputWrapper.length > 0) {
                                $preview.detach().insertAfter($phoneInputWrapper);
                                $preview.css({
                                    'display': 'block',
                                    'margin-top': '4px',
                                    'margin-bottom': '8px'
                                });
                            }
                        }

                        // Now inject OTP wrapper after the container (which contains phone-input-wrapper)
                        // This ensures it appears directly below the phone-input-wrapper
                        $blockPhoneContainer.after($phoneWrapper.detach());
                        $phoneWrapper.css({
                            'display': 'block',
                            'visibility': 'visible',
                            'position': 'static',
                            'left': 'auto',
                            'height': 'auto',
                            'overflow': 'visible',
                            'margin-top': '10px',
                            'width': '100%',
                            'clear': 'both'
                        });

                        // Ensure the phone-input-wrapper inside OTP section is hidden
                        $phoneWrapper.find('.phone-input-wrapper').css({
                            'display': 'none !important',
                            'visibility': 'hidden !important',
                            'height': '0',
                            'margin': '0',
                            'padding': '0',
                            'overflow': 'hidden'
                        });

                        // Ensure the button is visible
                        var $sendBtn = $phoneWrapper.find('#authyo_send_phone_otp');
                        if ($sendBtn.length > 0) {
                            $sendBtn.css({
                                'display': 'inline-block',
                                'visibility': 'visible',
                                'opacity': '1'
                            }).show();
                        }

                        injected = true;
                    } else {
                        // Classic checkout - try to find the form row
                        var $phoneFormRow = $phoneField.closest('.form-row, .woocommerce-billing-fields__field-wrapper, p.form-row');
                        if ($phoneFormRow.length > 0) {
                            // Move the wrapper after phone form row
                            $phoneFormRow.after($phoneWrapper.detach());
                            $phoneWrapper.css({
                                'display': 'block',
                                'visibility': 'visible',
                                'position': 'static',
                                'left': 'auto',
                                'height': 'auto',
                                'overflow': 'visible',
                                'margin-top': '10px'
                            });
                            injected = true;
                        } else {
                            // Fallback: try to find parent container
                            var $phoneParent = $phoneField.closest('div, fieldset, p, .woocommerce-billing-fields__field-wrapper');
                            if ($phoneParent.length > 0) {
                                $phoneParent.after($phoneWrapper.detach());
                                $phoneWrapper.css({
                                    'display': 'block',
                                    'visibility': 'visible',
                                    'position': 'static',
                                    'left': 'auto',
                                    'height': 'auto',
                                    'overflow': 'visible',
                                    'margin-top': '10px'
                                });
                                injected = true;
                            }
                        }
                    }
                }
            }
        }

        // Remove the hidden container if all fields are moved
        if ($container.length > 0 && ($container.find('#authyo-checkout-otp-email-wrapper').length === 0 && $container.find('#authyo-checkout-otp-phone-wrapper').length === 0)) {
            $container.remove();
        }

        // Remove any duplicates after injection - especially for email buttons
        removeDuplicateButtons();

        // Additional cleanup for email buttons specifically
        var $emailButtons = $('#authyo_send_email_otp');
        if ($emailButtons.length > 1) {
            $emailButtons.slice(1).each(function () {
                var $duplicate = $(this);
                var $wrapper = $duplicate.closest('#authyo-checkout-otp-email-wrapper');
                if ($wrapper.length > 0 && $wrapper.siblings('#authyo-checkout-otp-email-wrapper').length > 0) {
                    $wrapper.remove();
                } else {
                    $duplicate.closest('.authyo-checkout-otp-email-section, .authyo-checkout-otp-input-group, div').remove();
                }
            });
        }

        // Ensure buttons are visible and properly styled
        $('#authyo_send_email_otp, #authyo_send_phone_otp').each(function () {
            var $btn = $(this);
            if ($btn.length > 0) {
                $btn.css({
                    'display': 'inline-block !important',
                    'visibility': 'visible !important',
                    'opacity': '1',
                    'position': 'static'
                }).show();

                // Force visibility
                if ($btn.is(':hidden') || $btn.css('display') === 'none') {
                    $btn.css({
                        'display': 'inline-block !important',
                        'visibility': 'visible !important'
                    }).show();
                }
            }
        });

        // Also ensure the phone OTP wrapper and input group are visible
        var $phoneWrapper = $('#authyo-checkout-otp-phone-wrapper');
        if ($phoneWrapper.length > 0) {
            $phoneWrapper.css({
                'display': 'block !important',
                'visibility': 'visible !important',
                'clear': 'both',
                'width': '100%'
            }).show();

            // Ensure phone-input-wrapper inside OTP section is hidden
            $phoneWrapper.find('.phone-input-wrapper').css({
                'display': 'none !important',
                'visibility': 'hidden !important',
                'height': '0',
                'margin': '0',
                'padding': '0',
                'overflow': 'hidden'
            }).hide();

            var $phoneInputGroup = $phoneWrapper.find('.authyo-checkout-otp-input-group');
            if ($phoneInputGroup.length > 0) {
                $phoneInputGroup.css({
                    'display': 'flex !important',
                    'visibility': 'visible !important'
                }).show();
            }
        }

        return injected;
    }

    // Fallback: Create fields directly if container doesn't exist
    function createOtpFieldsFallback() {
        // Check if fields already exist and are properly placed (not in hidden container)
        var $existingEmail = $('#authyo-checkout-otp-email-wrapper');
        var $existingPhone = $('#authyo-checkout-otp-phone-wrapper');

        // Only create if fields don't exist OR if they're still in the hidden container
        var emailNeedsCreation = !($existingEmail.length > 0 && $existingEmail.closest('#authyo-checkout-otp-container').length === 0);
        var phoneNeedsCreation = !($existingPhone.length > 0 && $existingPhone.closest('#authyo-checkout-otp-container').length === 0);

        // Remove any duplicate buttons that might exist
        var $duplicateEmailButtons = $('#authyo_send_email_otp');
        if ($duplicateEmailButtons.length > 1) {
            // Keep only the first one, remove the rest
            $duplicateEmailButtons.slice(1).each(function () {
                $(this).closest('.authyo-checkout-otp-email-section, .authyo-checkout-otp-input-group, div').remove();
            });
            // Re-check after cleanup
            $existingEmail = $('#authyo-checkout-otp-email-wrapper');
            emailNeedsCreation = !($existingEmail.length > 0 && $existingEmail.closest('#authyo-checkout-otp-container').length === 0);
        }

        var $duplicatePhoneButtons = $('#authyo_send_phone_otp');
        if ($duplicatePhoneButtons.length > 1) {
            // Keep only the first one, remove the rest
            $duplicatePhoneButtons.slice(1).each(function () {
                $(this).closest('.authyo-checkout-otp-phone-section, .authyo-checkout-otp-input-group, div').remove();
            });
            // Re-check after cleanup
            $existingPhone = $('#authyo-checkout-otp-phone-wrapper');
            phoneNeedsCreation = !($existingPhone.length > 0 && $existingPhone.closest('#authyo-checkout-otp-container').length === 0);
        }

        // If both fields are already properly placed, don't create anything
        if (!emailNeedsCreation && !phoneNeedsCreation) {
            return;
        }

        var emailHtml = '';
        var phoneHtml = '';

        if (authyoOtpCheckout.emailEnabled && emailNeedsCreation) {
            // Check if button already exists before creating
            if ($('#authyo_send_email_otp').length > 0 && $('#authyo-checkout-otp-email-wrapper').closest('#authyo-checkout-otp-container').length === 0) {
                // Email OTP button already exists and is properly placed
            } else {
                var otpLength = authyoOtpCheckout.otpLength || 6;
                emailHtml = '<div id="authyo-checkout-otp-email-wrapper" class="authyo-checkout-otp-email-section form-row" style="display: block !important; visibility: visible !important; margin-top: 10px;">' +
                    '<div class="authyo-checkout-otp-input-group" style="display: flex; gap: 10px; align-items: center; margin-top: 5px;">' +
                    '<input type="text" id="authyo_checkout_email_otp" name="authyo_checkout_email_otp" class="input-text" placeholder="Enter OTP sent to your email" autocomplete="off" maxlength="' + otpLength + '" pattern="[0-9]*" inputmode="numeric" style="display: none; flex: 1;" />' +
                    '<button type="button" id="authyo_send_email_otp" class="button authyo-send-otp-btn" style="display: inline-block !important; visibility: visible !important; white-space: nowrap;">Send Email OTP</button>' +
                    '</div>' +
                    '<button type="button" id="authyo_resend_email_otp" class="button button-secondary resend-otp-btn" style="display: none; margin-top: 10px;">' +
                    '<span class="timer">Resend OTP</span>' +
                    '</button>' +
                    '<span id="authyo_email_otp_status" class="authyo-otp-status" style="display: block; margin-top: 5px;"></span>' +
                    '<input type="hidden" id="authyo_checkout_email_maskid" name="authyo_checkout_email_maskid" value="" />' +
                    '<input type="hidden" id="authyo_checkout_email_verified" name="authyo_checkout_email_verified" value="0" />' +
                    '</div>';
            }
        }

        if ((authyoOtpCheckout.whatsappEnabled || authyoOtpCheckout.smsEnabled || authyoOtpCheckout.voiceEnabled) && phoneNeedsCreation) {
            // Check if button already exists before creating
            if ($('#authyo_send_phone_otp').length > 0 && $('#authyo-checkout-otp-phone-wrapper').closest('#authyo-checkout-otp-container').length === 0) {
                // Phone OTP button already exists and is properly placed
            } else {
                var otpLength = authyoOtpCheckout.otpLength || 6;
                
                // Generate fallback methods HTML based on enabled methods (no method excluded initially)
                var fallbackMethodsHtml = generateFallbackMethodsHtml(null);
                // If only 1 method is enabled, create empty hidden section (will be populated later if needed)
                if (!fallbackMethodsHtml) {
                    fallbackMethodsHtml = '<div class="resend-otp-section authyo-hidden" style="display: none !important;"></div>';
                } else {
                    // Add authyo-hidden class to keep it hidden initially
                    fallbackMethodsHtml = fallbackMethodsHtml.replace('<div class="resend-otp-section">', '<div class="resend-otp-section authyo-hidden">');
                }
                
                phoneHtml = '<div id="authyo-checkout-otp-phone-wrapper" class="authyo-checkout-otp-phone-section form-row" style="display: block !important; visibility: visible !important; margin-top: 10px; clear: both; width: 100%;">' +
                    '<div class="phone-input-wrapper" style="display: none !important; visibility: hidden !important; height: 0 !important; margin: 0 !important; padding: 0 !important; overflow: hidden !important;">' +
                    '<select id="checkout-country-code" name="checkout_country_code" class="authyo-country-select" style="display: none;"></select>' +
                    '</div>' +
                    '<span id="checkout-number-preview" class="number-preview" style="display:none; font-size: 12px; color: #666; margin-bottom: 5px;"></span>' +
                    '<div class="authyo-checkout-otp-input-group" style="display: flex; gap: 10px; align-items: center; margin-top: 5px;">' +
                    '<input type="text" id="authyo_checkout_phone_otp" name="authyo_checkout_phone_otp" class="input-text" placeholder="Enter OTP sent to your phone" autocomplete="off" maxlength="' + otpLength + '" pattern="[0-9]*" inputmode="numeric" style="display: none; flex: 1;" />' +
                    '<button type="button" id="authyo_send_phone_otp" class="button authyo-send-otp-btn" style="display: inline-block !important; visibility: visible !important; white-space: nowrap;">Send Phone OTP</button>' +
                    '</div>' +
                    '<button type="button" id="authyo_resend_phone_otp" class="button button-secondary resend-otp-btn" style="display: none; margin-top: 10px;">' +
                    '<span class="timer">Resend OTP</span>' +
                    '</button>' +
                    fallbackMethodsHtml +
                    '<span id="authyo_phone_otp_status" class="authyo-otp-status" style="display: block; margin-top: 5px;"></span>' +
                    '<input type="hidden" id="authyo_checkout_phone_maskid" name="authyo_checkout_phone_maskid" value="" />' +
                    '</div>';
            }
        }

        // Try to inject email field
        if (emailHtml) {
            // Check if button already exists before injecting - prevent duplicates
            if ($('#authyo_send_email_otp').length > 0) {
                var $existingEmailBtn = $('#authyo_send_email_otp').first();
                var $existingEmailWrapper = $existingEmailBtn.closest('#authyo-checkout-otp-email-wrapper');
                if ($existingEmailWrapper.length > 0 && $existingEmailWrapper.closest('#authyo-checkout-otp-container').length === 0) {
                    emailHtml = ''; // Clear emailHtml to prevent duplicate injection
                }
            }

            if (emailHtml) {
                // Try multiple selectors for email field (classic and block-based)
                var $emailField = $('input[name="billing_email"], #billing_email, input#billing_email, #email, input#email, input[type="email"][id="email"]');

                if ($emailField.length > 0) {
                    // Check if it's block-based checkout
                    var $blockEmailContainer = $emailField.closest('.wc-block-components-text-input, .wc-block-components-address-form__email');

                    if ($blockEmailContainer.length > 0) {
                        // Block-based checkout - inject after the text input container
                        $blockEmailContainer.after(emailHtml);

                        // Ensure button is visible and remove duplicates if any
                        setTimeout(function () {
                            var $sendBtn = $('#authyo_send_email_otp');
                            // Remove duplicates if any
                            if ($sendBtn.length > 1) {
                                $sendBtn.slice(1).closest('.authyo-checkout-otp-email-section, .authyo-checkout-otp-input-group, div').remove();
                                $sendBtn = $('#authyo_send_email_otp').first();
                            }
                            $sendBtn.css({
                                'display': 'inline-block',
                                'visibility': 'visible',
                                'opacity': '1'
                            }).show();
                        }, 100);
                    } else {
                        // Classic checkout
                        var $emailFormRow = $emailField.closest('.form-row, .woocommerce-billing-fields__field-wrapper, p.form-row');
                        if ($emailFormRow.length > 0) {
                            $emailFormRow.after(emailHtml);
                        } else {
                            var $emailParent = $emailField.closest('div, fieldset, p, #contact');
                            if ($emailParent.length > 0) {
                                $emailParent.after(emailHtml);
                            }
                        }

                        // Ensure button is visible and remove duplicates if any
                        setTimeout(function () {
                            var $sendBtn = $('#authyo_send_email_otp');
                            // Remove duplicates if any
                            if ($sendBtn.length > 1) {
                                $sendBtn.slice(1).closest('.authyo-checkout-otp-email-section, .authyo-checkout-otp-input-group, div').remove();
                                $sendBtn = $('#authyo_send_email_otp').first();
                            }
                            $sendBtn.css({
                                'display': 'inline-block',
                                'visibility': 'visible',
                                'opacity': '1'
                            }).show();
                        }, 100);
                    }
                }
            }
        }

        // Try to inject phone field
        if (phoneHtml) {
            // Check if button already exists before injecting - prevent duplicates
            if ($('#authyo_send_phone_otp').length > 0) {
                var $existingPhoneBtn = $('#authyo_send_phone_otp').first();
                var $existingPhoneWrapper = $existingPhoneBtn.closest('#authyo-checkout-otp-phone-wrapper');
                if ($existingPhoneWrapper.length > 0 && $existingPhoneWrapper.closest('#authyo-checkout-otp-container').length === 0) {
                    phoneHtml = ''; // Clear phoneHtml to prevent duplicate injection
                }
            }

            if (phoneHtml) {
                // Try multiple selectors to find phone field - support both classic and block-based checkout
                // Priority: block-based checkout first (most specific)
                var $phoneField = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], .wc-block-components-address-form__phone input[type="tel"], input[name="billing_phone"]');

                if ($phoneField.length > 0) {

                    // Check if it's block-based checkout - look for the container div
                    var $blockPhoneContainer = $phoneField.closest('.wc-block-components-text-input.wc-block-components-address-form__phone, .wc-block-components-address-form__phone, .wc-block-components-text-input');
                    if ($blockPhoneContainer.length > 0) {
                        // Block-based checkout - inject after the container
                        $blockPhoneContainer.after(phoneHtml);
                        // Populate country dropdown after injection
                        setTimeout(function () {
                            populateCountrySelect();
                        }, 100);

                        // Ensure button is visible after injection
                        setTimeout(function () {
                            var $sendBtn = $('#authyo_send_phone_otp');
                            var $wrapper = $('#authyo-checkout-otp-phone-wrapper');
                            // Remove duplicates if any
                            if ($sendBtn.length > 1) {
                                $sendBtn.slice(1).closest('.authyo-checkout-otp-phone-section, .authyo-checkout-otp-input-group, div').remove();
                                $sendBtn = $('#authyo_send_phone_otp').first();
                            }
                            if ($sendBtn.length > 0) {
                                $sendBtn.css({
                                    'display': 'inline-block',
                                    'visibility': 'visible',
                                    'opacity': '1'
                                }).show();
                            }
                        }, 100);
                    } else {
                        // Classic checkout
                        var $phoneFormRow = $phoneField.closest('.form-row, .woocommerce-billing-fields__field-wrapper, p.form-row');
                        if ($phoneFormRow.length > 0) {
                            $phoneFormRow.after(phoneHtml);
                        } else {
                            $phoneField.closest('div, fieldset, p').after(phoneHtml);
                        }
                        // Populate country dropdown after injection
                        setTimeout(function () {
                            populateCountrySelect();
                        }, 100);
                    }
                }
            }
        }

        // Add hidden verification fields if they don't exist
        if ($('#authyo_checkout_email_verified').length === 0) {
            $('form.checkout').append('<input type="hidden" id="authyo_checkout_email_verified" name="authyo_checkout_email_verified" value="0" />');
        }
        if ($('#authyo_checkout_phone_verified').length === 0) {
            $('form.checkout').append('<input type="hidden" id="authyo_checkout_phone_verified" name="authyo_checkout_phone_verified" value="0" />');
        }

        // Remove any duplicates after fallback creation
        setTimeout(function () {
            removeDuplicateButtons();
        }, 200);
    }

    // Try to inject OTP fields with multiple attempts
    function tryInjectOtpFields(attempts) {
        attempts = attempts || 0;
        var maxAttempts = 10;

        if (attempts >= maxAttempts) {
            // Try fallback method
            createOtpFieldsFallback();
            return;
        }

        var injected = injectOtpFields();

        // If not injected and container exists, try again
        if (!injected && $('#authyo-checkout-otp-container').length > 0 && attempts < maxAttempts) {
            // Try again after a short delay
            setTimeout(function () {
                tryInjectOtpFields(attempts + 1);
            }, 200);
        } else if (!injected && attempts >= 3) {
            // After 3 attempts, try fallback
            createOtpFieldsFallback();
        }
    }

    // Function to check and inject OTP fields
    function attemptInjection() {
        // Update phone label text
        updatePhoneLabel();
        
        // Check if email field exists (block-based or classic)
        var $emailField = $('#email, input#email, input[type="email"][id="email"], input[name="billing_email"], #billing_email');
        var $existingEmail = $('#authyo-checkout-otp-email-wrapper');
        var emailWrapperExists = $existingEmail.length > 0;
        var emailWrapperPlaced = emailWrapperExists && $existingEmail.closest('#authyo-checkout-otp-container').length === 0;

        // Check if phone field exists (block-based or classic)
        var $phoneField = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], .wc-block-components-address-form__phone input[type="tel"], input[name="billing_phone"]');
        var $existingPhone = $('#authyo-checkout-otp-phone-wrapper');
        var phoneWrapperExists = $existingPhone.length > 0;
        var phoneWrapperPlaced = phoneWrapperExists && $existingPhone.closest('#authyo-checkout-otp-container').length === 0;

        // If email field exists but wrapper is not placed, use fallback immediately
        if ($emailField.length > 0 && authyoOtpCheckout.emailEnabled && !emailWrapperPlaced) {
            createOtpFieldsFallback();
        } else if ($phoneField.length > 0 && (authyoOtpCheckout.whatsappEnabled || authyoOtpCheckout.smsEnabled || authyoOtpCheckout.voiceEnabled) && !phoneWrapperPlaced) {
            createOtpFieldsFallback();
        } else if (!emailWrapperPlaced || !phoneWrapperPlaced) {
            // Try container method as backup
            tryInjectOtpFields(0);
        }
    }

    // Initial injection attempt - try multiple times with increasing delays
    setTimeout(function () {
        attemptInjection();
        // Clean up duplicates after initial injection
        setTimeout(function () {
            removeDuplicateButtons();
        }, 300);
    }, 500);

    // Try again after a longer delay (for slow-loading pages)
    setTimeout(function () {
        var $existingEmail = $('#authyo-checkout-otp-email-wrapper');
        var $existingPhone = $('#authyo-checkout-otp-phone-wrapper');
        if ($existingEmail.length === 0 || $existingEmail.closest('#authyo-checkout-otp-container').length > 0 ||
            $existingPhone.length === 0 || $existingPhone.closest('#authyo-checkout-otp-container').length > 0) {
            attemptInjection();
        }
        // Clean up duplicates after retry
        setTimeout(function () {
            removeDuplicateButtons();
        }, 300);
    }, 2000);

    // Try again after 5 seconds (for very slow pages)
    setTimeout(function () {
        var $existingEmail = $('#authyo-checkout-otp-email-wrapper');
        var $existingPhone = $('#authyo-checkout-otp-phone-wrapper');
        if ($existingEmail.length === 0 || $existingEmail.closest('#authyo-checkout-otp-container').length > 0 ||
            $existingPhone.length === 0 || $existingPhone.closest('#authyo-checkout-otp-container').length > 0) {
            createOtpFieldsFallback();
        }
        // Clean up duplicates after final retry
        setTimeout(function () {
            removeDuplicateButtons();
        }, 300);
    }, 5000);

    // Direct check for phone field - run immediately and on intervals
    function checkAndInjectPhoneField() {
        var $phoneField = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"]');
        var $phoneWrapper = $('#authyo-checkout-otp-phone-wrapper');

        if ($phoneField.length > 0 && (authyoOtpCheckout.whatsappEnabled || authyoOtpCheckout.smsEnabled || authyoOtpCheckout.voiceEnabled)) {
            // Phone field exists and phone OTP is enabled
            if ($phoneWrapper.length === 0 || $phoneWrapper.closest('#authyo-checkout-otp-container').length > 0) {
                // Wrapper doesn't exist or is in hidden container - need to inject
                // Try inject first, then fallback only if needed
                tryInjectOtpFields(0);
                // Only use fallback if injection didn't work and wrapper still doesn't exist
                setTimeout(function () {
                    var $checkWrapper = $('#authyo-checkout-otp-phone-wrapper');
                    if ($checkWrapper.length === 0 || $checkWrapper.closest('#authyo-checkout-otp-container').length > 0) {
                        createOtpFieldsFallback();
                    } else {
                        // Check for duplicates after injection
                        var $duplicates = $('#authyo_send_phone_otp');
                        if ($duplicates.length > 1) {
                            $duplicates.slice(1).closest('.authyo-checkout-otp-phone-section, .authyo-checkout-otp-input-group, div').remove();
                        }
                    }
                }, 500);
            } else {
                // Wrapper exists and is placed - just ensure button is visible
                var $sendBtn = $phoneWrapper.find('#authyo_send_phone_otp');
                if ($sendBtn.length > 0 && !$sendBtn.is(':visible')) {
                    $sendBtn.css({
                        'display': 'inline-block',
                        'visibility': 'visible',
                        'opacity': '1'
                    }).show();
                }
            }
        }
    }

    // Run check immediately
    checkAndInjectPhoneField();

    // Also run on intervals
    setInterval(checkAndInjectPhoneField, 2000);

    // Also try on checkout update (AJAX)
    $(document.body).on('updated_checkout', function () {
        setTimeout(function () {
            attemptInjection();
            // Update phone label after checkout update
            updatePhoneLabel();

            // Also ensure phone button is visible after checkout update
            setTimeout(function () {
                var $phoneWrapper = $('#authyo-checkout-otp-phone-wrapper');
                if ($phoneWrapper.length > 0) {
                    var $sendBtn = $phoneWrapper.find('#authyo_send_phone_otp');
                    if ($sendBtn.length > 0 && !$sendBtn.is(':visible')) {
                        $sendBtn.css({
                            'display': 'inline-block',
                            'visibility': 'visible',
                            'opacity': '1'
                        }).show();
                    }
                } else {
                    // Try to inject if wrapper doesn't exist
                    tryInjectOtpFields(0);
                }
            }, 500);
        }, 100);
    });

    // Also try when phone field is focused or changed (for dynamic forms)
    $(document).on('focus change input', '#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"]', function () {
        setTimeout(function () {
            var $phoneWrapper = $('#authyo-checkout-otp-phone-wrapper');
            if ($phoneWrapper.length === 0 || $phoneWrapper.closest('#authyo-checkout-otp-container').length > 0) {
                tryInjectOtpFields(0);
            }
        }, 300);
    });

    // Also try when DOM is fully ready
    // Function to update phone label text to "Phone*"
    function updatePhoneLabel() {
        // Update phone label text for all possible selectors
        var $phoneLabels = $('label[for="billing-phone"], label[for="billing_phone"], .billing_phone_field label, .billing-phone-field label, .wc-block-components-address-form__phone label, .wc-block-components-text-input.wc-block-components-address-form__phone label');
        
        $phoneLabels.each(function() {
            var $label = $(this);
            var currentText = $label.text().trim();
            
            // Only update if phone OTP validation is enabled
            if (authyoOtpCheckout.whatsappEnabled || authyoOtpCheckout.smsEnabled || authyoOtpCheckout.voiceEnabled) {
                // Remove "(optional)" and replace with "*" if not already present
                var newText = currentText.replace(/\s*\(optional\)/gi, '').replace(/\s*\(Optional\)/gi, '');
                
                // Add asterisk if not already present
                if (newText.indexOf('*') === -1) {
                    newText = newText.replace(/Phone/gi, 'Phone*');
                }
                
                // If text doesn't contain "Phone", just add "Phone*"
                if (newText.indexOf('Phone') === -1) {
                    newText = 'Phone*';
                }
                
                $label.text(newText);
                
                // Also add required class/attribute if needed
                $label.addClass('required');
            }
        });
    }

    $(document).ready(function () {
        // Update phone label immediately
        updatePhoneLabel();
        
        setTimeout(function () {
            attemptInjection();
            // Update label again after injection
            updatePhoneLabel();
        }, 1000);
    });

    // Try when window loads completely
    $(window).on('load', function () {
        setTimeout(function () {
            attemptInjection();
            // Update label again after injection
            updatePhoneLabel();
        }, 500);
    });

    // Email OTP variables
    var emailMaskId = '';
    var emailVerified = false;
    var emailOtpSent = false;
    var emailTimerInterval = null;
    var emailTimeLeft = authyoOtpCheckout.otpExpiry || 300;
    var emailInitialMethod = null;

    // Phone OTP variables
    var phoneMaskId = '';
    var phoneVerified = false;
    var phoneOtpSent = false;
    var phoneTimerInterval = null;
    var phoneTimeLeft = authyoOtpCheckout.otpExpiry || 300;
    var phoneInitialMethod = null;
    var countriesList = [];
    var selectedCountry = null;

    // Debug mode
    var debugMode = false;

    function debugLog(message, data) {
        // Debug logging removed for production
    }

    // Validation functions
    function validateEmail(email) {
        var re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }

    function validateMobile(mobile) {
        // Remove any non-digit characters except +
        mobile = mobile.replace(/[^\d+]/g, '');

        // Check if mobile number has at least 10 digits after +
        if (mobile.startsWith('+')) {
            return mobile.length >= 11 && mobile.length <= 16 && /^\d+$/.test(mobile.substring(1));
        }

        // Support for global mobile number lengths (7-15 digits)
        return mobile.length >= 7 && mobile.length <= 15 && /^\d+$/.test(mobile);
    }

    // Helper function to process mobile number with country code
    function processMobileNumber(phoneNumber) {
        if (!phoneNumber || typeof phoneNumber !== 'string') {
            return 'ERROR_INVALID_FORMAT';
        }

        // Try multiple selectors to get phone number if not provided
        if (!phoneNumber || phoneNumber.trim() === '') {
            phoneNumber = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]').val() || '';
        }

        if (!phoneNumber || phoneNumber.trim() === '') {
            return 'ERROR_INVALID_FORMAT';
        }

        // Remove all non-digit characters for processing
        var cleanMobile = phoneNumber.replace(/[^\d]/g, '');

        // If phone already has + and country code, validate and return as is
        if (phoneNumber.trim().startsWith('+')) {
            var withPlus = phoneNumber.replace(/[^\d+]/g, '');
            var digitsAfterPlus = withPlus.substring(1);
            if (digitsAfterPlus.length >= 7 && digitsAfterPlus.length <= 15 && /^\d+$/.test(digitsAfterPlus)) {
                return withPlus;
            } else {
                return 'ERROR_INVALID_FORMAT';
            }
        }

        // Check if we have digits to work with
        if (cleanMobile.length < 7 || cleanMobile.length > 15) {
            return 'ERROR_INVALID_FORMAT';
        }

        // Get country from dropdown directly (more reliable than variable)
        var $countrySelect = $('#checkout-country-code');
        var currentSelectedCountry = selectedCountry || ($countrySelect.length > 0 ? $countrySelect.val() : null);

        // Check if country is selected
        if (!currentSelectedCountry || currentSelectedCountry === '') {
            // If no country selected, check if input already has country code
            var originalPhone = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]').val() || '';
            if (originalPhone && originalPhone.trim().startsWith('+')) {
                // Input already has country code, use it
                return originalPhone.replace(/[^\d+]/g, '');
            } else {
                // No country selected and no country code in input - return error indicator
                return 'ERROR_NO_COUNTRY';
            }
        } else {
            // Country is selected, use country code
            var countryCode = currentSelectedCountry.replace(/[^\d]/g, ''); // Extract digits from country code
            if (!countryCode || countryCode.length === 0) {
                return 'ERROR_NO_COUNTRY';
            }
            var finalPhone = '+' + countryCode + cleanMobile;
            return finalPhone;
        }
    }

    // Show status message
    function showStatus($status, message, type, persistent) {
        $status.removeClass('success error info verified')
            .addClass(type)
            .text(message)
            .show();

        // Auto-hide success messages after 5 seconds (unless it's a verification message)
        if (type === 'success' && !persistent) {
            setTimeout(function () {
                $status.fadeOut(400, function () {
                    $status.removeClass('success error info').text('');
                });
            }, 5000);
        }
    }

    // Show verification success message (persistent)
    function showVerificationSuccess($status, message) {
        $status.removeClass('success error info')
            .addClass('success verified')
            .html('<span class="verified-icon">✓</span> ' + message)
            .show();
        // Don't auto-hide verification messages
    }

    // Function to reset email verification
    function resetEmailVerification() {
        // Reset verification state
        emailVerified = false;
        emailMaskId = '';
        emailOtpSent = false;
        emailTimeLeft = authyoOtpCheckout.otpExpiry || 300;

        // Clear hidden fields
        $('#authyo_checkout_email_verified').val('0');
        $('#authyo_checkout_email_maskid').val('');

        // Clear OTP field
        var $otpField = $('#authyo_checkout_email_otp');
        $otpField.val('').prop('disabled', false).removeClass('verified').hide();

        // Show OTP input group again (the container with OTP field and Send button)
        $('#authyo-checkout-otp-email-wrapper .authyo-checkout-otp-input-group').show();

        // Show "Send Email OTP" button - remove verified class/attribute
        var $emailBtn = $('#authyo_send_email_otp');
        $emailBtn.removeClass('authyo-verified')
            .removeAttr('data-verified')
            .text('Send Email OTP')
            .show()
            .css({
                'display': 'inline-block',
                'visibility': 'visible',
                'opacity': '1',
                'height': 'auto',
                'padding': '',
                'margin': '',
                'overflow': 'visible'
            });

        // Remove "Change email" link if it exists in wrapper
        $emailBtn.closest('.authyo-otp-button-wrapper').find('.authyo-change-email-link').remove();

        // Hide resend button
        ensureEmailResendButton().hide();

        // Clear status message
        $('#authyo_email_otp_status').removeClass('success error info verified').text('').hide();

        // Stop any running timers
        if (emailTimerInterval) {
            clearInterval(emailTimerInterval);
            emailTimerInterval = null;
        }
    }

    // Handle "Change email" click - use event delegation for dynamically created link
    $(document).on('click', '#authyo_change_email, .authyo-change-email-link', function (e) {
        e.preventDefault();
        resetEmailVerification();
    });

    // Function to reset phone verification
    function resetPhoneVerification() {
        // Reset verification state
        phoneVerified = false;
        phoneMaskId = '';
        phoneOtpSent = false;
        phoneTimeLeft = authyoOtpCheckout.otpExpiry || 300;

        // Clear hidden fields
        $('#authyo_checkout_phone_verified').val('0');
        $('#authyo_checkout_phone_maskid').val('');

        // Clear OTP field
        var $otpField = $('#authyo_checkout_phone_otp');
        $otpField.val('').prop('disabled', false).removeClass('verified').hide();

        // Show OTP input group again (the container with OTP field and Send button)
        $('#authyo-checkout-otp-phone-wrapper .authyo-checkout-otp-input-group').show();

        // Show "Send Phone OTP" button - remove verified class/attribute
        var $phoneBtn = $('#authyo_send_phone_otp');
        $phoneBtn.removeClass('authyo-verified')
            .removeAttr('data-verified')
            .text('Send Phone OTP')
            .show()
            .css({
                'display': 'inline-block',
                'visibility': 'visible',
                'opacity': '1',
                'height': 'auto',
                'padding': '',
                'margin': '',
                'overflow': 'visible'
            });

        // Remove "Change phone" link if it exists in wrapper
        $phoneBtn.closest('.authyo-otp-button-wrapper').find('.authyo-change-phone-link').remove();

        // Hide resend button
        ensurePhoneResendButton().hide();

        // Hide fallback methods section
        $('#authyo-checkout-otp-phone-wrapper .resend-otp-section').hide();

        // Clear status message
        $('#authyo_phone_otp_status').removeClass('success error info verified').text('').hide();

        // Stop any running timers
        if (phoneTimerInterval) {
            clearInterval(phoneTimerInterval);
            phoneTimerInterval = null;
        }
    }

    // Handle "Change phone" click - use event delegation for dynamically created link
    $(document).on('click', '#authyo_change_phone, .authyo-change-phone-link', function (e) {
        e.preventDefault();
        resetPhoneVerification();
    });

    // Helper: always get the visible resend buttons within checkout wrapper
    function ensureEmailResendButton() {
        var $wrapper = $('#authyo-checkout-otp-email-wrapper');
        var $btn = $wrapper.find('[id="authyo_resend_email_otp"]');
        if ($btn.length === 0) {
            $btn = $('<button type="button" id="authyo_resend_email_otp" class="button button-secondary resend-otp-btn" style="display: block !important; visibility: visible !important; margin-top: 10px !important;"><span class="timer">' + (authyoOtpCheckout.messages.resend || 'Resend OTP') + '</span></button>');
            // Try to place after the button wrapper or input group
            var $buttonWrapper = $wrapper.find('.authyo-otp-button-wrapper').first();
            var $group = $wrapper.find('.authyo-checkout-otp-input-group').first();
            if ($buttonWrapper.length > 0) {
                $btn.insertAfter($buttonWrapper);
            } else if ($group.length > 0) {
                $btn.insertAfter($group);
            } else {
                $wrapper.append($btn);
            }
        } else if ($btn.closest('#authyo-checkout-otp-email-wrapper').length === 0) {
            // Button exists but is outside wrapper, move it
            var $buttonWrapper = $wrapper.find('.authyo-otp-button-wrapper').first();
            var $group = $wrapper.find('.authyo-checkout-otp-input-group').first();
            if ($buttonWrapper.length > 0) {
                $btn.detach().insertAfter($buttonWrapper);
            } else if ($group.length > 0) {
                $btn.detach().insertAfter($group);
            } else {
                $wrapper.append($btn.detach());
            }
        }
        // Always ensure button is visible with inline style
        var existingStyle = $btn.attr('style') || '';
        var newStyle = 'display: block !important; visibility: visible !important; opacity: 1 !important; margin-top: 10px !important;';
        $btn.attr('style', newStyle + ' ' + existingStyle).show();
        return $btn;
    }

    function ensurePhoneResendButton() {
        var $wrapper = $('#authyo-checkout-otp-phone-wrapper');

        // Ensure wrapper is visible first
        if ($wrapper.length > 0) {
            $wrapper.css({
                'display': 'block',
                'visibility': 'visible',
                'opacity': '1'
            }).show();
        }

        var $btn = $wrapper.find('[id="authyo_resend_phone_otp"]');
        if ($btn.length === 0) {
            $btn = $('<button type="button" id="authyo_resend_phone_otp" class="button button-secondary resend-otp-btn" style="display: block !important; visibility: visible !important; opacity: 1 !important; margin-top: 10px !important;"><span class="timer">' + (authyoOtpCheckout.messages.resend || 'Resend OTP') + '</span></button>');
            // Try to place after the button wrapper or input group
            var $buttonWrapper = $wrapper.find('.authyo-otp-button-wrapper').first();
            var $group = $wrapper.find('.authyo-checkout-otp-input-group').first();
            if ($buttonWrapper.length > 0) {
                $btn.insertAfter($buttonWrapper);
            } else if ($group.length > 0) {
                $btn.insertAfter($group);
            } else {
                $wrapper.append($btn);
            }
        } else if ($btn.closest('#authyo-checkout-otp-phone-wrapper').length === 0) {
            // Button exists but is outside wrapper, move it
            var $buttonWrapper = $wrapper.find('.authyo-otp-button-wrapper').first();
            var $group = $wrapper.find('.authyo-checkout-otp-input-group').first();
            if ($buttonWrapper.length > 0) {
                $btn.detach().insertAfter($buttonWrapper);
            } else if ($group.length > 0) {
                $btn.detach().insertAfter($group);
            } else {
                $wrapper.append($btn.detach());
            }
        }
        // Always ensure button is visible with inline style
        var existingStyle = $btn.attr('style') || '';
        var newStyle = 'display: block !important; visibility: visible !important; opacity: 1 !important; margin-top: 10px !important;';

        // Always apply visibility styles, but preserve other styles if they exist
        if (existingStyle && !existingStyle.includes('display: block !important')) {
            // Merge styles, but prioritize our visibility styles
            existingStyle = existingStyle.replace(/display\s*:\s*[^;!]*;?/gi, '').trim();
            existingStyle = existingStyle.replace(/visibility\s*:\s*[^;!]*;?/gi, '').trim();
            existingStyle = existingStyle.replace(/opacity\s*:\s*[^;!]*;?/gi, '').trim();
            existingStyle = existingStyle.replace(/margin-top\s*:\s*[^;!]*;?/gi, '').trim();
            if (existingStyle && !existingStyle.endsWith(';')) {
                existingStyle += ';';
            }
            $btn.attr('style', newStyle + (existingStyle ? ' ' + existingStyle : ''));
        } else {
            // If no existing style or style doesn't have our visibility rules, set it
            if (!existingStyle.includes('display: block !important')) {
                $btn.attr('style', newStyle);
            }
        }

        // Force show the button
        $btn.show();
        return $btn;
    }

    // Start timer for email OTP
    function startEmailTimer() {
        emailTimeLeft = authyoOtpCheckout.otpExpiry || 300;
        var $resendBtn = ensureEmailResendButton();

        if ($resendBtn.length === 0) {
            return;
        }

        $resendBtn.prop('disabled', true);
        $('.authyo-checkout-otp-email-section .resend-otp-section').addClass('authyo-hidden').hide();

        // Ensure the timer span exists BEFORE showing the button
        var $timerSpan = $resendBtn.find('.timer');
        if ($timerSpan.length === 0) {
            $timerSpan = $('<span class="timer">' + (authyoOtpCheckout.messages.resend || 'Resend OTP') + '</span>');
            $resendBtn.append($timerSpan);
        }

        // Ensure wrapper is visible first
        var $wrapper = $('#authyo-checkout-otp-email-wrapper');
        if ($wrapper.length > 0) {
            $wrapper.css({
                'display': 'block',
                'visibility': 'visible',
                'opacity': '1'
            }).show();
        }

        // Explicitly show the button - remove inline display:none and set proper styles
        // Get current style, remove display:none, add our styles
        var currentStyle = $resendBtn.attr('style') || '';
        currentStyle = currentStyle.replace(/display\s*:\s*none[^;]*;?/gi, '').trim();
        var newStyle = 'display: block !important; visibility: visible !important; opacity: 1 !important; margin-top: 10px !important;';
        if (currentStyle && !currentStyle.endsWith(';')) {
            currentStyle += ';';
        }
        $resendBtn.attr('style', newStyle + (currentStyle ? ' ' + currentStyle : ''));
        $resendBtn.show();

        // Update display immediately
        updateEmailTimerDisplay();

        clearInterval(emailTimerInterval);
        emailTimerInterval = setInterval(function () {
            emailTimeLeft--;
            updateEmailTimerDisplay();

            if (emailTimeLeft <= 0) {
                clearInterval(emailTimerInterval);
                $resendBtn.prop('disabled', false).find('.timer').text(authyoOtpCheckout.messages.resend);
                $resendBtn.css({
                    'display': 'block',
                    'visibility': 'visible'
                }).show();

                // Change "Send Email OTP" button text from "OTP sent successfully!" to "Resend OTP"
                var $sendBtn = $('#authyo_send_email_otp');
                if ($sendBtn.length > 0 && !emailVerified) {
                    var currentText = $sendBtn.text().trim();
                    var sentMessage = authyoOtpCheckout.messages.sent || 'OTP sent successfully!';
                    if (currentText === sentMessage || currentText.includes('sent successfully')) {
                        $sendBtn.prop('disabled', false)
                            .text(authyoOtpCheckout.messages.resend || 'Resend OTP')
                            .show()
                            .css({
                                'display': 'inline-block',
                                'visibility': 'visible',
                                'opacity': '1'
                            });
                    }
                }

                // For email, no fallback methods to show
            }
        }, 1000);
    }

    // Update email timer display
    function updateEmailTimerDisplay() {
        var minutes = Math.floor(emailTimeLeft / 60);
        var seconds = emailTimeLeft % 60;
        var $resendBtn = ensureEmailResendButton();
        if ($resendBtn.length > 0) {
            var $timerSpan = $resendBtn.find('.timer');
            if ($timerSpan.length === 0) {
                // Create timer span if it doesn't exist
                $timerSpan = $('<span class="timer"></span>');
                $resendBtn.append($timerSpan);
            }

            // Update timer text - format: "Resend OTP (0:47s)"
            var timerText = (authyoOtpCheckout.messages.resend || 'Resend OTP') + ' (' + minutes + ':' + (seconds < 10 ? '0' : '') + seconds + 's)';
            $timerSpan.text(timerText);

            // Ensure button is visible during timer - override inline styles
            if (emailTimeLeft > 0) {
                // Ensure wrapper is visible
                var $wrapper = $('#authyo-checkout-otp-email-wrapper');
                if ($wrapper.length > 0) {
                    $wrapper.css({
                        'display': 'block',
                        'visibility': 'visible',
                        'opacity': '1'
                    }).show();
                }

                // Get current style, ensure display:block is set with !important
                var currentStyle = $resendBtn.attr('style') || '';
                currentStyle = currentStyle.replace(/display\s*:\s*none[^;]*;?/gi, '').trim();
                var newStyle = 'display: block !important; visibility: visible !important; opacity: 1 !important; margin-top: 10px !important;';
                if (currentStyle && !currentStyle.endsWith(';')) {
                    currentStyle += ';';
                }
                $resendBtn.attr('style', newStyle + (currentStyle ? ' ' + currentStyle : ''));
                $resendBtn.show();

            }
        }
    }

    // Start timer for phone OTP
    function startPhoneTimer() {
        phoneTimeLeft = authyoOtpCheckout.otpExpiry || 300;
        var $resendBtn = ensurePhoneResendButton();

        if ($resendBtn.length === 0) {
            return;
        }

        $resendBtn.prop('disabled', true);
        $('.authyo-checkout-otp-phone-section .resend-otp-section').addClass('authyo-hidden').hide();

        // Ensure the timer span exists BEFORE showing the button
        var $timerSpan = $resendBtn.find('.timer');
        if ($timerSpan.length === 0) {
            $timerSpan = $('<span class="timer">' + (authyoOtpCheckout.messages.resend || 'Resend OTP') + '</span>');
            $resendBtn.append($timerSpan);
        }

        // Ensure wrapper is visible first
        var $wrapper = $('#authyo-checkout-otp-phone-wrapper');
        if ($wrapper.length > 0) {
            $wrapper.css({
                'display': 'block',
                'visibility': 'visible',
                'opacity': '1'
            }).show();
        }

        // Explicitly show the button - remove inline display:none and set proper styles
        // Get current style, remove display:none, add our styles
        var currentStyle = $resendBtn.attr('style') || '';
        currentStyle = currentStyle.replace(/display\s*:\s*none[^;]*;?/gi, '').trim();
        var newStyle = 'display: block !important; visibility: visible !important; opacity: 1 !important; margin-top: 10px !important;';
        if (currentStyle && !currentStyle.endsWith(';')) {
            currentStyle += ';';
        }
        $resendBtn.attr('style', newStyle + (currentStyle ? ' ' + currentStyle : ''));
        $resendBtn.show();

        // Update display immediately
        updatePhoneTimerDisplay();

        clearInterval(phoneTimerInterval);
        phoneTimerInterval = setInterval(function () {
            phoneTimeLeft--;
            updatePhoneTimerDisplay();

            if (phoneTimeLeft <= 0) {
                clearInterval(phoneTimerInterval);
                $resendBtn.prop('disabled', false).find('.timer').text(authyoOtpCheckout.messages.resend);
                $resendBtn.css({
                    'display': 'block',
                    'visibility': 'visible'
                }).show();

                // Change "Send Phone OTP" button text from "OTP sent successfully!" to "Resend OTP"
                var $sendBtn = $('#authyo_send_phone_otp');
                if ($sendBtn.length > 0 && !phoneVerified) {
                    var currentText = $sendBtn.text().trim();
                    var sentMessage = authyoOtpCheckout.messages.sent || 'OTP sent successfully!';
                    if (currentText === sentMessage || currentText.includes('sent successfully')) {
                        $sendBtn.prop('disabled', false)
                            .text(authyoOtpCheckout.messages.resend || 'Resend OTP')
                            .show()
                            .css({
                                'display': 'inline-block',
                                'visibility': 'visible',
                                'opacity': '1'
                            });
                    }
                }

                // Show fallback methods after timer completes for phone (only if not verified)
                if (!phoneVerified) {
                    showPhoneResendMethods(phoneInitialMethod);
                }
            }
        }, 1000);
    }

    // Update phone timer display
    function updatePhoneTimerDisplay() {
        var minutes = Math.floor(phoneTimeLeft / 60);
        var seconds = phoneTimeLeft % 60;
        var $resendBtn = ensurePhoneResendButton();
        if ($resendBtn.length > 0) {
            var $timerSpan = $resendBtn.find('.timer');
            if ($timerSpan.length === 0) {
                // Create timer span if it doesn't exist
                $timerSpan = $('<span class="timer"></span>');
                $resendBtn.append($timerSpan);
            }

            // Update timer text - format: "Resend OTP (0:47s)"
            var timerText = (authyoOtpCheckout.messages.resend || 'Resend OTP') + ' (' + minutes + ':' + (seconds < 10 ? '0' : '') + seconds + 's)';
            $timerSpan.text(timerText);

            // Ensure button is visible during timer - override inline styles
            if (phoneTimeLeft > 0) {
                // Ensure wrapper is visible
                var $wrapper = $('#authyo-checkout-otp-phone-wrapper');
                if ($wrapper.length > 0) {
                    $wrapper.css({
                        'display': 'block',
                        'visibility': 'visible',
                        'opacity': '1'
                    }).show();
                }

                // Get current style, ensure display:block is set with !important
                var currentStyle = $resendBtn.attr('style') || '';
                currentStyle = currentStyle.replace(/display\s*:\s*none[^;]*;?/gi, '').trim();
                var newStyle = 'display: block !important; visibility: visible !important; opacity: 1 !important; margin-top: 10px !important;';
                if (currentStyle && !currentStyle.endsWith(';')) {
                    currentStyle += ';';
                }
                $resendBtn.attr('style', newStyle + (currentStyle ? ' ' + currentStyle : ''));
                $resendBtn.show();

            }
        }
    }

    // Helper function to generate fallback methods HTML based on enabled methods
    // excludeMethod: optional, method to exclude from the list (e.g., the one that was just used)
    // Returns HTML string if 2+ total methods are enabled (and at least 1 after exclusion), null otherwise
    function generateFallbackMethodsHtml(excludeMethod) {
        excludeMethod = excludeMethod || null;
        
        // Check which methods are enabled
        var whatsappEnabled = authyoOtpCheckout.whatsappEnabled || false;
        var smsEnabled = authyoOtpCheckout.smsEnabled || false;
        var voiceEnabled = authyoOtpCheckout.voiceEnabled || false;

        // Count total enabled methods
        var totalEnabledCount = 0;
        if (whatsappEnabled) totalEnabledCount++;
        if (smsEnabled) totalEnabledCount++;
        if (voiceEnabled) totalEnabledCount++;

        // If only 1 or 0 methods are enabled total, don't show fallback section at all
        if (totalEnabledCount <= 1) {
            return null;
        }

        // Build array of enabled methods (excluding the one that was just used if provided)
        var enabledMethods = [];
        
        if (whatsappEnabled && excludeMethod !== 'whatsapp') {
            enabledMethods.push({ method: 'whatsapp', label: 'WhatsApp' });
        }
        if (smsEnabled && excludeMethod !== 'sms') {
            enabledMethods.push({ method: 'sms', label: 'SMS' });
        }
        if (voiceEnabled && excludeMethod !== 'voice') {
            enabledMethods.push({ method: 'voice', label: 'Voicecall' });
        }

        // If no methods available after exclusion, don't show fallback section
        if (enabledMethods.length === 0) {
            return null;
        }

        // Generate HTML for enabled methods
        var buttonsHtml = '';
        enabledMethods.forEach(function(method) {
            buttonsHtml += '<button type="button" class="resend-method-btn" data-method="' + method.method + '">' + method.label + '</button>';
        });

        return '<div class="resend-otp-section">' +
            '<p class="resend-text">Didn\'t receive OTP? Try Resend OTP via:</p>' +
            '<div class="resend-methods-grid">' +
            buttonsHtml +
            '</div>' +
            '</div>';
    }

    // Show resend methods for phone after timer completes
    function showPhoneResendMethods(usedMethod) {
        phoneInitialMethod = usedMethod;

        // Check if we have enough enabled methods to show fallback (need at least 2)
        var fallbackHtml = generateFallbackMethodsHtml(usedMethod);
        
        // If only 1 method is enabled, don't show fallback section at all
        if (!fallbackHtml) {
            return;
        }

        // Always show fallback methods for phone OTP (since we're in phone OTP section)
        // No need to validate mobile - we're already in the phone OTP flow
        // Use multiple selectors to find the section - try both ID and class selectors
        var $wrapper = $('#authyo-checkout-otp-phone-wrapper');
        if ($wrapper.length === 0) {
            $wrapper = $('.authyo-checkout-otp-phone-section');
        }

        var $resendSection = $wrapper.find('.resend-otp-section');

        // Also try direct selector as fallback
        if ($resendSection.length === 0) {
            $resendSection = $('#authyo-checkout-otp-phone-wrapper .resend-otp-section, .authyo-checkout-otp-phone-section .resend-otp-section');
        }

        // If section doesn't exist, create it with only enabled methods
        if ($resendSection.length === 0) {
            // Try to insert after the resend button, or append to wrapper
            var $resendBtn = $wrapper.find('#authyo_resend_phone_otp');
            if ($resendBtn.length > 0) {
                $resendBtn.after(fallbackHtml);
            } else {
                $wrapper.append(fallbackHtml);
            }

            $resendSection = $wrapper.find('.resend-otp-section');
        } else {
            // Section exists, update it with only enabled methods
            // First check total enabled methods
            var whatsappEnabled = authyoOtpCheckout.whatsappEnabled || false;
            var smsEnabled = authyoOtpCheckout.smsEnabled || false;
            var voiceEnabled = authyoOtpCheckout.voiceEnabled || false;
            
            var totalEnabledCount = 0;
            if (whatsappEnabled) totalEnabledCount++;
            if (smsEnabled) totalEnabledCount++;
            if (voiceEnabled) totalEnabledCount++;
            
            // If only 1 or 0 methods are enabled total, hide the section
            if (totalEnabledCount <= 1) {
                $resendSection.addClass('authyo-hidden').hide();
                return;
            }
            
            // Remove all existing buttons
            $resendSection.find('.resend-method-btn').remove();
            
            // Add only enabled method buttons (excluding the one that was just used)
            var $methodsGrid = $resendSection.find('.resend-methods-grid');
            if ($methodsGrid.length === 0) {
                $resendSection.append('<div class="resend-methods-grid"></div>');
                $methodsGrid = $resendSection.find('.resend-methods-grid');
            }
            
            var buttonsAdded = 0;
            if (whatsappEnabled && usedMethod !== 'whatsapp') {
                $methodsGrid.append('<button type="button" class="resend-method-btn" data-method="whatsapp">WhatsApp</button>');
                buttonsAdded++;
            }
            if (smsEnabled && usedMethod !== 'sms') {
                $methodsGrid.append('<button type="button" class="resend-method-btn" data-method="sms">SMS</button>');
                buttonsAdded++;
            }
            if (voiceEnabled && usedMethod !== 'voice') {
                $methodsGrid.append('<button type="button" class="resend-method-btn" data-method="voice">Voicecall</button>');
                buttonsAdded++;
            }
            
            // If no buttons were added, hide the section
            if (buttonsAdded === 0) {
                $resendSection.addClass('authyo-hidden').hide();
                return;
            }
        }

        if ($resendSection.length === 0) {
            return;
        }

        // Remove hidden class and show - matching login/register form pattern
        $resendSection.removeClass('authyo-hidden');

        // Force show with inline style using !important to override any CSS
        var currentStyle = $resendSection.attr('style') || '';
        // Remove any existing display/visibility/opacity rules
        currentStyle = currentStyle.replace(/display\s*:\s*[^;!]*;?/gi, '').trim();
        currentStyle = currentStyle.replace(/visibility\s*:\s*[^;!]*;?/gi, '').trim();
        currentStyle = currentStyle.replace(/opacity\s*:\s*[^;!]*;?/gi, '').trim();

        // Set inline style with !important
        var newStyle = 'display: block !important; visibility: visible !important; opacity: 1 !important;';
        if (currentStyle && !currentStyle.endsWith(';')) {
            currentStyle += ';';
        }
        $resendSection.attr('style', newStyle + (currentStyle ? ' ' + currentStyle : ''));

        // Also use jQuery show() as backup
        $resendSection.show();

        // Ensure wrapper is visible
        if ($wrapper.length > 0) {
            $wrapper.css({
                'display': 'block',
                'visibility': 'visible',
                'opacity': '1'
            }).show();
        }

        // Enable all resend method buttons
        var $methodButtons = $resendSection.find('.resend-method-btn');
        $methodButtons.prop('disabled', false);

        // Disable the method that was just used
        if (usedMethod) {
            $resendSection.find('.resend-method-btn[data-method="' + usedMethod + '"]').prop('disabled', true);
        }

        // Force visibility check
        setTimeout(function () {
            if (!$resendSection.is(':visible')) {
                $resendSection.removeClass('authyo-hidden').css({
                    'display': 'block !important',
                    'visibility': 'visible !important',
                    'opacity': '1 !important'
                }).show();
            }
        }, 100);
    }

    // Fetch country list
    function fetchCountryList() {
        // Ensure we have the required objects
        if (typeof authyoOtpCheckout === 'undefined' || !authyoOtpCheckout.ajaxurl || !authyoOtpCheckout.nonce) {
            // Use fallback country list if AJAX is not available
            if (typeof countriesList === 'undefined' || countriesList.length === 0) {
                countriesList = getFallbackCountryList();
                populateCountrySelect();
                // Ensure dropdown is visible after populating
                ensureCountryDropdownVisible();
            }
            return;
        }
        

        // Ensure AJAX URL is valid
        var ajaxUrl = authyoOtpCheckout.ajaxurl;
        if (!ajaxUrl || ajaxUrl === '' || ajaxUrl === 'undefined') {
            if (typeof countriesList === 'undefined' || countriesList.length === 0) {
                countriesList = getFallbackCountryList();
                populateCountrySelect();
                ensureCountryDropdownVisible();
            }
            return;
        }

        // Use a timeout to prevent hanging requests
        var requestTimeout = setTimeout(function () {
            if (typeof countriesList === 'undefined' || countriesList.length === 0) {
                countriesList = getFallbackCountryList();
                populateCountrySelect();
                ensureCountryDropdownVisible();
            }
        }, 10000); // 10 second timeout

        $.ajax({
            url: ajaxUrl,
            type: 'POST',
            data: {
                action: 'authyo_otp_auth_get_country_list',
                nonce: authyoOtpCheckout.nonce
            },
            timeout: 8000, // 8 second timeout
            success: function (response) {
                clearTimeout(requestTimeout);
                if (response && response.success && response.data && response.data.countries && Array.isArray(response.data.countries) && response.data.countries.length > 0) {
                    countriesList = response.data.countries;
                    // Populate immediately if dropdown exists, otherwise it will be populated when dropdown is created
                    populateCountrySelect();
                    // Ensure dropdown is visible after populating
                    ensureCountryDropdownVisible();
                } else {
                    // If response format is unexpected, use fallback
                    if (typeof countriesList === 'undefined' || countriesList.length === 0) {
                        countriesList = getFallbackCountryList();
                        populateCountrySelect();
                        ensureCountryDropdownVisible();
                    }
                }
            },
            error: function (xhr, status, error) {
                clearTimeout(requestTimeout);
                // Use fallback country list on failure
                if (typeof countriesList === 'undefined' || countriesList.length === 0) {
                    countriesList = getFallbackCountryList();
                    populateCountrySelect();
                    ensureCountryDropdownVisible();
                }
            }
        });
    }

    // Fallback country list (used when API is unavailable)
    function getFallbackCountryList() {
        return [
            { name: 'United States', countryCode: '+1', code: '1' },
            { name: 'Canada', countryCode: '+1', code: '1' },
            { name: 'United Kingdom', countryCode: '+44', code: '44' },
            { name: 'India', countryCode: '+91', code: '91' },
            { name: 'Australia', countryCode: '+61', code: '61' },
            { name: 'Germany', countryCode: '+49', code: '49' },
            { name: 'France', countryCode: '+33', code: '33' },
            { name: 'Italy', countryCode: '+39', code: '39' },
            { name: 'Spain', countryCode: '+34', code: '34' },
            { name: 'China', countryCode: '+86', code: '86' },
            { name: 'Japan', countryCode: '+81', code: '81' },
            { name: 'South Korea', countryCode: '+82', code: '82' },
            { name: 'Singapore', countryCode: '+65', code: '65' },
            { name: 'Malaysia', countryCode: '+60', code: '60' },
            { name: 'Indonesia', countryCode: '+62', code: '62' },
            { name: 'Philippines', countryCode: '+63', code: '63' },
            { name: 'Thailand', countryCode: '+66', code: '66' },
            { name: 'Vietnam', countryCode: '+84', code: '84' },
            { name: 'UAE', countryCode: '+971', code: '971' },
            { name: 'Saudi Arabia', countryCode: '+966', code: '966' },
            { name: 'Brazil', countryCode: '+55', code: '55' },
            { name: 'Mexico', countryCode: '+52', code: '52' },
            { name: 'South Africa', countryCode: '+27', code: '27' },
            { name: 'Nigeria', countryCode: '+234', code: '234' }
        ];
    }

    // Populate country select dropdown
    function populateCountrySelect() {
        // Initialize countriesList if not defined
        if (typeof countriesList === 'undefined') {
            countriesList = [];
        }

        if (countriesList.length === 0) {
            // If country list is empty, try fetching it again or use fallback
            if (typeof authyoOtpCheckout !== 'undefined' && authyoOtpCheckout.ajaxurl) {
                fetchCountryList();
            } else {
                // Use fallback if AJAX is not available
                countriesList = getFallbackCountryList();
            }
            // Wait a bit and try again
            setTimeout(function () {
                populateCountrySelect();
            }, 500);
            return;
        }

        var $countrySelect = $('#checkout-country-code');

        // If dropdown doesn't exist yet, wait a bit and try again
        if ($countrySelect.length === 0) {
            setTimeout(function () {
                populateCountrySelect();
            }, 500);
            return;
        }

        // Don't repopulate if already populated (avoid unnecessary DOM manipulation)
        var currentOptionCount = $countrySelect.find('option').length;
        if (currentOptionCount > 1) {
            // Ensure dropdown is visible even if already populated
            $countrySelect.css({
                'display': 'block',
                'visibility': 'visible',
                'opacity': '1'
            }).show();
            return; // Already populated
        }

        $countrySelect.empty();
        $countrySelect.append('<option value="">Select Country</option>');

        $.each(countriesList, function (index, country) {
            if (!country) return;

            // Handle different API response formats
            var countryCode = country.countryCode || country.code || '';
            var countryName = country.name || '';

            // Ensure countryCode has + prefix
            if (countryCode && !countryCode.toString().startsWith('+')) {
                countryCode = '+' + countryCode.toString();
            }

            if (countryCode && countryName) {
                // Escape HTML to prevent XSS
                var safeName = $('<div>').text(countryName).html();
                var safeCode = $('<div>').text(countryCode.toString()).html();
                $countrySelect.append('<option value="' + safeCode + '" data-code="' + safeCode + '" data-name="' + safeName + '">' + safeName + ' (' + safeCode + ')</option>');
            }
        });

        // Ensure dropdown is visible after populating
        $countrySelect.css({
            'display': 'block',
            'visibility': 'visible',
            'opacity': '1'
        }).show();
    }

    // Show/hide country dropdown based on input type
    function toggleCountryDropdown(value) {
        var $countrySelect = $('#checkout-country-code');

        // For checkout phone field, always show country dropdown
        // The phone field is specifically for phone numbers, so country selection is always needed
        if ($countrySelect.length > 0) {
            $countrySelect.css({
                'display': 'block !important',
                'visibility': 'visible !important',
                'opacity': '1 !important',
                'height': 'auto !important',
                'width': 'auto !important'
            }).show();

            // Ensure it's populated if empty
            var optionCount = $countrySelect.find('option').length;
            if (optionCount <= 1 && (typeof countriesList !== 'undefined' && countriesList.length > 0)) {
                populateCountrySelect();
            } else if (optionCount <= 1) {
                // If empty and no country list, fetch it
                fetchCountryList();
            }
        }
    }

    // Handle country selection - use event delegation for dynamically created elements
    $(document).on('change', '#checkout-country-code', function () {
        selectedCountry = $(this).val();
        // Try multiple selectors for phone field
        var phone = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]').val() || '';
        toggleCountryDropdown(phone);
        updatePhonePreview();
    });

    // Update phone preview
    function updatePhonePreview() {
        // Try multiple selectors for phone field
        var phone = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]').val() || '';
        var $preview = $('#checkout-number-preview');
        var isMobile = validateMobile(phone);

        if (isMobile && selectedCountry) {
            var cleanMobile = phone.replace(/\D/g, '');
            var countryCode = selectedCountry.replace(/[^\d]/g, '');
            $preview.text('Will send to: +' + countryCode + cleanMobile).show();
        } else if (isMobile && !selectedCountry) {
            $preview.text('Please select a country for SMS delivery').show();
        } else {
            $preview.hide();
        }
    }

    // Monitor billing phone input - use event delegation for dynamically created fields
    $(document).on('input focus', '#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]', function () {
        var phone = $(this).val() || '';
        toggleCountryDropdown(phone);
        updatePhonePreview();

        // Ensure country dropdown is visible when phone field is focused or has value
        var $countrySelect = $('#checkout-country-code');
        if ($countrySelect.length > 0) {
            $countrySelect.css({
                'display': 'block !important',
                'visibility': 'visible !important',
                'opacity': '1 !important',
                'height': 'auto !important'
            }).show().removeClass('authyo-hidden');
            // Populate dropdown if it's empty
            if ($countrySelect.find('option').length <= 1) {
                if (typeof countriesList !== 'undefined' && countriesList.length > 0) {
                    populateCountrySelect();
                } else {
                    fetchCountryList();
                }
            }
        }

        // Ensure fallback methods are hidden for invalid inputs
        var isMobile = validateMobile(phone);
        if (!isMobile) {
            $('.authyo-checkout-otp-phone-section .resend-otp-section').addClass('authyo-hidden').hide();
        }
    });

    // Fetch country list on page load
    fetchCountryList();

    // Periodically check and populate country dropdown if it exists but is empty (handles dynamic injection)
    var countryDropdownCheckInterval = setInterval(function () {
        var $countrySelect = $('#checkout-country-code');
        if ($countrySelect.length > 0) {
            // Always ensure dropdown is visible
            $countrySelect.css({
                'display': 'block !important',
                'visibility': 'visible !important',
                'opacity': '1 !important'
            }).show().removeClass('authyo-hidden');

            // Check if dropdown has only the default option (empty) or no options at all
            var optionCount = $countrySelect.find('option').length;
            if (optionCount <= 1) {
                if (typeof countriesList !== 'undefined' && countriesList.length > 0) {
                    populateCountrySelect();
                } else {
                    fetchCountryList();
                }
            }
        }
    }, 1000); // Check every second

    // Clear interval after 30 seconds to avoid infinite checking
    setTimeout(function () {
        clearInterval(countryDropdownCheckInterval);
    }, 30000);

    // Function to ensure country dropdown is properly placed and visible
    function ensureCountryDropdownVisible() {
        var $phoneField = $('input[name="billing_phone"], #billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[type="tel"][name="billing_phone"]');
        var $countrySelect = $('#checkout-country-code');

        // If country select doesn't exist, try to find it in the hidden container
        if ($countrySelect.length === 0) {
            $countrySelect = $('#authyo-checkout-otp-container #checkout-country-code');
        }

        // Populate country dropdown if it exists but is empty
        if ($countrySelect.length > 0) {
            // Always ensure dropdown is visible - use !important in inline styles
            $countrySelect.attr('style',
                'display: block !important; ' +
                'visibility: visible !important; ' +
                'opacity: 1 !important; ' +
                'height: auto !important; ' +
                'position: relative !important; ' +
                'z-index: 1 !important;'
            ).show().removeClass('authyo-hidden');

            var optionCount = $countrySelect.find('option').length;
            if (optionCount <= 1) {
                if (typeof countriesList !== 'undefined' && countriesList.length > 0) {
                    populateCountrySelect();
                } else {
                    fetchCountryList();
                }
            }
        }

        if ($phoneField.length > 0 && $countrySelect.length > 0) {
            // Check if it's block-based checkout
            var $blockPhoneContainer = $phoneField.closest('.wc-block-components-text-input.wc-block-components-address-form__phone, .wc-block-components-address-form__phone');

            if ($blockPhoneContainer.length > 0) {
                // Block-based checkout handling
                // Check if country select is already in the phone container
                if ($blockPhoneContainer.find('#checkout-country-code').length === 0) {
                    // Country select is not in phone container, move it there
                    var $phoneInputWrapper = $blockPhoneContainer.find('.phone-input-wrapper');

                    if ($phoneInputWrapper.length === 0) {
                        // Create wrapper
                        var $label = $blockPhoneContainer.find('label');
                        $phoneInputWrapper = $('<div class="phone-input-wrapper" style="display: grid; grid-template-columns: 25% 75%; gap: 10px; align-items: stretch; width: 100%; margin-top: 8px; margin-bottom: 10px; clear: both;"></div>');

                        // Move country select first (25% width)
                        $countrySelect.detach().prependTo($phoneInputWrapper);
                        // Populate country dropdown after moving
                        populateCountrySelect();
                        $countrySelect.css({
                            'display': 'block',
                            'visibility': 'visible',
                            'width': '100%',
                            'min-width': '0',
                            'max-width': '100%'
                        });

                        // Move phone input (75% width)
                        $phoneField.detach().appendTo($phoneInputWrapper);
                        $phoneField.css({
                            'width': '100%',
                            'min-width': '0',
                            'max-width': '100%'
                        });

                        // Add wrapper to container
                        if ($label.length > 0) {
                            $label.after($phoneInputWrapper);
                        } else {
                            $blockPhoneContainer.append($phoneInputWrapper);
                        }
                    } else {
                        // Wrapper exists, just add country select
                        if ($phoneInputWrapper.find('#checkout-country-code').length === 0) {
                            $countrySelect.detach().prependTo($phoneInputWrapper);
                            $countrySelect.css({
                                'display': 'block',
                                'visibility': 'visible',
                                'width': 'auto',
                                'min-width': '150px',
                                'flex-shrink': '0'
                            });
                        }
                    }
                }

                // Always show country dropdown with strong CSS rules
                $countrySelect.attr('style',
                    'display: block !important; ' +
                    'visibility: visible !important; ' +
                    'opacity: 1 !important; ' +
                    'height: auto !important; ' +
                    'width: auto !important; ' +
                    'min-width: 130px !important; ' +
                    'max-width: 200px !important; ' +
                    'position: relative !important; ' +
                    'z-index: 1 !important;'
                ).show().removeClass('authyo-hidden');

                // Ensure the phone number preview appears directly below the phone-input-wrapper
                var $preview = $('#checkout-number-preview');
                var $phoneInputWrapper = $blockPhoneContainer.find('.phone-input-wrapper');
                if ($preview.length > 0 && $phoneInputWrapper && $phoneInputWrapper.length > 0) {
                    $preview.detach().insertAfter($phoneInputWrapper);
                    $preview.css({
                        'display': 'block',
                        'margin-top': '4px',
                        'margin-bottom': '8px'
                    });
                }
            } else {
                // Classic checkout handling
                var $phoneFormRow = $phoneField.closest('.form-row, .woocommerce-billing-fields__field-wrapper, p.form-row, .billing_phone_field, .billing-phone-field');

                if ($phoneFormRow.length > 0) {
                    // Check if country select is already in the form row
                    if ($phoneFormRow.find('#checkout-country-code').length === 0) {
                        // Check if there's already a phone-input-wrapper
                        var $phoneInputWrapper = $phoneFormRow.find('.phone-input-wrapper');

                        if ($phoneInputWrapper.length === 0) {
                            // Create wrapper for classic checkout
                            $phoneInputWrapper = $('<div class="phone-input-wrapper" style="display: flex; gap: 10px; align-items: center; margin-bottom: 10px; width: 100%;"></div>');

                            // Move country select first
                            $countrySelect.detach().prependTo($phoneInputWrapper);
                            // Populate country dropdown after moving
                            populateCountrySelect();
                            $countrySelect.css({
                                'display': 'block',
                                'visibility': 'visible',
                                'width': 'auto',
                                'min-width': '150px',
                                'max-width': '200px',
                                'flex-shrink': '0'
                            });

                            // Move phone input next to country select
                            $phoneField.detach().appendTo($phoneInputWrapper);
                            $phoneField.css({
                                'flex': '1',
                                'width': 'auto',
                                'min-width': '0'
                            });

                            // Insert wrapper before the form row ends or after label
                            var $label = $phoneFormRow.find('label');
                            if ($label.length > 0) {
                                $label.after($phoneInputWrapper);
                            } else {
                                $phoneFormRow.prepend($phoneInputWrapper);
                            }
                        } else {
                            // Wrapper exists, just add country select if not present
                            if ($phoneInputWrapper.find('#checkout-country-code').length === 0) {
                                $countrySelect.detach().prependTo($phoneInputWrapper);
                                $countrySelect.css({
                                    'display': 'block',
                                    'visibility': 'visible',
                                    'width': 'auto',
                                    'min-width': '150px',
                                    'max-width': '200px',
                                    'flex-shrink': '0'
                                });
                            }
                        }
                    }

                    // Always show country dropdown with strong CSS rules for classic checkout
                    $countrySelect.attr('style',
                        'display: block !important; ' +
                        'visibility: visible !important; ' +
                        'opacity: 1 !important; ' +
                        'height: auto !important; ' +
                        'width: auto !important; ' +
                        'min-width: 150px !important; ' +
                        'max-width: 200px !important; ' +
                        'position: relative !important; ' +
                        'z-index: 1 !important;'
                    ).show().removeClass('authyo-hidden');
                } else {
                    // Fallback: if form row not found, try to place it near the phone field
                    var $phoneParent = $phoneField.parent();
                    if ($phoneParent.length > 0 && $phoneParent.find('#checkout-country-code').length === 0) {
                        // Create a simple wrapper
                        var $phoneInputWrapper = $phoneParent.find('.phone-input-wrapper');
                        if ($phoneInputWrapper.length === 0) {
                            $phoneInputWrapper = $('<div class="phone-input-wrapper" style="display: flex; gap: 10px; align-items: center; margin-bottom: 10px; width: 100%;"></div>');
                            $countrySelect.detach().prependTo($phoneInputWrapper);
                            populateCountrySelect();
                            $phoneField.detach().appendTo($phoneInputWrapper);
                            $phoneParent.prepend($phoneInputWrapper);
                        } else {
                            if ($phoneInputWrapper.find('#checkout-country-code').length === 0) {
                                $countrySelect.detach().prependTo($phoneInputWrapper);
                            }
                        }
                    }

                    // Always ensure visibility
                    $countrySelect.attr('style',
                        'display: block !important; ' +
                        'visibility: visible !important; ' +
                        'opacity: 1 !important; ' +
                        'height: auto !important; ' +
                        'position: relative !important; ' +
                        'z-index: 1 !important;'
                    ).show().removeClass('authyo-hidden');
                }
            }
        } else if ($countrySelect.length > 0) {
            // Country select exists but phone field not found yet - still ensure it's visible
            $countrySelect.attr('style',
                'display: block !important; ' +
                'visibility: visible !important; ' +
                'opacity: 1 !important; ' +
                'height: auto !important; ' +
                'position: relative !important; ' +
                'z-index: 1 !important;'
            ).show().removeClass('authyo-hidden');
        }
    }

    // Run country dropdown check immediately and on intervals
    ensureCountryDropdownVisible(); // Run immediately
    setTimeout(ensureCountryDropdownVisible, 100); // Run after 100ms
    setTimeout(ensureCountryDropdownVisible, 500); // Run after 500ms
    setInterval(ensureCountryDropdownVisible, 2000); // Run every 2 seconds

    // Also run after checkout updates
    $(document.body).on('updated_checkout', function () {
        updatePhoneLabel();
        setTimeout(ensureCountryDropdownVisible, 100);
        setTimeout(ensureCountryDropdownVisible, 500);
    });

    // Run when DOM is ready (in case jQuery ready fired before our code)
    $(document).ready(function () {
        setTimeout(ensureCountryDropdownVisible, 100);
        setTimeout(ensureCountryDropdownVisible, 1000);
    });

    // Run on window load as well
    $(window).on('load', function () {
        setTimeout(ensureCountryDropdownVisible, 100);
        setTimeout(ensureCountryDropdownVisible, 500);
    });

    // ===== EMAIL OTP FUNCTIONS =====

    // Send email OTP - use event delegation for dynamically created buttons
    $(document).on('click', '#authyo_send_email_otp', function (e) {
        e.preventDefault();

        var $button = $(this);
        var $status = $('#authyo_email_otp_status');
        var $otpField = $('#authyo_checkout_email_otp');
        var $resendBtn = ensureEmailResendButton();

        // Get email from email field - works for both guest and logged-in users, classic and block-based checkout
        var $emailInput = $('input[name="billing_email"], #billing_email, input#billing_email, #email, input#email, input[type="email"][id="email"]');
        var email = '';

        // Try multiple methods to get email (handles both guest checkout and logged-in users)
        if ($emailInput.length > 0) {
            // Method 1: Get from input value (works for guest users and editable fields)
            email = $emailInput.val() || '';

            // Method 2: If empty, try to get from value attribute (for readonly fields)
            if (!email || email.trim() === '') {
                email = $emailInput.attr('value') || '';
            }

            // Method 3: If still empty, try to get from data attribute or default value
            if (!email || email.trim() === '') {
                email = $emailInput.data('value') || $emailInput.prop('defaultValue') || '';
            }
        }

        // Method 4: Try to get from WooCommerce checkout object (for logged-in users)
        if ((!email || email.trim() === '') && typeof wc_checkout_params !== 'undefined') {
            // This will be handled by the AJAX call which can access server-side checkout object
        }

        if (!email || !validateEmail(email)) {
            showStatus($status, authyoOtpCheckout.messages.enter_email, 'error');
            return;
        }

        // Verify nonce exists
        if (!authyoOtpCheckout.nonce) {
            showStatus($status, 'Security token is missing. Please refresh the page.', 'error');
            return;
        }

        $button.prop('disabled', true).text(authyoOtpCheckout.messages.sending);
        $status.removeClass('success error info').text('');

        // Prepare AJAX data
        var ajaxData = {
            action: 'authyo_otp_auth_send_checkout_otp',
            nonce: authyoOtpCheckout.nonce,
            type: 'email',
            email: email,
            billing_email: email  // Send both for compatibility
        };

        // Send both email field names for compatibility (block-based uses 'email', classic uses 'billing_email')
        $.ajax({
            url: authyoOtpCheckout.ajaxurl,
            type: 'POST',
            data: ajaxData,
            success: function (response) {
                if (response.success) {
                    emailMaskId = response.data.maskId;
                    emailInitialMethod = response.data.method || 'email';
                    emailOtpSent = true;
                    showStatus($status, response.data.message || authyoOtpCheckout.messages.sent, 'success');
                    $('#authyo_checkout_email_maskid').val(emailMaskId);

                    // Set maxlength based on configured OTP length
                    var otpLength = authyoOtpCheckout.otpLength || 6;
                    $otpField.attr('maxlength', otpLength);

                    // Update placeholder based on delivery method
                    var placeholderText = 'Please enter OTP sent on your Email';
                    $otpField.attr('placeholder', placeholderText);

                    // Manually trigger input handler if field already has value (for paste scenarios)
                    setTimeout(function () {
                        if ($otpField.val().length > 0) {
                            handleEmailOtpInput();
                        }
                    }, 100);

                    $otpField.show();

                    // Show button with "OTP sent successfully!" message and "Change email" link (only if not already verified)
                    if (!emailVerified) {
                        // Create a wrapper for button and change email link
                        var $buttonWrapper = $button.parent();
                        if (!$buttonWrapper.hasClass('authyo-otp-button-wrapper')) {
                            $button.wrap('<span class="authyo-otp-button-wrapper" style="display: inline-flex; align-items: center; gap: 10px;"></span>');
                            $buttonWrapper = $button.parent();
                        }

                        // Add or update "Change email" link
                        var $changeEmailLink = $buttonWrapper.find('.authyo-change-email-link');
                        if ($changeEmailLink.length === 0) {
                            $changeEmailLink = $('<a href="#" class="authyo-change-email-link" style="font-size: 13px; color: #2271b1; text-decoration: underline; cursor: pointer;">Change email</a>');
                            $buttonWrapper.append($changeEmailLink);
                        }

                        // Update button text and show
                        $button.prop('disabled', false).text(authyoOtpCheckout.messages.sent).show().css({
                            'display': 'inline-block',
                            'visibility': 'visible'
                        });
                        $buttonWrapper.show();
                    } else {
                        // If already verified, keep it hidden
                        $button.hide().css({
                            'display': 'none !important',
                            'visibility': 'hidden !important'
                        });
                    }

                    // Ensure resend button exists and is visible
                    var $resendBtn = ensureEmailResendButton();

                    // Ensure wrapper is visible
                    var $emailWrapper = $('#authyo-checkout-otp-email-wrapper');
                    if ($emailWrapper.length > 0) {
                        $emailWrapper.css({
                            'display': 'block',
                            'visibility': 'visible',
                            'opacity': '1'
                        }).show();
                    }

                    // Show resend button with timer (initially disabled)
                    $resendBtn.prop('disabled', true);
                    var btnStyle = 'display: block !important; visibility: visible !important; opacity: 1 !important; margin-top: 10px !important;';
                    $resendBtn.attr('style', btnStyle).show();
                    startEmailTimer();
                } else {
                    showStatus($status, response.data.message || authyoOtpCheckout.messages.error, 'error');
                    $button.text('Send Email OTP');
                }
            },
            error: function (xhr, status, error) {
                var errorMessage = authyoOtpCheckout.messages.error;
                if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    errorMessage = xhr.responseJSON.data.message;
                } else if (xhr.responseText) {
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (response.data && response.data.message) {
                            errorMessage = response.data.message;
                        }
                    } catch (e) {
                        // If not JSON, use default message
                    }
                }

                showStatus($status, errorMessage, 'error');
                $button.prop('disabled', false).text('Send Email OTP');
            }
        });
    });

    // Resend email OTP - use event delegation for dynamically created buttons
    $(document).on('click', '#authyo_resend_email_otp', function (e) {
        e.preventDefault();

        var $button = $(this);
        var $status = $('#authyo_email_otp_status');
        // Get email - works for both classic and block-based checkout
        var $emailInput = $('input[name="billing_email"], #billing_email, input#billing_email, #email, input#email, input[type="email"][id="email"]');
        var email = '';

        // Get email using multiple methods (works for both guest and logged-in users)
        if ($emailInput.length > 0) {
            email = $emailInput.val() || $emailInput.attr('value') || $emailInput.data('value') || '';
        }

        if (!email || !validateEmail(email)) {
            showStatus($status, authyoOtpCheckout.messages.enter_email, 'error');
            return;
        }

        $button.prop('disabled', true).find('.timer').text(authyoOtpCheckout.messages.sending);

        $.ajax({
            url: authyoOtpCheckout.ajaxurl,
            type: 'POST',
            data: {
                action: 'authyo_otp_auth_send_checkout_otp',
                nonce: authyoOtpCheckout.nonce,
                type: 'email',
                email: email
            },
            success: function (response) {
                if (response.success) {
                    emailMaskId = response.data.maskId;
                    emailInitialMethod = response.data.method || 'email';
                    showStatus($status, response.data.message || authyoOtpCheckout.messages.sent, 'success');
                    $('#authyo_checkout_email_maskid').val(emailMaskId);
                    startEmailTimer();
                } else {
                    showStatus($status, response.data.message || authyoOtpCheckout.messages.error, 'error');
                    $button.find('.timer').text(authyoOtpCheckout.messages.resend);
                }
            },
            error: function () {
                showStatus($status, authyoOtpCheckout.messages.error, 'error');
                $button.find('.timer').text(authyoOtpCheckout.messages.resend);
            },
            complete: function () {
                $button.prop('disabled', false);
            }
        });
    });

    // Function to handle email OTP input
    function handleEmailOtpInput() {
        var $otpField = $('#authyo_checkout_email_otp');
        if ($otpField.length === 0) {
            return;
        }

        // Only allow numeric input
        var otp = $otpField.val().replace(/[^0-9]/g, '');
        if ($otpField.val() !== otp) {
            $otpField.val(otp);
        }

        var $status = $('#authyo_email_otp_status');
        // Get email - works for both classic and block-based checkout
        var $emailInput = $('input[name="billing_email"], #billing_email, input#billing_email, #email, input#email, input[type="email"][id="email"]');
        var email = '';

        // Get email using multiple methods (works for both guest and logged-in users)
        if ($emailInput.length > 0) {
            email = $emailInput.val() || $emailInput.attr('value') || $emailInput.data('value') || '';
        }

        // Fallback: Try to get email from form data if direct input not found
        if (!email) {
            var formData = $('form.checkout').serializeArray();
            for (var i = 0; i < formData.length; i++) {
                if (formData[i].name === 'billing_email' || formData[i].name === 'email') {
                    email = formData[i].value;
                    break;
                }
            }
        }

        // Get OTP length from settings (default to 6 if not set) - ensure it's a number
        var otpLength = parseInt(authyoOtpCheckout.otpLength, 10) || 6;

        // Get maskId from variable or hidden field
        var currentEmailMaskId = emailMaskId || $('#authyo_checkout_email_maskid').val() || '';

        // Enhanced debug logging on every keystroke - only log when length matches to reduce noise
        var lengthMatches = parseInt(otp.length, 10) === parseInt(otpLength, 10);

        // Auto-verify when configured number of digits are entered
        if (lengthMatches && currentEmailMaskId && !emailVerified && email) {
            verifyOtp('email', email, otp, currentEmailMaskId, $status, function () {
                emailVerified = true;
                $('#authyo_checkout_email_verified').val('1');

                // Clear and stop the email timer
                if (emailTimerInterval) {
                    clearInterval(emailTimerInterval);
                    emailTimerInterval = null;
                }

                // Clear the periodic OTP check interval
                if (typeof emailOtpCheckInterval !== 'undefined' && emailOtpCheckInterval) {
                    clearInterval(emailOtpCheckInterval);
                    emailOtpCheckInterval = null;
                }

                // Hide OTP input group (OTP field and "Send OTP" button)
                $('#authyo-checkout-otp-email-wrapper .authyo-checkout-otp-input-group').hide();

                // Hide "Send Email OTP" button (which shows "OTP sent successfully!")
                // Add class and data attribute to ensure it stays hidden via CSS
                var $sendBtn = $('#authyo_send_email_otp');
                $sendBtn.addClass('authyo-verified')
                    .attr('data-verified', 'true')
                    .hide()
                    .css({
                        'display': 'none',
                        'visibility': 'hidden',
                        'opacity': '0',
                        'height': '0',
                        'padding': '0',
                        'margin': '0',
                        'overflow': 'hidden'
                    });

                // Hide resend button and remove timer display
                var $resendBtn = ensureEmailResendButton();
                if ($resendBtn.length > 0) {
                    // Remove timer span if it exists
                    $resendBtn.find('.timer').remove();
                    // Reset button text to just "Resend OTP" without timer
                    $resendBtn.prop('disabled', false).text(authyoOtpCheckout.messages.resend || 'Resend OTP');
                    // Hide the resend button completely
                    $resendBtn.hide();
                }

                // Hide OTP field itself
                $otpField.hide();

                // Show persistent verification message with "Change email" option
                var verifiedMessage = authyoOtpCheckout.messages.email_verified || 'This email is verified';
                var changeEmailLink = '<a href="#" id="authyo_change_email" class="authyo-change-email-link">Change email</a>';
                showVerificationSuccess($status, verifiedMessage + ' ' + changeEmailLink);

                // Remove any duplicate "Change email" links that are outside the status message
                var $statusLink = $('#authyo_email_otp_status .authyo-change-email-link');
                $('.authyo-change-email-link').not($statusLink).remove();
            });
        } else if (otp.length > otpLength) {
            // Limit input to configured OTP length
            $otpField.val(otp.substring(0, otpLength));
        } else if (lengthMatches) {
        }
    }

    // Verify email OTP as user types - use event delegation for dynamically created fields
    $(document).on('input keyup paste change', '#authyo_checkout_email_otp', function (e) {
        handleEmailOtpInput();
    });

    // Periodic check for OTP field value (backup for cases where events don't fire)
    var emailOtpCheckInterval = setInterval(function () {
        var $otpField = $('#authyo_checkout_email_otp');
        if ($otpField.length > 0 && $otpField.is(':visible') && !emailVerified) {
            var currentValue = $otpField.val().replace(/[^0-9]/g, '');
            var otpLength = authyoOtpCheckout.otpLength || 6;

            if (currentValue.length === otpLength) {
                handleEmailOtpInput();
            }
        } else if (emailVerified) {
            // Stop checking once verified
            clearInterval(emailOtpCheckInterval);
        }
    }, 500); // Check every 500ms

    // ===== PHONE OTP FUNCTIONS =====

    // Send phone OTP - use event delegation for dynamically created buttons
    $(document).on('click', '#authyo_send_phone_otp', function (e) {
        e.preventDefault();

        var $button = $(this);
        var $status = $('#authyo_phone_otp_status');
        var $otpField = $('#authyo_checkout_phone_otp');
        var $resendBtn = $('#authyo_resend_phone_otp');
        // Try multiple selectors to get phone number
        var phone = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]').val() || '';

        // Validate phone number exists
        if (!phone || phone.trim() === '') {
            showStatus($status, authyoOtpCheckout.messages.enter_phone || 'Please enter a valid phone number.', 'error');
            return;
        }

        // Process mobile number with country code
        var processedPhone = processMobileNumber(phone);

        if (processedPhone === 'ERROR_NO_COUNTRY') {
            showStatus($status, 'Please select a country for your mobile number to enable SMS delivery.', 'error');
            return;
        }

        if (processedPhone === 'ERROR_INVALID_FORMAT') {
            showStatus($status, 'Please enter a valid phone number (7-15 digits).', 'error');
            return;
        }

        // Validate the processed phone number
        if (!processedPhone || processedPhone.length < 8) {
            showStatus($status, 'Please enter a valid phone number.', 'error');
            return;
        }

        $button.prop('disabled', true).text(authyoOtpCheckout.messages.sending);
        $status.removeClass('success error info').text('');

        $.ajax({
            url: authyoOtpCheckout.ajaxurl,
            type: 'POST',
            data: {
                action: 'authyo_otp_auth_send_checkout_otp',
                nonce: authyoOtpCheckout.nonce,
                type: 'phone',
                phone: processedPhone,
                contact: processedPhone,
                billing_phone: phone
            },
            success: function (response) {
                if (response.success) {
                    phoneMaskId = response.data.maskId;
                    phoneInitialMethod = response.data.method || 'whatsapp';
                    phoneOtpSent = true;
                    showStatus($status, response.data.message || authyoOtpCheckout.messages.sent, 'success');
                    $('#authyo_checkout_phone_maskid').val(phoneMaskId);

                    // Set maxlength based on configured OTP length
                    var otpLength = authyoOtpCheckout.otpLength || 6;
                    $otpField.attr('maxlength', otpLength);

                    // Update placeholder based on delivery method
                    var method = response.data.method || 'whatsapp';
                    var placeholderText = 'Please enter OTP sent to your phone';
                    if (method === 'whatsapp') {
                        placeholderText = 'Please enter OTP sent on WhatsApp';
                    } else if (method === 'sms') {
                        placeholderText = 'Please enter OTP sent on SMS';
                    } else if (method === 'voice') {
                        placeholderText = 'Please enter OTP sent on Voicecall';
                    }
                    $otpField.attr('placeholder', placeholderText);

                    // Manually trigger input handler if field already has value (for paste scenarios)
                    setTimeout(function () {
                        if ($otpField.val().length > 0) {
                            handlePhoneOtpInput();
                        }
                    }, 100);

                    $otpField.show();

                    // Show button with "OTP sent successfully!" message and "Change phone" link (only if not already verified)
                    if (!phoneVerified) {
                        // Create a wrapper for button and change phone link
                        var $buttonWrapper = $button.parent();
                        if (!$buttonWrapper.hasClass('authyo-otp-button-wrapper')) {
                            $button.wrap('<span class="authyo-otp-button-wrapper" style="display: inline-flex; align-items: center; gap: 10px;"></span>');
                            $buttonWrapper = $button.parent();
                        }

                        // Add or update "Change phone" link
                        var $changePhoneLink = $buttonWrapper.find('.authyo-change-phone-link');
                        if ($changePhoneLink.length === 0) {
                            $changePhoneLink = $('<a href="#" class="authyo-change-phone-link" style="font-size: 13px; color: #2271b1; text-decoration: underline; cursor: pointer;">Change phone</a>');
                            $buttonWrapper.append($changePhoneLink);
                        }

                        // Update button text and show
                        $button.prop('disabled', false).text(authyoOtpCheckout.messages.sent).show().css({
                            'display': 'inline-block',
                            'visibility': 'visible'
                        });
                        $buttonWrapper.show();
                    } else {
                        // If already verified, keep it hidden
                        $button.hide().css({
                            'display': 'none !important',
                            'visibility': 'hidden !important'
                        });
                    }

                    // Ensure resend button exists and is visible (let ensurePhoneResendButton handle creation)
                    var $resendBtn = ensurePhoneResendButton();

                    // Ensure wrapper is visible
                    var $phoneWrapper = $('#authyo-checkout-otp-phone-wrapper');
                    if ($phoneWrapper.length > 0) {
                        $phoneWrapper.css({
                            'display': 'block',
                            'visibility': 'visible',
                            'opacity': '1'
                        }).show();
                    }

                    // Show resend button with timer (initially disabled)
                    $resendBtn.prop('disabled', true);
                    var btnStyle = 'display: block !important; visibility: visible !important; opacity: 1 !important; margin-top: 10px !important;';
                    $resendBtn.attr('style', btnStyle).show();
                    startPhoneTimer();
                } else {
                    showStatus($status, response.data.message || authyoOtpCheckout.messages.error, 'error');
                    $button.text('Send Phone OTP');
                }
            },
            error: function () {
                showStatus($status, authyoOtpCheckout.messages.error, 'error');
                $button.prop('disabled', false).text('Send Phone OTP');
            }
        });
    });

    // Resend phone OTP (uses primary method) - use event delegation for dynamically created buttons
    $(document).on('click', '#authyo_resend_phone_otp', function (e) {
        e.preventDefault();

        var $button = $(this);
        var $status = $('#authyo_phone_otp_status');
        // Try multiple selectors to get phone number (same as Send Phone OTP handler)
        var phone = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]').val() || '';

        if (!phone || phone.trim() === '' || !validateMobile(phone)) {
            showStatus($status, authyoOtpCheckout.messages.enter_phone || 'Please enter a valid phone number.', 'error');
            return;
        }

        // Process mobile number with country code
        var processedPhone = processMobileNumber(phone);

        if (processedPhone === 'ERROR_NO_COUNTRY') {
            showStatus($status, 'Please select a country for your mobile number to enable SMS delivery.', 'error');
            return;
        }

        $button.prop('disabled', true).find('.timer').text(authyoOtpCheckout.messages.sending);

        $.ajax({
            url: authyoOtpCheckout.ajaxurl,
            type: 'POST',
            data: {
                action: 'authyo_otp_auth_send_checkout_otp',
                nonce: authyoOtpCheckout.nonce,
                type: 'phone',
                phone: processedPhone
            },
            success: function (response) {
                if (response.success) {
                    phoneMaskId = response.data.maskId;
                    phoneInitialMethod = response.data.method || 'whatsapp';
                    showStatus($status, response.data.message || authyoOtpCheckout.messages.sent, 'success');
                    $('#authyo_checkout_phone_maskid').val(phoneMaskId);

                    // Update placeholder based on delivery method
                    var method = response.data.method || 'whatsapp';
                    var placeholderText = 'Please enter OTP sent to your phone';
                    if (method === 'whatsapp') {
                        placeholderText = 'Please enter OTP sent on WhatsApp';
                    } else if (method === 'sms') {
                        placeholderText = 'Please enter OTP sent on SMS';
                    } else if (method === 'voice') {
                        placeholderText = 'Please enter OTP sent on Voicecall';
                    }
                    $('#authyo_checkout_phone_otp').attr('placeholder', placeholderText);

                    startPhoneTimer();
                } else {
                    showStatus($status, response.data.message || authyoOtpCheckout.messages.error, 'error');
                    $button.find('.timer').text(authyoOtpCheckout.messages.resend);
                }
            },
            error: function () {
                showStatus($status, authyoOtpCheckout.messages.error, 'error');
                $button.find('.timer').text(authyoOtpCheckout.messages.resend);
            },
            complete: function () {
                $button.prop('disabled', false);
            }
        });
    });

    // Handle fallback method button click for phone
    $(document).on('click', '.authyo-checkout-otp-phone-section .resend-method-btn', function (e) {
        e.preventDefault();
        e.stopPropagation();

        var $btn = $(this);

        if ($btn.prop('disabled')) {
            return;
        }

        var method = $btn.data('method');
        // Try multiple selectors to get phone number (same as Send Phone OTP handler)
        var phone = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]').val() || '';

        if (!phone || phone.trim() === '' || !method) {
            showStatus($('#authyo_phone_otp_status'), authyoOtpCheckout.messages.enter_phone || 'Please enter a valid phone number.', 'error');
            return;
        }

        // Validate that fallback methods can only be used with mobile numbers
        var isMobile = validateMobile(phone);

        if (!isMobile) {
            showStatus($('#authyo_phone_otp_status'), authyoOtpCheckout.messages.enter_phone || 'Please enter a valid phone number.', 'error');
            return;
        }

        // Process mobile number with country code
        var processedPhone = processMobileNumber(phone);

        if (processedPhone === 'ERROR_NO_COUNTRY') {
            showStatus($('#authyo_phone_otp_status'), 'Please select a country for your mobile number to enable SMS delivery.', 'error');
            return;
        }

        // Store original button text
        var originalText = $btn.text() || (method === 'sms' ? 'SMS' : method === 'whatsapp' ? 'WhatsApp' : 'Voicecall');
        $btn.prop('disabled', true).text('Sending...');

        $.ajax({
            url: authyoOtpCheckout.ajaxurl,
            type: 'POST',
            data: {
                action: 'authyo_otp_auth_resend_checkout_fallback',
                nonce: authyoOtpCheckout.nonce,
                type: 'phone',
                phone: processedPhone,
                contact: processedPhone,
                billing_phone: phone,
                method: method
            },
            success: function (response) {
                $btn.prop('disabled', false).text(originalText);
                if (response.success) {
                    phoneMaskId = response.data.maskId;
                    phoneInitialMethod = response.data.method || method;
                    showStatus($('#authyo_phone_otp_status'), response.data.message, 'success');
                    $('#authyo_checkout_phone_maskid').val(phoneMaskId);

                    // Update placeholder based on fallback method used
                    var placeholderText = 'Please enter OTP sent to your phone';
                    if (method === 'whatsapp') {
                        placeholderText = 'Please enter OTP sent on WhatsApp';
                    } else if (method === 'sms') {
                        placeholderText = 'Please enter OTP sent on SMS';
                    } else if (method === 'voice') {
                        placeholderText = 'Please enter OTP sent on Voicecall';
                    }
                    $('#authyo_checkout_phone_otp').attr('placeholder', placeholderText);

                    startPhoneTimer();
                } else {
                    showStatus($('#authyo_phone_otp_status'), response.data.message, 'error');
                }
            },
            error: function () {
                $btn.prop('disabled', false).text(originalText);
                showStatus($('#authyo_phone_otp_status'), 'Connection error. Please check your internet connection and try again.', 'error');
            }
        });
    });

    // Function to handle phone OTP input
    function handlePhoneOtpInput() {
        var $otpField = $('#authyo_checkout_phone_otp');
        if ($otpField.length === 0) {
            return;
        }

        // Only allow numeric input
        var otp = $otpField.val().replace(/[^0-9]/g, '');
        if ($otpField.val() !== otp) {
            $otpField.val(otp);
        }

        var $status = $('#authyo_phone_otp_status');
        // Try multiple selectors to get phone number
        var phone = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]').val() || '';

        // Get OTP length from settings (default to 6 if not set) - ensure it's a number
        var otpLength = parseInt(authyoOtpCheckout.otpLength, 10) || 6;

        // Get maskId from variable or hidden field
        var currentPhoneMaskId = phoneMaskId || $('#authyo_checkout_phone_maskid').val() || '';

        var lengthMatches = parseInt(otp.length, 10) === parseInt(otpLength, 10);

        // Auto-verify when configured number of digits are entered
        if (lengthMatches && currentPhoneMaskId && !phoneVerified && phone) {
            // Process phone number for verification
            var processedPhone = processMobileNumber(phone);

            if (processedPhone === 'ERROR_NO_COUNTRY') {
                showStatus($status, 'Please select a country for your mobile number to enable SMS delivery.', 'error');
                return;
            }

            if (processedPhone === 'ERROR_INVALID_FORMAT') {
                showStatus($status, 'Please enter a valid phone number (7-15 digits).', 'error');
                return;
            }

            if (processedPhone && processedPhone.length >= 8) {
                verifyOtp('phone', processedPhone, otp, currentPhoneMaskId, $status, function () {
                    phoneVerified = true;
                    $('#authyo_checkout_phone_verified').val('1');

                    // Clear and stop the phone timer
                    if (phoneTimerInterval) {
                        clearInterval(phoneTimerInterval);
                        phoneTimerInterval = null;
                    }

                    // Clear the periodic OTP check interval
                    if (typeof phoneOtpCheckInterval !== 'undefined' && phoneOtpCheckInterval) {
                        clearInterval(phoneOtpCheckInterval);
                        phoneOtpCheckInterval = null;
                    }

                    // Hide OTP input group (OTP field and "Send OTP" button)
                    $('#authyo-checkout-otp-phone-wrapper .authyo-checkout-otp-input-group').hide();

                    // Hide "Send Phone OTP" button (which shows "OTP sent successfully!")
                    // Add class and data attribute to ensure it stays hidden via CSS
                    var $sendBtn = $('#authyo_send_phone_otp');
                    $sendBtn.addClass('authyo-verified')
                        .attr('data-verified', 'true')
                        .hide()
                        .css({
                            'display': 'none',
                            'visibility': 'hidden',
                            'opacity': '0',
                            'height': '0',
                            'padding': '0',
                            'margin': '0',
                            'overflow': 'hidden'
                        });

                    // Hide resend button and remove timer display
                    var $resendBtn = ensurePhoneResendButton();
                    if ($resendBtn.length > 0) {
                        // Remove timer span if it exists
                        $resendBtn.find('.timer').remove();
                        // Reset button text to just "Resend OTP" without timer
                        $resendBtn.prop('disabled', false).text(authyoOtpCheckout.messages.resend || 'Resend OTP');
                        // Hide the resend button completely
                        $resendBtn.hide();
                    }

                    // Hide OTP field itself
                    $otpField.hide();

                    // Hide "Change phone" link from button wrapper and resend methods section after verification
                    $('.authyo-change-phone-link').not('#authyo_change_phone').hide();
                    var $resendSection = $('#authyo-checkout-otp-phone-wrapper .resend-otp-section, .authyo-checkout-otp-phone-section .resend-otp-section');
                    $resendSection.addClass('authyo-hidden').hide().css({
                        'display': 'none',
                        'visibility': 'hidden',
                        'opacity': '0'
                    });

                    // Show persistent verification message with "Change phone" option
                    var verifiedMessage = authyoOtpCheckout.messages.phone_verified || 'This phone number is verified';
                    var changePhoneLink = '<a href="#" id="authyo_change_phone" class="authyo-change-phone-link">Change phone</a>';
                    showVerificationSuccess($status, verifiedMessage + ' ' + changePhoneLink);

                    // Remove any duplicate "Change phone" links that are outside the status message
                    var $statusLink = $('#authyo_phone_otp_status .authyo-change-phone-link');
                    $('.authyo-change-phone-link').not($statusLink).remove();
                });
            } else {
                showStatus($status, 'Please enter a valid phone number.', 'error');
            }
        } else if (otp.length > otpLength) {
            // Limit input to configured OTP length
            $otpField.val(otp.substring(0, otpLength));
        } else if (lengthMatches) {
        }
    }

    // Verify phone OTP as user types - use event delegation for dynamically created fields
    $(document).on('input keyup paste change', '#authyo_checkout_phone_otp', function (e) {
        handlePhoneOtpInput();
    });

    // Periodic check for phone OTP field value (backup for cases where events don't fire)
    var phoneOtpCheckInterval = setInterval(function () {
        var $otpField = $('#authyo_checkout_phone_otp');
        if ($otpField.length > 0 && $otpField.is(':visible') && !phoneVerified) {
            var currentValue = $otpField.val().replace(/[^0-9]/g, '');
            var otpLength = authyoOtpCheckout.otpLength || 6;

            if (currentValue.length === otpLength) {
                handlePhoneOtpInput();
            }
        } else if (phoneVerified) {
            // Stop checking once verified
            clearInterval(phoneOtpCheckInterval);
        }
    }, 500); // Check every 500ms

    // Verify OTP function
    function verifyOtp(type, contact, otp, maskId, $status, callback) {
        // Validate inputs
        if (!otp || !maskId || !contact) {
            showStatus($status, 'Missing verification data. Please try again.', 'error');
            return;
        }

        $status.removeClass('success error info verified').addClass('info').text(authyoOtpCheckout.messages.verifying).show();

        $.ajax({
            url: authyoOtpCheckout.ajaxurl,
            type: 'POST',
            data: {
                action: 'authyo_otp_auth_verify_checkout_otp',
                nonce: authyoOtpCheckout.nonce,
                type: type,
                contact: contact,
                otp: otp,
                maskId: maskId
            },
            success: function (response) {
                if (response.success && response.data.verified) {
                    // Call callback first to set verified flag
                    if (callback) callback();
                    // The callback will show the verification message
                } else {
                    var errorMsg = response.data && response.data.message
                        ? response.data.message
                        : 'Invalid OTP. Please try again.';
                    showStatus($status, errorMsg, 'error');

                    // If error message indicates expiration, update the send button to "Resend OTP"
                    if (errorMsg && (errorMsg.toLowerCase().includes('expired') || errorMsg.toLowerCase().includes('session expired'))) {
                        var $sendBtn = type === 'email' ? $('#authyo_send_email_otp') : $('#authyo_send_phone_otp');
                        if ($sendBtn.length > 0) {
                            var currentText = $sendBtn.text().trim();
                            var sentMessage = authyoOtpCheckout.messages.sent || 'OTP sent successfully!';
                            if (currentText === sentMessage || currentText.includes('sent successfully')) {
                                // Change button text to "Resend OTP"
                                $sendBtn.prop('disabled', false)
                                    .text(authyoOtpCheckout.messages.resend || 'Resend OTP')
                                    .removeClass('authyo-verified')
                                    .removeAttr('data-verified')
                                    .show()
                                    .css({
                                        'display': 'inline-block',
                                        'visibility': 'visible',
                                        'opacity': '1'
                                    });

                                // Clear the maskId to indicate expiration
                                if (type === 'email') {
                                    $('#authyo_checkout_email_maskid').val('');
                                    emailMaskId = '';
                                } else {
                                    $('#authyo_checkout_phone_maskid').val('');
                                    phoneMaskId = '';
                                }
                            }
                        }
                    }
                }
            },
            error: function (xhr, status, error) {

                var errorMessage = authyoOtpCheckout.messages.error;
                if (xhr.responseJSON && xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    errorMessage = xhr.responseJSON.data.message;
                } else if (xhr.responseText) {
                    try {
                        var response = JSON.parse(xhr.responseText);
                        if (response.data && response.data.message) {
                            errorMessage = response.data.message;
                        }
                    } catch (e) {
                        // If not JSON, use default message
                    }
                }
                showStatus($status, errorMessage, 'error');

                // If error message indicates expiration, update the send button to "Resend OTP"
                if (errorMessage && (errorMessage.toLowerCase().includes('expired') || errorMessage.toLowerCase().includes('session expired'))) {
                    var $sendBtn = type === 'email' ? $('#authyo_send_email_otp') : $('#authyo_send_phone_otp');
                    if ($sendBtn.length > 0) {
                        var currentText = $sendBtn.text().trim();
                        var sentMessage = authyoOtpCheckout.messages.sent || 'OTP sent successfully!';
                        if (currentText === sentMessage || currentText.includes('sent successfully')) {
                            // Change button text to "Resend OTP"
                            $sendBtn.prop('disabled', false)
                                .text(authyoOtpCheckout.messages.resend || 'Resend OTP')
                                .removeClass('authyo-verified')
                                .removeAttr('data-verified')
                                .show()
                                .css({
                                    'display': 'inline-block',
                                    'visibility': 'visible',
                                    'opacity': '1'
                                });

                            // Clear the maskId to indicate expiration
                            if (type === 'email') {
                                $('#authyo_checkout_email_maskid').val('');
                                emailMaskId = '';
                            } else {
                                $('#authyo_checkout_phone_maskid').val('');
                                phoneMaskId = '';
                            }
                        }
                    }
                }
            }
        });
    }

    // Prevent form submission if OTPs are not verified - Multiple hooks for reliability
    // Hook 1: Before checkout validation (WooCommerce block-based checkout)
    $(document.body).on('checkout_place_order', function (e, data) {
        var isValid = validateOtpBeforeCheckout();
        if (!isValid) {
            e.preventDefault();
            e.stopImmediatePropagation();
            return false;
        }
        return isValid;
    });

    // Hook 2: On form submit (additional check for classic checkout)
    $(document).on('submit', 'form.checkout, form.woocommerce-checkout', function (e) {
        if (!validateOtpBeforeCheckout()) {
            e.preventDefault();
            e.stopImmediatePropagation();
            e.stopPropagation();
            return false;
        }
    });

    // Hook 3: Before checkout processing (WooCommerce specific)
    $(document.body).on('checkout_place_order_checkout', function (e) {
        if (!validateOtpBeforeCheckout()) {
            e.preventDefault();
            e.stopImmediatePropagation();
            e.stopPropagation();
            return false;
        }
    });

    // Hook 4: Intercept Place Order button click (most reliable)
    $(document).on('click', 'button[name="woocommerce_checkout_place_order"], button#place_order, button[type="submit"][name="woocommerce_checkout_place_order"], button.wc-block-components-checkout-place-order-button', function (e) {
        // Ensure hidden fields exist before validation
        ensureHiddenFieldsExist();

        if (!validateOtpBeforeCheckout()) {
            e.preventDefault();
            e.stopImmediatePropagation();
            e.stopPropagation();
            return false;
        }
    });

    // Function to ensure hidden fields always exist
    function ensureHiddenFieldsExist() {
        var emailEnabled = authyoOtpCheckout.emailEnabled;
        var phoneMethodsEnabled = authyoOtpCheckout.whatsappEnabled || authyoOtpCheckout.smsEnabled || authyoOtpCheckout.voiceEnabled;

        // Find all possible checkout forms
        var $forms = $('form.checkout, form.woocommerce-checkout, form.wc-block-checkout__form');

        if ($forms.length === 0) {
            // For block-based checkout, try to find the checkout container
            var $checkoutContainer = $('.wc-block-checkout, .woocommerce-checkout');
            if ($checkoutContainer.length > 0) {
                // Create a hidden form element to store our fields
                if ($('#authyo-checkout-hidden-fields').length === 0) {
                    $checkoutContainer.prepend('<div id="authyo-checkout-hidden-fields" style="display:none;"></div>');
                }
                var $hiddenContainer = $('#authyo-checkout-hidden-fields');

                if (emailEnabled && $('#authyo_checkout_email_verified').length === 0) {
                    $hiddenContainer.append('<input type="hidden" id="authyo_checkout_email_verified" name="authyo_checkout_email_verified" value="0" />');
                    $hiddenContainer.append('<input type="hidden" id="authyo_checkout_email_maskid" name="authyo_checkout_email_maskid" value="" />');
                }
                if (phoneMethodsEnabled && $('#authyo_checkout_phone_verified').length === 0) {
                    $hiddenContainer.append('<input type="hidden" id="authyo_checkout_phone_verified" name="authyo_checkout_phone_verified" value="0" />');
                    $hiddenContainer.append('<input type="hidden" id="authyo_checkout_phone_maskid" name="authyo_checkout_phone_maskid" value="" />');
                }
            }
        } else {
            // For classic checkout, add to form
            $forms.each(function () {
                var $form = $(this);
                if (emailEnabled && $form.find('#authyo_checkout_email_verified').length === 0) {
                    $form.append('<input type="hidden" id="authyo_checkout_email_verified" name="authyo_checkout_email_verified" value="0" />');
                    $form.append('<input type="hidden" id="authyo_checkout_email_maskid" name="authyo_checkout_email_maskid" value="" />');
                }
                if (phoneMethodsEnabled && $form.find('#authyo_checkout_phone_verified').length === 0) {
                    $form.append('<input type="hidden" id="authyo_checkout_phone_verified" name="authyo_checkout_phone_verified" value="0" />');
                    $form.append('<input type="hidden" id="authyo_checkout_phone_maskid" name="authyo_checkout_phone_maskid" value="" />');
                }
            });
        }
    }

    // Ensure hidden fields exist on page load and before checkout updates
    $(document).ready(function () {
        ensureHiddenFieldsExist();
    });

    // Also ensure fields exist before checkout AJAX updates
    $(document.body).on('checkout_place_order', function () {
        ensureHiddenFieldsExist();
    });

    $(document.body).on('updated_checkout', function () {
        ensureHiddenFieldsExist();
        updatePhoneLabel();

        // Ensure "Send Email OTP" button stays hidden if email is already verified
        var emailVerified = $('#authyo_checkout_email_verified').val() === '1';
        if (emailVerified) {
            var $emailBtn = $('#authyo_send_email_otp');
            $emailBtn.addClass('authyo-verified')
                .attr('data-verified', 'true')
                .hide()
                .css({
                    'display': 'none',
                    'visibility': 'hidden',
                    'opacity': '0',
                    'height': '0',
                    'padding': '0',
                    'margin': '0',
                    'overflow': 'hidden'
                });
        }

        // Ensure "Send Phone OTP" button stays hidden if phone is already verified
        var phoneVerified = $('#authyo_checkout_phone_verified').val() === '1';
        if (phoneVerified) {
            var $phoneBtn = $('#authyo_send_phone_otp');
            $phoneBtn.addClass('authyo-verified')
                .attr('data-verified', 'true')
                .hide()
                .css({
                    'display': 'none',
                    'visibility': 'hidden',
                    'opacity': '0',
                    'height': '0',
                    'padding': '0',
                    'margin': '0',
                    'overflow': 'hidden'
                });

            // Hide fallback methods section if phone is already verified
            var $resendSection = $('#authyo-checkout-otp-phone-wrapper .resend-otp-section, .authyo-checkout-otp-phone-section .resend-otp-section');
            $resendSection.addClass('authyo-hidden').hide().css({
                'display': 'none',
                'visibility': 'hidden',
                'opacity': '0'
            });
        }

        // Ensure phone-input-wrapper inside OTP section is always hidden
        $('#authyo-checkout-otp-phone-wrapper .phone-input-wrapper').css({
            'display': 'none !important',
            'visibility': 'hidden !important',
            'height': '0',
            'margin': '0',
            'padding': '0',
            'overflow': 'hidden'
        }).hide();

        // Ensure OTP wrapper is positioned correctly (directly below phone-input-wrapper)
        var $phoneWrapper = $('#authyo-checkout-otp-phone-wrapper');
        if ($phoneWrapper.length > 0) {
            $phoneWrapper.css({
                'clear': 'both',
                'width': '100%',
                'margin-top': '10px',
                'display': 'block'
            });
        }
    });

    // Validation function
    function validateOtpBeforeCheckout() {
        // Check if OTP is enabled from settings (not just field existence)
        var emailEnabled = authyoOtpCheckout.emailEnabled;
        var phoneMethodsEnabled = authyoOtpCheckout.whatsappEnabled || authyoOtpCheckout.smsEnabled || authyoOtpCheckout.voiceEnabled;

        // If neither email nor phone OTP is enabled, allow submission
        if (!emailEnabled && !phoneMethodsEnabled) {
            return true; // OTP not enabled, allow submission
        }

        // Get email - works for both classic and block-based checkout, guest and logged-in users
        var $emailInput = $('input[name="billing_email"], #billing_email, input#billing_email, #email, input#email, input[type="email"][id="email"]');
        var email = '';

        // Try multiple methods to get email (handles both guest checkout and logged-in users)
        if ($emailInput.length > 0) {
            email = $emailInput.val() || $emailInput.attr('value') || $emailInput.data('value') || '';
        }

        // Try multiple selectors to get phone number
        var phone = $('#billing-phone, input#billing-phone, input[type="tel"][id="billing-phone"], input[name="billing_phone"]').val() || '';

        var hasError = false;
        var errorMessages = [];

        // Check email verification - REQUIRED if email is enabled and email exists
        if (emailEnabled) {
            if (!email || email.trim() === '') {
                // Email is required but missing
                errorMessages.push('Email address is required.');
                hasError = true;
            } else {
                var emailVerified = $('#authyo_checkout_email_verified').val() === '1';
                var emailMaskId = $('#authyo_checkout_email_maskid').val() || '';

                if (!emailVerified) {
                    var errorMsg = 'Please verify your email address with the OTP sent to your email before placing the order.';
                    showStatus($('#authyo_email_otp_status'), errorMsg, 'error');
                    errorMessages.push(errorMsg);
                    hasError = true;

                    // Show the OTP field if hidden (only if not verified)
                    var emailVerified = $('#authyo_checkout_email_verified').val() === '1';
                    if (!emailVerified) {
                        $('#authyo_checkout_email_otp').show();
                        $('#authyo_send_email_otp').removeClass('authyo-verified').removeAttr('data-verified').show();
                        $('#authyo-checkout-otp-email-wrapper').show();
                    }
                } else if (!emailMaskId) {
                    var errorMsg = 'Email OTP session expired. Please request a new OTP.';
                    showStatus($('#authyo_email_otp_status'), errorMsg, 'error');
                    errorMessages.push(errorMsg);
                    hasError = true;

                    // Replace "OTP sent successfully!" with "Resend OTP" on the send button
                    var $sendBtn = $('#authyo_send_email_otp');
                    if ($sendBtn.length > 0) {
                        // Check if button currently shows "OTP sent successfully!" message
                        var currentText = $sendBtn.text().trim();
                        var sentMessage = authyoOtpCheckout.messages.sent || 'OTP sent successfully!';
                        if (currentText === sentMessage || currentText.includes('sent successfully')) {
                            // Change button text to "Resend OTP"
                            $sendBtn.prop('disabled', false)
                                .text(authyoOtpCheckout.messages.resend || 'Resend OTP')
                                .removeClass('authyo-verified')
                                .removeAttr('data-verified')
                                .show()
                                .css({
                                    'display': 'inline-block',
                                    'visibility': 'visible',
                                    'opacity': '1'
                                });

                            // Show the OTP field and wrapper if hidden
                            $('#authyo_checkout_email_otp').show();
                            $('#authyo-checkout-otp-email-wrapper').show();
                        }
                    }
                }
            }
        }

        // Check phone verification - REQUIRED if phone methods are enabled and phone exists
        if (phoneMethodsEnabled) {
            if (!phone || phone.trim() === '') {
                // Phone is optional in WooCommerce, so only check if phone field has value
                // If phone field is empty, skip verification
            } else {
                var phoneVerified = $('#authyo_checkout_phone_verified').val() === '1';
                var phoneMaskId = $('#authyo_checkout_phone_maskid').val() || '';

                if (!phoneVerified) {
                    var errorMsg = 'Please verify your phone number with the OTP sent to your phone before placing the order.';
                    showStatus($('#authyo_phone_otp_status'), errorMsg, 'error');
                    errorMessages.push(errorMsg);
                    hasError = true;

                    // Show the OTP field if hidden
                    $('#authyo_checkout_phone_otp').show();
                    $('#authyo_send_phone_otp').show();
                    $('#authyo-checkout-otp-phone-wrapper').show();
                } else if (!phoneMaskId) {
                    var errorMsg = 'Phone OTP session expired. Please request a new OTP.';
                    showStatus($('#authyo_phone_otp_status'), errorMsg, 'error');
                    errorMessages.push(errorMsg);
                    hasError = true;

                    // Replace "OTP sent successfully!" with "Resend OTP" on the send button
                    var $sendBtn = $('#authyo_send_phone_otp');
                    if ($sendBtn.length > 0) {
                        // Check if button currently shows "OTP sent successfully!" message
                        var currentText = $sendBtn.text().trim();
                        var sentMessage = authyoOtpCheckout.messages.sent || 'OTP sent successfully!';
                        if (currentText === sentMessage || currentText.includes('sent successfully')) {
                            // Change button text to "Resend OTP"
                            $sendBtn.prop('disabled', false)
                                .text(authyoOtpCheckout.messages.resend || 'Resend OTP')
                                .removeClass('authyo-verified')
                                .removeAttr('data-verified')
                                .show()
                                .css({
                                    'display': 'inline-block',
                                    'visibility': 'visible',
                                    'opacity': '1'
                                });

                            // Show the OTP field and wrapper if hidden
                            $('#authyo_checkout_phone_otp').show();
                            $('#authyo-checkout-otp-phone-wrapper').show();
                        }
                    }
                }
            }
        }

        if (hasError) {

            // Scroll to first OTP field
            var $firstOtpField = $('#authyo-checkout-otp-email-wrapper').length > 0
                ? $('#authyo-checkout-otp-email-wrapper')
                : $('#authyo-checkout-otp-phone-wrapper');
            if ($firstOtpField.length > 0) {
                $('html, body').animate({
                    scrollTop: $firstOtpField.offset().top - 100
                }, 500);
            }

            // Show WooCommerce notice
            if (typeof wc_checkout_params !== 'undefined') {
                $('form.checkout').trigger('checkout_error', [errorMessages.join(' ')]);
            }

            // Also show alert for better visibility
            alert('OTP Verification Required\n\n' + errorMessages.join('\n') + '\n\nPlease verify your email/phone before placing the order.');

            return false;
        }

        // Ensure hidden fields are present and have correct values before submission
        var $emailVerifiedField = $('#authyo_checkout_email_verified');
        var $phoneVerifiedField = $('#authyo_checkout_phone_verified');

        if (emailEnabled && $emailVerifiedField.length === 0) {
            // Create the field if missing
            $('form.checkout, form.woocommerce-checkout').append('<input type="hidden" id="authyo_checkout_email_verified" name="authyo_checkout_email_verified" value="' + ($('#authyo_checkout_email_verified').val() || '0') + '" />');
        }

        if (phoneMethodsEnabled && $phoneVerifiedField.length === 0) {
            // Create the field if missing
            $('form.checkout, form.woocommerce-checkout').append('<input type="hidden" id="authyo_checkout_phone_verified" name="authyo_checkout_phone_verified" value="' + ($('#authyo_checkout_phone_verified').val() || '0') + '" />');
        }

        return true;
    }
});