jQuery(document).ready(function($) {
    // Prevent multiple initialization
    if (window.authyoAuthFormInitialized) {
        return;
    }
    window.authyoAuthFormInitialized = true;
    
    // Critical dependency check - ensure authyoOtpAuth is defined
    if (typeof authyoOtpAuth === 'undefined') {
        // Create a minimal fallback object to prevent crashes
        window.authyoOtpAuth = {
            ajaxurl: (typeof ajaxurl !== 'undefined') ? ajaxurl : '/wp-admin/admin-ajax.php',
            nonce: '',
            otpExpiry: 300,
            debug: true,
            messages: {
                login: 'Login',
                sending: 'Sending...',
                verifying: 'Verifying...',
                resend: 'Resend OTP',
                enter_email: 'Please enter a valid email or mobile number.',
                all_fields_required: 'All fields are required.',
                mobile_preview: 'Mobile number will be sent as: +'
            }
        };
    }
    
    // Match the actual generic auth form ID used in the template
    var $form = $('#authyo-otp-authentication-for-woocommerce-form');
    var $email = $('#auth-email');
    var $numberPreview = $('#auth-number-preview');
    var $otpFields = $('.otp-fields');
    var $sendOtpBtn = $('#send-auth-otp');
    var $verifyOtpBtn = $('#verify-auth-otp');
    var $resendOtpBtn = $('#resend-auth-otp');
    var $message = $('#auth-message');
    var timerInterval;
    var timeLeft = 300; // 5 minutes in seconds
    var messageTimeout;
    
    // Debug mode - set to false in production
    var debugMode = typeof authyoOtpAuth !== 'undefined' && authyoOtpAuth.debug === true;
    
    function debugLog(message, data) {
        // Debug logging disabled for production
    }

    function showMessage(message, isError = false) {
        // Clear any pending timeout to prevent duplicate animations
        if (messageTimeout) {
            clearTimeout(messageTimeout);
            messageTimeout = null;
        }
        
        // Stop any ongoing animations and clear the queue
        $message.stop(true, true);
        
        // Update and show the message
        $message.html(message).removeClass('success error').addClass(isError ? 'error' : 'success').css('display', 'block').css('opacity', '1');
        
        // Auto-hide messages after 3 seconds
        messageTimeout = setTimeout(function() {
            $message.fadeOut(400, function() {
                // Clear the message content, remove classes, and hide after fade out completes
                $message.empty().removeClass('success error').hide();
            });
            messageTimeout = null;
        }, 3000);
    }

    function toggleOTPFields(show) {
        $otpFields.toggle(show);
        $sendOtpBtn.toggle(!show);
        $verifyOtpBtn.toggle(show);
        if (show) {
            startTimer();
        }
    }

    function startTimer() {
        timeLeft = authyoOtpAuth.otpExpiry; // Use configured expiry time
        $resendOtpBtn.prop('disabled', true);
        $('.resend-otp-section').addClass('authyo-hidden'); // Hide fallback during timer
        updateTimerDisplay();
        
        clearInterval(timerInterval);
        timerInterval = setInterval(function() {
            timeLeft--;
            updateTimerDisplay();
            
            if (timeLeft <= 0) {
                clearInterval(timerInterval);
                $resendOtpBtn.prop('disabled', false).find('.timer').text(authyoOtpAuth.messages.resend);
                // Show resend methods after timer completes (only for mobile)
                showResendMethods();
            }
        }, 1000);
    }
    
    // Show resend methods only for mobile numbers, not emails
    function showResendMethods() {
        // Get current email/mobile value
        var currentValue = $email.val().trim();
        
        // Check email FIRST - if it's an email, never show fallback methods
        var isEmail = validateEmail(currentValue);
        if (isEmail) {
            // Hide fallback methods for email - use both class and inline style
            $('.resend-otp-section').addClass('authyo-hidden').hide();
            return;
        }
        
        // Only check if mobile after confirming it's not an email
        var isMobile = validateMobile(currentValue);
        if (isMobile) {
            // Show the resend section for mobile numbers - remove class and inline style
            $('.resend-otp-section').removeClass('authyo-hidden').show();
            
            // Enable all resend method buttons
            $('.resend-method-btn').prop('disabled', false);
        } else {
            // Hide for invalid input
            $('.resend-otp-section').addClass('authyo-hidden').hide();
        }
    }

    function updateTimerDisplay() {
        var minutes = Math.floor(timeLeft / 60);
        var seconds = timeLeft % 60;
        $resendOtpBtn.find('.timer').text(authyoOtpAuth.messages.resend + ' (' + minutes + ':' + (seconds < 10 ? '0' : '') + seconds + 's)');
    }

    function validateEmail(email) {
        var re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    }

    function validateMobile(mobile) {
        // Remove any non-digit characters except +
        mobile = mobile.replace(/[^\d+]/g, '');
        
        // Check if mobile number has at least 10 digits after +
        if (mobile.startsWith('+')) {
            return mobile.length >= 11 && /^\d+$/.test(mobile.substring(1));
        }
        
        // If no + prefix, check if it has 10-15 digits (standard mobile format or country code + mobile)
        return mobile.length >= 10 && mobile.length <= 15 && /^\d+$/.test(mobile);
    }

    // Show/hide number preview message
    $email.on('input', function() {
        var value = $(this).val().trim();
        var isEmail = validateEmail(value);
        var isMobile = validateMobile(value);
        
        if (isMobile) {
            $numberPreview.show().text(authyoOtpAuth.messages.mobile_preview + value.replace(/\D/g, ''));
        } else if (isEmail) {
            $numberPreview.hide();
        } else {
            $numberPreview.hide();
        }
    });

    $sendOtpBtn.on('click', function() {
        var email = $email.val().trim();
        
        if (!email) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        var isEmail = validateEmail(email);
        var isMobile = validateMobile(email);
        
        if (!isEmail && !isMobile) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        // Process mobile number - add + prefix only when sending
        if (isMobile) {
            // Remove all non-digit characters except +
            var cleanMobile = email.replace(/[^\d+]/g, '');
            // Only add + if it doesn't already have one
            email = cleanMobile.startsWith('+') ? cleanMobile : '+' + cleanMobile;
        }

        var $btn = $(this);
        $btn.prop('disabled', true).text(authyoOtpAuth.messages.sending);

        $.post(authyoOtpAuth.ajaxurl, {
            action: 'send_auth_otp',
            email: email,
            nonce: authyoOtpAuth.nonce
        }, function(response) {
            $btn.prop('disabled', false).text(authyoOtpAuth.messages.login);
            if (response.success) {
                // Check if this is a bypass message
                if (response.data.message && response.data.message.includes('bypassed for testing')) {
                    showMessage('Testing Mode: OTP verification bypassed for this test number. You can enter any 6-digit code to proceed.', false);
                } else {
                    showMessage(response.data.message, false);
                }
                $('#auth-mask-id').val(response.data.maskId);
                toggleOTPFields(true);
                
                // CRITICAL: Hide fallback methods immediately for email, show for mobile
                if (isEmail) {
                    // Email: Hide fallback methods completely with class AND inline style
                    $('.resend-otp-section').addClass('authyo-hidden').hide();
                } else {
                    // Mobile: Keep hidden during timer, will show after timer completes
                    $('.resend-otp-section').addClass('authyo-hidden').hide();
                }
            } else {
                showMessage(response.data.message, true);
            }
        });
    });

    $verifyOtpBtn.on('click', function() {
        var email = $email.val().trim();
        var otp = $('#auth-otp').val().trim();
        var maskId = $('#auth-mask-id').val().trim();

        if (!email || !otp || !maskId) {
            showMessage(authyoOtpAuth.messages.all_fields_required, true);
            return;
        }

        // Process mobile number - add + prefix only when sending
        var isMobile = validateMobile(email);
        if (isMobile) {
            // Remove all non-digit characters except +
            var cleanMobile = email.replace(/[^\d+]/g, '');
            // Only add + if it doesn't already have one
            email = cleanMobile.startsWith('+') ? cleanMobile : '+' + cleanMobile;
        }

        var $btn = $(this);
        $btn.prop('disabled', true).text(authyoOtpAuth.messages.verifying);

        $.post(authyoOtpAuth.ajaxurl, {
            action: 'authyo_otp_auth',
            email: email,
            otp: otp,
            maskId: maskId,
            _wpnonce: authyoOtpAuth.nonce
        }, function(response) {
            $btn.prop('disabled', false).text(authyoOtpAuth.messages.login);
            if (response.success) {
                showMessage(response.data.message, false);
                window.location.href = response.data.redirect;
            } else {
                showMessage(response.data.message, true);
            }
        });
    });

    // Handle resend method button clicks
    // Scope the event handler to only work within the auth form container
    $(document).on('click', '.authyo-otp-auth-form .resend-method-btn', function(e) {
        e.preventDefault(); // Prevent any default button behavior
        e.stopPropagation(); // Stop event from bubbling to other handlers
        
        var $btn = $(this);
        var method = $btn.data('method');
        var email = $email.val().trim();
        
        debugLog('Fallback button clicked', {
            method: method,
            email: email,
            buttonDisabled: $btn.prop('disabled'),
            ajaxurl: authyoOtpAuth.ajaxurl,
            hasNonce: !!authyoOtpAuth.nonce
        });

        if (!email || !method) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        // Validate that fallback methods can only be used with mobile numbers
        var isMobile = validateMobile(email);
        var isEmail = validateEmail(email);
        
        if (isEmail) {
            showMessage('Fallback methods (SMS, WhatsApp, Voice) are only available for mobile numbers.', true);
            return;
        }
        
        if (!isMobile) {
            showMessage(authyoOtpAuth.messages.enter_email, true);
            return;
        }

        // Process mobile number - add + prefix only when sending
        if (isMobile) {
            var cleanMobile = email.replace(/[^\d+]/g, '');
            email = cleanMobile.startsWith('+') ? cleanMobile : '+' + cleanMobile;
        }

        $btn.prop('disabled', true).text('Sending...');

        $.post(authyoOtpAuth.ajaxurl, {
            action: 'authyo_otp_auth_resend_fallback',
            email: email,
            method: method,
            action_type: 'auth',
            nonce: authyoOtpAuth.nonce
        }, function(response) {
            $btn.prop('disabled', false).text(method.charAt(0).toUpperCase() + method.slice(1));
            if (response.success) {
                showMessage(response.data.message, false);
                $('#auth-mask-id').val(response.data.maskId);
                startTimer();
            } else {
                showMessage(response.data.message, true);
            }
        }).fail(function(jqXHR, textStatus, errorThrown) {
            debugLog('AJAX request failed', {
                status: textStatus,
                error: errorThrown,
                response: jqXHR.responseText
            });
            $btn.prop('disabled', false).text(method.charAt(0).toUpperCase() + method.slice(1));
            showMessage('Connection error. Please check your internet connection and try again.', true);
        });
    });
}); 