<?php
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Create statistics table on plugin activation
 */
function aura_chat_create_statistics_table() {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'aura_statistics';
    $charset_collate = $wpdb->get_charset_collate();
    
    $sql = "CREATE TABLE $table_name (
        id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
        event_type varchar(50) NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY event_type_idx (event_type),
        KEY created_at_idx (created_at),
        KEY event_date_idx (event_type, created_at)
    ) $charset_collate;";
    
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

/**
 * Record a statistic event
 */
function aura_record_statistic($event_type) {
    global $wpdb;
    
    $valid_types = array('click', 'chat_open', 'conversion');
    if (!in_array($event_type, $valid_types, true)) {
        return false;
    }
    
    $table_name = $wpdb->prefix . 'aura_statistics';
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table for plugin statistics
    return $wpdb->insert(
        $table_name,
        array(
            'event_type' => sanitize_text_field($event_type),
            'created_at' => current_time('mysql')
        ),
        array('%s', '%s')
    );
}


/**
 * Get statistics with date filtering
 */
function aura_get_statistics($period = '7days') {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'aura_statistics';
    
    // Calculate date range using plugin timezone for consistency
    $date_value = '';
    $has_date_filter = true;
    
    switch ($period) {
        case 'today':
            $date_value = aura_chat_current_time('Y-m-d');
            break;
        case '7days':
            $date_value = aura_chat_current_time('Y-m-d 00:00:00', strtotime('-7 days'));
            break;
        case '14days':
            $date_value = aura_chat_current_time('Y-m-d 00:00:00', strtotime('-14 days'));
            break;
        case '30days':
            $date_value = aura_chat_current_time('Y-m-d 00:00:00', strtotime('-30 days'));
            break;
        case '90days':
            $date_value = aura_chat_current_time('Y-m-d 00:00:00', strtotime('-90 days'));
            break;
        case 'all':
        default:
            $has_date_filter = false;
            break;
    }
    
    // Get totals with proper prepared statements
    if ($has_date_filter && $period === 'today') {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $totals = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT event_type, COUNT(*) as count FROM {$wpdb->prefix}aura_statistics WHERE DATE(created_at) = %s GROUP BY event_type",
                $date_value
            )
        );
    } elseif ($has_date_filter) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $totals = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT event_type, COUNT(*) as count FROM {$wpdb->prefix}aura_statistics WHERE created_at >= %s GROUP BY event_type",
                $date_value
            )
        );
    } else {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $totals = $wpdb->get_results(
            "SELECT event_type, COUNT(*) as count FROM {$wpdb->prefix}aura_statistics GROUP BY event_type"
        );
    }
    
    $total_clicks = 0;
    $total_chat_opens = 0;
    $total_conversions = 0;
    
    foreach ($totals as $row) {
        switch ($row->event_type) {
            case 'click':
                $total_clicks = (int) $row->count;
                break;
            case 'chat_open':
                $total_chat_opens = (int) $row->count;
                break;
            case 'conversion':
                $total_conversions = (int) $row->count;
                break;
        }
    }
    
    // Get daily breakdown with proper prepared statements
    if ($has_date_filter && $period === 'today') {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $daily_data = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT DATE(created_at) as date, event_type, COUNT(*) as count FROM {$wpdb->prefix}aura_statistics WHERE DATE(created_at) = %s GROUP BY DATE(created_at), event_type ORDER BY date ASC",
                $date_value
            )
        );
    } elseif ($has_date_filter) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $daily_data = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT DATE(created_at) as date, event_type, COUNT(*) as count FROM {$wpdb->prefix}aura_statistics WHERE created_at >= %s GROUP BY DATE(created_at), event_type ORDER BY date ASC",
                $date_value
            )
        );
    } else {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $daily_data = $wpdb->get_results(
            "SELECT DATE(created_at) as date, event_type, COUNT(*) as count FROM {$wpdb->prefix}aura_statistics GROUP BY DATE(created_at), event_type ORDER BY date ASC"
        );
    }
    
    // Format daily data
    $daily_breakdown = array();
    foreach ($daily_data as $row) {
        $date = $row->date;
        if (!isset($daily_breakdown[$date])) {
            $daily_breakdown[$date] = array(
                'date' => $date,
                'clicks' => 0,
                'chat_opens' => 0,
                'conversions' => 0
            );
        }
        switch ($row->event_type) {
            case 'click':
                $daily_breakdown[$date]['clicks'] = (int) $row->count;
                break;
            case 'chat_open':
                $daily_breakdown[$date]['chat_opens'] = (int) $row->count;
                break;
            case 'conversion':
                $daily_breakdown[$date]['conversions'] = (int) $row->count;
                break;
        }
    }
    
    return array(
        'total_clicks' => $total_clicks,
        'total_chat_opens' => $total_chat_opens,
        'total_conversions' => $total_conversions,
        'daily_data' => array_values($daily_breakdown)
    );
}

/**
 * Reset all statistics
 */
function aura_reset_statistics() {
    global $wpdb;
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange
    return $wpdb->query("TRUNCATE TABLE {$wpdb->prefix}aura_statistics");
}

// AJAX Handlers

/**
 * AJAX: Track event
 */
function aura_ajax_track_event() {
    // Verify nonce for tracking - stop execution if invalid
    if (!check_ajax_referer('aura_track_event', 'nonce', false)) {
        wp_send_json_error(array('message' => esc_html__('Invalid security token', 'aura-chat-button')));
        return;
    }
    
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- sanitized below
    $event_type = isset($_POST['event_type']) ? sanitize_text_field(wp_unslash($_POST['event_type'])) : '';
    
    if (empty($event_type)) {
        wp_send_json_error(array('message' => esc_html__('Invalid event type', 'aura-chat-button')));
        return;
    }
    
    $result = aura_record_statistic($event_type);
    
    if ($result) {
        wp_send_json_success(array('message' => esc_html__('Event recorded', 'aura-chat-button')));
    } else {
        wp_send_json_error(array('message' => esc_html__('Failed to record event', 'aura-chat-button')));
    }
}
add_action('wp_ajax_aura_track_event', 'aura_ajax_track_event');
add_action('wp_ajax_nopriv_aura_track_event', 'aura_ajax_track_event');

/**
 * AJAX: Get statistics
 */
function aura_ajax_get_statistics() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => esc_html__('Unauthorized', 'aura-chat-button')));
        return;
    }
    
    check_ajax_referer('aura_get_stats', 'nonce');
    
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- sanitized below
    $period = isset($_POST['period']) ? sanitize_text_field(wp_unslash($_POST['period'])) : '7days';
    $stats = aura_get_statistics($period);
    
    wp_send_json_success($stats);
}
add_action('wp_ajax_aura_get_statistics', 'aura_ajax_get_statistics');

/**
 * AJAX: Reset statistics
 */
function aura_ajax_reset_statistics() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(array('message' => esc_html__('Unauthorized', 'aura-chat-button')));
        return;
    }
    
    check_ajax_referer('aura_reset_stats', 'nonce');
    
    $result = aura_reset_statistics();
    
    if ($result !== false) {
        wp_send_json_success(array('message' => esc_html__('Statistics reset successfully', 'aura-chat-button')));
    } else {
        wp_send_json_error(array('message' => esc_html__('Failed to reset statistics', 'aura-chat-button')));
    }
}
add_action('wp_ajax_aura_reset_statistics', 'aura_ajax_reset_statistics');
