<?php
if (!defined('ABSPATH')) {
    exit;
}

// Register Settings
function aura_chat_register_settings() {
    register_setting(
        'aura_chat_options_group',
        'aura_chat_options',
        array(
            'type' => 'array',
            'sanitize_callback' => 'aura_chat_sanitize_options',
            'default' => array(),
        )
    );
}
add_action('admin_init', 'aura_chat_register_settings');

/**
 * Sanitize plugin options
 *
 * @param array $input The input options array
 * @return array Sanitized options
 */
function aura_chat_sanitize_options($input) {
    // Get current options to merge and preserve existing values
    $existing = get_option('aura_chat_options', array());
    if (!is_array($existing)) {
        $existing = array();
    }
    
    $sanitized = $existing;
    
    if (!is_array($input)) {
        return $sanitized;
    }
    
    // 1. Text fields (Single line)
    $text_fields = array(
        'phone_number', 'timezone', 'time_start', 'time_end', 
        'button_style', 'position', 'tooltip_visibility'
    );
    
    foreach ($text_fields as $field) {
        if (isset($input[$field])) {
            $sanitized[$field] = sanitize_text_field($input[$field]);
        }
    }

    // 2. Message/Textarea fields (Preserve line breaks)
    $message_fields = array(
        'welcome_message', 'button_text', 'prefilled_message',
        'greeting_message', 'offline_message', 'offline_subtitle', 'tooltip_text'
    );

    foreach ($message_fields as $m_field) {
        if (isset($input[$m_field])) {
            $sanitized[$m_field] = sanitize_textarea_field($input[$m_field]);
        }
    }
    
    // 3. URL fields
    if (isset($input['agent_avatar'])) {
        $sanitized['agent_avatar'] = esc_url_raw($input['agent_avatar']);
    }
    
    // 4. Color fields
    if (isset($input['custom_button_color'])) {
        $sanitized['custom_button_color'] = sanitize_hex_color($input['custom_button_color']);
    }
    
    // 5. Integer fields (with specific clamping)
    $int_fields = array(
        'button_size_desktop', 'button_size_mobile', 'auto_popup',
        'position_offset_x', 'position_offset_y', 'mobile_position_offset_x',
        'mobile_position_offset_y', 'mobile_auto_popup_delay'
    );
    
    foreach ($int_fields as $field) {
        if (isset($input[$field])) {
            $val = intval($input[$field]);
            
            // Apply specific limits (Based on User Requirements)
            if ($field === 'button_size_desktop') {
                $val = max(50, min(70, $val));
            } elseif ($field === 'button_size_mobile') {
                $val = max(50, min(70, $val));
            } elseif ($field === 'mobile_position_offset_x') {
                $val = max(-30, min(40, $val));
            } elseif ($field === 'mobile_position_offset_y') {
                $val = max(-40, min(40, $val));
            } elseif (strpos($field, 'position_offset') !== false) {
                $val = max(-40, min(40, $val));
            }
            
            $sanitized[$field] = $val;
        }
    }
    
    // 6. Checkbox/Toggle fields (explicit 1/0)
    $checkbox_fields = array(
        'enable_availability',
        'hide_on_mobile',
        'mobile_tooltip_hidden'
    );
    
    foreach ($checkbox_fields as $cb_field) {
        $sanitized[$cb_field] = (isset($input[$cb_field]) && $input[$cb_field] == '1') ? 1 : 0;
    }
    
    // 7. Array fields (multi-select/checkbox groups)
    // Available Days
    $valid_days = array('Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun');
    if (isset($input['available_days']) && is_array($input['available_days'])) {
        $sanitized['available_days'] = array_intersect($input['available_days'], $valid_days);
    } else {
        $sanitized['available_days'] = array();
    }
    
    // Page Visibility
    $valid_pages = array('home', 'pages', 'posts', 'categories');
    if (isset($input['hide_on']) && is_array($input['hide_on'])) {
        $sanitized['hide_on'] = array_intersect($input['hide_on'], $valid_pages);
    } else {
        $sanitized['hide_on'] = array();
    }
    
    // WooCommerce Visibility
    $valid_woo_pages = array('shop', 'product', 'cart', 'checkout', 'account', 'product_category');
    if (isset($input['woo_pages']) && is_array($input['woo_pages'])) {
        $sanitized['woo_pages'] = array_intersect($input['woo_pages'], $valid_woo_pages);
    } else {
        $sanitized['woo_pages'] = array();
    }
    
    return $sanitized;
}

// Add Admin Menu
function aura_chat_add_admin_menu() {
    add_menu_page(
        'Aura Chat Button',
        'Aura Chat Button',
        'manage_options',
        'aura_chat',
        'aura_chat_options_page',
        'dashicons-admin-comments',
        100
    );
}
add_action('admin_menu', 'aura_chat_add_admin_menu');

// Enqueue Admin Scripts
function aura_chat_admin_scripts($hook) {
    // Only load on our plugin's admin page
    if ($hook !== 'toplevel_page_aura_chat') {
        return;
    }
    
    wp_enqueue_style('aura-admin-style', AURA_CHAT_URL . 'assets/css/aura-style.css', array(), '1.0.6');
    wp_enqueue_media();
    
    // Chart.js for statistics (local file, no CDN dependency)
    wp_enqueue_script('chartjs', AURA_CHAT_URL . 'assets/js/chart.min.js', array(), '4.5.1', true);
    
    wp_enqueue_script('aura-admin-script', AURA_CHAT_URL . 'assets/js/aura-admin.js', array('jquery', 'chartjs'), '1.0.6', true);
    
    // Pass nonces and translations for admin scripts
    wp_localize_script('aura-admin-script', 'auraAdmin', array(
        'resetNonce' => wp_create_nonce('aura_reset_stats'),
        'statsNonce' => wp_create_nonce('aura_get_stats'),
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'i18n' => array(
            'chooseImage' => esc_html__('Choose Image', 'aura-chat-button'),
            'removeImage' => esc_attr__('Remove Image', 'aura-chat-button'),
            'uploadImage' => esc_html__('Upload Image', 'aura-chat-button'),
            'changeImage' => esc_html__('Change Image', 'aura-chat-button'),
            'changesSaved' => esc_html__('Changes saved', 'aura-chat-button'),
            'chatOpens' => esc_html__('Chat Opens', 'aura-chat-button'),
            'conversions' => esc_html__('Conversions', 'aura-chat-button'),
            'months' => array(
                esc_html__('January', 'aura-chat-button'),
                esc_html__('February', 'aura-chat-button'),
                esc_html__('March', 'aura-chat-button'),
                esc_html__('April', 'aura-chat-button'),
                esc_html__('May', 'aura-chat-button'),
                esc_html__('June', 'aura-chat-button'),
                esc_html__('July', 'aura-chat-button'),
                esc_html__('August', 'aura-chat-button'),
                esc_html__('September', 'aura-chat-button'),
                esc_html__('October', 'aura-chat-button'),
                esc_html__('November', 'aura-chat-button'),
                esc_html__('December', 'aura-chat-button'),
            ),
        ),
    ));
}
add_action('admin_enqueue_scripts', 'aura_chat_admin_scripts');

// Enqueue Frontend Scripts
function aura_chat_enqueue_scripts() {
    // Frontend Styles & Scripts
    wp_enqueue_style('aura-frontend-style', AURA_CHAT_URL . 'assets/css/aura-frontend-style.css', array(), '1.0.6');
    
    // Add dynamic inline styles
    $dynamic_css = aura_generate_dynamic_css();
    if (!empty($dynamic_css)) {
        wp_add_inline_style('aura-frontend-style', $dynamic_css);
    }
    
    wp_enqueue_script('aura-frontend-script', AURA_CHAT_URL . 'assets/js/aura-script.js', array('jquery'), '1.0.6', true);

    // Pass PHP variables to JS
    $options = get_option('aura_chat_options');
    
    $prefilled_message = isset($options['prefilled_message']) && !empty($options['prefilled_message']) ? $options['prefilled_message'] : __('Hello, I have a question about your services.', 'aura-chat-button');
    
    wp_localize_script('aura-frontend-script', 'auraParams', array(
        'phoneNumber' => isset($options['phone_number']) ? $options['phone_number'] : '',
        'prefilledMessage' => $prefilled_message,
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'trackNonce' => wp_create_nonce('aura_track_event'),
    ));
}
add_action('wp_enqueue_scripts', 'aura_chat_enqueue_scripts');

/**
 * Generate dynamic CSS for the chat button widget
 * 
 * @return string Generated CSS string with escaped values
 */
function aura_generate_dynamic_css() {
    $options = get_option('aura_chat_options');
    
    // Button Sizes
    $button_size_desktop = isset($options['button_size_desktop']) ? intval($options['button_size_desktop']) : 60;
    $button_size_mobile = isset($options['button_size_mobile']) ? intval($options['button_size_mobile']) : 55;
    
    // Ensure sizes are within limits
    $button_size_desktop = max(50, min(70, $button_size_desktop));
    $button_size_mobile = max(50, min(70, $button_size_mobile));
    
    // Calculate icon sizes (Golden Mean: 52% for perfect balance) - FIXED: Comment says proportional but Style 1 needs specifically balanced size
    $icon_size_desktop = round($button_size_desktop * 0.49);
    $icon_size_mobile = round($button_size_mobile * 0.60);
    
    // Button Color
    $custom_color = isset($options['custom_button_color']) ? $options['custom_button_color'] : '#2ecc71';
    
    // Generate colors based on custom selection
    $main_color = $custom_color;
    $colors = array(
        'main' => $main_color,
        'light' => aura_adjust_brightness($main_color, 20),
        'lighter' => aura_adjust_brightness($main_color, 35),
        'dark' => aura_adjust_brightness($main_color, -15),
        'darker' => aura_adjust_brightness($main_color, -25)
    );
    
    // Escape all color values
    $c_main = esc_attr($colors['main']);
    $c_light = esc_attr($colors['light']);
    $c_lighter = esc_attr($colors['lighter']);
    $c_dark = esc_attr($colors['dark']);
    $c_darker = esc_attr($colors['darker']);
    $c_lighter_hover = esc_attr(aura_adjust_brightness($colors['lighter'], 15));
    $c_darker_hover = esc_attr(aura_adjust_brightness($colors['darker'], -10));
    
    // Helper function to convert hex to rgba
    $hex_to_rgba = function($hex, $alpha = 1) {
        $hex = ltrim($hex, '#');
        if (strlen($hex) == 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        return sprintf('rgba(%d, %d, %d, %.2f)', $r, $g, $b, $alpha);
    };
    
    // Generate rgba colors for box-shadow
    $rgba_light_80 = $hex_to_rgba($colors['light'], 0.5);
    $rgba_main_66 = $hex_to_rgba($colors['main'], 0.4);
    $rgba_main_4d = $hex_to_rgba($colors['main'], 0.3);
    $rgba_light_b3 = $hex_to_rgba($colors['light'], 0.7);
    $rgba_main_80 = $hex_to_rgba($colors['main'], 0.5);
    $rgba_main_99 = $hex_to_rgba($colors['main'], 0.6);
    $rgba_main_33 = $hex_to_rgba($colors['main'], 0.2);
    $rgba_main_1a = $hex_to_rgba($colors['main'], 0.1);
    $rgba_dark_4d = $hex_to_rgba($colors['dark'], 0.4);
    
    // Proportional Scaling Calculations for Styles 4 & 5 (Pill Styles)
    // Scale Factors (relative to button size)
    $pill_font_scale = 0.25; // Balanced text size
    $pill_padding_right_scale = 0.45; // Generous right padding
    $pill_gap_scale = 0.18; // Gap between icon and text start
    
    // Desktop Proportional Values
    $pill_font_size_desktop = round($button_size_desktop * $pill_font_scale);
    $pill_padding_right_desktop = round($button_size_desktop * $pill_padding_right_scale);
    $pill_icon_offset_desktop = round($button_size_desktop * $pill_gap_scale);
    $pill_radius_style4_desktop = round($button_size_desktop * 0.25);
    
    // Mobile Proportional Values - Ensure compact readability even at small sizes
    $pill_font_size_mobile = max(12, round($button_size_mobile * $pill_font_scale));
    $pill_padding_right_mobile = round($button_size_mobile * 0.35); // Slightly tighter on mobile
    $pill_icon_offset_mobile = round($button_size_mobile * $pill_gap_scale);
    $pill_radius_style4_mobile = round($button_size_mobile * 0.25);

    $css = "
        .aura-fab {
            width: {$button_size_desktop}px;
            height: {$button_size_desktop}px;
        }
        @keyframes aura-emerald-pulse-style1 {
            0% { transform: scale(1); box-shadow: 0 10px 30px {$rgba_main_4d}; }
            50% { transform: scale(1.02); box-shadow: 0 15px 45px {$rgba_main_66}; }
            100% { transform: scale(1); box-shadow: 0 10px 30px {$rgba_main_4d}; }
        }
        
        .aura-fab.style-1 svg.aura-fab-icon-chat,
        .aura-fab.style-1 svg.aura-fab-icon-close,
        .aura-fab.style-5 svg.aura-fab-icon-chat,
        .aura-fab.style-5 svg.aura-fab-icon-close,
        .aura-fab.style-6 svg.aura-fab-icon-chat,
        .aura-fab.style-6 svg.aura-fab-icon-close {
            width: {$icon_size_desktop}px !important;
            height: {$icon_size_desktop}px !important;
            color: #ffffff !important;
            fill: #ffffff !important;
        }
        /* Style 2 icon size - balanced with Style 1 (49% of button) */
        .aura-fab.style-2 svg.aura-fab-icon-chat,
        .aura-fab.style-2 svg.aura-fab-icon-close {
            width: {$icon_size_desktop}px !important;
            height: {$icon_size_desktop}px !important;
        }
        @media (max-width: 768px) {
            .aura-fab {
                width: {$button_size_mobile}px;
                height: {$button_size_mobile}px;
            }
            /* Style 1 mobile icon size */
            .aura-fab.style-1 svg.aura-fab-icon-chat,
            .aura-fab.style-1 svg.aura-fab-icon-close,
            .aura-fab.style-5 svg.aura-fab-icon-chat,
            .aura-fab.style-5 svg.aura-fab-icon-close,
            .aura-fab.style-6 svg.aura-fab-icon-chat,
            .aura-fab.style-6 svg.aura-fab-icon-close {
                width: {$icon_size_mobile}px !important;
                height: {$icon_size_mobile}px !important;
            }
            /* Style 2 mobile icon size - balanced */
            .aura-fab.style-2 svg.aura-fab-icon-chat,
            .aura-fab.style-2 svg.aura-fab-icon-close {
                width: {$icon_size_mobile}px !important;
                height: {$icon_size_mobile}px !important;
            }
        }
        
        /* Style 1 - Elegant Embossed 3D */
        .aura-fab.style-1 {
            border-radius: 50% !important;
            background: linear-gradient(145deg, {$c_light} 0%, {$c_main} 50%, {$c_dark} 100%) !important;
            border: 4px solid !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
            position: relative !important;
            transition: all 0.3s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
            animation: aura-emerald-pulse-style1 4s ease-in-out infinite !important;
        }
        .aura-fab.style-1::before {
            content: '' !important;
            position: absolute !important;
            inset: 0 !important;
            border-radius: 50% !important;
            background: linear-gradient(145deg, rgba(255, 255, 255, 0.2) 0%, transparent 40%) !important;
            pointer-events: none !important;
            z-index: 0 !important;
        }
        @media (hover: hover) {
            .aura-fab.style-1:hover {
                transform: translateY(-3px) scale(1.04) !important;
                background: linear-gradient(145deg, {$c_lighter} 0%, {$c_light} 50%, {$c_main} 100%) !important;
            }
        }
        .aura-fab.style-1:active {
            transform: translateY(-1px) scale(1.01) !important;
            background: linear-gradient(145deg, {$c_dark} 0%, {$c_main} 50%, {$c_light} 100%) !important;
            border-top-color: {$c_darker} !important;
            border-left-color: {$c_darker} !important;
            border-bottom-color: {$c_lighter} !important;
            border-right-color: {$c_lighter} !important;
        }

        /* Style 2 - Modern Square (Original Design) */
        .aura-fab.style-2 {
            background: linear-gradient(145deg, {$c_light} 0%, {$c_main} 50%, {$c_dark} 100%) !important;
            border-radius: 12px !important;
            border: none !important;
            box-shadow: 
                inset 0 2px 8px rgba(255, 255, 255, 0.5),
                inset 0 -2px 8px rgba(0, 0, 0, 0.2),
                0 10px 20px {$rgba_main_33} !important;
        }
        @media (hover: hover) {
            .aura-fab.style-2:hover {
                background: linear-gradient(145deg, {$c_lighter} 0%, {$c_light} 50%, {$c_main} 100%) !important;
            }
        }
        .aura-fab.style-2:active {
            background: linear-gradient(145deg, {$c_dark} 0%, {$c_main} 50%, {$c_light} 100%) !important;
            border-top-color: {$c_darker} !important;
            border-left-color: {$c_darker} !important;
            border-bottom-color: {$c_lighter} !important;
            border-right-color: {$c_lighter} !important;
        }
        

        /* Tooltip */
        .aura-fab-tooltip {
            background: linear-gradient(145deg, {$c_lighter} 0%, {$c_light} 20%, {$c_main} 50%, {$c_dark} 80%, {$c_darker} 100%) !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
        }
        .aura-fab-tooltip::before {
            display: none !important;
        }
        
        /* Style 3 (Premium Pill) */
        #aura-fab.style-3 {
            display: inline-flex !important;
            align-items: center !important;
            justify-content: flex-start !important;
            width: auto !important;
            min-width: calc({$button_size_desktop}px * 1.5) !important;
            max-width: calc(100vw - 40px) !important;
            height: {$button_size_desktop}px !important;
            padding: 0 {$pill_padding_right_desktop}px 0 calc({$button_size_desktop}px * 0.92) !important;
            border-radius: {$button_size_desktop}px !important;
            box-sizing: border-box !important;
            overflow: visible !important;
            background: linear-gradient(135deg, {$c_light} 0%, {$c_main} 50%, {$c_dark} 100%) !important;
            border: 3px solid !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
            box-shadow: 0 8px 25px {$rgba_main_33} !important;
            gap: 0 !important;
            transition: all 0.3s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
            position: relative;
            white-space: nowrap !important;
        }
        #aura-fab.style-3 svg.aura-fab-icon-chat,
        #aura-fab.style-3 svg.aura-fab-icon-close,
        #aura-fab.style-4 svg.aura-fab-icon-chat,
        #aura-fab.style-4 svg.aura-fab-icon-close {
            position: absolute !important;
            left: calc({$button_size_desktop}px / 2) !important;
            top: 50% !important;
            transform: translate(-50%, -50%) !important;
            margin: 0 !important;
            width: {$icon_size_desktop}px !important;
            height: {$icon_size_desktop}px !important;
            filter: drop-shadow(0 2px 4px rgba(0, 0, 0, 0.3)) !important;
            z-index: 2 !important;
        }
        .aura-fab.style-3 .aura-fab-tooltip, 
        .aura-fab.style-4 .aura-fab-tooltip {
            position: relative !important;
            display: inline-flex !important;
            align-items: center !important;
            background: transparent !important;
            border: none !important;
            box-shadow: none !important;
            padding: 0 !important;
            margin: 0 !important;
            opacity: 1 !important;
            visibility: visible !important;
            transform: none !important;
            pointer-events: none !important;
            right: auto !important;
            left: auto !important;
            inset: auto !important;
            white-space: nowrap !important;
            line-height: 1.2 !important;
            height: auto !important;
            flex-shrink: 0 !important;
            max-width: none !important;
        }
        .aura-fab.style-3 .aura-fab-tooltip.aura-tooltip-vis-none,
        .aura-fab.style-4 .aura-fab-tooltip.aura-tooltip-vis-none {
            display: none !important;
        }
        .aura-fab.style-3 .aura-fab-tooltip {
            color: white !important;
            font-weight: 700 !important;
            font-size: {$pill_font_size_desktop}px !important;
            text-shadow: 0 1px 3px rgba(0, 0, 0, 0.4) !important;
            letter-spacing: 0.4px !important;
        }
        @media (hover: hover) {
            .aura-fab.style-3:hover {
                transform: translateY(-3px) scale(1.02) !important;
                box-shadow: 0 12px 30px {$rgba_main_4d} !important;
            }
        }
        .aura-fab.style-3:active {
            transform: translateY(1px) scale(0.98) !important;
        }
        .aura-fab.style-3:active, .aura-fab.style-3.active {
            transform: translateY(2px) scale(0.98) !important;
            box-shadow:
                inset 0 4px 12px rgba(0, 0, 0, 0.3),
                inset 0 -2px 8px rgba(255, 255, 255, 0.2) !important;
        }

        /* Style 4 (Business Pill) */
        #aura-fab.style-4 {
            display: inline-flex !important;
            align-items: center !important;
            justify-content: flex-start !important;
            width: auto !important;
            min-width: calc({$button_size_desktop}px * 1.5) !important;
            height: {$button_size_desktop}px !important;
            padding: 0 {$pill_padding_right_desktop}px 0 calc({$button_size_desktop}px * 0.92) !important;
            border-radius: {$pill_radius_style4_desktop}px !important;
            box-sizing: border-box !important;
            overflow: visible !important;
            background: linear-gradient(135deg, {$c_light} 0%, {$c_main} 50%, {$c_dark} 100%) !important;
            border: 3px solid !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
            box-shadow: 0 8px 25px {$rgba_main_33} !important;
            gap: 0 !important;
            transition: all 0.3s cubic-bezier(0.34, 1.56, 0.64, 1) !important;
            position: relative;
            white-space: nowrap !important;
        }
        #aura-fab.style-4 svg.aura-fab-icon-chat,
        #aura-fab.style-4 svg.aura-fab-icon-close {
            position: absolute !important;
            left: calc({$button_size_desktop}px / 2) !important;
            top: 50% !important;
            transform: translate(-50%, -50%) !important;
            margin: 0 !important;
            width: {$icon_size_desktop}px !important;
            height: {$icon_size_desktop}px !important;
            filter: drop-shadow(0 2px 4px rgba(0, 0, 0, 0.3)) !important;
            z-index: 2 !important;
        }
        @media (hover: hover) {
            .aura-fab.style-4:hover {
                transform: translateY(-3px) scale(1.02) !important;
                box-shadow: 0 12px 30px {$rgba_main_4d} !important;
            }
        }
        .aura-fab.style-4:active {
            transform: translateY(1px) scale(0.98) !important;
        }
        .aura-fab.style-4:active, .aura-fab.style-4.active {
            transform: translateY(2px) scale(0.98) !important;
            box-shadow:
                inset 0 4px 12px rgba(0, 0, 0, 0.3),
                inset 0 -2px 8px rgba(255, 255, 255, 0.2) !important;
        }
        .aura-fab.style-4:active::before {
            border-top-color: rgba(0, 0, 0, 0.2) !important;
            border-left-color: rgba(0, 0, 0, 0.2) !important;
            border-bottom-color: rgba(255, 255, 255, 0.4) !important;
            border-right-color: rgba(255, 255, 255, 0.4) !important;
        }

        /* Style 5 - Executive 3D Square (Clean Refinement) */
        .aura-fab.style-5 {
            border-radius: 14px !important;
            padding: 0 !important;
            margin: 0 !important;
            box-sizing: border-box !important;
            background: linear-gradient(145deg, {$c_light} 0%, {$c_main} 50%, {$c_dark} 100%) !important;
            background-clip: padding-box !important;
            border: 5px solid !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
            position: relative !important;
            overflow: visible !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
            box-shadow: 
                /* CLEAN 3D Topography (Outer Shadow Removed) */
                inset 0 4px 10px rgba(255, 255, 255, 0.3),
                inset 0 -4px 10px rgba(0, 0, 0, 0.2) !important;
            transition: all 0.3s ease !important;
        }
        .aura-fab.style-5::before,
        .aura-fab.style-5 .aura-fab-inner::before,
        .aura-fab.style-5 .aura-fab-inner::after {
            display: none !important;
        }
        .aura-fab.style-5 .aura-fab-inner {
            position: relative !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
        }
        @media (hover: hover) {
            .aura-fab.style-5:hover {
                transform: none !important;
                box-shadow: 
                    inset 0 4px 10px rgba(255, 255, 255, 0.3),
                    inset 0 -4px 10px rgba(0, 0, 0, 0.2) !important;
            }
            .aura-fab.style-5:hover svg {
                filter: 
                    drop-shadow(0 4px 6px rgba(0, 0, 0, 0.5))
                    drop-shadow(0 -1px 2px rgba(255, 255, 255, 0.6))
                    brightness(1.2) !important;
            }
        }
        /* Style 5 Tooltip Refinement */
        .aura-fab.style-5 .aura-fab-tooltip {
            background: linear-gradient(145deg, {$c_light} 0%, {$c_main} 50%, {$c_dark} 100%) !important;
            border: 4px solid !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
            box-shadow: none !important;
            color: white !important;
            padding: 10px 20px !important;
            border-radius: 14px !important;
            font-weight: 700 !important;
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.3) !important;
        }
        .aura-fab.style-5:active {
            transform: translateY(0) scale(0.95) !important;
        }

        /* Icon Embossing - Making icons look carved/physical */
        .aura-fab.style-5 svg.aura-fab-icon-chat,
        .aura-fab.style-5 svg.aura-fab-icon-close {
            filter: 
                drop-shadow(0 2px 3px rgba(0, 0, 0, 0.4))
                drop-shadow(0 -1px 1px rgba(255, 255, 255, 0.5))
                brightness(1.1) !important;
            transform-style: preserve-3d !important;
        }

        /* Style 6 - Modern Glass (Full Sync with Style 1-5) */
        .aura-fab.style-6 {
            border-radius: 50% !important;
            background: linear-gradient(145deg, {$colors['light']} 0%, {$colors['main']} 50%, {$colors['dark']} 100%) !important;
            backdrop-filter: blur(12px) !important;
            -webkit-backdrop-filter: blur(12px) !important;
            border: 4px solid {$colors['lighter']} !important;
            box-shadow: 
                inset 0 4px 10px rgba(255, 255, 255, 0.15),
                inset 0 -4px 10px rgba(0, 0, 0, 0.3),
                0 8px 20px -10px rgba(0, 0, 0, 0.4) !important;
            transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275) !important;
            display: flex !important;
            align-items: center !important;
            justify-content: center !important;
        }
        @media (hover: hover) {
            .aura-fab.style-6:hover {
                background: linear-gradient(145deg, {$colors['main']} 0%, {$colors['dark']} 50%, {$colors['darker']} 100%) !important;
                transform: translateY(-5px) scale(1.05) !important;
                border-color: white !important;
            }
        }
        .aura-fab.style-6:active {
            transform: translateY(-2px) scale(0.98) !important;
        }
        
        .aura-fab.style-6 .aura-fab-tooltip {
            background: linear-gradient(145deg, {$colors['light']} 0%, {$colors['main']} 50%, {$colors['dark']} 100%) !important;
            backdrop-filter: blur(12px) !important;
            -webkit-backdrop-filter: blur(12px) !important;
            border: 3px solid {$colors['lighter']} !important;
            box-shadow: 
                inset 0 3px 8px rgba(255, 255, 255, 0.12),
                inset 0 -3px 8px rgba(0, 0, 0, 0.25),
                0 6px 15px -8px rgba(0, 0, 0, 0.3) !important;
            color: #ffffff !important;
            padding: 8px 18px !important;
            border-radius: 100px !important;
            font-size: 13px !important;
            font-weight: 700 !important;
            letter-spacing: 0.1px !important;
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.3) !important;
        }

        @media (max-width: 768px) {
            /* Style 3 mobile adjustments */
            #aura-fab.style-3 {
                width: auto !important;
                min-width: calc({$button_size_mobile}px * 1.5) !important;
                height: {$button_size_mobile}px !important;
                padding: 0 {$pill_padding_right_mobile}px 0 calc({$button_size_mobile}px * 0.92) !important;
                border-radius: {$button_size_mobile}px !important;
                display: flex !important;
                align-items: center !important;
                justify-content: flex-start !important;
                white-space: nowrap !important;
                overflow: visible !important;
            }
            #aura-fab.style-3.mobile-hidden {
                width: {$button_size_mobile}px !important;
                min-width: {$button_size_mobile}px !important;
                padding: 0 !important;
            }
            #aura-fab.style-3 .aura-fab-tooltip,
            #aura-fab.style-4 .aura-fab-tooltip {
                position: relative !important;
                display: inline-flex !important;
                align-items: center !important;
                justify-content: center !important;
                font-size: {$pill_font_size_mobile}px !important;
                font-weight: 700 !important;
                opacity: 1 !important;
                visibility: visible !important;
                transform: none !important;
                inset: auto !important;
                margin: 0 !important;
                padding: 0 !important;
                height: {$button_size_mobile}px !important;
                line-height: {$button_size_mobile}px !important;
                z-index: 3 !important;
            }
            #aura-fab.style-3 svg, #aura-fab.style-3.mobile-hidden svg {
                left: calc({$button_size_mobile}px / 2) !important;
                top: 50% !important;
                transform: translate(-50%, -50%) !important;
                width: {$icon_size_mobile}px !important;
                height: {$icon_size_mobile}px !important;
            }
            #aura-fab.style-3.mobile-hidden svg {
                left: 50% !important;
            }

            /* Style 4 mobile adjustments */
            #aura-fab.style-4 {
                width: auto !important;
                min-width: calc({$button_size_mobile}px * 1.5) !important;
                height: {$button_size_mobile}px !important;
                padding: 0 {$pill_padding_right_mobile}px 0 calc({$button_size_mobile}px * 0.92) !important;
                border-radius: {$pill_radius_style4_mobile}px !important;
                display: flex !important;
                align-items: center !important;
                justify-content: flex-start !important;
                white-space: nowrap !important;
                overflow: visible !important;
            }
            #aura-fab.style-4.mobile-hidden {
                width: {$button_size_mobile}px !important;
                min-width: {$button_size_mobile}px !important;
                padding: 0 !important;
            }
            #aura-fab.style-4 .aura-fab-tooltip {
                position: relative !important;
                display: inline-flex !important;
                align-items: center !important;
                justify-content: center !important;
                font-size: {$pill_font_size_mobile}px !important;
                font-weight: 700 !important;
                opacity: 1 !important;
                visibility: visible !important;
                transform: none !important;
                inset: auto !important;
                margin: 0 !important;
                padding: 0 !important;
                height: {$button_size_mobile}px !important;
                line-height: {$button_size_mobile}px !important;
                z-index: 3 !important;
            }
            #aura-fab.style-4 svg, #aura-fab.style-4.mobile-hidden svg {
                left: calc({$button_size_mobile}px / 2) !important;
                top: 50% !important;
                transform: translate(-50%, -50%) !important;
                width: {$icon_size_mobile}px !important;
                height: {$icon_size_mobile}px !important;
            }
            #aura-fab.style-4.mobile-hidden svg {
                left: 50% !important;
            }

            /* Style 5 mobile adjustments */
            .aura-fab.style-5 {
                width: {$button_size_mobile}px !important;
                height: {$button_size_mobile}px !important;
                border: 4px solid !important;
                border-top-color: {$c_lighter} !important;
                border-left-color: {$c_lighter} !important;
                border-bottom-color: {$c_darker} !important;
                border-right-color: {$c_darker} !important;
                box-shadow: 
                    /* Preserve refined 3D without mobile global shadow override */
                    inset 0 4px 8px rgba(255, 255, 255, 0.3),
                    inset 0 -4px 8px rgba(0, 0, 0, 0.2) !important;
            }
            .aura-fab.style-5 .aura-fab-tooltip {
                padding: 6px 14px !important;
                border-radius: 12px !important;
                font-size: 13px !important;
                font-weight: 700 !important;
                letter-spacing: 0.1px !important;
            }
        }
        @keyframes aura-emerald-pulse-style2 {
            0% { transform: scale(1); box-shadow: 0 10px 30px {$rgba_main_4d}; }
            50% { transform: scale(1.02); box-shadow: 0 15px 45px {$rgba_main_66}; }
            100% { transform: scale(1); box-shadow: 0 10px 30px {$rgba_main_4d}; }
        }
    ";
    
    return $css;
}

/**
 * Check if widget should be hidden on current page based on visibility settings
 * Logic: Show ONLY on selected pages
 */
function aura_should_hide_on_page($options) {
    $show_on = isset($options['hide_on']) ? (array)$options['hide_on'] : array();
    $woo_pages = isset($options['woo_pages']) ? (array)$options['woo_pages'] : array('shop', 'product', 'cart', 'checkout', 'account', 'product_category');
    
    // Check WooCommerce pages first (if WooCommerce is active)
    if (class_exists('WooCommerce')) {
        // Shop page
        if (in_array('shop', $woo_pages) && is_shop()) {
            return false; // Show on shop page
        }
        
        // Single product page
        if (in_array('product', $woo_pages) && is_product()) {
            return false; // Show on product pages
        }
        
        // Cart page
        if (in_array('cart', $woo_pages) && is_cart()) {
            return false; // Show on cart page
        }
        
        // Checkout page
        if (in_array('checkout', $woo_pages) && (is_checkout() || is_checkout_pay_page())) {
            return false; // Show on checkout page
        }
        
        // My Account page
        if (in_array('account', $woo_pages) && is_account_page()) {
            return false; // Show on account page
        }
        
        // Product category pages
        if (in_array('product_category', $woo_pages) && is_product_category()) {
            return false; // Show on product category pages
        }
        
        // If we're on a WooCommerce page but it's not in the selected list, check if it should be hidden
        if (is_woocommerce() || is_cart() || is_checkout() || is_account_page()) {
            // We're on a WooCommerce page, and it wasn't matched above, so hide it
            // unless it matches a regular page type
        }
    }
    
    // If nothing selected in both lists, hide everywhere
    if (empty($show_on) && empty($woo_pages)) {
        return true;
    }
    
    // Check if current page is in the "show" list
    if (in_array('home', $show_on) && (is_home() || is_front_page())) {
        return false; // Show on homepage
    }
    
    if (in_array('posts', $show_on) && is_single() && get_post_type() === 'post') {
        return false; // Show on blog posts
    }
    
    if (in_array('pages', $show_on) && is_page()) {
        // Don't show on WooCommerce special pages if they weren't explicitly selected
        if (class_exists('WooCommerce')) {
            if (is_cart() || is_checkout() || is_account_page()) {
                return true; // Hide on WooCommerce special pages unless explicitly selected
            }
        }
        return false; // Show on static pages
    }
    
    if (in_array('categories', $show_on) && is_category()) {
        return false; // Show on category pages
    }

    // If current page is not in the list, hide it
    return true;
}
