<?php
/**
 * Aura Chat Button - SVG Icons
 * Centralized icon management system
 * 
 * All icon output should be sanitized through wp_kses() with a strict allowlist
 * at the point of echo. The aura_get_icon() function returns raw SVG markup
 * from files in assets/icons/.
 * 
 * @package Aura_Chat_Button
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get allowed HTML for SVG icons
 *
 * @return array Allowed HTML tags and attributes for wp_kses
 */
function aura_get_allowed_svg_html() {
    return array(
        'svg' => array(
            'class' => true,
            'viewbox' => true,
            'fill' => true,
            'xmlns' => true,
            'style' => true,
            'width' => true,
            'height' => true,
            'id' => true,
            'data-icon' => true,
        ),
        'path' => array(
            'd' => true,
            'fill' => true,
        ),
    );
}

/**
 * Get SVG icon markup from file
 * 
 * This function reads an SVG file from assets/icons/ and adds classes/attributes.
 * Note: The output should be sanitized using wp_kses() when echoed for late escaping.
 * 
 * @param string $name  Icon name (filename without .svg)
 * @param string $class Optional CSS classes
 * @param array  $attrs Optional HTML attributes
 * @return string SVG HTML markup or empty string if not found
 */
function aura_get_icon($name, $class = '', $attrs = array()) {
    // Map of icon names to actual filenames if they differ
    $name = sanitize_key($name);
    
    // Construct the path to the SVG file
    $file_path = plugin_dir_path(dirname(__FILE__)) . 'assets/icons/' . $name . '.svg';
    
    if (!file_exists($file_path)) {
        return '';
    }
    
    $svg = file_get_contents($file_path);
    if (!$svg) {
        return '';
    }

    // Build attributes string
    $class_attr = 'aura-icon aura-icon-' . esc_attr($name);
    if (!empty($class)) {
        $class_attr .= ' ' . esc_attr($class);
    }
    
    $extra_attrs = ' class="' . $class_attr . '" fill="currentColor"';
    $allowed_attrs = array('style', 'width', 'height', 'id', 'data-icon');
    foreach ($attrs as $key => $value) {
        if (in_array($key, $allowed_attrs, true)) {
            $extra_attrs .= ' ' . esc_attr($key) . '="' . esc_attr($value) . '"';
        }
    }
    
    // Inject attributes into the <svg tag
    // We use a simple string replacement for reliability
    $svg = preg_replace('/^<svg/i', '<svg' . $extra_attrs, $svg);
    
    return $svg;
}

