<?php
/**
 * Aura Chat Button - Admin Page
 * 
 * SECURITY NOTE: Icon output is sanitized using wp_kses() at the point of echo
 * with a strict allowlist of SVG-specific tags and attributes.
 * 
 * @package Aura_Chat_Button
 * @see aura_get_icon() in includes/icons.php
 * @see aura_get_allowed_svg_html() for allowed SVG tags/attributes
 */

if (!defined('ABSPATH')) {
    exit;
}

// Admin Page HTML
function aura_chat_options_page() {
    $options = get_option('aura_chat_options');
    $valid_styles = array('style-1', 'style-2', 'style-3', 'style-4', 'style-5', 'style-6');
    $current_style = isset($options['button_style']) && !empty($options['button_style']) ? $options['button_style'] : 'style-1';
    
    
    // Final validation
    if (!in_array($current_style, $valid_styles, true)) $current_style = 'style-1';
    // Only show notice if settings were updated and we're on our plugin's page
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Only checking if settings were saved, no data processing
    $settings_updated = isset($_GET['settings-updated']) && sanitize_text_field(wp_unslash($_GET['settings-updated'])) === 'true' && isset($_GET['page']) && sanitize_text_field(wp_unslash($_GET['page'])) === 'aura_chat';
    ?>
    <div class="aura-admin-wrapper">
        <h1><span class="header-icon"><?php echo wp_kses(aura_get_icon('chat'), aura_get_allowed_svg_html()); ?></span> Aura Chat Button</h1>
        
        <?php if ($settings_updated): ?>
        <div class="aura-notice aura-notice-success" id="aura-save-notice">
            <?php echo wp_kses(aura_get_icon('check-circle'), aura_get_allowed_svg_html()); ?>
            <span><?php esc_html_e('Changes saved', 'aura-chat-button'); ?></span>
        </div>
        <?php endif; ?>

        <!-- Settings Tabs Navigation -->
        <div class="aura-settings-tabs">
            <button type="button" class="aura-tab-button active" data-tab="general"><?php esc_html_e('General Settings', 'aura-chat-button'); ?></button>
            <?php if (class_exists('WooCommerce')): ?>
            <button type="button" class="aura-tab-button" data-tab="woocommerce"><?php esc_html_e('WooCommerce', 'aura-chat-button'); ?></button>
            <?php endif; ?>
            <button type="button" class="aura-tab-button" data-tab="mobile"><?php esc_html_e('Mobile Settings', 'aura-chat-button'); ?></button>
            <button type="button" class="aura-tab-button" data-tab="statistics"><?php esc_html_e('Statistics', 'aura-chat-button'); ?></button>
        </div>

        <form method="post" action="options.php">
            <?php settings_fields('aura_chat_options_group'); ?>

            <!-- General Settings Tab -->
            <div id="general-settings" class="aura-tab-content active">
                <div class="aura-grid-container">
                <!-- Left Column: Main Settings -->
                <div class="aura-column-main">
                    
                    <!-- Profile & Messages -->
                    <div class="aura-card">
                        <h2><?php echo wp_kses(aura_get_icon('user-tie'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Profile & Messages', 'aura-chat-button'); ?></h2>
                        <div class="form-grid">
                            <div class="form-group">
                                <label><?php esc_html_e('Phone Number', 'aura-chat-button'); ?></label>
                                <input type="text" name="aura_chat_options[phone_number]" value="<?php echo isset($options['phone_number']) ? esc_attr($options['phone_number']) : ''; ?>" placeholder="<?php esc_attr_e('e.g. 905551234567', 'aura-chat-button'); ?>">
                                <p class="description"><?php esc_html_e('With country code (e.g., 90 for TR)', 'aura-chat-button'); ?></p>
                            </div>
                            <div class="form-group">
                                <label><?php esc_html_e('Avatar', 'aura-chat-button'); ?></label>
                                <div class="upload-group">
                                    <input type="hidden" name="aura_chat_options[agent_avatar]" value="<?php echo isset($options['agent_avatar']) ? esc_attr($options['agent_avatar']) : ''; ?>" class="avatar-input">
                                    <button type="button" class="button upload-avatar-btn"><?php echo !empty($options['agent_avatar']) ? esc_html__('Change', 'aura-chat-button') : esc_html__('Upload', 'aura-chat-button'); ?></button>
                                    <?php if(!empty($options['agent_avatar'])): ?>
                                        <div class="avatar-preview-wrapper">
                                            <img src="<?php echo esc_url($options['agent_avatar']); ?>" class="avatar-preview">
                                            <button type="button" class="button remove-avatar-btn" title="<?php esc_attr_e('Remove', 'aura-chat-button'); ?>" aria-label="<?php esc_attr_e('Remove', 'aura-chat-button'); ?>"><?php echo wp_kses(aura_get_icon('times'), aura_get_allowed_svg_html()); ?></button>
                                        </div>
                                    <?php else: ?>
                                        <div class="avatar-preview-wrapper avatar-placeholder">
                                            <div class="avatar-placeholder-icon"><?php echo wp_kses(aura_get_icon('chat'), aura_get_allowed_svg_html()); ?></div>
                                        </div>
                                        <img src="" class="avatar-preview" style="display:none;">
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="form-group">
                                <label for="welcome_message"><?php esc_html_e('Chat Header', 'aura-chat-button'); ?></label>
                                <input type="text" id="welcome_message" name="aura_chat_options[welcome_message]" value="<?php echo isset($options['welcome_message']) ? esc_attr($options['welcome_message']) : esc_attr__('Support Team', 'aura-chat-button'); ?>" placeholder="<?php esc_attr_e('Support Team', 'aura-chat-button'); ?>">
                            </div>
                            <div class="form-group">
                                <label for="button_text"><?php esc_html_e('Button Text', 'aura-chat-button'); ?></label>
                                <input type="text" id="button_text" name="aura_chat_options[button_text]" value="<?php echo isset($options['button_text']) ? esc_attr($options['button_text']) : esc_attr__('Start Chat', 'aura-chat-button'); ?>" placeholder="<?php esc_attr_e('Start Chat', 'aura-chat-button'); ?>">
                            </div>
                            <div class="form-group full-width">
                                <label for="greeting_message"><?php esc_html_e('Greeting Message', 'aura-chat-button'); ?></label>
                                <input type="text" id="greeting_message" name="aura_chat_options[greeting_message]" value="<?php echo isset($options['greeting_message']) ? esc_attr($options['greeting_message']) : esc_attr__('Hello! 👋 How can we help you today?', 'aura-chat-button'); ?>" placeholder="<?php esc_attr_e('Hello! 👋 How can we help you today?', 'aura-chat-button'); ?>">
                                <p class="description"><?php esc_html_e('Message shown in the chat bubble when online.', 'aura-chat-button'); ?></p>
                            </div>
                            <div class="form-group full-width">
                                <label for="prefilled_message"><?php esc_html_e('Prefilled Message', 'aura-chat-button'); ?></label>
                                <textarea id="prefilled_message" name="aura_chat_options[prefilled_message]" rows="3" placeholder="<?php esc_attr_e('Hello, I need help with...', 'aura-chat-button'); ?>"><?php echo isset($options['prefilled_message']) ? esc_textarea($options['prefilled_message']) : esc_attr__('Hello, I have a question about your services.', 'aura-chat-button'); ?></textarea>
                            </div>
                            <div class="form-group">
                                <label for="offline_message"><?php esc_html_e('Offline Title', 'aura-chat-button'); ?></label>
                                <input type="text" id="offline_message" name="aura_chat_options[offline_message]" value="<?php echo isset($options['offline_message']) ? esc_attr($options['offline_message']) : ''; ?>" placeholder="<?php esc_attr_e('We are currently offline. Please leave a message.', 'aura-chat-button'); ?>">
                            </div>
                            <div class="form-group">
                                <label for="offline_subtitle"><?php esc_html_e('Offline Subtitle', 'aura-chat-button'); ?></label>
                                <input type="text" id="offline_subtitle" name="aura_chat_options[offline_subtitle]" value="<?php echo isset($options['offline_subtitle']) ? esc_attr($options['offline_subtitle']) : ''; ?>" placeholder="<?php esc_attr_e('You can still leave us a message!', 'aura-chat-button'); ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Button Appearance -->
                    <div class="aura-card">
                        <h2><?php echo wp_kses(aura_get_icon('palette'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Button Styles', 'aura-chat-button'); ?></h2>
                        <div class="form-grid">


                            <input type="hidden" id="custom_button_color" name="aura_chat_options[custom_button_color]" value="<?php echo isset($options['custom_button_color']) ? esc_attr($options['custom_button_color']) : '#2ecc71'; ?>">
                            <div class="form-group">
                                <label for="button_size_desktop"><?php esc_html_e('Size (px)', 'aura-chat-button'); ?></label>
                                <?php $bs_desktop = isset($options['button_size_desktop']) ? intval($options['button_size_desktop']) : 60; ?>
                                <input type="number" id="button_size_desktop" name="aura_chat_options[button_size_desktop]" value="<?php echo esc_attr(max(50, min(70, $bs_desktop))); ?>" min="50" max="70" step="1" placeholder="60">
                            </div>
                            <div class="form-group">
                                <label for="tooltip_text"><?php esc_html_e('Label Text', 'aura-chat-button'); ?></label>
                                <input type="text" id="tooltip_text" name="aura_chat_options[tooltip_text]" value="<?php echo isset($options['tooltip_text']) ? esc_attr($options['tooltip_text']) : esc_attr__('Need Help?', 'aura-chat-button'); ?>" placeholder="<?php esc_attr_e('Need Help?', 'aura-chat-button'); ?>">
                            </div>
                            <div class="form-group">
                                <label for="tooltip_visibility"><?php esc_html_e('Label Visibility', 'aura-chat-button'); ?></label>
                                <select id="tooltip_visibility" name="aura_chat_options[tooltip_visibility]">
                                    <option value="hover" <?php selected(isset($options['tooltip_visibility']) ? $options['tooltip_visibility'] : 'always', 'hover'); ?>><?php esc_html_e('On Hover', 'aura-chat-button'); ?></option>
                                    <option value="always" <?php selected(isset($options['tooltip_visibility']) ? $options['tooltip_visibility'] : 'always', 'always'); ?>><?php esc_html_e('Always', 'aura-chat-button'); ?></option>
                                    <option value="none" <?php selected(isset($options['tooltip_visibility']) ? $options['tooltip_visibility'] : 'always', 'none'); ?>><?php esc_html_e('Hidden', 'aura-chat-button'); ?></option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label for="position"><?php esc_html_e('Position', 'aura-chat-button'); ?></label>
                                <select id="position" name="aura_chat_options[position]">
                                    <?php $current_pos = isset($options['position']) ? $options['position'] : 'right'; ?>
                                    <option value="right" <?php selected($current_pos, 'right'); ?>><?php esc_html_e('Bottom Right', 'aura-chat-button'); ?></option>
                                    <option value="left" <?php selected($current_pos, 'left'); ?>><?php esc_html_e('Bottom Left', 'aura-chat-button'); ?></option>
                                </select>
                            </div>

                            <div class="form-group">
                                <label for="position_offset_x"><?php esc_html_e('Horizontal Position (px)', 'aura-chat-button'); ?></label>
                                <?php $pos_x = isset($options['position_offset_x']) ? intval($options['position_offset_x']) : 0; ?>
                                <input type="number" id="position_offset_x" name="aura_chat_options[position_offset_x]" value="<?php echo esc_attr(max(-40, min(40, $pos_x))); ?>" min="-40" max="40" step="1" placeholder="0">
                            </div>
                            <div class="form-group">
                                <label for="position_offset_y"><?php esc_html_e('Vertical Position (px)', 'aura-chat-button'); ?></label>
                                <?php $pos_y = isset($options['position_offset_y']) ? intval($options['position_offset_y']) : 0; ?>
                                <input type="number" id="position_offset_y" name="aura_chat_options[position_offset_y]" value="<?php echo esc_attr(max(-40, min(40, $pos_y))); ?>" min="-40" max="40" step="1" placeholder="0">
                            </div>

                            <div class="form-group full-width" style="margin-top: 20px; border-top: 1px dashed #e2e8f0; padding-top: 20px;">
                                <label for="auto_popup"><?php esc_html_e('Auto Open (seconds)', 'aura-chat-button'); ?></label>
                                <input type="number" id="auto_popup" name="aura_chat_options[auto_popup]" value="<?php echo isset($options['auto_popup']) ? esc_attr($options['auto_popup']) : ''; ?>" placeholder="<?php esc_attr_e('0 = disabled', 'aura-chat-button'); ?>">
                                <p class="description">
                                    <?php esc_html_e('Delay (seconds) before auto-open. Shown once every 24 hours per visitor for better UX.', 'aura-chat-button'); ?>
                                </p>
                            </div>
                            <div class="form-group full-width" style="margin-top: 20px; border-top: 1px dashed #e2e8f0; padding-top: 20px;">
                                <label><?php esc_html_e('Button Colors', 'aura-chat-button'); ?></label>
                                <div class="color-presets">
                                    <?php
                                    $presets = [
                                        '#2ecc71',
                                        '#8C3573',
                                        '#BA8E23',
                                        '#803120',
                                        '#0084FF',
                                        '#E74C3C',
                                        '#FF5C00',
                                        '#9B59B6',
                                        '#1ABC9C',
                                        '#2C3E50'
                                    ];
                                    foreach ($presets as $color) {
                                        echo '<div class="color-preset-item" style="background-color: ' . esc_attr($color) . ';" data-color="' . esc_attr($color) . '"></div>';
                                    }
                                    ?>
                                </div>
                            </div>

                        </div>
                    </div>


                </div>

                <!-- Right Column: Advanced Settings -->
                <div class="aura-column-side">
                    
                    <!-- Scheduling -->
                    <div class="aura-card">
                        <h2><?php echo wp_kses(aura_get_icon('clock'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Business Hours & Scheduling', 'aura-chat-button'); ?></h2>
                        
                        <div class="form-group">
                            <label>
                                <input type="checkbox" name="aura_chat_options[enable_availability]" value="1" <?php checked(isset($options['enable_availability']) ? $options['enable_availability'] : 0, 1); ?>>
                                <?php esc_html_e('Enable Business Hours', 'aura-chat-button'); ?>
                            </label>
                            <p class="description">
                                <?php esc_html_e('Automatically manage the chat button visibility based on your working days and hours.', 'aura-chat-button'); ?>
                                <br>
                                <span style="color: #dc3232; font-weight: 500;">
                                    <?php esc_html_e('Important: If unchecked, the button will be online 24/7.', 'aura-chat-button'); ?>
                                </span>
                            </p>
                        </div>

                        <div class="form-group">
                            <label for="timezone"><?php esc_html_e('Time Zone', 'aura-chat-button'); ?></label>
                            <select id="timezone" name="aura_chat_options[timezone]">
                                <?php
                                $current_timezone = isset($options['timezone']) ? $options['timezone'] : get_option('timezone_string', 'UTC');
                                if (empty($current_timezone)) {
                                    $current_timezone = 'UTC';
                                }
                                
                                // Use WordPress default timezone function - output is pre-escaped by WordPress
                                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- wp_timezone_choice returns escaped HTML
                                echo wp_timezone_choice($current_timezone);
                                ?>
                            </select>
                            <p class="description" style="color: #dc3232;"><?php esc_html_e('Crucial: Select your local time zone to ensure the scheduler works accurately exactly when you expect.', 'aura-chat-button'); ?></p>
                        </div>

                        <div class="form-group">
                            <label><?php esc_html_e('Available Days', 'aura-chat-button'); ?></label>
                            <div class="days-grid">
                                <?php
                                $days = [
                                    'Mon' => __('Monday', 'aura-chat-button'),
                                    'Tue' => __('Tuesday', 'aura-chat-button'), 
                                    'Wed' => __('Wednesday', 'aura-chat-button'),
                                    'Thu' => __('Thursday', 'aura-chat-button'),
                                    'Fri' => __('Friday', 'aura-chat-button'),
                                    'Sat' => __('Saturday', 'aura-chat-button'),
                                    'Sun' => __('Sunday', 'aura-chat-button')
                                ];
                                $default_days = array('Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun');
                                $saved_days = isset($options['available_days']) ? $options['available_days'] : $default_days;
                                foreach ($days as $day => $full_name) {
                                    $is_checked = in_array($day, $saved_days, true);
                                    $is_weekend = ($day === 'Sat' || $day === 'Sun');
                                    echo '<label class="day-chip ' . ($is_checked ? 'active' : 'inactive') . ' ' . ($is_weekend ? 'weekend' : '') . '">';
                                    echo '<input type="checkbox" name="aura_chat_options[available_days][]" value="' . esc_attr($day) . '" ' . ($is_checked ? 'checked' : '') . '>';
                                    echo '<span class="day-name">' . esc_html($full_name) . '</span>';
                                    echo '<span class="day-check">' . wp_kses(aura_get_icon('check'), aura_get_allowed_svg_html()) . '</span>';
                                    echo '</label>';
                                }
                                ?>
                            </div>
                        </div>

                        <div class="form-grid">
                            <div class="form-group">
                                <label for="time_start"><?php esc_html_e('Start Time', 'aura-chat-button'); ?></label>
                                <input type="time" id="time_start" name="aura_chat_options[time_start]" value="<?php echo isset($options['time_start']) ? esc_attr($options['time_start']) : '09:00'; ?>">
                            </div>
                            <div class="form-group">
                                <label for="time_end"><?php esc_html_e('End Time', 'aura-chat-button'); ?></label>
                                <input type="time" id="time_end" name="aura_chat_options[time_end]" value="<?php echo isset($options['time_end']) ? esc_attr($options['time_end']) : '18:00'; ?>">
                            </div>
                        </div>
                    </div>

                    <!-- Page Visibility -->
                    <div class="aura-card">
                        <h2><?php echo wp_kses(aura_get_icon('eye'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Page Visibility', 'aura-chat-button'); ?></h2>
                        <p class="description"><?php esc_html_e('Show on selected pages.', 'aura-chat-button'); ?></p>

                        <?php
                        $default_pages = array('home', 'pages', 'posts', 'categories');
                        $hide_on = isset($options['hide_on']) ? (array)$options['hide_on'] : $default_pages;
                        ?>

                        <div class="hide-options-grid">
                            <label class="hide-option-item <?php echo in_array('home', $hide_on) ? 'active' : 'inactive'; ?>">
                                <input type="checkbox" name="aura_chat_options[hide_on][]" value="home" <?php checked(in_array('home', $hide_on)); ?>>
                                <?php echo wp_kses(aura_get_icon('home'), aura_get_allowed_svg_html()); ?>
                                <span><?php esc_html_e('Homepage', 'aura-chat-button'); ?></span>
                                <small><?php esc_html_e('Front page of your site', 'aura-chat-button'); ?></small>
                            </label>

                            <label class="hide-option-item <?php echo in_array('pages', $hide_on) ? 'active' : 'inactive'; ?>">
                                <input type="checkbox" name="aura_chat_options[hide_on][]" value="pages" <?php checked(in_array('pages', $hide_on)); ?>>
                                <?php echo wp_kses(aura_get_icon('file'), aura_get_allowed_svg_html()); ?>
                                <span><?php esc_html_e('Static Pages', 'aura-chat-button'); ?></span>
                                <small><?php esc_html_e('About, Contact, etc.', 'aura-chat-button'); ?></small>
                            </label>

                            <label class="hide-option-item <?php echo in_array('posts', $hide_on) ? 'active' : 'inactive'; ?>">
                                <input type="checkbox" name="aura_chat_options[hide_on][]" value="posts" <?php checked(in_array('posts', $hide_on)); ?>>
                                <?php echo wp_kses(aura_get_icon('newspaper'), aura_get_allowed_svg_html()); ?>
                                <span><?php esc_html_e('Blog Posts', 'aura-chat-button'); ?></span>
                                <small><?php esc_html_e('All single blog posts', 'aura-chat-button'); ?></small>
                            </label>

                            <label class="hide-option-item <?php echo in_array('categories', $hide_on) ? 'active' : 'inactive'; ?>">
                                <input type="checkbox" name="aura_chat_options[hide_on][]" value="categories" <?php checked(in_array('categories', $hide_on)); ?>>
                                <?php echo wp_kses(aura_get_icon('folder'), aura_get_allowed_svg_html()); ?>
                                <span><?php esc_html_e('Categories', 'aura-chat-button'); ?></span>
                                <small><?php esc_html_e('Category listing pages', 'aura-chat-button'); ?></small>
                            </label>
                        </div>
                    </div>


                    <!-- Position Preview -->
                    <div class="aura-card">
                        <h2><?php echo wp_kses(aura_get_icon('desktop'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Desktop Preview', 'aura-chat-button'); ?></h2>
                        <div id="desktop-preview-container" style="display: flex; justify-content: center; padding: 15px 0;">
                            <!-- Desktop Monitor Mockup -->
                            <div style="position: relative;">
                                <!-- Monitor Frame -->
                                <div style="background: linear-gradient(180deg, #2d3748 0%, #1a202c 100%); border-radius: 10px; padding: 8px 8px 20px 8px; box-shadow: 0 10px 30px -5px rgba(0, 0, 0, 0.3);">
                                    <!-- Screen -->
                                    <div id="desktop-preview-box" style="position: relative; width: 220px; height: 130px; background: linear-gradient(180deg, #ffffff 0%, #f8fafc 100%); border-radius: 4px; overflow: hidden;">
                                        <!-- Browser Bar -->
                                        <div style="height: 20px; background: #f1f5f9; border-bottom: 1px solid #e2e8f0; display: flex; align-items: center; padding: 0 8px; gap: 4px;">
                                            <span style="width: 6px; height: 6px; background: #ef4444; border-radius: 50%;"></span>
                                            <span style="width: 6px; height: 6px; background: #f59e0b; border-radius: 50%;"></span>
                                            <span style="width: 6px; height: 6px; background: #22c55e; border-radius: 50%;"></span>
                                            <div style="flex: 1; margin-left: 8px; height: 10px; background: #e2e8f0; border-radius: 3px;"></div>
                                        </div>
                                        
                                        <!-- Page Content -->
                                        <div style="padding: 10px;">
                                            <div style="height: 5px; background: #cbd5e1; border-radius: 2px; width: 50%; margin-bottom: 6px;"></div>
                                            <div style="height: 3px; background: #e2e8f0; border-radius: 2px; width: 85%; margin-bottom: 4px;"></div>
                                            <div style="height: 3px; background: #e2e8f0; border-radius: 2px; width: 70%; margin-bottom: 4px;"></div>
                                            <div style="height: 3px; background: #e2e8f0; border-radius: 2px; width: 80%; margin-bottom: 8px;"></div>
                                            <div style="height: 25px; background: #e2e8f0; border-radius: 4px; width: 100%;"></div>
                                        </div>
                                        
                                        <!-- Chat Widget -->
                                        <div class="position-preview-widget" id="preview-widget" style="position: absolute; width: 28px; height: 28px; background: linear-gradient(145deg, #2ecc71 0%, #25D366 50%, #128C7E 100%); border-radius: 50%; display: flex; align-items: center; justify-content: center; color: white; bottom: 8px; right: 8px; box-shadow: 0 3px 10px rgba(37, 211, 102, 0.4); border: 2px solid white; z-index: 1;">
                                            <?php echo wp_kses(aura_get_icon('chat', '', ['style' => 'width: 14px; height: 14px;']), aura_get_allowed_svg_html()); ?>
                                        </div>
                                    </div>
                                </div>
                                <!-- Monitor Stand -->
                                <div style="width: 40px; height: 20px; background: linear-gradient(180deg, #4a5568 0%, #2d3748 100%); margin: 0 auto; border-radius: 0 0 4px 4px;"></div>
                                <div style="width: 70px; height: 6px; background: linear-gradient(180deg, #4a5568 0%, #2d3748 100%); margin: 0 auto; border-radius: 0 0 6px 6px;"></div>
                            </div>
                        </div>

                        <!-- Compact Style Selector -->
                        <div style="margin-top: 20px; border-top: 1px dashed #e2e8f0; padding-top: 20px;">
                            <label style="display: block; margin-bottom: 12px; font-weight: 600; color: #475569; font-size: 13px;"><?php esc_html_e('Button Styles', 'aura-chat-button'); ?></label>
                            <div class="style-selector-grid">
                                <?php
                                $styles = [
                                    'style-1' => __('Style 1', 'aura-chat-button'),
                                    'style-2' => __('Style 2', 'aura-chat-button'),
                                    'style-3' => __('Style 3', 'aura-chat-button'),
                                    'style-4' => __('Style 4', 'aura-chat-button'),
                                    'style-5' => __('Style 5', 'aura-chat-button'),
                                    'style-6' => __('Style 6', 'aura-chat-button'),
                                ];
                                foreach ($styles as $val => $label) :
                                    $active_class = ($current_style === $val) ? 'active' : '';
                                    $is_pill = ($val === 'style-3' || $val === 'style-4');
                                ?>
                                <div class="style-option <?php echo esc_attr($active_class); ?>" data-style="<?php echo esc_attr($val); ?>">
                                    <div class="style-preview-canvas <?php echo esc_attr($val); ?> <?php echo $is_pill ? 'is-pill' : ''; ?>">
                                        <div class="style-preview-button">
                                            <?php echo wp_kses(aura_get_icon('chat'), aura_get_allowed_svg_html()); ?>
                                            <span class="preview-label-text"><?php echo (isset($options['tooltip_text']) && !empty($options['tooltip_text'])) ? esc_html($options['tooltip_text']) : esc_html__('Need Help?', 'aura-chat-button'); ?></span>
                                        </div>
                                    </div>
                                    <span class="style-name"><?php echo esc_html($label); ?></span>
                                    <div class="style-check-mark"><?php echo wp_kses(aura_get_icon('check'), aura_get_allowed_svg_html()); ?></div>
                                </div>
                                <?php endforeach; ?>
                            </div>
                            <input type="hidden" id="button_style" name="aura_chat_options[button_style]" value="<?php echo esc_attr($current_style); ?>">
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- WooCommerce Settings Tab -->
        <?php if (class_exists('WooCommerce')): ?>
        <div id="woocommerce-settings" class="aura-tab-content">
            <div class="aura-grid-container">
                <div class="aura-column-main">
                    <div class="aura-card">
                        <div class="woo-premium-header">
                            <h2 style="margin: 0;"><?php echo wp_kses(aura_get_icon('chat'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('WooCommerce Page Visibility', 'aura-chat-button'); ?></h2>

                        </div>
                        <div class="woo-info-box">
                            <p class="description"><?php esc_html_e('Enable the chat button on specific WooCommerce pages to boost customer engagement and sales.', 'aura-chat-button'); ?></p>
                        </div>

                        <?php
                        $default_woo_pages = array('shop', 'product', 'cart', 'checkout', 'account', 'product_category');
                        $woo_pages = isset($options['woo_pages']) ? (array)$options['woo_pages'] : $default_woo_pages;
                        ?>

                        <div class="woo-premium-grid">
                            <label class="woo-toggle-item <?php echo in_array('shop', $woo_pages) ? 'active' : ''; ?>" data-color="emerald">
                                <input type="checkbox" name="aura_chat_options[woo_pages][]" value="shop" <?php checked(in_array('shop', $woo_pages)); ?>>
                                <div class="woo-toggle-content">
                                    <div class="woo-toggle-icon">
                                        <?php echo wp_kses(aura_get_icon('home'), aura_get_allowed_svg_html()); ?>
                                    </div>
                                    <div class="woo-toggle-info">
                                        <span class="woo-toggle-title"><?php esc_html_e('Shop Page', 'aura-chat-button'); ?></span>
                                        <span class="woo-toggle-desc"><?php esc_html_e('Main store page with all products', 'aura-chat-button'); ?></span>
                                    </div>
                                </div>
                                <div class="woo-toggle-switch"></div>
                            </label>

                            <label class="woo-toggle-item <?php echo in_array('product', $woo_pages) ? 'active' : ''; ?>" data-color="emerald">
                                <input type="checkbox" name="aura_chat_options[woo_pages][]" value="product" <?php checked(in_array('product', $woo_pages)); ?>>
                                <div class="woo-toggle-content">
                                    <div class="woo-toggle-icon">
                                        <?php echo wp_kses(aura_get_icon('file'), aura_get_allowed_svg_html()); ?>
                                    </div>
                                    <div class="woo-toggle-info">
                                        <span class="woo-toggle-title"><?php esc_html_e('Product Pages', 'aura-chat-button'); ?></span>
                                        <span class="woo-toggle-desc"><?php esc_html_e('Product detail and purchase pages', 'aura-chat-button'); ?></span>
                                    </div>
                                </div>
                                <div class="woo-toggle-switch"></div>
                            </label>

                            <label class="woo-toggle-item <?php echo in_array('cart', $woo_pages) ? 'active' : ''; ?>" data-color="emerald">
                                <input type="checkbox" name="aura_chat_options[woo_pages][]" value="cart" <?php checked(in_array('cart', $woo_pages)); ?>>
                                <div class="woo-toggle-content">
                                    <div class="woo-toggle-icon">
                                        <?php echo wp_kses(aura_get_icon('chat'), aura_get_allowed_svg_html()); ?>
                                    </div>
                                    <div class="woo-toggle-info">
                                        <span class="woo-toggle-title"><?php esc_html_e('Shopping Cart', 'aura-chat-button'); ?></span>
                                        <span class="woo-toggle-desc"><?php esc_html_e('Cart page before checkout', 'aura-chat-button'); ?></span>
                                    </div>
                                </div>
                                <div class="woo-toggle-switch"></div>
                            </label>

                            <label class="woo-toggle-item <?php echo in_array('checkout', $woo_pages) ? 'active' : ''; ?>" data-color="emerald">
                                <input type="checkbox" name="aura_chat_options[woo_pages][]" value="checkout" <?php checked(in_array('checkout', $woo_pages)); ?>>
                                <div class="woo-toggle-content">
                                    <div class="woo-toggle-icon">
                                        <?php echo wp_kses(aura_get_icon('check-circle'), aura_get_allowed_svg_html()); ?>
                                    </div>
                                    <div class="woo-toggle-info">
                                        <span class="woo-toggle-title"><?php esc_html_e('Checkout Page', 'aura-chat-button'); ?></span>
                                        <span class="woo-toggle-desc"><?php esc_html_e('Payment and order completion', 'aura-chat-button'); ?></span>
                                    </div>
                                </div>
                                <div class="woo-toggle-switch"></div>
                            </label>

                            <label class="woo-toggle-item <?php echo in_array('account', $woo_pages) ? 'active' : ''; ?>" data-color="emerald">
                                <input type="checkbox" name="aura_chat_options[woo_pages][]" value="account" <?php checked(in_array('account', $woo_pages)); ?>>
                                <div class="woo-toggle-content">
                                    <div class="woo-toggle-icon">
                                        <?php echo wp_kses(aura_get_icon('user-tie'), aura_get_allowed_svg_html()); ?>
                                    </div>
                                    <div class="woo-toggle-info">
                                        <span class="woo-toggle-title"><?php esc_html_e('My Account', 'aura-chat-button'); ?></span>
                                        <span class="woo-toggle-desc"><?php esc_html_e('Customer dashboard and orders', 'aura-chat-button'); ?></span>
                                    </div>
                                </div>
                                <div class="woo-toggle-switch"></div>
                            </label>

                            <label class="woo-toggle-item <?php echo in_array('product_category', $woo_pages) ? 'active' : ''; ?>" data-color="emerald">
                                <input type="checkbox" name="aura_chat_options[woo_pages][]" value="product_category" <?php checked(in_array('product_category', $woo_pages)); ?>>
                                <div class="woo-toggle-content">
                                    <div class="woo-toggle-icon">
                                        <?php echo wp_kses(aura_get_icon('folder'), aura_get_allowed_svg_html()); ?>
                                    </div>
                                    <div class="woo-toggle-info">
                                        <span class="woo-toggle-title"><?php esc_html_e('Product Categories', 'aura-chat-button'); ?></span>
                                        <span class="woo-toggle-desc"><?php esc_html_e('Category archive and listing pages', 'aura-chat-button'); ?></span>
                                    </div>
                                </div>
                                <div class="woo-toggle-switch"></div>
                            </label>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Mobile Settings Tab -->
        <div id="mobile-settings" class="aura-tab-content">
            <div class="aura-grid-container">
                <!-- Left Column: Mobile Specific Settings -->
                <div class="aura-column-main">
                    <div class="aura-card">
                        <h2><?php echo wp_kses(aura_get_icon('mobile-alt'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Mobile Appearance', 'aura-chat-button'); ?></h2>


                        <div class="form-grid">
                            <div class="form-group">
                                <label for="mobile_position_offset_x"><?php esc_html_e('Horizontal Position (px)', 'aura-chat-button'); ?></label>
                                <?php $m_pos_x = isset($options['mobile_position_offset_x']) ? intval($options['mobile_position_offset_x']) : -10; ?>
                                <input type="number" id="mobile_position_offset_x" name="aura_chat_options[mobile_position_offset_x]" value="<?php echo esc_attr(max(-30, min(40, $m_pos_x))); ?>" min="-30" max="40" step="1" placeholder="-10">
                                <p class="description"><?php esc_html_e('Fine-tune horizontal position (-30 to +30).', 'aura-chat-button'); ?></p>
                            </div>
                            <div class="form-group">
                                <label for="mobile_position_offset_y"><?php esc_html_e('Vertical Position (px)', 'aura-chat-button'); ?></label>
                                <?php $m_pos_y = isset($options['mobile_position_offset_y']) ? intval($options['mobile_position_offset_y']) : -15; ?>
                                <input type="number" id="mobile_position_offset_y" name="aura_chat_options[mobile_position_offset_y]" value="<?php echo esc_attr(max(-40, min(40, $m_pos_y))); ?>" min="-40" max="40" step="1" placeholder="-15">
                                <p class="description"><?php esc_html_e('Fine-tune vertical position (-40 to +40).', 'aura-chat-button'); ?></p>
                            </div>
                            <div class="form-group">
                                <label for="mobile_button_size"><?php esc_html_e('Mobile Button Size (px)', 'aura-chat-button'); ?></label>
                                <?php $bs_mobile = isset($options['button_size_mobile']) ? intval($options['button_size_mobile']) : 55; ?>
                                <input type="number" id="mobile_button_size" name="aura_chat_options[button_size_mobile]" value="<?php echo esc_attr(max(50, min(70, $bs_mobile))); ?>" min="50" max="70" step="1">
                                <p class="description"><?php esc_html_e('Floating button size (50-70px).', 'aura-chat-button'); ?></p>
                            </div>
                            <div class="form-group">
                                <label for="mobile_auto_popup_delay"><?php esc_html_e('Auto Open Delay (seconds)', 'aura-chat-button'); ?></label>
                                <input type="number" id="mobile_auto_popup_delay" name="aura_chat_options[mobile_auto_popup_delay]" value="<?php echo isset($options['mobile_auto_popup_delay']) ? esc_attr($options['mobile_auto_popup_delay']) : ''; ?>" placeholder="<?php esc_attr_e('Use desktop setting', 'aura-chat-button'); ?>">
                                <p class="description"><?php esc_html_e('Different delay for mobile users.', 'aura-chat-button'); ?></p>
                            </div>
                        </div>
                    </div>

                    <div class="aura-card">
                        <h2><?php echo wp_kses(aura_get_icon('eye-slash'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Visibility Options', 'aura-chat-button'); ?></h2>

                        <div class="form-grid">
                            <label class="visibility-option-card">
                                <input type="checkbox" name="aura_chat_options[hide_on_mobile]" value="1" <?php checked(isset($options['hide_on_mobile']) ? $options['hide_on_mobile'] : 0, 1); ?>>
                                <div>
                                    <span class="visibility-option-title"><?php esc_html_e('Hide Button on Mobile', 'aura-chat-button'); ?></span>
                                    <span class="visibility-option-desc"><?php esc_html_e('Completely hide the button on mobile devices.', 'aura-chat-button'); ?></span>
                                </div>
                            </label>

                            <label class="visibility-option-card">
                                <input type="checkbox" name="aura_chat_options[mobile_tooltip_hidden]" value="1" <?php checked(isset($options['mobile_tooltip_hidden']) ? $options['mobile_tooltip_hidden'] : 0, 1); ?>>
                                <div>
                                    <span class="visibility-option-title"><?php esc_html_e('Hide Button Label on Mobile', 'aura-chat-button'); ?></span>
                                    <span class="visibility-option-desc"><?php esc_html_e('Hide the tooltip text to save screen space.', 'aura-chat-button'); ?></span>
                                </div>
                            </label>
                        </div>
                    </div>
                </div>

                <!-- Right Column: Mobile Preview -->
                <div class="aura-column-side">
                    <div class="aura-card" style="position: sticky; top: 50px;">
                        <h2><?php echo wp_kses(aura_get_icon('eye'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Live Preview', 'aura-chat-button'); ?></h2>
                        
                        <!-- Phone Mockup -->
                        <div style="display: flex; justify-content: center; padding: 20px 0;">
                            <div style="position: relative; width: 180px; height: 360px; background: linear-gradient(180deg, #1a1a2e 0%, #16213e 100%); border-radius: 32px; padding: 8px; box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.4), inset 0 1px 0 rgba(255,255,255,0.1);">
                                <!-- Phone Notch -->
                                <div style="position: absolute; top: 8px; left: 50%; transform: translateX(-50%); width: 80px; height: 24px; background: #0a0a0f; border-radius: 0 0 16px 16px; z-index: 10;"></div>
                                
                                <!-- Phone Screen -->
                                <div style="width: 100%; height: 100%; background: linear-gradient(180deg, #f8fafc 0%, #e2e8f0 100%); border-radius: 24px; position: relative; overflow: hidden;">
                                    <!-- Status Bar -->
                                    <div style="height: 32px; background: rgba(0,0,0,0.03); display: flex; align-items: center; justify-content: space-between; padding: 0 16px; font-size: 10px; color: #64748b;">
                                        <span>9:41</span>
                                        <div style="display: flex; gap: 4px; align-items: center;">
                                            <?php echo wp_kses(aura_get_icon('signal', '', ['style' => 'width: 8px; height: 8px;']), aura_get_allowed_svg_html()); ?>
                                            <?php echo wp_kses(aura_get_icon('wifi', '', ['style' => 'width: 8px; height: 8px;']), aura_get_allowed_svg_html()); ?>
                                            <?php echo wp_kses(aura_get_icon('battery-full', '', ['style' => 'width: 8px; height: 8px;']), aura_get_allowed_svg_html()); ?>
                                        </div>
                                    </div>
                                    
                                    <!-- Content Lines -->
                                    <div style="padding: 16px;">
                                        <div style="height: 12px; background: #cbd5e1; border-radius: 6px; margin-bottom: 10px; width: 70%;"></div>
                                        <div style="height: 8px; background: #e2e8f0; border-radius: 4px; margin-bottom: 6px; width: 100%;"></div>
                                        <div style="height: 8px; background: #e2e8f0; border-radius: 4px; margin-bottom: 6px; width: 85%;"></div>
                                        <div style="height: 8px; background: #e2e8f0; border-radius: 4px; margin-bottom: 16px; width: 60%;"></div>
                                        
                                        <div style="height: 60px; background: #e2e8f0; border-radius: 8px; margin-bottom: 12px;"></div>
                                        
                                        <div style="height: 8px; background: #e2e8f0; border-radius: 4px; margin-bottom: 6px; width: 90%;"></div>
                                        <div style="height: 8px; background: #e2e8f0; border-radius: 4px; width: 75%;"></div>
                                    </div>
                                    
                                    <!-- Chat Button Preview -->
                                    <div class="mobile-position-preview-widget" style="position: absolute; width: 44px; height: 44px; background: linear-gradient(145deg, #34d77a, #25D366, #1a9e4a); border-radius: 50%; display: flex; align-items: center; justify-content: center; color: white; bottom: 16px; right: 12px; box-shadow: 0 4px 15px rgba(37, 211, 102, 0.4), 0 2px 6px rgba(0,0,0,0.15); border: 3px solid; border-top-color: #5ce89a; border-left-color: #5ce89a; border-bottom-color: #1a9e4a; border-right-color: #1a9e4a; z-index: 5;">
                                        <?php echo wp_kses(aura_get_icon('chat', '', ['style' => 'width: 22px; height: 22px; filter: drop-shadow(0 1px 2px rgba(0,0,0,0.2));']), aura_get_allowed_svg_html()); ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <p style="text-align: center; font-size: 12px; color: #64748b; margin: 0;"><?php esc_html_e('Button position updates in real-time', 'aura-chat-button'); ?></p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Statistics Tab - Advanced -->
        <div id="statistics-settings" class="aura-tab-content">
            <!-- Chart.js loaded via wp_enqueue_script in settings-functions.php -->
            
            <!-- Pro Header -->
            <div class="stats-pro-header">
                <div class="stats-header-left">
                    <h2 class="stats-title"><?php echo wp_kses(aura_get_icon('chart-area'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Analytics Dashboard', 'aura-chat-button'); ?></h2>
                </div>
                <div class="stats-header-right">
                    <div class="stats-period-selector">
                        <button type="button" class="period-btn" data-period="today"><?php esc_html_e('Today', 'aura-chat-button'); ?></button>
                        <button type="button" class="period-btn active" data-period="7days"><?php esc_html_e('7 Days', 'aura-chat-button'); ?></button>
                        <button type="button" class="period-btn" data-period="14days"><?php esc_html_e('14 Days', 'aura-chat-button'); ?></button>
                        <button type="button" class="period-btn" data-period="30days"><?php esc_html_e('30 Days', 'aura-chat-button'); ?></button>
                        <button type="button" class="period-btn" data-period="90days"><?php esc_html_e('90 Days', 'aura-chat-button'); ?></button>
                        <button type="button" class="period-btn" data-period="all"><?php esc_html_e('All Time', 'aura-chat-button'); ?></button>
                    </div>
                    <button type="button" id="reset-stats-btn" class="stats-reset-btn" aria-label="<?php esc_attr_e('Reset Statistics', 'aura-chat-button'); ?>" title="<?php esc_attr_e('Reset Statistics', 'aura-chat-button'); ?>">
                        <?php echo wp_kses(aura_get_icon('redo-alt'), aura_get_allowed_svg_html()); ?>
                    </button>
                </div>
            </div>

            <!-- Pro Stats Cards -->
            <div class="stats-cards-pro">
                <div class="stat-card-pro stat-card-chats">
                    <div class="stat-card-glow"></div>
                    <div class="stat-card-content">
                        <div class="stat-card-header">
                            <div class="stat-icon-pro">
                                <?php echo wp_kses(aura_get_icon('comments'), aura_get_allowed_svg_html()); ?>
                            </div>
                            <div class="stat-trend" id="trend-chats">
                                <?php echo wp_kses(aura_get_icon('arrow-up'), aura_get_allowed_svg_html()); ?>
                                <span>0%</span>
                            </div>
                        </div>
                        <div class="stat-card-body">
                            <span class="stat-number" id="stat-chat-opens">0</span>
                            <span class="stat-name"><?php esc_html_e('Chat Opens', 'aura-chat-button'); ?></span>
                        </div>
                        <div class="stat-card-footer">
                            <div class="stat-sparkline" id="sparkline-chats"></div>
                        </div>
                    </div>
                </div>

                <div class="stat-card-pro stat-card-conversions">
                    <div class="stat-card-glow"></div>
                    <div class="stat-card-content">
                        <div class="stat-card-header">
                            <div class="stat-icon-pro">
                                <?php echo wp_kses(aura_get_icon('chat'), aura_get_allowed_svg_html()); ?>
                            </div>
                            <div class="stat-trend stat-trend-success" id="trend-conversions">
                                <?php echo wp_kses(aura_get_icon('arrow-up'), aura_get_allowed_svg_html()); ?>
                                <span>0%</span>
                            </div>
                        </div>
                        <div class="stat-card-body">
                            <span class="stat-number" id="stat-conversions">0</span>
                            <span class="stat-name"><?php esc_html_e('Conversions', 'aura-chat-button'); ?></span>
                        </div>
                        <div class="stat-card-footer">
                            <div class="stat-sparkline" id="sparkline-conversions"></div>
                        </div>
                    </div>
                </div>

                <div class="stat-card-pro stat-card-rate">
                    <div class="stat-card-glow"></div>
                    <div class="stat-card-content">
                        <div class="stat-card-header">
                            <div class="stat-icon-pro">
                                <?php echo wp_kses(aura_get_icon('percentage'), aura_get_allowed_svg_html()); ?>
                            </div>
                        </div>
                        <div class="stat-card-body">
                            <span class="stat-number" id="stat-rate">0<small>%</small></span>
                            <span class="stat-name"><?php esc_html_e('Conversion Rate', 'aura-chat-button'); ?></span>
                        </div>
                        <div class="stat-card-footer">
                            <div class="stat-progress-bar">
                                <div class="stat-progress-fill" id="rate-progress"></div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Pro Charts -->
            <div class="chart-card-pro chart-card-main chart-card-full">
                <div class="chart-card-header">
                    <div class="chart-title">
                        <?php echo wp_kses(aura_get_icon('chart-line'), aura_get_allowed_svg_html()); ?>
                        <span><?php esc_html_e('Performance Overview', 'aura-chat-button'); ?></span>
                    </div>
                    <div class="chart-legend-inline">
                        <span class="legend-dot legend-chats"></span> <?php esc_html_e('Chats', 'aura-chat-button'); ?>
                        <span class="legend-dot legend-conversions"></span> <?php esc_html_e('Conversions', 'aura-chat-button'); ?>
                    </div>
                </div>
                <div class="chart-card-body">
                    <canvas id="trends-chart"></canvas>
                </div>
                <div id="stats-no-data" class="chart-empty-state" style="display: none;">
                    <div class="empty-icon"><?php echo wp_kses(aura_get_icon('chart-area'), aura_get_allowed_svg_html()); ?></div>
                    <h3><?php esc_html_e('No Data Yet', 'aura-chat-button'); ?></h3>
                    <p><?php esc_html_e('Start tracking by enabling the button on your site', 'aura-chat-button'); ?></p>
                </div>
            </div>

            <!-- Fallback -->
            <div id="chartjs-fallback" class="chart-empty-state" style="display: none;">
                <div class="empty-icon"><?php echo wp_kses(aura_get_icon('exclamation-triangle'), aura_get_allowed_svg_html()); ?></div>
                <h3><?php esc_html_e('Charts Unavailable', 'aura-chat-button'); ?></h3>
                <p><?php esc_html_e('Please refresh the page to load charts', 'aura-chat-button'); ?></p>
            </div>
        </div>

        <div class="aura-sticky-footer">
            <?php submit_button(__('Save Changes', 'aura-chat-button'), 'primary', 'submit', false, array('class' => 'aura-save-btn')); ?>
        </div>
        </form>
    </div>
    
    <!-- Reset Statistics Confirmation Modal -->
    <div id="reset-stats-modal" class="aura-modal" style="display: none;">
        <div class="aura-modal-content">
            <h3><?php echo wp_kses(aura_get_icon('exclamation-triangle'), aura_get_allowed_svg_html()); ?> <?php esc_html_e('Reset Statistics', 'aura-chat-button'); ?></h3>
            <p><?php esc_html_e('Are you sure you want to delete all statistics data? This action cannot be undone.', 'aura-chat-button'); ?></p>
            <div class="aura-modal-actions">
                <button type="button" id="confirm-reset-btn" class="button button-primary"><?php esc_html_e('Yes, Reset', 'aura-chat-button'); ?></button>
                <button type="button" id="cancel-reset-btn" class="button button-secondary"><?php esc_html_e('Cancel', 'aura-chat-button'); ?></button>
            </div>
        </div>
    </div>
    <?php
}
