// SVG Icon Helper
var auraSvgIcons = {
    'chat': '<svg class="aura-icon" viewBox="0 0 448 512" fill="currentColor"><path d="M380.9 97.1C339 55.1 283.2 32 223.9 32c-122.4 0-222 99.6-222 222 0 39.1 10.2 77.3 29.6 111L0 480l117.7-30.9c32.4 17.7 68.9 27 106.1 27h.1c122.3 0 224.1-99.6 224.1-222 0-59.3-25.2-115-67.1-157zm-157 341.6c-33.2 0-65.7-8.9-94-25.7l-6.7-4-69.8 18.3L72 359.2l-4.4-7c-18.5-29.4-28.2-63.3-28.2-98.2 0-101.7 82.8-184.5 184.6-184.5 49.3 0 95.6 19.2 130.4 54.1 34.8 34.9 56.2 81.2 56.1 130.5 0 101.8-84.9 184.6-186.6 184.6zm101.2-138.2c-5.5-2.8-32.8-16.2-37.9-18-5.1-1.9-8.8-2.8-12.5 2.8-3.7 5.6-14.3 18-17.6 21.8-3.2 3.7-6.5 4.2-12 1.4-32.6-16.3-54-29.1-75.5-66-5.7-9.8 5.7-9.1 16.3-30.3 1.8-3.7.9-6.9-.5-9.7-1.4-2.8-12.5-30.1-17.1-41.2-4.5-10.8-9.1-9.3-12.5-9.5-3.2-.2-6.9-.2-10.6-.2-3.7 0-9.7 1.4-14.8 6.9-5.1 5.6-19.4 19-19.4 46.3 0 27.3 19.9 53.7 22.6 57.4 2.8 3.7 39.1 59.7 94.8 83.8 35.2 15.2 49 16.5 66.6 13.9 10.7-1.6 32.8-13.4 37.4-26.4 4.6-13 4.6-24.1 3.2-26.4-1.3-2.5-5-3.9-10.5-6.6z"/></svg>',
    'times': '<svg class="aura-icon" viewBox="0 0 352 512" fill="currentColor"><path d="M242.72 256l100.07-100.07c12.28-12.28 12.28-32.19 0-44.48l-22.24-22.24c-12.28-12.28-32.19-12.28-44.48 0L176 189.28 75.93 89.21c-12.28-12.28-32.19-12.28-44.48 0L9.21 111.45c-12.28 12.28-12.28 32.19 0 44.48L109.28 256 9.21 356.07c-12.28 12.28-12.28 32.19 0 44.48l22.24 22.24c12.28 12.28 32.2 12.28 44.48 0L176 322.72l100.07 100.07c12.28 12.28 32.2 12.28 44.48 0l22.24-22.24c12.28-12.28 12.28-32.19 0-44.48L242.72 256z"/></svg>',
    'check': '<svg class="aura-icon" viewBox="0 0 512 512" fill="currentColor"><path d="M173.898 439.404l-166.4-166.4c-9.997-9.997-9.997-26.206 0-36.204l36.203-36.204c9.997-9.998 26.207-9.998 36.204 0L192 312.69 432.095 72.596c9.997-9.997 26.207-9.997 36.204 0l36.203 36.204c9.997 9.997 9.997 26.206 0 36.204l-294.4 294.401c-9.998 9.997-26.207 9.997-36.204-.001z"/></svg>',
    'spinner': '<svg class="aura-icon aura-spin" viewBox="0 0 512 512" fill="currentColor"><path d="M304 48c0 26.51-21.49 48-48 48s-48-21.49-48-48 21.49-48 48-48 48 21.49 48 48zm-48 368c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48-21.49-48-48-48zm208-208c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48-21.49-48-48-48zM96 256c0-26.51-21.49-48-48-48S0 229.49 0 256s21.49 48 48 48 48-21.49 48-48zm12.922 99.078c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48c0-26.509-21.491-48-48-48zm294.156 0c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48c0-26.509-21.49-48-48-48zM108.922 60.922c-26.51 0-48 21.49-48 48s21.49 48 48 48 48-21.49 48-48-21.491-48-48-48z"/></svg>'
};

function auraGetIcon(name, style) {
    var icon = auraSvgIcons[name] || '';
    if (style && icon) {
        icon = icon.replace('class="aura-icon"', 'class="aura-icon" style="' + style + '"');
    }
    return icon;
}

jQuery(document).ready(function ($) {
    // Media Uploader
    var mediaUploader;

    $(document).on('click', '.upload-avatar-btn', function (e) {
        e.preventDefault();
        var button = $(this);
        var inputField = button.prev('input');

        if (mediaUploader) {
            mediaUploader.open();
            return;
        }

        mediaUploader = wp.media.frames.file_frame = wp.media({
            title: auraAdmin.i18n.chooseImage,
            button: { text: auraAdmin.i18n.chooseImage },
            multiple: false
        });

        mediaUploader.on('select', function () {
            var attachment = mediaUploader.state().get('selection').first().toJSON();
            inputField.val(attachment.url);

            var previewWrapper = button.siblings('.avatar-preview-wrapper');
            var previewImg = previewWrapper.length > 0 ? previewWrapper.find('.avatar-preview') : button.siblings('.avatar-preview');

            if (previewWrapper.length === 0) {
                var wrapper = $('<div class="avatar-preview-wrapper"></div>');
                previewImg = $('<img class="avatar-preview">');
                var removeBtn = $('<button type="button" class="button remove-avatar-btn" title="' + auraAdmin.i18n.removeImage + '">' + auraGetIcon('times') + '</button>');
                wrapper.append(previewImg).append(removeBtn);
                button.after(wrapper);
            } else if (previewWrapper.hasClass('avatar-placeholder')) {
                previewWrapper.removeClass('avatar-placeholder').html('');
                previewImg = $('<img class="avatar-preview">');
                var removeBtn = $('<button type="button" class="button remove-avatar-btn" title="' + auraAdmin.i18n.removeImage + '">' + auraGetIcon('times') + '</button>');
                previewWrapper.append(previewImg).append(removeBtn);
            } else {
                previewImg = previewWrapper.find('.avatar-preview');
                if (previewImg.length === 0) {
                    previewImg = $('<img class="avatar-preview">');
                    previewWrapper.prepend(previewImg);
                }
                if (previewWrapper.find('.remove-avatar-btn').length === 0) {
                    var removeBtn = $('<button type="button" class="button remove-avatar-btn" title="' + auraAdmin.i18n.removeImage + '">' + auraGetIcon('times') + '</button>');
                    previewWrapper.append(removeBtn);
                }
            }

            previewImg.attr('src', attachment.url).show();
            previewWrapper.show();
            button.text(auraAdmin.i18n.changeImage);
        });

        mediaUploader.open();
    });

    // Remove Avatar
    $(document).on('click', '.remove-avatar-btn', function (e) {
        e.preventDefault();
        var wrapper = $(this).closest('.avatar-preview-wrapper');
        var uploadGroup = wrapper.closest('.upload-group');
        var inputField = uploadGroup.find('.avatar-input');
        var uploadBtn = uploadGroup.find('.upload-avatar-btn');

        inputField.val('');
        wrapper.remove();

        var placeholder = $('<div class="avatar-preview-wrapper avatar-placeholder"><div class="avatar-placeholder-icon">' + auraGetIcon('chat') + '</div></div>');
        uploadBtn.after(placeholder);
        uploadBtn.text(auraAdmin.i18n.uploadImage);
    });

    // Position Fine-Tuning Live Preview
    function updatePositionPreview(triggerInput) {
        var $inputX = $('#position_offset_x');
        var $inputY = $('#position_offset_y');
        var offsetX = parseInt($inputX.val()) || 0;
        var offsetY = parseInt($inputY.val()) || 0;
        var position = $('#position').val() || 'right';
        var size = parseInt($('#button_size_desktop').val()) || 60;
        var buttonStyle = $('#button_style').val() || 'style-1';
        var isPill = (buttonStyle === 'style-3' || buttonStyle === 'style-4');

        var previewWidget = $('#desktop-preview-box .position-preview-widget');
        if (previewWidget.length === 0) {
            $('#desktop-preview-box').append('<div class="position-preview-widget" style="position: absolute; width: 28px; height: 28px; background: #25D366; border-radius: 50%; display: flex; align-items: center; justify-content: center; color: white; z-index: 1; box-shadow: 0 3px 10px rgba(37, 211, 102, 0.2); border: 2px solid white; pointer-events: none;">' + auraGetIcon('chat', 'width: 14px; height: 14px;') + '</div>');
            previewWidget = $('#desktop-preview-box .position-preview-widget');
        }

        var previewSize = Math.round(size * 0.45);
        var iconSize = Math.round(previewSize * 0.5);
        var borderRadius = (buttonStyle === 'style-3' ? previewSize + 'px' : (buttonStyle === 'style-4' ? Math.round(previewSize * 0.25) + 'px' : (buttonStyle === 'style-5' ? Math.round(previewSize * 0.28) + 'px' : (buttonStyle === 'style-6' ? '50%' : previewSize + 'px'))));

        // Styling first to measure dimensions
        if (isPill) {
            previewWidget.css({
                width: 'auto',
                height: previewSize + 'px',
                'border-radius': borderRadius,
                'padding-left': (previewSize * 0.92) + 'px',
                'padding-right': (previewSize * 0.4) + 'px',
                'font-size': (previewSize * 0.45) + 'px'
            });
        } else {
            previewWidget.css({
                width: previewSize + 'px',
                height: previewSize + 'px',
                'padding': '0'
            });
        }

        var w = previewWidget.outerWidth();
        var h = previewWidget.outerHeight();

        // Calculate Real-World Boundaries (Scale 4:1)
        // Desktop Mockup: 220px wide, 130px high. Browser bar: 20px. Usable height: 110px.
        // baseOffset: 10px. Initial Right/Left = 10. Initial Bottom = 10.
        // translateX(tx) moves RIGHT. 
        // Right Edge Hit: baseOffset - tx = 0 -> tx = 10 -> offsetX = 40.
        var maxOffsetX = 40;
        // Left Edge Hit: tx = -(boxW - baseOffset - w) -> offsetX = -(220 - 10 - w) * 4.
        // Also enforce a hard negative limit of -40 for perspective
        var minOffsetX = Math.max(-40, -(220 - 10 - w) * 4);

        // Bottom Edge Hit: baseOffset - ty = 0 -> ty = 10 -> offsetY = 40.
        var maxOffsetY = 40;
        // Top Edge Hit (20px bar): ty = -(130 - 20 - 10 - h) -> offsetY = -(100 - h) * 4.
        var minOffsetY = Math.max(-40, -(100 - h) * 4);

        // Precise Clamping
        var clampedX = Math.max(minOffsetX, Math.min(maxOffsetX, offsetX));
        var clampedY = Math.max(minOffsetY, Math.min(maxOffsetY, offsetY));

        if (triggerInput && (offsetX !== clampedX || offsetY !== clampedY)) {
            if (offsetX !== clampedX) $inputX.val(Math.round(clampedX));
            if (offsetY !== clampedY) $inputY.val(Math.round(clampedY));
            offsetX = clampedX;
            offsetY = clampedY;
        }

        previewWidget.find('svg, .aura-icon').css({
            width: iconSize + 'px',
            height: iconSize + 'px'
        });

        if (position === 'right') {
            previewWidget.css({
                right: '10px',
                left: 'auto',
                bottom: '10px',
                transform: 'translate(' + (offsetX * -0.25) + 'px, ' + (offsetY * -0.25) + 'px)'
            });
        } else {
            previewWidget.css({
                left: '10px',
                right: 'auto',
                bottom: '10px',
                transform: 'translate(' + (offsetX * 0.25) + 'px, ' + (offsetY * -0.25) + 'px)'
            });
        }

        updatePreviewButtonStyle();
    }

    $('#position_offset_x, #position_offset_y, #position, #button_size_desktop').on('input change', function () {
        updatePositionPreview(true);
    });

    // Update preview labels when text changes
    $('#tooltip_text').on('input change', function () {
        var text = $(this).val() || '';
        $('.preview-label-text').text(text);
        updatePositionPreview();
    });

    // Mobile Position Fine-Tuning Live Preview
    function updateMobilePositionPreview(triggerInput) {
        var $inputX = $('#mobile_position_offset_x');
        var $inputY = $('#mobile_position_offset_y');
        var mobileOffsetX = parseInt($inputX.val()) || -10;
        var mobileOffsetY = parseInt($inputY.val()) || -15;
        var mobilePosition = $('#position').val() || 'right';
        var buttonSize = parseInt($('#mobile_button_size').val()) || 55;
        var buttonStyle = $('#button_style').val() || 'style-1';
        var isPill = (buttonStyle === 'style-3' || buttonStyle === 'style-4');

        // Mobile mockup constants
        var baseMX = 12; // Initial spacing
        var baseMY = 16;
        var mockupW = 180;
        var mockupH = 328; // Screen height inside phone (360 - notch area)

        $('.mobile-position-preview-widget').each(function () {
            var widget = $(this);
            var isPhoneMockup = widget.closest('[style*="border-radius: 24px"]').length > 0;

            // Scaling for phone mockup
            var previewSize = isPhoneMockup ? Math.round(buttonSize * 0.7) : Math.round(buttonSize * 0.5);
            var iconSize = Math.round(previewSize * 0.5);

            // Style widget first to get accurate width/height
            if (isPill) {
                widget.css({
                    'width': 'auto',
                    'min-width': Math.round(previewSize * 1.2) + 'px',
                    'height': previewSize + 'px',
                    'padding-left': Math.round(previewSize * 0.92) + 'px',
                    'padding-right': Math.round(previewSize * 0.45) + 'px',
                    'border-radius': (buttonStyle === 'style-3' ? previewSize + 'px' : (buttonStyle === 'style-4' ? Math.round(previewSize * 0.25) + 'px' : (buttonStyle === 'style-5' ? Math.round(previewSize * 0.28) + 'px' : '50%'))),
                    'display': 'inline-flex',
                    'align-items': 'center',
                    'justify-content': 'flex-start'
                });
            } else {
                widget.css({
                    width: previewSize + 'px',
                    height: previewSize + 'px',
                    'padding': '0'
                });
            }

            var w = widget.outerWidth();
            var h = widget.outerHeight();

            // Mobile Clamping: 0.4 scaling is used in translate()
            // Right Edge Hit: baseMX - (mox * 0.4) = 0 -> mox = baseMX / 0.4 = 12 / 0.4 = 30.
            var maxCX = 30; // baseMX / 0.4

            // Left Edge Hit: baseMX - (mox * 0.4) + w = mockupW -> mox * 0.4 = baseMX + w - mockupW
            // Enforce -40 hard limit for mobile too
            var minCX = Math.max(-40, (baseMX + w - mockupW) / 0.4);

            // Bottom Edge Hit: baseMY - (moy * 0.4) = 0 -> moy = baseMY / 0.4 = 16 / 0.4 = 40.
            var maxCY = 40; // baseMY / 0.4

            // Top Edge Hit: ty = -(mockupH - baseMY - h) -> moy = (baseMY + h - mockupH) / 0.4
            var minCY = Math.max(-40, (baseMY + h - mockupH) / 0.4);

            var clampedMX = Math.max(minCX, Math.min(maxCX, mobileOffsetX));
            var clampedMY = Math.max(minCY, Math.min(maxCY, mobileOffsetY));

            if (triggerInput && (mobileOffsetX !== clampedMX || mobileOffsetY !== clampedMY)) {
                if (mobileOffsetX !== clampedMX) $inputX.val(Math.round(clampedMX));
                if (mobileOffsetY !== clampedMY) $inputY.val(Math.round(clampedMY));
                mobileOffsetX = clampedMX;
                mobileOffsetY = clampedMY;
            }

            widget.find('svg, i, .aura-icon').each(function () {
                $(this).css({
                    width: iconSize + 'px',
                    height: iconSize + 'px',
                });
            });

            if (mobilePosition === 'right') {
                widget.css({
                    right: baseMX + 'px',
                    left: 'auto',
                    bottom: baseMY + 'px',
                    transform: 'translate(' + (mobileOffsetX * -0.4) + 'px, ' + (mobileOffsetY * -0.4) + 'px)'
                });
            } else {
                widget.css({
                    left: baseMX + 'px',
                    right: 'auto',
                    bottom: baseMY + 'px',
                    transform: 'translate(' + (mobileOffsetX * 0.4) + 'px, ' + (mobileOffsetY * -0.4) + 'px)'
                });
            }
        });

        updatePreviewButtonStyle();
    }

    $('#mobile_position_offset_x, #mobile_position_offset_y, #position, #mobile_button_size').on('input change', function () {
        updateMobilePositionPreview(true);
    });

    updatePositionPreview();
    updateMobilePositionPreview();
    $('.position-preview-widget[style*="fixed"], .mobile-position-preview-widget[style*="fixed"]').remove();

    // Tab functionality
    $('.aura-tab-button').on('click', function () {
        const tabId = $(this).data('tab');
        $('.aura-tab-button').removeClass('active');
        $('.aura-tab-content').removeClass('active');
        $(this).addClass('active');
        $('#' + tabId + '-settings').addClass('active');
        localStorage.setItem('auraActiveTab', tabId);
    });

    const activeTab = localStorage.getItem('auraActiveTab');
    if (activeTab && (activeTab === 'general' || activeTab === 'mobile' || activeTab === 'statistics')) {
        $('.aura-tab-button').removeClass('active');
        $('.aura-tab-content').removeClass('active');
        $('.aura-tab-button[data-tab="' + activeTab + '"]').addClass('active');
        $('#' + activeTab + '-settings').addClass('active');
    }



    // Hide option toggle
    $('.hide-option-item input[type="checkbox"]').on('change', function () {
        var item = $(this).closest('.hide-option-item');
        if ($(this).is(':checked')) {
            item.removeClass('inactive').addClass('active');
        } else {
            item.removeClass('active').addClass('inactive');
        }
    });

    // Days selector toggle
    $('.day-chip input[type="checkbox"]').on('change', function () {
        var chip = $(this).closest('.day-chip');
        if ($(this).is(':checked')) {
            chip.removeClass('inactive').addClass('active');
        } else {
            chip.removeClass('active').addClass('inactive');
        }
    });

    // Save notice - only show if PHP rendered the notice (after page reload)
    const notice = $('#aura-save-notice');
    if (notice.length) {
        setTimeout(function () { notice.addClass('show'); }, 100);
        setTimeout(function () {
            notice.removeClass('show');
            setTimeout(function () { notice.remove(); }, 300);
        }, 4000);
    }

    // Color adjustment helper (CSS-like brightness)
    function adjustColor(color, percent) {
        var num = parseInt(color.replace("#", ""), 16),
            amt = Math.round(2.55 * percent),
            R = (num >> 16) + amt,
            G = (num >> 8 & 0x00FF) + amt,
            B = (num & 0x0000FF) + amt;
        return "#" + (0x1000000 + (R < 255 ? R < 1 ? 0 : R : 255) * 0x10000 + (G < 255 ? G < 1 ? 0 : G : 255) * 0x100 + (B < 255 ? B < 1 ? 0 : B : 255)).toString(16).slice(1);
    }

    $('#custom_button_color').on('change', function () {
        updatePreviewButtonStyle();
    });

    // Color Preset Click Handler
    $(document).on('click', '.color-preset-item', function () {
        var color = $(this).data('color');
        $('#custom_button_color').val(color).trigger('change');

        $('.color-preset-item').removeClass('active');
        $(this).addClass('active');
    });

    // Initial Active Preset Check
    var initialColor = $('#custom_button_color').val().toLowerCase();
    $('.color-preset-item').each(function () {
        if ($(this).data('color').toLowerCase() === initialColor) {
            $(this).addClass('active');
        }
    });

    function updatePreviewButtonStyle() {
        var customColor = $('#custom_button_color').val() || '#2ecc71';
        var buttonStyle = $('#button_style').val() || 'style-1';

        // Generate color variants like PHP aura_adjust_brightness
        var c = {
            main: customColor,
            light: adjustColor(customColor, 20),
            lighter: adjustColor(customColor, 35),
            dark: adjustColor(customColor, -15),
            darker: adjustColor(customColor, -25)
        };

        var isRound = (buttonStyle === 'style-1');
        var isPill = (buttonStyle === 'style-3' || buttonStyle === 'style-4');

        $('.position-preview-widget, .mobile-position-preview-widget').each(function () {
            var widget = $(this);
            var isMobile = widget.hasClass('mobile-position-preview-widget');
            var widgetHeight = parseInt(widget.css('height')) || (isMobile ? 38 : 28);
            widget.css({ 'outline': 'none', 'outline-offset': '0' });

            var borderRadius = isRound ? '50%' : (isPill ? (widgetHeight + 'px') : '10px');

            // Style-specific styling
            if (buttonStyle === 'style-3') {
                // Style 3 (Old Style 4)
                widget.css({
                    'border-radius': borderRadius,
                    'background': 'linear-gradient(to bottom, ' + c.lighter + ' 0%, ' + c.main + ' 40%, ' + c.dark + ' 100%)',
                    'border': 'none',
                    'box-shadow': '0 4px 15px rgba(0,0,0,0.2), inset 0 1px 0 rgba(255,255,255,0.5), inset 0 -1px 0 rgba(0,0,0,0.2)'
                });
            } else if (buttonStyle === 'style-4') {
                // Style 4 (Business Pill)
                widget.css({
                    'border-radius': '10px',
                    'background': 'linear-gradient(145deg, ' + c.light + ', ' + c.main + ', ' + c.dark + ')',
                    'border': '3px solid',
                    'border-top-color': c.lighter,
                    'border-left-color': c.lighter,
                    'border-bottom-color': c.darker,
                    'border-right-color': c.darker,
                    'box-shadow': '0 4px 15px rgba(0,0,0,0.2)'
                });
            } else if (buttonStyle === 'style-5') {
                // Style 5 (Executive 3D Square - Refined)
                widget.css({
                    'border-radius': Math.round(widgetHeight * 0.25) + 'px',
                    'background': 'linear-gradient(145deg, ' + c.light + ', ' + c.main + ', ' + c.dark + ')',
                    'border': '4px solid',
                    'border-top-color': c.lighter,
                    'border-left-color': c.lighter,
                    'border-bottom-color': c.darker,
                    'border-right-color': c.darker,
                    'box-shadow': 'none'
                });
            } else if (buttonStyle === 'style-6') {
                // Style 6 (Futuristic Glass - Round)
                var hexToRgba = function (hex, alpha) {
                    var r = parseInt(hex.slice(1, 3), 16),
                        g = parseInt(hex.slice(3, 5), 16),
                        b = parseInt(hex.slice(5, 7), 16);
                    return 'rgba(' + r + ', ' + g + ', ' + b + ', ' + alpha + ')';
                };

                widget.css({
                    'border-radius': '50%',
                    'background': 'linear-gradient(145deg, ' + c.light + ' 0%, ' + c.main + ' 50%, ' + c.dark + ' 100%)',
                    'backdrop-filter': 'blur(12px)',
                    '-webkit-backdrop-filter': 'blur(12px)',
                    'border': '4px solid ' + c.lighter,
                    'box-shadow': 'inset 0 4px 10px rgba(255, 255, 255, 0.15), inset 0 -4px 10px rgba(0, 0, 0, 0.3), 0 8px 20px -10px rgba(0, 0, 0, 0.4)'
                });

                // Tooltip styling for Style 6 preview
                widget.find('.aura-fab-tooltip').css({
                    'background': 'linear-gradient(145deg, ' + c.light + ' 0%, ' + c.main + ' 50%, ' + c.dark + ' 100%)',
                    'backdrop-filter': 'blur(12px)',
                    '-webkit-backdrop-filter': 'blur(12px)',
                    'border': '3px solid ' + c.lighter,
                    'box-shadow': 'inset 0 3px 8px rgba(255, 255, 255, 0.12), inset 0 -3px 8px rgba(0, 0, 0, 0.25), 0 6px 15px -8px rgba(0, 0, 0, 0.3)',
                    'color': '#ffffff',
                    'padding': '10px 20px',
                    'border-radius': '100px',
                    'font-weight': '700',
                    'text-shadow': '0 1px 2px rgba(0, 0, 0, 0.3)',
                    'opacity': '1',
                    'visibility': 'visible'
                });
            } else {
                // Standard styles (1, 2)
                widget.css({
                    'border-radius': borderRadius,
                    'background': 'linear-gradient(145deg, ' + c.light + ', ' + c.main + ', ' + c.dark + ')',
                    'border': '3px solid',
                    'border-top-color': c.lighter,
                    'border-left-color': c.lighter,
                    'border-bottom-color': c.darker,
                    'border-right-color': c.darker,
                    'box-shadow': '0 4px 15px rgba(0,0,0,0.2), 0 2px 6px rgba(0,0,0,0.15)'
                });
            }

            // Position icons for pill styles
            if (isPill) {
                widget.find('svg, .aura-icon').css({
                    'left': (widgetHeight / 2) + 'px',
                    'transform': 'translate(-50%, -50%)',
                    'position': 'absolute',
                    'top': '50%'
                });
            } else {
                widget.find('svg, .aura-icon').css({
                    'left': '50%',
                    'transform': 'translate(-50%, -50%)',
                    'position': 'absolute',
                    'top': '50%'
                });
            }
        });
    }

    // Visual Style Selector Interaction
    $('.style-option').on('click', function () {
        var $this = $(this);
        var style = $this.data('style');

        // UI update
        $('.style-option').removeClass('active');
        $this.addClass('active');

        // Update hidden field and trigger existing logic
        $('#button_style').val(style).trigger('change');

        // Force immediate preview refresh
        updatePositionPreview();
        if (typeof updateMobilePositionPreview === 'function') {
            updateMobilePositionPreview();
        }
    });

    $('#button_style').on('change', function () {
        updatePreviewButtonStyle();
    });

    updatePreviewButtonStyle();
});


// Advanced Statistics Dashboard
(function ($) {
    $(document).ready(function () {
        var statsLoaded = false;
        var trendsChart = null;
        var currentPeriod = '7days';

        // Advanced color palette with enhanced glow effects
        var colors = {
            chats: {
                main: '#8b5cf6',
                light: 'rgba(139, 92, 246, 0.25)',
                gradient: ['rgba(139, 92, 246, 0.45)', 'rgba(139, 92, 246, 0.02)'],
                glow: 'rgba(139, 92, 246, 0.7)',
                shadow: 'rgba(139, 92, 246, 0.4)'
            },
            conversions: {
                main: '#06b6d4',
                light: 'rgba(6, 182, 212, 0.25)',
                gradient: ['rgba(6, 182, 212, 0.45)', 'rgba(6, 182, 212, 0.02)'],
                glow: 'rgba(6, 182, 212, 0.7)',
                shadow: 'rgba(6, 182, 212, 0.4)'
            }
        };

        // Period selector
        $('.period-btn').on('click', function () {
            $('.period-btn').removeClass('active');
            $(this).addClass('active');
            currentPeriod = $(this).data('period');
            loadStatistics();
        });

        // Tab click
        $('.aura-tab-button[data-tab="statistics"]').on('click', function () {
            if (!statsLoaded) {
                setTimeout(loadStatistics, 100);
                statsLoaded = true;
            }
        });

        // Auto-load if active
        if ($('.aura-tab-button[data-tab="statistics"]').hasClass('active')) {
            setTimeout(loadStatistics, 100);
            statsLoaded = true;
        }

        // Reset handlers
        $('#reset-stats-btn').on('click', function () { $('#reset-stats-modal').show(); });
        $('#cancel-reset-btn').on('click', function () { $('#reset-stats-modal').hide(); });
        $('#confirm-reset-btn').on('click', resetStatistics);
        $('#reset-stats-modal').on('click', function (e) { if (e.target === this) $(this).hide(); });

        function loadStatistics() {
            // Show loading state
            $('.stat-number').addClass('loading');

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: { action: 'aura_get_statistics', period: currentPeriod, nonce: typeof auraAdmin !== 'undefined' ? auraAdmin.statsNonce : '' },
                success: function (response) {
                    if (response.success) {
                        updateDashboard(response.data);
                    }
                },
                error: function () {
                    console.error('Failed to load statistics');
                    $('.stat-number').removeClass('loading').text('0');
                }
            });
        }

        function updateDashboard(data) {
            var chats = data.total_chat_opens || 0;
            var conversions = data.total_conversions || 0;
            var total = chats + conversions;
            var rate = chats > 0 ? Math.round((conversions / chats) * 100) : 0;

            // Animate numbers
            animateNumber('#stat-chat-opens', chats);
            animateNumber('#stat-conversions', conversions);
            animateNumber('#stat-rate', rate, '%');

            // Update conversion rate progress
            $('#rate-progress').css('width', Math.min(rate, 100) + '%');

            // Render sparklines
            renderSparklines(data.daily_data || []);

            // Render chart
            if (typeof Chart !== 'undefined') {
                renderTrendsChart(data.daily_data || []);
            } else {
                $('#chartjs-fallback').show();
                $('.chart-card-pro').hide();
            }
        }

        function animateNumber(selector, target, suffix) {
            var $el = $(selector);
            $el.removeClass('loading');
            var current = parseInt($el.text()) || 0;

            $({ val: current }).animate({ val: target }, {
                duration: 800,
                easing: 'swing',
                step: function () {
                    $el.text(Math.round(this.val) + (suffix || ''));
                },
                complete: function () {
                    $el.text(target + (suffix || ''));
                }
            });
        }

        function renderSparklines(dailyData) {
            var containers = {
                chats: '#sparkline-chats',
                conversions: '#sparkline-conversions'
            };

            $.each(containers, function (key, selector) {
                var $container = $(selector);
                $container.empty();

                if (!dailyData || dailyData.length === 0) return;

                var dataKey = key === 'chats' ? 'chat_opens' : key;
                var values = dailyData.slice(-7).map(function (d) { return d[dataKey] || 0; });
                var max = Math.max.apply(null, values) || 1;

                values.forEach(function (val) {
                    var height = Math.max((val / max) * 100, 10);
                    $container.append('<div class="spark-bar" style="height: ' + height + '%"></div>');
                });
            });
        }

        function createGradient(ctx, colorStart, colorEnd, height) {
            var gradient = ctx.createLinearGradient(0, 0, 0, height || 300);
            gradient.addColorStop(0, colorStart);
            gradient.addColorStop(0.4, colorEnd);
            gradient.addColorStop(1, 'rgba(255, 255, 255, 0)');
            return gradient;
        }

        // Enhanced glow plugin with animated effect
        var glowPlugin = {
            id: 'glowEffect',
            beforeDatasetsDraw: function (chart) {
                var ctx = chart.ctx;
                ctx.save();
                chart.data.datasets.forEach(function (dataset, i) {
                    var meta = chart.getDatasetMeta(i);
                    if (!meta.hidden && dataset.borderColor) {
                        ctx.shadowColor = dataset.glowColor || dataset.borderColor;
                        ctx.shadowBlur = 20;
                        ctx.shadowOffsetX = 0;
                        ctx.shadowOffsetY = 6;
                    }
                });
            },
            afterDatasetsDraw: function (chart) {
                chart.ctx.restore();
            }
        };

        // Custom background plugin for professional look
        var backgroundPlugin = {
            id: 'customBackground',
            beforeDraw: function (chart) {
                var ctx = chart.ctx;
                var chartArea = chart.chartArea;
                if (!chartArea) return;

                ctx.save();

                // Subtle gradient background
                var bgGradient = ctx.createLinearGradient(0, chartArea.top, 0, chartArea.bottom);
                bgGradient.addColorStop(0, 'rgba(248, 250, 252, 0.8)');
                bgGradient.addColorStop(1, 'rgba(255, 255, 255, 0.4)');

                ctx.fillStyle = bgGradient;
                ctx.fillRect(chartArea.left, chartArea.top, chartArea.right - chartArea.left, chartArea.bottom - chartArea.top);

                ctx.restore();
            }
        };

        function renderTrendsChart(dailyData) {
            var canvas = document.getElementById('trends-chart');
            if (!canvas) return;

            var ctx = canvas.getContext('2d');
            var $noData = $('#stats-no-data');

            if (!dailyData || dailyData.length === 0) {
                $(canvas).hide();
                $noData.show();
                if (trendsChart) { trendsChart.destroy(); trendsChart = null; }
                return;
            }

            $(canvas).show();
            $noData.hide();

            var labels = dailyData.map(function (d) { return formatDate(d.date); });
            var chatsData = dailyData.map(function (d) { return d.chat_opens || 0; });
            var conversionsData = dailyData.map(function (d) { return d.conversions || 0; });

            if (trendsChart) trendsChart.destroy();

            var chartHeight = canvas.parentElement.offsetHeight || 300;

            trendsChart = new Chart(ctx, {
                type: 'line',
                plugins: [backgroundPlugin, glowPlugin],
                data: {
                    labels: labels,
                    datasets: [
                        {
                            label: auraAdmin.i18n.chatOpens,
                            data: chatsData,
                            borderColor: colors.chats.main,
                            backgroundColor: createGradient(ctx, colors.chats.gradient[0], colors.chats.gradient[1], chartHeight),
                            glowColor: colors.chats.glow,
                            fill: true,
                            tension: 0.4,
                            borderWidth: 3,
                            pointRadius: 4,
                            pointHoverRadius: 10,
                            pointBackgroundColor: colors.chats.main,
                            pointBorderColor: '#ffffff',
                            pointBorderWidth: 2,
                            pointHoverBorderWidth: 4,
                            pointHoverBackgroundColor: '#ffffff',
                            pointHoverBorderColor: colors.chats.main,
                            pointStyle: 'circle',
                            borderCapStyle: 'round',
                            borderJoinStyle: 'round'
                        },
                        {
                            label: auraAdmin.i18n.conversions,
                            data: conversionsData,
                            borderColor: colors.conversions.main,
                            backgroundColor: createGradient(ctx, colors.conversions.gradient[0], colors.conversions.gradient[1], chartHeight),
                            glowColor: colors.conversions.glow,
                            fill: true,
                            tension: 0.4,
                            borderWidth: 3,
                            pointRadius: 4,
                            pointHoverRadius: 10,
                            pointBackgroundColor: colors.conversions.main,
                            pointBorderColor: '#ffffff',
                            pointBorderWidth: 2,
                            pointHoverBorderWidth: 4,
                            pointHoverBackgroundColor: '#ffffff',
                            pointHoverBorderColor: colors.conversions.main,
                            pointStyle: 'circle',
                            borderCapStyle: 'round',
                            borderJoinStyle: 'round'
                        }
                    ]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    animation: {
                        duration: 1500,
                        easing: 'easeOutQuart',
                        delay: function (context) {
                            return context.dataIndex * 80;
                        }
                    },
                    interaction: {
                        intersect: false,
                        mode: 'index',
                        axis: 'x'
                    },
                    plugins: {
                        legend: { display: false },
                        tooltip: {
                            enabled: true,
                            backgroundColor: 'rgba(15, 23, 42, 0.96)',
                            titleFont: { family: "'Inter', sans-serif", size: 13, weight: '600' },
                            bodyFont: { family: "'Inter', sans-serif", size: 12, weight: '500' },
                            footerFont: { family: "'Inter', sans-serif", size: 11 },
                            padding: { top: 14, bottom: 14, left: 16, right: 16 },
                            cornerRadius: 14,
                            displayColors: true,
                            usePointStyle: true,
                            boxWidth: 8,
                            boxHeight: 8,
                            boxPadding: 6,
                            borderColor: 'rgba(139, 92, 246, 0.3)',
                            borderWidth: 1,
                            caretSize: 8,
                            caretPadding: 12,
                            callbacks: {
                                title: function (context) {
                                    return '📅 ' + context[0].label;
                                },
                                label: function (context) {
                                    var icon = context.datasetIndex === 0 ? '💬' : '✅';
                                    return icon + ' ' + context.dataset.label + ': ' + context.parsed.y;
                                },
                                labelPointStyle: function () {
                                    return { pointStyle: 'circle', rotation: 0 };
                                }
                            }
                        }
                    },
                    scales: {
                        x: {
                            grid: {
                                display: true,
                                color: 'rgba(226, 232, 240, 0.4)',
                                lineWidth: 1,
                                drawTicks: false
                            },
                            border: { display: false },
                            ticks: {
                                font: { family: "'Inter', sans-serif", size: 11, weight: '500' },
                                color: '#94a3b8',
                                padding: 12,
                                maxRotation: 0
                            }
                        },
                        y: {
                            beginAtZero: true,
                            border: { display: false },
                            grid: {
                                color: 'rgba(226, 232, 240, 0.5)',
                                drawBorder: false,
                                lineWidth: 1,
                                drawTicks: false
                            },
                            ticks: {
                                font: { family: "'Inter', sans-serif", size: 11, weight: '500' },
                                color: '#94a3b8',
                                padding: 16,
                                stepSize: 1,
                                callback: function (value) {
                                    if (Number.isInteger(value)) {
                                        return value.toLocaleString();
                                    }
                                    return '';
                                }
                            }
                        }
                    },
                    elements: {
                        line: {
                            capBezierPoints: true
                        }
                    },
                    layout: {
                        padding: {
                            top: 20,
                            right: 20,
                            bottom: 10,
                            left: 10
                        }
                    }
                }
            });
        }

        function formatDate(dateStr) {
            var d = new Date(dateStr);
            var months = auraAdmin.i18n.months || ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
            return months[d.getMonth()] + ' ' + d.getDate();
        }

        function resetStatistics() {
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: { action: 'aura_reset_statistics', nonce: typeof auraAdmin !== 'undefined' ? auraAdmin.resetNonce : '' },
                success: function (response) {
                    $('#reset-stats-modal').hide();
                    showNotice(response.success ? 'Statistics reset successfully' : 'Failed to reset', response.success ? 'success' : 'error');
                    if (response.success) loadStatistics();
                },
                error: function () {
                    $('#reset-stats-modal').hide();
                    showNotice('Failed to reset statistics', 'error');
                }
            });
        }

        function showNotice(message, type) {
            var cls = type === 'success' ? 'aura-notice-success' : 'aura-notice-error';
            var icon = type === 'success' ? 'fa-check-circle' : 'fa-exclamation-circle';
            var $notice = $('<div class="aura-notice ' + cls + '"><i class="fas ' + icon + '"></i><span>' + message + '</span></div>');
            $('body').append($notice);
            setTimeout(function () { $notice.addClass('show'); }, 100);
            setTimeout(function () { $notice.removeClass('show'); setTimeout(function () { $notice.remove(); }, 300); }, 3000);
        }
    });

    // WooCommerce Premium Toggle System
    $('.woo-toggle-item').each(function () {
        var $item = $(this);
        var $checkbox = $item.find('input[type="checkbox"]');

        // Set initial state
        if ($checkbox.is(':checked')) {
            $item.addClass('active');
        }

        // Handle click on the entire item
        $item.on('click', function (e) {
            // Prevent double triggering when clicking on checkbox directly
            if ($(e.target).is('input[type="checkbox"]')) {
                return;
            }

            e.preventDefault();

            // Toggle checkbox
            $checkbox.prop('checked', !$checkbox.prop('checked'));

            // Toggle active class with animation
            if ($checkbox.prop('checked')) {
                $item.addClass('active');
            } else {
                $item.removeClass('active');
            }
        });

        // Handle direct checkbox changes
        $checkbox.on('change', function () {
            if ($(this).prop('checked')) {
                $item.addClass('active');
            } else {
                $item.removeClass('active');
            }
        });
    });
})(jQuery);