<?php
if (!defined('ABSPATH')) {
    exit;
}

// Register Settings
function aura_chat_register_settings() {
    register_setting(
        'aura_chat_options_group',
        'aura_chat_options',
        array(
            'type' => 'array',
            'sanitize_callback' => 'aura_chat_sanitize_options',
            'default' => array(),
        )
    );
}
add_action('admin_init', 'aura_chat_register_settings');

/**
 * Sanitize plugin options
 *
 * @param array $input The input options array
 * @return array Sanitized options
 */
function aura_chat_sanitize_options($input) {
    $sanitized = array();
    
    if (!is_array($input)) {
        return $sanitized;
    }
    
    // Text fields
    $text_fields = array(
        'phone_number', 'welcome_message', 'button_text', 'prefilled_message',
        'greeting_message', 'offline_message', 'offline_subtitle', 'tooltip_text', 'timezone',
        'time_start', 'time_end', 'button_style', 'position',
        'tooltip_visibility'
    );
    
    foreach ($text_fields as $field) {
        if (isset($input[$field])) {
            $sanitized[$field] = sanitize_text_field($input[$field]);
        }
    }
    
    // URL fields
    if (isset($input['agent_avatar'])) {
        $sanitized['agent_avatar'] = esc_url_raw($input['agent_avatar']);
    }
    
    // Color fields
    if (isset($input['custom_button_color'])) {
        $sanitized['custom_button_color'] = sanitize_hex_color($input['custom_button_color']);
    }
    
    // Integer fields
    $int_fields = array(
        'button_size_desktop', 'button_size_mobile', 'auto_popup',
        'position_offset_x', 'position_offset_y', 'mobile_position_offset_x',
        'mobile_position_offset_y', 'mobile_auto_popup_delay'
    );
    
    foreach ($int_fields as $field) {
        if (isset($input[$field])) {
            $sanitized[$field] = intval($input[$field]);
        }
    }
    
    // Checkbox fields (boolean)
    // Unchecked checkboxes are not sent in form data, so we explicitly set them to 0
    $sanitized['enable_availability'] = isset($input['enable_availability']) && $input['enable_availability'] == '1' ? 1 : 0;
    $sanitized['hide_on_mobile'] = isset($input['hide_on_mobile']) && $input['hide_on_mobile'] == '1' ? 1 : 0;
    $sanitized['mobile_tooltip_hidden'] = isset($input['mobile_tooltip_hidden']) && $input['mobile_tooltip_hidden'] == '1' ? 1 : 0;
    
    // Array fields
    if (isset($input['available_days']) && is_array($input['available_days'])) {
        $valid_days = array('Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun');
        $sanitized['available_days'] = array_intersect($input['available_days'], $valid_days);
    }
    
    if (isset($input['hide_on']) && is_array($input['hide_on'])) {
        $valid_pages = array('home', 'pages', 'posts', 'categories');
        $sanitized['hide_on'] = array_intersect($input['hide_on'], $valid_pages);
    }
    
    return $sanitized;
}

// Add Admin Menu
function aura_chat_add_admin_menu() {
    add_menu_page(
        'Aura Chat Button',
        'Aura Chat Button',
        'manage_options',
        'aura_chat',
        'aura_chat_options_page',
        'dashicons-admin-comments',
        100
    );
}
add_action('admin_menu', 'aura_chat_add_admin_menu');

// Enqueue Admin Scripts
function aura_chat_admin_scripts($hook) {
    // Only load on our plugin's admin page
    if ($hook !== 'toplevel_page_aura_chat') {
        return;
    }
    
    wp_enqueue_style('aura-admin-style', AURA_CHAT_URL . 'assets/css/aura-style.css', array(), '1.0.0');
    wp_enqueue_media();
    
    // Chart.js for statistics (local file, no CDN dependency)
    wp_enqueue_script('chartjs', AURA_CHAT_URL . 'assets/js/chart.min.js', array(), '4.5.1', true);
    
    wp_enqueue_script('aura-admin-script', AURA_CHAT_URL . 'assets/js/aura-admin.js', array('jquery', 'chartjs'), '1.0.0', true);
    
    // Pass nonces and translations for admin scripts
    wp_localize_script('aura-admin-script', 'auraAdmin', array(
        'resetNonce' => wp_create_nonce('aura_reset_stats'),
        'statsNonce' => wp_create_nonce('aura_get_stats'),
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'i18n' => array(
            'chooseImage' => esc_html__('Choose Image', 'aura-chat-button'),
            'removeImage' => esc_attr__('Remove Image', 'aura-chat-button'),
            'uploadImage' => esc_html__('Upload Image', 'aura-chat-button'),
            'changeImage' => esc_html__('Change Image', 'aura-chat-button'),
            'changesSaved' => esc_html__('Changes saved', 'aura-chat-button'),
            'chatOpens' => esc_html__('Chat Opens', 'aura-chat-button'),
            'conversions' => esc_html__('Conversions', 'aura-chat-button'),
            'months' => array(
                esc_html__('January', 'aura-chat-button'),
                esc_html__('February', 'aura-chat-button'),
                esc_html__('March', 'aura-chat-button'),
                esc_html__('April', 'aura-chat-button'),
                esc_html__('May', 'aura-chat-button'),
                esc_html__('June', 'aura-chat-button'),
                esc_html__('July', 'aura-chat-button'),
                esc_html__('August', 'aura-chat-button'),
                esc_html__('September', 'aura-chat-button'),
                esc_html__('October', 'aura-chat-button'),
                esc_html__('November', 'aura-chat-button'),
                esc_html__('December', 'aura-chat-button'),
            ),
        ),
    ));
}
add_action('admin_enqueue_scripts', 'aura_chat_admin_scripts');

// Enqueue Frontend Scripts
function aura_chat_enqueue_scripts() {
    // Frontend Styles & Scripts
    wp_enqueue_style('aura-frontend-style', AURA_CHAT_URL . 'assets/css/aura-frontend-style.css', array(), '1.0.0');
    
    // Add dynamic inline styles
    $dynamic_css = aura_generate_dynamic_css();
    if (!empty($dynamic_css)) {
        wp_add_inline_style('aura-frontend-style', $dynamic_css);
    }
    
    wp_enqueue_script('aura-frontend-script', AURA_CHAT_URL . 'assets/js/aura-script.js', array('jquery'), '1.0.0', true);

    // Pass PHP variables to JS
    $options = get_option('aura_chat_options');
    
    $prefilled_message = isset($options['prefilled_message']) && !empty($options['prefilled_message']) ? $options['prefilled_message'] : __('Hello, I have a question about your services.', 'aura-chat-button');
    
    wp_localize_script('aura-frontend-script', 'auraParams', array(
        'phoneNumber' => isset($options['phone_number']) ? $options['phone_number'] : '',
        'prefilledMessage' => $prefilled_message,
        'isAdmin' => current_user_can('manage_options'),
        'positionOffsetX' => isset($options['position_offset_x']) ? intval($options['position_offset_x']) : 0,
        'positionOffsetY' => isset($options['position_offset_y']) ? intval($options['position_offset_y']) : 0,
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'trackNonce' => wp_create_nonce('aura_track_event'),
    ));
}
add_action('wp_enqueue_scripts', 'aura_chat_enqueue_scripts');

/**
 * Generate dynamic CSS for the chat button widget
 * 
 * @return string Generated CSS string with escaped values
 */
function aura_generate_dynamic_css() {
    $options = get_option('aura_chat_options');
    
    // Button Sizes
    $button_size_desktop = isset($options['button_size_desktop']) ? intval($options['button_size_desktop']) : 70;
    $button_size_mobile = isset($options['button_size_mobile']) ? intval($options['button_size_mobile']) : 56;
    
    // Ensure sizes are within limits
    $button_size_desktop = max(50, min(70, $button_size_desktop));
    $button_size_mobile = max(40, min(70, $button_size_mobile));
    
    // Calculate icon sizes
    $icon_size_desktop = round($button_size_desktop * 0.49);
    $icon_size_mobile = round($button_size_mobile * 0.5);
    
    // Button Color
    $custom_color = isset($options['custom_button_color']) ? $options['custom_button_color'] : '#2ecc71';
    
    // Generate colors based on custom selection
    $main_color = $custom_color;
    $colors = array(
        'main' => $main_color,
        'light' => aura_adjust_brightness($main_color, 20),
        'lighter' => aura_adjust_brightness($main_color, 35),
        'dark' => aura_adjust_brightness($main_color, -15),
        'darker' => aura_adjust_brightness($main_color, -25)
    );
    
    // Escape all color values
    $c_main = esc_attr($colors['main']);
    $c_light = esc_attr($colors['light']);
    $c_lighter = esc_attr($colors['lighter']);
    $c_dark = esc_attr($colors['dark']);
    $c_darker = esc_attr($colors['darker']);
    $c_lighter_hover = esc_attr(aura_adjust_brightness($colors['lighter'], 15));
    $c_darker_hover = esc_attr(aura_adjust_brightness($colors['darker'], -10));
    
    // Helper function to convert hex to rgba
    $hex_to_rgba = function($hex, $alpha = 1) {
        $hex = ltrim($hex, '#');
        if (strlen($hex) == 3) {
            $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
        }
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        return sprintf('rgba(%d, %d, %d, %.2f)', $r, $g, $b, $alpha);
    };
    
    // Generate rgba colors for box-shadow
    $rgba_light_80 = $hex_to_rgba($colors['light'], 0.5);
    $rgba_main_66 = $hex_to_rgba($colors['main'], 0.4);
    $rgba_main_4d = $hex_to_rgba($colors['main'], 0.3);
    $rgba_light_b3 = $hex_to_rgba($colors['light'], 0.7);
    $rgba_main_80 = $hex_to_rgba($colors['main'], 0.5);
    $rgba_main_99 = $hex_to_rgba($colors['main'], 0.6);
    $rgba_dark_4d = $hex_to_rgba($colors['dark'], 0.3);
    
    $css = "
        .aura-fab {
            width: {$button_size_desktop}px;
            height: {$button_size_desktop}px;
        }
        .aura-fab svg.aura-fab-icon-chat,
        .aura-fab svg.aura-fab-icon-close {
            width: {$icon_size_desktop}px !important;
            height: {$icon_size_desktop}px !important;
        }
        @media (max-width: 768px) {
            .aura-fab {
                width: {$button_size_mobile}px;
                height: {$button_size_mobile}px;
            }
            .aura-fab svg.aura-fab-icon-chat,
            .aura-fab svg.aura-fab-icon-close {
                width: {$icon_size_mobile}px !important;
                height: {$icon_size_mobile}px !important;
            }
        }
        
        /* Style 1 - Elegant Round */
        .aura-fab.style-1 {
            border-radius: 50% !important;
            background: linear-gradient(135deg, {$c_lighter} 0%, {$c_light} 25%, {$c_main} 55%, {$c_dark} 80%, {$c_darker} 100%) !important;
            border: 4px solid !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
        }
        .aura-fab.style-1:hover {
            background: linear-gradient(135deg, {$c_light} 0%, {$c_lighter} 30%, {$c_main} 60%, {$c_dark} 100%) !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_dark} !important;
            border-right-color: {$c_dark} !important;
        }
        .aura-fab.style-1:active {
            background: linear-gradient(135deg, {$c_dark} 0%, {$c_main} 40%, {$c_light} 100%) !important;
            border-top-color: {$c_darker} !important;
            border-left-color: {$c_darker} !important;
            border-bottom-color: {$c_lighter} !important;
            border-right-color: {$c_lighter} !important;
        }
        
        /* Style 2 */
        .aura-fab.style-2 {
            background: linear-gradient(145deg, {$c_light} 0%, {$c_main} 50%, {$c_dark} 100%) !important;
            border-color: {$c_light} !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
        }
        .aura-fab.style-2:hover {
            background: linear-gradient(145deg, {$c_lighter} 0%, {$c_light} 50%, {$c_main} 100%) !important;
        }
        .aura-fab.style-2:active {
            background: linear-gradient(145deg, {$c_dark} 0%, {$c_main} 50%, {$c_light} 100%) !important;
            border-top-color: {$c_darker} !important;
            border-left-color: {$c_darker} !important;
            border-bottom-color: {$c_lighter} !important;
            border-right-color: {$c_lighter} !important;
        }
        .aura-fab.style-2::after {
            border-color: {$c_darker} !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
        }
        
        /* Style 3 */
        .aura-fab.style-3 {
            background: linear-gradient(145deg, {$c_light} 0%, {$c_main} 50%, {$c_dark} 100%) !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
        }
        .aura-fab.style-3:hover {
            background: linear-gradient(145deg, {$c_lighter} 0%, {$c_light} 50%, {$c_main} 100%) !important;
        }
        .aura-fab.style-3:active {
            background: linear-gradient(145deg, {$c_dark} 0%, {$c_main} 50%, {$c_light} 100%) !important;
            border-top-color: {$c_darker} !important;
            border-left-color: {$c_darker} !important;
            border-bottom-color: {$c_lighter} !important;
            border-right-color: {$c_lighter} !important;
        }
        
        /* Tooltip */
        .aura-fab-tooltip {
            background: linear-gradient(145deg, {$c_lighter} 0%, {$c_light} 20%, {$c_main} 50%, {$c_dark} 80%, {$c_darker} 100%) !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
        }
        .aura-fab-tooltip::before {
            display: none !important;
        }
        
        /* Style 4 - Deluxe Square */
        .aura-fab.style-4 {
            border-radius: 16px !important;
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.25) 0%, transparent 50%), linear-gradient(145deg, {$c_lighter} 0%, {$c_light} 25%, {$c_main} 50%, {$c_dark} 75%, {$c_darker} 100%) !important;
            border: 4px solid !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_light} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
        }
        .aura-fab.style-4:hover {
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.3) 0%, transparent 50%), linear-gradient(145deg, {$c_lighter} 0%, {$c_light} 30%, {$c_main} 60%, {$c_dark} 100%) !important;
            border-top-color: {$c_lighter_hover} !important;
            border-left-color: {$c_lighter} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker_hover} !important;
        }
        .aura-fab.style-4:active {
            background: linear-gradient(135deg, rgba(255, 255, 255, 0.15) 0%, transparent 50%), linear-gradient(145deg, {$c_dark} 0%, {$c_main} 40%, {$c_light} 100%) !important;
            border-top-color: {$c_darker} !important;
            border-left-color: {$c_dark} !important;
            border-bottom-color: {$c_lighter} !important;
            border-right-color: {$c_light} !important;
        }
        
        /* Style-1 Tooltip - Premium Framed Border with Dynamic Colors */
        .aura-fab.style-1 .aura-fab-tooltip.tooltip-style-1,
        .aura-fab-tooltip.tooltip-style-1 {
            background: linear-gradient(145deg, {$c_lighter} 0%, {$c_light} 15%, {$c_main} 30%, {$c_main} 45%, {$c_dark} 60%, {$c_dark} 75%, {$c_darker} 90%, {$c_darker} 100%) !important;
            border: 4px solid !important;
            border-top-color: {$c_lighter} !important;
            border-left-color: {$c_light} !important;
            border-bottom-color: {$c_darker} !important;
            border-right-color: {$c_darker} !important;
            outline: 2px solid rgba(255, 255, 255, 0.5) !important;
            outline-offset: -6px !important;
            box-shadow: none !important;
        }
        .aura-fab.style-1 .aura-fab-tooltip.tooltip-style-1::before,
        .aura-fab-tooltip.tooltip-style-1::before {
            display: none !important;
        }
        .aura-fab.style-1:hover .aura-fab-tooltip.tooltip-style-1.aura-tooltip-vis-hover,
        .aura-fab-tooltip.tooltip-style-1.aura-tooltip-vis-always {
            outline-color: rgba(255, 255, 255, 0.7) !important;
            box-shadow: none !important;
        }
        
        /* Style-4 Tooltip - Premium Signature Noir Glass (unique to Style-4) */
        .aura-fab.style-4 .aura-fab-tooltip.tooltip-style-4,
        .aura-fab-tooltip.tooltip-style-4 {
            /* Style-4 Bowl (ÇANAK): recessed cavity + depth (matches button palette) */
            background:
                radial-gradient(160px 110px at 28% 18%, rgba(255, 255, 255, 0.22) 0%, transparent 62%),
                radial-gradient(220px 140px at 55% 70%, rgba(0, 0, 0, 0.32) 0%, transparent 58%),
                linear-gradient(145deg, {$c_dark} 0%, {$c_main} 52%, {$c_dark} 100%) !important;

            /* Outer frame stays clean; engraved rim is handled by ::after */
            border: 2px solid rgba(255, 255, 255, 0.16) !important;
            outline: 1px solid rgba(0, 0, 0, 0.28) !important;
            outline-offset: -3px !important;

            /* Sunken feel */
            box-shadow:
                inset 0 12px 26px rgba(0, 0, 0, 0.22),
                inset 0 -10px 22px rgba(255, 255, 255, 0.10),
                inset 14px 0 20px rgba(0, 0, 0, 0.16),
                inset -14px 0 20px rgba(255, 255, 255, 0.06),
                0 18px 48px rgba(0, 0, 0, 0.16) !important;

            text-shadow:
                0 2px 10px rgba(0, 0, 0, 0.38),
                0 1px 2px rgba(0, 0, 0, 0.26) !important;
        }

        /* Style-4 Tooltip Engraved Rim (OYMA KENARLIK) — uses ::after slot, no shimmer */
        .aura-fab.style-4 .aura-fab-tooltip.tooltip-style-4::after,
        .aura-fab-tooltip.tooltip-style-4::after {
            border-radius: 12px !important;
            background: transparent !important;

            box-shadow:
                /* Engraved rim (symmetric): concentric, no directional mixing */
                inset 0 0 0 1px rgba(0, 0, 0, 0.28),
                inset 0 0 0 2px rgba(255, 255, 255, 0.12),
                inset 0 0 0 4px rgba(0, 0, 0, 0.18),
                inset 0 0 0 5px rgba(255, 255, 255, 0.08) !important;

            opacity: 1 !important;
            filter: none !important;
            animation: none !important;
        }

        .aura-fab.style-4:hover .aura-fab-tooltip.tooltip-style-4::after,
        .aura-fab-tooltip.tooltip-style-4.aura-tooltip-vis-always::after {
            box-shadow:
                /* Stronger (still symmetric) engraving on reveal */
                inset 0 0 0 1px rgba(0, 0, 0, 0.32),
                inset 0 0 0 2px rgba(255, 255, 255, 0.14),
                inset 0 0 0 4px rgba(0, 0, 0, 0.22),
                inset 0 0 0 5px rgba(255, 255, 255, 0.10) !important;
        }

        /* Hover/always: slightly brighter rim + deeper glass */
        .aura-fab.style-4:hover .aura-fab-tooltip.tooltip-style-4.aura-tooltip-vis-hover,
        .aura-fab-tooltip.tooltip-style-4.aura-tooltip-vis-always {
            border-color: rgba(255, 255, 255, 0.14) !important;
            outline-color: rgba(0, 0, 0, 0.72) !important;
            box-shadow:
                inset 0 1px 0 rgba(255, 255, 255, 0.12),
                inset 0 -12px 30px rgba(0, 0, 0, 0.6),
                inset 12px 0 22px rgba(0, 0, 0, 0.5),
                inset -12px 0 22px rgba(255, 255, 255, 0.06),
                0 22px 56px rgba(0, 0, 0, 0.32) !important;
        }
        
        /* Style-4 Button - Remove shadows */
        .aura-fab.style-4 {
            box-shadow: none !important;
        }
        .aura-fab.style-4:hover {
            box-shadow: none !important;
        }
        .aura-fab.style-4:active {
            box-shadow: none !important;
        }
    ";
    
    return $css;
}

/**
 * Check if widget should be hidden on current page based on visibility settings
 * Logic: Show ONLY on selected pages
 */
function aura_should_hide_on_page($options) {
    $show_on = isset($options['hide_on']) ? (array)$options['hide_on'] : array();
    
    // If nothing selected, hide everywhere
    if (empty($show_on)) {
        return true;
    }
    
    // Check if current page is in the "show" list
    if (in_array('home', $show_on) && (is_home() || is_front_page())) {
        return false; // Show on homepage
    }
    
    if (in_array('posts', $show_on) && is_single() && get_post_type() === 'post') {
        return false; // Show on blog posts
    }
    
    if (in_array('pages', $show_on) && is_page()) {
        return false; // Show on static pages
    }
    
    if (in_array('categories', $show_on) && is_category()) {
        return false; // Show on category pages
    }

    // If current page is not in the list, hide it
    return true;
}
