<?php
/**
 * Aura Chat Button - Frontend Display
 * 
 * SECURITY NOTE: Icon output is sanitized using wp_kses() at the point of echo
 * with a strict allowlist of SVG-specific tags and attributes.
 * 
 * @package Aura_Chat_Button
 * @see aura_get_icon() in includes/icons.php
 * @see aura_get_allowed_svg_html() for allowed SVG tags/attributes
 */

if (!defined('ABSPATH')) {
    exit;
}

function aura_chat_display_widget() {
    $options = get_option('aura_chat_options');
    
    // Check page visibility
    if (aura_should_hide_on_page($options)) {
        return;
    }
    
    // Settings
    $phone = isset($options['phone_number']) ? $options['phone_number'] : '';
    
    $avatar = isset($options['agent_avatar']) ? $options['agent_avatar'] : '';

    $welcome_title = isset($options['welcome_message']) && !empty($options['welcome_message']) ? $options['welcome_message'] : __('Support Team', 'aura-chat-button');
    $position = isset($options['position']) ? $options['position'] : 'right';
    $auto_popup = isset($options['auto_popup']) ? intval($options['auto_popup']) : 0;
    $tooltip_vis = isset($options['tooltip_visibility']) ? $options['tooltip_visibility'] : 'always';
    
    $mobile_tooltip_hidden = isset($options['mobile_tooltip_hidden']) ? $options['mobile_tooltip_hidden'] : 0;
    $hide_on_mobile = isset($options['hide_on_mobile']) ? intval($options['hide_on_mobile']) : 0;
    $mobile_auto_popup_delay = isset($options['mobile_auto_popup_delay']) ? intval($options['mobile_auto_popup_delay']) : 0;
    
    // Button Color
    $custom_color = isset($options['custom_button_color']) ? $options['custom_button_color'] : '#2ecc71';
    
    // Generate palette from custom color
    $main_color = $custom_color;
    $colors = array(
        'main' => $main_color,
        'light' => aura_adjust_brightness($main_color, 20),
        'lighter' => aura_adjust_brightness($main_color, 35),
        'dark' => aura_adjust_brightness($main_color, -15),
        'darker' => aura_adjust_brightness($main_color, -25)
    );
    
    // Availability Logic with WordPress Timezone Support
    $is_offline = false;
    $offline_message = isset($options['offline_message']) && !empty($options['offline_message']) 
        ? $options['offline_message'] 
        : __('We are currently offline. Please leave a message.', 'aura-chat-button');
    
    if (isset($options['enable_availability']) && $options['enable_availability']) {
        // Use WordPress timezone functions
        $current_day = aura_chat_current_time('D'); // Mon, Tue... (using plugin timezone)
        $current_time = aura_chat_current_time('H:i'); // Current time in plugin timezone
        
        $available_days = isset($options['available_days']) ? $options['available_days'] : array('Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun');
        $time_start = isset($options['time_start']) ? $options['time_start'] : '00:00';
        $time_end = isset($options['time_end']) ? $options['time_end'] : '23:59';

        if (!in_array($current_day, $available_days, true)) {
            $is_offline = true; // Not available today
        }
        if ($current_time < $time_start || $current_time > $time_end) {
            $is_offline = true; // Not available at this time
        }
    }

    // If no phone number is set, don't render
    if (empty($phone)) {
        return;
    }

    // Dynamic Styles
    $style_pos = ($position === 'left') ? 'left: 20px;' : 'right: 20px;';

    // Position Offsets
    $offset_x = isset($options['position_offset_x']) ? intval($options['position_offset_x']) : 0;
    $offset_y = isset($options['position_offset_y']) ? intval($options['position_offset_y']) : 0;

    // Mobile Position Offsets
    $mobile_offset_x = isset($options['mobile_position_offset_x']) ? intval($options['mobile_position_offset_x']) : 0;
    $mobile_offset_y = isset($options['mobile_position_offset_y']) ? intval($options['mobile_position_offset_y']) : 0;

    // Apply offsets if they exist
    if ($offset_x != 0 || $offset_y != 0) {
        $style_pos .= sprintf(' transform: translate(%dpx, %dpx);', $offset_x, $offset_y);
    }

    // Expose mobile offsets and button size via CSS variables for robust mobile positioning
    $style_pos .= sprintf(' --mx: %dpx; --my: %dpx; --aura-button-size-mobile: %dpx;', $mobile_offset_x, $mobile_offset_y, $button_size_mobile);

    // Mobile auto popup delay - use desktop setting if empty
    if ($mobile_auto_popup_delay === '' || $mobile_auto_popup_delay === 0) {
        $mobile_auto_popup_delay = $auto_popup;
    }
    
    // Button Sizes (customizable)
    $button_size_desktop = isset($options['button_size_desktop']) ? intval($options['button_size_desktop']) : 70;
    $button_size_mobile = isset($options['button_size_mobile']) ? intval($options['button_size_mobile']) : 56;
    
    // Ensure sizes are within limits
    $button_size_desktop = max(50, min(70, $button_size_desktop));
    $button_size_mobile = max(40, min(70, $button_size_mobile));
    
    // Calculate icon sizes (proportional to button size)
    $icon_size_desktop = round($button_size_desktop * 0.49); // ~49% of button
    $icon_size_mobile = round($button_size_mobile * 0.5); // ~50% of button
    
    
    // Mobile hide check - if enabled, don't render on mobile
    if ($hide_on_mobile) {
        $style_pos .= ' data-hide-mobile="1"';
    }
    
    ?>
    <div id="aura-chat-widget" class="aura-widget-container <?php echo esc_attr($position); ?>-position <?php echo $hide_on_mobile ? 'hide-on-mobile' : ''; ?>"
         data-auto-popup="<?php echo esc_attr($auto_popup); ?>"
         data-mobile-auto-popup="<?php echo esc_attr($mobile_auto_popup_delay); ?>"
         style="<?php echo esc_attr($style_pos); ?>">
        
        <!-- Chat Box -->
        <div class="aura-chat-box" id="aura-chat-box">
            <div class="aura-chat-header">
                <div class="aura-header-info" style="display: flex; align-items: center;">
                    <div class="aura-avatar-wrapper">
                        <?php if (!empty($avatar)): ?>
                            <img src="<?php echo esc_url($avatar); ?>" alt="Avatar" class="aura-header-avatar">
                        <?php else: ?>
                            <div class="aura-avatar-placeholder">
                                <?php echo wp_kses(aura_get_icon('chat'), aura_get_allowed_svg_html()); ?>
                            </div>
                        <?php endif; ?>
                        <span class="<?php echo $is_offline ? 'aura-offline-indicator' : 'aura-online-indicator'; ?>"></span>
                    </div>
                    <div>
                        <span class="aura-title"><?php echo esc_html($welcome_title); ?></span>
                    </div>
                </div>
                <button class="aura-close-btn" id="aura-close-chat"><?php echo wp_kses(aura_get_icon('times'), aura_get_allowed_svg_html()); ?></button>
            </div>
            
            <div class="aura-chat-body">
                <?php if ($is_offline): ?>
                    <!-- Offline Mode - Simple Chat Redirect -->
                    <div class="aura-offline-content">
                        <div class="aura-offline-icon">
                            <?php echo wp_kses(aura_get_icon('clock'), aura_get_allowed_svg_html()); ?>
                        </div>
                        <div class="aura-offline-title"><?php echo esc_html($offline_message); ?></div>
                        <?php 
                        $time_start = isset($options['time_start']) ? $options['time_start'] : '09:00';
                        $time_end = isset($options['time_end']) ? $options['time_end'] : '18:00';
                        ?>
                        <div class="aura-offline-hours">
                            <?php echo wp_kses(aura_get_icon('clock'), aura_get_allowed_svg_html()); ?> <?php echo esc_html($time_start); ?> - <?php echo esc_html($time_end); ?>
                        </div>
                        <?php 
                        $offline_subtitle = isset($options['offline_subtitle']) && !empty($options['offline_subtitle']) 
                            ? $options['offline_subtitle'] 
                            : __('You can still leave us a message!', 'aura-chat-button');
                        ?>
                        <p class="aura-offline-subtitle"><?php echo esc_html($offline_subtitle); ?></p>
                    </div>
                <?php else: ?>
                    <!-- Online Mode -->
                    <?php 
                    $greeting_message = isset($options['greeting_message']) && !empty($options['greeting_message']) 
                        ? $options['greeting_message'] 
                        : __('Hello! 👋 How can we help you today?', 'aura-chat-button');
                    ?>
                    <div class="aura-message-bubble incoming">
                        <?php echo nl2br(esc_html($greeting_message)); ?>
                        <span class="aura-time"><?php echo esc_html(aura_chat_current_time('H:i')); ?></span>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="aura-chat-footer">
                <?php if ($is_offline): ?>
                <?php
                /* translators: 1: Start time (e.g., 09:00), 2: End time (e.g., 18:00) */
                $offline_chat_message = sprintf(__("Hello, I'm writing outside of working hours (%1\$s - %2\$s). Please get back to me when you're available.", 'aura-chat-button'), $time_start, $time_end);
                ?>
                <button class="aura-start-chat-btn aura-offline-btn" id="aura-offline-chat" data-offline-message="<?php echo esc_attr($offline_chat_message); ?>">
                    <?php echo wp_kses(aura_get_icon('chat'), aura_get_allowed_svg_html()); ?>
                    <?php esc_html_e('Send Message', 'aura-chat-button'); ?>
                </button>
                <?php else: ?>
                <button class="aura-start-chat-btn" id="aura-start-chat">
                    <?php echo wp_kses(aura_get_icon('paper-plane'), aura_get_allowed_svg_html()); ?>
                    <?php echo (isset($options['button_text']) && !empty($options['button_text'])) ? esc_html($options['button_text']) : esc_html__('Start Chat', 'aura-chat-button'); ?>
                </button>
                <?php endif; ?>
            </div>
        </div>

        <!-- Floating Button -->
        <?php 
        $valid_styles = array('style-1', 'style-2', 'style-3', 'style-4');
        $button_style = isset($options['button_style']) && !empty($options['button_style']) && in_array($options['button_style'], $valid_styles) ? $options['button_style'] : 'style-1';
        ?>
        <button class="aura-fab <?php echo esc_attr($button_style); ?>" id="aura-fab" aria-label="Open Chat">
            <?php echo wp_kses(aura_get_icon('chat', 'aura-fab-icon-chat'), aura_get_allowed_svg_html()); ?>
            <?php echo wp_kses(aura_get_icon('times', 'aura-fab-icon-close'), aura_get_allowed_svg_html()); ?>
            <span class="aura-fab-tooltip tooltip-<?php echo esc_attr($button_style); ?> aura-tooltip-vis-<?php echo esc_attr($tooltip_vis); ?> <?php echo $mobile_tooltip_hidden ? 'mobile-hidden' : ''; ?>" style="cursor: pointer;"><?php echo (isset($options['tooltip_text']) && !empty($options['tooltip_text'])) ? esc_html($options['tooltip_text']) : esc_html__('Need Help?', 'aura-chat-button'); ?></span>
        </button>
    </div>
    <?php
}
add_action('wp_footer', 'aura_chat_display_widget');

// Helper function to adjust color brightness
function aura_adjust_brightness($hex, $percent) {
    $hex = ltrim($hex, '#');
    
    if (strlen($hex) == 3) {
        $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
    }
    
    $r = hexdec(substr($hex, 0, 2));
    $g = hexdec(substr($hex, 2, 2));
    $b = hexdec(substr($hex, 4, 2));
    
    $r = max(0, min(255, $r + ($r * $percent / 100)));
    $g = max(0, min(255, $g + ($g * $percent / 100)));
    $b = max(0, min(255, $b + ($b * $percent / 100)));
    
    return sprintf('#%02x%02x%02x', $r, $g, $b);
}
