jQuery(document).ready(function ($) {
    const fab = $('#aura-fab');
    const chatBox = $('#aura-chat-box');
    const closeBtn = $('#aura-close-chat');
    const startChatBtn = $('#aura-start-chat');
    const widgetContainer = $('#aura-chat-widget');

    // Statistics tracking function
    function trackEvent(eventType) {
        if (typeof auraParams !== 'undefined' && auraParams.ajaxUrl) {
            $.ajax({
                url: auraParams.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'aura_track_event',
                    event_type: eventType,
                    page_url: window.location.href,
                    nonce: auraParams.trackNonce || ''
                }
            });
        }
    }

    const autoPopupDelay = Number(widgetContainer.data('auto-popup')) || 0;
    const mobileAutoPopupDelay = Number(widgetContainer.data('mobile-auto-popup')) || 0;
    const isMobileViewport = window.matchMedia('(max-width: 768px)').matches;
    // Use mobile delay if set and > 0, otherwise use desktop delay
    const effectiveDelay = (isMobileViewport && mobileAutoPopupDelay > 0) ? mobileAutoPopupDelay : autoPopupDelay;
    if (effectiveDelay > 0) {
        setTimeout(function () {
            if (!chatBox.hasClass('active') && !sessionStorage.getItem('aura_chat_closed')) {
                chatBox.addClass('active');
                fab.addClass('active');
            }
        }, effectiveDelay * 1000);
    }

    // Toggle Chat Box
    fab.on('click', function () {
        chatBox.toggleClass('active');
        fab.toggleClass('active');
        // Track click event
        trackEvent('click');
        // Track chat open if opening
        if (chatBox.hasClass('active')) {
            trackEvent('chat_open');
        }
    });

    // Close Chat Box
    closeBtn.on('click', function () {
        chatBox.removeClass('active');
        fab.removeClass('active');
        sessionStorage.setItem('aura_chat_closed', 'true'); // Remember user closed it
    });

    // Helper to open chat
    function openChat(phone) {
        const message = encodeURIComponent(auraParams.prefilledMessage);

        if (!phone) {
            alert('Phone number is not configured.');
            return;
        }

        // Use universal wa.me deep-link that works on both mobile and desktop
        const url = `https://wa.me/${phone}?text=${message}`;

        window.open(url, '_blank');
    }

    // Start Chat
    startChatBtn.on('click', function () {
        const phone = auraParams.phoneNumber;
        // Track conversion event
        trackEvent('conversion');
        openChat(phone);
    });

    // Close on click outside
    $(document).on('click', function (e) {
        if (!$(e.target).closest('#aura-chat-widget').length) {
            chatBox.removeClass('active');
            fab.removeClass('active');
        }
    });

    // Offline Chat Button - Opens chat with offline message
    $('#aura-offline-chat').on('click', function() {
        const phone = auraParams.phoneNumber;
        const offlineMessage = $(this).data('offline-message') || 'Hello, I am writing outside of working hours.';
        
        if (!phone) {
            alert('Phone number is not configured.');
            return;
        }
        
        const url = `https://wa.me/${phone}?text=${encodeURIComponent(offlineMessage)}`;
        window.open(url, '_blank');
        
        // Track conversion
        trackEvent('conversion');
    });

    // Apply initial position offsets if set (RTL-aware)
    if (typeof auraParams !== 'undefined' && (auraParams.positionOffsetX !== 0 || auraParams.positionOffsetY !== 0)) {
        const isRTL = widgetContainer.hasClass('aura-rtl');
        const offsetX = isRTL ? -auraParams.positionOffsetX : auraParams.positionOffsetX;
        widgetContainer.css('transform', `translate(${offsetX}px, ${auraParams.positionOffsetY}px)`);
    }

    // Drag functionality for position fine-tuning (only for admins)
    if (typeof auraParams !== 'undefined' && auraParams.isAdmin) {
        setupDragFunctionality();
    }
});

function setupDragFunctionality() {
    const widget = $('#aura-chat-widget');
    const isRTL = widget.hasClass('aura-rtl');
    let isDragging = false;
    let startX, startY, startLeft, startBottom;

    // Make button draggable
    widget.on('mousedown', function(e) {
        if (e.target.id === 'aura-fab' || e.target.id === 'aura-chat-box' || $(e.target).closest('#aura-chat-widget').length) {
            isDragging = true;
            startX = e.clientX;
            startY = e.clientY;

            // Get current position
            const currentTransform = widget.css('transform');
            if (currentTransform && currentTransform !== 'none') {
                // Parse existing transform to get current offsets
                const matrix = currentTransform.match(/^matrix\((.+)\)$/);
                if (matrix) {
                    const values = matrix[1].split(', ');
                    startLeft = parseFloat(values[4]) || 0;
                    startBottom = parseFloat(values[5]) || 0;
                }
            } else {
                startLeft = 0;
                startBottom = 0;
            }

            widget.css('cursor', 'grabbing');
            e.preventDefault();
        }
    });

    $(document).on('mousemove', function(e) {
        if (isDragging) {
            // RTL-aware: reverse X direction for RTL languages
            const dx = isRTL ? -(e.clientX - startX) : (e.clientX - startX);
            const dy = e.clientY - startY;

            // Calculate new position
            const newLeft = startLeft + dx;
            const newBottom = startBottom + dy;

            // Apply transform
            widget.css('transform', `translate(${newLeft}px, ${newBottom}px)`);

            // Update admin values in real-time
            $('#position_offset_x').val(Math.round(newLeft));
            $('#position_offset_y').val(Math.round(newBottom));
        }
    });

    $(document).on('mouseup', function() {
        if (isDragging) {
            isDragging = false;
            widget.css('cursor', '');
        }
    });

    // Add visual indicator for drag mode
    if (widget.length) {
        widget.append('<div class="aura-drag-hint">Drag to position</div>');
    }
}
