<?php
/**
 * Fired when the plugin is uninstalled.
 *
 * When populating this file, consider the following flow
 * of control:
 *
 * - This method should be static
 * - Check if the $_REQUEST content actually is the plugin name
 * - Run an admin referrer check to make sure it goes through authentication
 * - Verify the output of $_GET makes sense
 * - Repeat with other user roles. Best directly by using the links/query string parameters.
 * - Repeat things for multisite. Once for a single site in the network, once sitewide.
 *
 * @package    Audit_Export
 */

// If uninstall not called from WordPress, then exit.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

/**
 * Uninstall the Audit Export plugin.
 */
class Audit_Export_Uninstaller {

	/**
	 * Run the uninstall process.
	 */
	public static function uninstall() {
		// Check if we should preserve data
		$preserve_data = get_option( 'audit_export_preserve_data_on_uninstall', false );
		
		if ( ! $preserve_data ) {
			self::delete_database_tables();
			self::delete_options();
			self::delete_transients();
			self::delete_files();
		}
		
		self::clear_scheduled_events();
		self::remove_capabilities();
		
		// Clear any cached data
		wp_cache_flush();
	}

	/**
	 * Delete database tables.
	 */
	private static function delete_database_tables() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'audit_export_reports';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct schema change necessary for uninstall, caching not applicable.
		$wpdb->query( $wpdb->prepare( 'DROP TABLE IF EXISTS %i', $table_name ) );
	}

	/**
	 * Delete plugin options.
	 */
	private static function delete_options() {
		// Core settings
		delete_option( 'audit_export_settings' );
		delete_option( 'audit_export_remote_post_settings' );
		delete_option( 'audit_export_db_version' );
		delete_option( 'audit_export_last_cron_run' );
		delete_option( 'audit_export_preserve_data_on_uninstall' );
		
		// For multisite, delete from all sites
		if ( is_multisite() ) {
			$sites = get_sites();
			foreach ( $sites as $site ) {
				switch_to_blog( $site->blog_id );
				
				delete_option( 'audit_export_settings' );
				delete_option( 'audit_export_remote_post_settings' );
				delete_option( 'audit_export_db_version' );
				delete_option( 'audit_export_last_cron_run' );
				delete_option( 'audit_export_preserve_data_on_uninstall' );
				
				restore_current_blog();
			}
		}
	}

	/**
	 * Delete transients.
	 */
	private static function delete_transients() {
		global $wpdb;
		
		// Delete transients with our prefix
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct query necessary for transient cleanup on uninstall.
		$wpdb->query( 
			"DELETE FROM {$wpdb->options} 
			WHERE option_name LIKE '_transient_audit_export_%' 
			OR option_name LIKE '_transient_timeout_audit_export_%'"
		);
		
		// For multisite
		if ( is_multisite() ) {
			$sites = get_sites();
			foreach ( $sites as $site ) {
				switch_to_blog( $site->blog_id );
				
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct query necessary for transient cleanup on uninstall.
				$wpdb->query( 
					"DELETE FROM {$wpdb->options} 
					WHERE option_name LIKE '_transient_audit_export_%' 
					OR option_name LIKE '_transient_timeout_audit_export_%'"
				);
				
				restore_current_blog();
			}
		}
	}

	/**
	 * Delete exported files.
	 */
	private static function delete_files() {
		$settings = get_option( 'audit_export_settings', array() );
		
		if ( empty( $settings['save_to_filesystem'] ) ) {
			return;
		}
		
		$filesystem_type = isset( $settings['filesystem_type'] ) ? $settings['filesystem_type'] : 'uploads';
		$filesystem_path = isset( $settings['filesystem_path'] ) ? $settings['filesystem_path'] : 'audit-export';
		
		// Get upload directory
		$upload_dir = wp_upload_dir();
		
		if ( $filesystem_type === 'uploads' ) {
			$base_dir = $upload_dir['basedir'];
		} else {
			$base_dir = WP_CONTENT_DIR;
		}
		
		$export_dir = $base_dir . '/' . $filesystem_path;
		
		// Delete directory and contents
		if ( is_dir( $export_dir ) ) {
			self::delete_directory( $export_dir );
		}
	}

	/**
	 * Clear scheduled events.
	 */
	private static function clear_scheduled_events() {
		// Clear all scheduled events for our plugin
		wp_clear_scheduled_hook( 'audit_export_cron_hook' );
		wp_clear_scheduled_hook( 'audit_export_process_queue' );
		
		// Clear any remaining cron events that might have our audit IDs
		$cron_array = _get_cron_array();
		
		if ( is_array( $cron_array ) ) {
			foreach ( $cron_array as $timestamp => $hooks ) {
				if ( isset( $hooks['audit_export_process_queue'] ) ) {
					foreach ( $hooks['audit_export_process_queue'] as $key => $event ) {
						wp_unschedule_event( $timestamp, 'audit_export_process_queue', $event['args'] );
					}
				}
			}
		}
	}

	/**
	 * Remove capabilities from roles.
	 */
	private static function remove_capabilities() {
		$capabilities = array(
			'audit_export_view_reports',
			'audit_export_process_reports',
			'audit_export_manage_settings',
		);
		
		// Get all roles
		$roles = wp_roles()->roles;
		
		foreach ( $roles as $role_name => $role_info ) {
			$role = get_role( $role_name );
			
			if ( $role ) {
				foreach ( $capabilities as $cap ) {
					$role->remove_cap( $cap );
				}
			}
		}
	}

	/**
	 * Recursively delete a directory and its contents.
	 *
	 * @param string $dir The directory to delete.
	 * @return bool True on success, false on failure.
	 */
	private static function delete_directory( $dir ) {
		if ( ! is_dir( $dir ) ) {
			return false;
		}
		
		$files = array_diff( scandir( $dir ), array( '.', '..' ) );
		
		foreach ( $files as $file ) {
			$path = $dir . '/' . $file;
			
			if ( is_dir( $path ) ) {
				self::delete_directory( $path );
			} else {
				// phpcs:ignore WordPress.WP.AlternativeFunctions.unlink_unlink -- Direct file deletion needed during uninstall.
				unlink( $path );
			}
		}
		
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_rmdir -- Direct directory deletion needed during uninstall.
		return rmdir( $dir );
	}
}

// Run the uninstaller
Audit_Export_Uninstaller::uninstall();