<?php
/**
 * Service class for handling audit export reports.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/services
 */

/**
 * Audit export report service.
 */
class Audit_Export_Report {

	/**
	 * Get report data from database.
	 *
	 * @param string $audit_name The audit name.
	 * @return array|null The report data or null if not found.
	 */
	public function get_report_data( $audit_name ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'audit_export_reports';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct query necessary for custom audit data table.
		$result = $wpdb->get_row( $wpdb->prepare(
			'SELECT audit_data FROM %i WHERE audit_name = %s ORDER BY id DESC LIMIT 1',
			$table_name,
			$audit_name
		) );

		if ( $result ) {
			return json_decode( $result->audit_data, true );
		}

		return null;
	}

	/**
	 * Save report data to database.
	 *
	 * @param string $audit_name The audit name.
	 * @param array  $data       The audit data.
	 * @return bool True on success, false on failure.
	 */
	public function save_report_data( $audit_name, $data ) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'audit_export_reports';
		
		// Delete existing report
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct query necessary for custom audit data table.
		$wpdb->delete( $table_name, array( 'audit_name' => $audit_name ) );
		
		// Insert new report
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct query necessary for custom audit data table.
		$result = $wpdb->insert(
			$table_name,
			array(
				'audit_name' => $audit_name,
				'audit_data' => wp_json_encode( $data ),
				'created_by' => get_current_user_id(),
			),
			array( '%s', '%s', '%d' )
		);
		
		return $result !== false;
	}

	/**
	 * Clear report data.
	 *
	 * @param string $audit_name The audit name.
	 * @return bool True on success, false on failure.
	 */
	public function clear_report_data( $audit_name ) {
		global $wpdb;
		
		$table_name = $wpdb->prefix . 'audit_export_reports';
		
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct query necessary for custom audit data table.
		$result = $wpdb->delete( $table_name, array( 'audit_name' => $audit_name ) );
		
		return $result !== false;
	}

	/**
	 * Get last processed date for an audit.
	 *
	 * @param string $audit_name The audit name.
	 * @return string|null The last processed date or null.
	 */
	public function get_last_processed_date( $audit_name ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'audit_export_reports';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct query necessary for custom audit data table.
		$result = $wpdb->get_var( $wpdb->prepare(
			'SELECT created_at FROM %i WHERE audit_name = %s ORDER BY id DESC LIMIT 1',
			$table_name,
			$audit_name
		) );

		return $result;
	}

	/**
	 * Get all available reports.
	 *
	 * @return array Array of available reports.
	 */
	public function get_available_reports() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'audit_export_reports';

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery -- Direct query necessary for custom audit data table.
		$results = $wpdb->get_results(
			$wpdb->prepare(
				'SELECT audit_name, created_at, created_by FROM %i ORDER BY audit_name, created_at DESC',
				$table_name
			)
		);

		$reports = array();
		foreach ( $results as $result ) {
			if ( ! isset( $reports[ $result->audit_name ] ) ) {
				$reports[ $result->audit_name ] = array(
					'audit_name' => $result->audit_name,
					'last_run' => $result->created_at,
					'created_by' => $result->created_by,
				);
			}
		}

		return array_values( $reports );
	}

	/**
	 * Export report as CSV.
	 *
	 * @param string $audit_name The audit name.
	 * @param array  $data       The audit data.
	 * @param array  $headers    The CSV headers.
	 * @return string The CSV content.
	 */
	public function export_as_csv( $audit_name, $data, $headers ) {
		$csv_output = '';
		
		// Add headers
		$csv_output .= implode( ',', array_map( array( $this, 'escape_csv_field' ), $headers ) ) . "\n";
		
		// Add data rows
		foreach ( $data as $row ) {
			$csv_output .= implode( ',', array_map( array( $this, 'escape_csv_field' ), $row ) ) . "\n";
		}
		
		return $csv_output;
	}

	/**
	 * Escape CSV field.
	 *
	 * @param string $field The field to escape.
	 * @return string The escaped field.
	 */
	private function escape_csv_field( $field ) {
		$field = str_replace( '"', '""', $field );
		if ( strpos( $field, ',' ) !== false || strpos( $field, '"' ) !== false || strpos( $field, "\n" ) !== false ) {
			$field = '"' . $field . '"';
		}
		return $field;
	}

	/**
	 * Save CSV to filesystem.
	 *
	 * @param string $audit_name The audit name.
	 * @param string $csv_content The CSV content.
	 * @return string|false The file path on success, false on failure.
	 */
	public function save_csv_to_filesystem( $audit_name, $csv_content ) {
		$settings = get_option( 'audit_export_settings', array() );
		
		if ( empty( $settings['save_to_filesystem'] ) ) {
			return false;
		}
		
		$filesystem_type = isset( $settings['filesystem_type'] ) ? $settings['filesystem_type'] : 'uploads';
		$filesystem_path = isset( $settings['filesystem_path'] ) ? $settings['filesystem_path'] : 'audit-export';
		
		// Get upload directory
		$upload_dir = wp_upload_dir();
		
		if ( $filesystem_type === 'uploads' ) {
			$base_dir = $upload_dir['basedir'];
		} else {
			$base_dir = WP_CONTENT_DIR;
		}
		
		$export_dir = $base_dir . '/' . $filesystem_path;
		
		// Create directory if it doesn't exist
		if ( ! wp_mkdir_p( $export_dir ) ) {
			return false;
		}
		
		// Generate filename
		$filename = sanitize_file_name( $audit_name . '_' . gmdate( 'Y-m-d_H-i-s' ) . '.csv' );
		$file_path = $export_dir . '/' . $filename;
		
		// Save file
		$result = file_put_contents( $file_path, $csv_content );
		
		return $result !== false ? $file_path : false;
	}

	/**
	 * Process audit and save results.
	 *
	 * @param string $audit_name    The audit name.
	 * @param object $audit_manager The audit manager instance.
	 * @return array Result array with success status and message.
	 */
	public function process_audit( $audit_name, $audit_manager ) {
		$audit = $audit_manager->get_audit( $audit_name );
		
		if ( ! $audit ) {
			return array(
				'success' => false,
				/* translators: %s: audit name */
				'message' => sprintf( __( 'Audit "%s" not found.', 'audit-export' ), $audit_name ),
			);
		}
		
		try {
			// Run the audit
			$audit_data = $audit->get_audit_data();
			
			// Save to database
			$saved = $this->save_report_data( $audit_name, $audit_data );
			
			if ( ! $saved ) {
				return array(
					'success' => false,
					'message' => __( 'Failed to save audit data to database.', 'audit-export' ),
				);
			}
			
			// Save to filesystem if enabled
			$settings = get_option( 'audit_export_settings', array() );
			if ( ! empty( $settings['save_to_filesystem'] ) ) {
				$csv_content = $this->export_as_csv( $audit_name, $audit_data['data'], $audit_data['headers'] );
				$file_path = $this->save_csv_to_filesystem( $audit_name, $csv_content );
				
				if ( $file_path ) {
					$audit_data['file_path'] = $file_path;
				}
			}
			
			// Fire action for other plugins to hook into
			do_action( 'audit_export_complete', $audit_name, $audit_data );
			
			return array(
				'success' => true,
				/* translators: %s: audit label */
				'message' => sprintf( __( 'Audit "%s" completed successfully.', 'audit-export' ), $audit->get_label() ),
				'data' => $audit_data,
			);
			
		} catch ( Exception $e ) {
			return array(
				'success' => false,
				/* translators: %s: error message */
				'message' => sprintf( __( 'Error processing audit: %s', 'audit-export' ), $e->getMessage() ),
			);
		}
	}
}