<?php
/**
 * Service class for remote posting functionality.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/services
 */

/**
 * Audit export remote post service.
 */
class Audit_Export_Remote_Post {

	/**
	 * Check if remote posting is enabled.
	 *
	 * @return bool True if enabled, false otherwise.
	 */
	public function is_enabled() {
		$settings = get_option( 'audit_export_remote_post_settings', array() );
		return ! empty( $settings['enable_remote_post'] );
	}

	/**
	 * Maybe post audit data to remote endpoint.
	 *
	 * @param string $audit_name The audit name.
	 * @param array  $audit_data The audit data.
	 */
	public function maybe_post_audit( $audit_name, $audit_data ) {
		if ( ! $this->is_enabled() ) {
			return;
		}
		
		$this->post_audit_data( $audit_name, $audit_data );
	}

	/**
	 * Post audit data to remote endpoint.
	 *
	 * @param string $audit_name The audit name.
	 * @param array  $audit_data The audit data.
	 * @return bool True on success, false on failure.
	 */
	public function post_audit_data( $audit_name, $audit_data ) {
		$settings = get_option( 'audit_export_remote_post_settings', array() );

		$remote_url = isset( $settings['remote_url'] ) ? $settings['remote_url'] : '';

		// Check if SiteDash integration should override
		if ( class_exists( 'SiteDash_Client' ) ) {
			$sitedash_client = new SiteDash_Client();

			if ( $sitedash_client->is_authenticated() ) {
				$sitedash_endpoint = $sitedash_client->get_report_endpoint();

				if ( $sitedash_endpoint ) {
					$remote_url = $sitedash_endpoint;
					// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
					error_log( 'Audit Export Remote Post: Using SiteDash endpoint for ' . $audit_name );
				}
			}
		}

		if ( empty( $remote_url ) ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
			error_log( 'Audit Export Remote Post: No remote URL configured.' );
			return false;
		}

		// Build post data
		$post_data = $this->build_post_data( $audit_name, $audit_data, $settings );

		// Add SiteDash specific data to post body if available
		if ( isset( $sitedash_client ) && $sitedash_client->is_authenticated() ) {
			$sitedash_settings = new SiteDash_Settings();
			$sitedash_config = $sitedash_settings->get_settings();

			if ( ! empty( $sitedash_config['siteid'] ) ) {
				$post_data['siteId'] = $sitedash_config['siteid'];
			}

			if ( ! empty( $sitedash_config['envid'] ) ) {
				$post_data['environmentId'] = $sitedash_config['envid'];
			}

			// Add report type for SiteDash
			$post_data['reportType'] = $audit_name;
		}

		// Allow filtering of the URL
		$remote_url = apply_filters( 'audit_export_post_url', $remote_url );

		// Allow filtering of the post data
		$post_data = apply_filters( 'audit_export_post_data', $post_data, $audit_name );

		// Prepare request arguments (this will JSON encode the body)
		$args = $this->build_request_args( $settings, $post_data );

		// Add SiteDash authentication headers if available
		if ( isset( $sitedash_client ) && $sitedash_client->is_authenticated() ) {
			$sitedash_headers = $sitedash_client->get_auth_headers();

			if ( ! isset( $args['headers'] ) ) {
				$args['headers'] = array();
			}

			$args['headers'] = array_merge( $args['headers'], $sitedash_headers );
		}

		// Allow filtering of request args
		$args = apply_filters( 'audit_export_post_request_args', $args, $audit_name );

		// Make the request
		$response = wp_remote_post( $remote_url, $args );

		// Handle response
		if ( is_wp_error( $response ) ) {
			$error_message = $response->get_error_message();
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
			error_log( "Audit Export Remote Post: Error posting {$audit_name}: {$error_message}" );
			return false;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );

		if ( $response_code >= 200 && $response_code < 300 ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
			error_log( "Audit Export Remote Post: Successfully posted {$audit_name}. Response code: {$response_code}" );

			// Log response if debug mode is enabled
			if ( ! empty( $settings['debug_mode'] ) ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
				error_log( "Audit Export Remote Post Debug: Response body: " . substr( $response_body, 0, 500 ) );
			}

			return true;
		} else {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
			error_log( "Audit Export Remote Post: Failed to post {$audit_name}. Response code: {$response_code}" );

			if ( ! empty( $settings['debug_mode'] ) ) {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
				error_log( "Audit Export Remote Post Debug: Response body: " . substr( $response_body, 0, 500 ) );
			}

			return false;
		}
	}

	/**
	 * Register core audit types with the manager.
	 *
	 * @param Audit_Export_Audit_Manager $manager The audit manager.
	 */
	private function register_core_audits( $manager ) {
		$audits = array(
			'site_report' => 'Audit_Export_Site_Report',
			'content_types' => 'Audit_Export_Content_Types',
			'users_roles' => 'Audit_Export_Users_Roles',
			'plugins' => 'Audit_Export_Plugins',
			'themes' => 'Audit_Export_Themes',
			'menus' => 'Audit_Export_Menus',
			'widgets' => 'Audit_Export_Widgets',
			'taxonomies' => 'Audit_Export_Taxonomies',
		);

		foreach ( $audits as $id => $class ) {
			if ( class_exists( $class ) ) {
				$manager->register_audit( $id, $class );
			}
		}
	}

	/**
	 * Build post data.
	 *
	 * @param string $audit_name The audit name.
	 * @param array  $audit_data The audit data.
	 * @param array  $settings   The remote post settings.
	 * @return array The post data.
	 */
	private function build_post_data( $audit_name, $audit_data, $settings ) {
		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
		error_log( 'Audit Export: Starting build_post_data for ' . $audit_name );
		// phpcs:disable WordPress.PHP.DevelopmentFunctions -- Intentional logging for debugging remote posts.
		error_log( 'Audit Export: Input audit_data: ' . print_r( $audit_data, true ) );
		// phpcs:enable WordPress.PHP.DevelopmentFunctions

		// Get WordPress version
		global $wp_version;

		// Prepare site name for use in identifiers
		$site_name = isset( $settings['site_name'] ) ? $settings['site_name'] : get_bloginfo( 'name' );
		$site_name_clean = str_replace( ' ', '_', $site_name );
		$site_name_clean = preg_replace( '/[^a-zA-Z0-9_]/', '', $site_name_clean );

		// Get host and IP information
		$host = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : 'unknown';
		$server_ip = isset( $_SERVER['SERVER_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_ADDR'] ) ) :
			( isset( $_SERVER['REMOTE_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ) : 'unknown' );

		// Build site info
		$site_info = array(
			'site_label' => $site_name,
			'site_name' => strtolower( $site_name_clean ),
			'cms' => 'WordPress',
			'version' => $wp_version,
			'version_major' => explode( '.', $wp_version )[0],
			'php_version' => phpversion(),
			'host' => $host,
			'server_ip_addr' => $server_ip,
			'date_generated' => current_time( 'Y-m-d H:i:s' ),
			'site_id' => '', // This will be populated by SiteDash if available
		);

		// phpcs:disable WordPress.PHP.DevelopmentFunctions -- Intentional logging for debugging remote posts.
		error_log( 'Audit Export: Built site_info: ' . print_r( $site_info, true ) );
		// phpcs:enable WordPress.PHP.DevelopmentFunctions

		// Allow filtering of site info
		$site_info = apply_filters( 'audit_export_post_site_info', $site_info );

		// Get headers from the audit instance
		$headers = array();
		if ( class_exists( 'Audit_Export_Audit_Manager' ) ) {
			$manager = new Audit_Export_Audit_Manager();
			
			// Re-register audits to ensure they're available
			$this->register_core_audits( $manager );
			
			$audit_instance = $manager->get_audit( $audit_name );
			if ( $audit_instance ) {
				$headers = $audit_instance->get_headers();
				// phpcs:disable WordPress.PHP.DevelopmentFunctions -- Intentional logging for debugging remote posts.
				error_log( 'Audit Export: Retrieved headers from audit instance for ' . $audit_name . ': ' . print_r( $headers, true ) );
				// phpcs:enable WordPress.PHP.DevelopmentFunctions
			} else {
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
				error_log( 'Audit Export: Could not get audit instance for ' . $audit_name );
			}
		}

		// Allow filtering of headers
		$headers = apply_filters( 'audit_export_get_headers', $headers, $audit_name );

		// phpcs:disable WordPress.PHP.DevelopmentFunctions -- Intentional logging for debugging remote posts.
		error_log( 'Audit Export: Headers: ' . print_r( $headers, true ) );
		// phpcs:enable WordPress.PHP.DevelopmentFunctions

		// Create a single combined data array with headers as the first row
		$combined_data = array();

		// First add the headers
		if ( ! empty( $headers ) ) {
			$combined_data[] = $headers;
		}

		// Add the audit data rows
		if ( ! empty( $audit_data ) && is_array( $audit_data ) ) {
			// Handle different possible data structures
			if ( isset( $audit_data['data'] ) && is_array( $audit_data['data'] ) ) {
				// If data is in a nested 'data' key
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
				error_log( 'Audit Export: Found data in nested "data" key with ' . count( $audit_data['data'] ) . ' rows' );
				foreach ( $audit_data['data'] as $row ) {
					if ( is_array( $row ) ) {
						$combined_data[] = $row;
					}
				}
			} elseif ( isset( $audit_data['rows'] ) && is_array( $audit_data['rows'] ) ) {
				// Alternative structure with 'rows' key
				// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
				error_log( 'Audit Export: Found data in "rows" key with ' . count( $audit_data['rows'] ) . ' rows' );
				foreach ( $audit_data['rows'] as $row ) {
					if ( is_array( $row ) ) {
						$combined_data[] = $row;
					}
				}
			} else {
				// If data is directly in the array
				// Check if first element is an array (indicating array of rows)
				$first_key = array_key_first( $audit_data );
				if ( $first_key !== null && is_numeric( $first_key ) && is_array( $audit_data[$first_key] ) ) {
					// It's an indexed array of rows
					// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
					error_log( 'Audit Export: Data is indexed array with ' . count( $audit_data ) . ' rows' );
					foreach ( $audit_data as $row ) {
						if ( is_array( $row ) ) {
							$combined_data[] = $row;
						}
					}
				} else {
					// It's a single row, wrap it
					// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
					error_log( 'Audit Export: Data appears to be single row, wrapping it' );
					$combined_data[] = $audit_data;
				}
			}
		}

		// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for debugging remote posts.
		error_log( 'Audit Export: Combined data has ' . count( $combined_data ) . ' rows' );

		// Prepare sample data for backend (headers + first data row)
		$sample_data = array();
		if ( ! empty( $headers ) ) {
			$sample_data[] = $headers;
		}
		// Add first data row (skip headers row at index 0 if it exists)
		if ( count( $combined_data ) > 1 ) {
			$sample_data[] = $combined_data[1];
		} elseif ( count( $combined_data ) === 1 && empty( $headers ) ) {
			// If no headers and only one row, include that row
			$sample_data[] = $combined_data[0];
		}

		// phpcs:disable WordPress.PHP.DevelopmentFunctions -- Intentional logging for debugging remote posts.
		error_log( 'Audit Export: Sample data for backend: ' . print_r( $sample_data, true ) );
		// phpcs:enable WordPress.PHP.DevelopmentFunctions

		// Build the complete report data structure
		$report_data = array(
			'audit' => $audit_name,
			'author' => (string) get_current_user_id(),
			'date' => (string) time(),
			'headers' => $headers, // Include headers explicitly
			'sample' => $sample_data, // Include sample data for backend validation
			'data' => $combined_data, // Full data set
			'row_count' => count( $combined_data ) - ( ! empty( $headers ) ? 1 : 0 ), // Actual data rows excluding headers
		);

		// Build the post data
		$post_data = array(
			'site_info' => $site_info,
			'report_data' => $report_data,
		);

		// phpcs:disable WordPress.PHP.DevelopmentFunctions -- Intentional logging for debugging remote posts.
		error_log( 'Audit Export: Built post_data before filters: ' . print_r( $post_data, true ) );
		// phpcs:enable WordPress.PHP.DevelopmentFunctions

		// Allow other plugins to alter the post data
		$post_data = apply_filters( 'audit_export_build_post_data', $post_data, $audit_name );

		// phpcs:disable WordPress.PHP.DevelopmentFunctions -- Intentional logging for debugging remote posts.
		error_log( 'Audit Export: Final post_data after filters: ' . print_r( $post_data, true ) );
		// phpcs:enable WordPress.PHP.DevelopmentFunctions

		return $post_data;
	}

	/**
	 * Build request arguments.
	 *
	 * @param array $settings  The remote post settings.
	 * @param array $post_data The post data.
	 * @return array The request arguments.
	 */
	private function build_request_args( $settings, $post_data ) {
		$args = array(
			'method' => 'POST',
			'timeout' => isset( $settings['timeout'] ) ? (int) $settings['timeout'] : 300,
			'headers' => array(
				'Content-Type' => 'application/json',
				'User-Agent' => 'WordPress Audit Export Plugin',
			),
			'body' => wp_json_encode( $post_data ),
			'sslverify' => isset( $settings['verify_ssl'] ) ? (bool) $settings['verify_ssl'] : true,
		);
		
		// Add authentication
		$auth_type = isset( $settings['authentication_type'] ) ? $settings['authentication_type'] : 'none';
		
		switch ( $auth_type ) {
			case 'basic':
				$username = isset( $settings['username'] ) ? $settings['username'] : '';
				$password = isset( $settings['password'] ) ? $settings['password'] : '';
				
				if ( ! empty( $username ) && ! empty( $password ) ) {
					$args['headers']['Authorization'] = 'Basic ' . base64_encode( $username . ':' . $password );
				}
				break;
				
			case 'bearer':
				$token = isset( $settings['token'] ) ? $settings['token'] : '';
				
				if ( ! empty( $token ) ) {
					$args['headers']['Authorization'] = 'Bearer ' . $token;
				}
				break;
		}
		
		return $args;
	}

	/**
	 * Test remote connection.
	 *
	 * @return array Result array with success status and message.
	 */
	public function test_connection() {
		$settings = get_option( 'audit_export_remote_post_settings', array() );
		
		$remote_url = isset( $settings['remote_url'] ) ? $settings['remote_url'] : '';
		
		if ( empty( $remote_url ) ) {
			return array(
				'success' => false,
				'message' => __( 'No remote URL configured.', 'audit-export' ),
			);
		}
		
		// Build test data
		$test_data = array(
			'test' => true,
			'timestamp' => current_time( 'mysql' ),
			'site_url' => get_site_url(),
		);
		
		// Prepare request
		$args = $this->build_request_args( $settings, $test_data );
		$args['body'] = wp_json_encode( $test_data );
		
		// Make request
		$response = wp_remote_post( $remote_url, $args );
		
		if ( is_wp_error( $response ) ) {
			return array(
				'success' => false,
				/* translators: %s: error message */
				'message' => sprintf( __( 'Connection failed: %s', 'audit-export' ), $response->get_error_message() ),
			);
		}
		
		$response_code = wp_remote_retrieve_response_code( $response );
		
		if ( $response_code >= 200 && $response_code < 300 ) {
			return array(
				'success' => true,
				/* translators: %d: HTTP response code */
				'message' => sprintf( __( 'Connection successful. Response code: %d', 'audit-export' ), $response_code ),
			);
		} else {
			return array(
				'success' => false,
				/* translators: %d: HTTP response code */
				'message' => sprintf( __( 'Connection failed. Response code: %d', 'audit-export' ), $response_code ),
			);
		}
	}

	/**
	 * Force post a specific audit.
	 *
	 * @param string $audit_name The audit name.
	 * @return array Result array with success status and message.
	 */
	public function force_post_audit( $audit_name ) {
		if ( ! $this->is_enabled() ) {
			return array(
				'success' => false,
				'message' => __( 'Remote posting is not enabled.', 'audit-export' ),
			);
		}
		
		$report_service = new Audit_Export_Report();
		$audit_data = $report_service->get_report_data( $audit_name );
		
		if ( empty( $audit_data ) ) {
			return array(
				'success' => false,
				/* translators: %s: audit name */
				'message' => sprintf( __( 'No data found for audit "%s".', 'audit-export' ), $audit_name ),
			);
		}
		
		$result = $this->post_audit_data( $audit_name, $audit_data );
		
		if ( $result ) {
			return array(
				'success' => true,
				/* translators: %s: audit name */
				'message' => sprintf( __( 'Successfully posted audit "%s" to remote endpoint.', 'audit-export' ), $audit_name ),
			);
		} else {
			return array(
				'success' => false,
				/* translators: %s: audit name */
				'message' => sprintf( __( 'Failed to post audit "%s" to remote endpoint.', 'audit-export' ), $audit_name ),
			);
		}
	}
}