<?php
/**
 * Service class for handling cron functionality.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/services
 */

/**
 * Audit export cron service.
 */
class Audit_Export_Cron {

	/**
	 * The audit manager instance.
	 *
	 * @var Audit_Export_Audit_Manager
	 */
	private $audit_manager;

	/**
	 * Constructor.
	 *
	 * @param Audit_Export_Audit_Manager $audit_manager The audit manager instance.
	 */
	public function __construct( $audit_manager ) {
		$this->audit_manager = $audit_manager;
	}

	/**
	 * Add custom cron interval.
	 *
	 * @param array $schedules Existing schedules.
	 * @return array Modified schedules.
	 */
	public function add_cron_interval( $schedules ) {
		$settings = get_option( 'audit_export_settings', array() );
		$frequency = isset( $settings['cron_frequency'] ) ? (int) $settings['cron_frequency'] : 1440;
		
		$schedules['audit_export_interval'] = array(
			'interval' => $frequency * 60, // Convert minutes to seconds
			/* translators: %d: number of minutes */
			'display' => sprintf( __( 'Every %d minutes', 'audit-export' ), $frequency ),
		);
		
		return $schedules;
	}

	/**
	 * Schedule cron events.
	 */
	public function schedule_events() {
		$settings = get_option( 'audit_export_settings', array() );
		
		// Only schedule if cron is enabled
		if ( empty( $settings['enable_cron'] ) ) {
			// Clear existing events if cron is disabled
			wp_clear_scheduled_hook( 'audit_export_cron_hook' );
			return;
		}
		
		// Check if event is already scheduled
		if ( ! wp_next_scheduled( 'audit_export_cron_hook' ) ) {
			$frequency = ! empty( $settings['run_on_every_cron'] ) ? 'hourly' : 'audit_export_interval';
			wp_schedule_event( time(), $frequency, 'audit_export_cron_hook' );
		}
	}

	/**
	 * Process audits via cron.
	 */
	public function process_audits() {
		$settings = get_option( 'audit_export_settings', array() );
		
		// Check if cron processing is enabled
		if ( empty( $settings['enable_cron'] ) ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for cron debugging.
			error_log( 'Audit Export: Cron processing is disabled in settings.' );
			return;
		}
		
		$run_every_cron = ! empty( $settings['run_on_every_cron'] );
		$frequency = isset( $settings['cron_frequency'] ) ? (int) $settings['cron_frequency'] : 1440;
		$last_run = get_option( 'audit_export_last_cron_run', 0 );
		
		// Determine whether we should run based on the settings
		$time_elapsed = ( time() - $last_run ) >= ( $frequency * 60 );
		$should_run = $run_every_cron || $time_elapsed;
		
		if ( ! $should_run ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for cron debugging.
			error_log( 'Audit Export: Skipping cron - frequency condition not met.' );
			return;
		}
		
		// Get queue timeout
		$queue_timeout = isset( $settings['queue_timeout'] ) ? (int) $settings['queue_timeout'] : 120;
		
		// Set time limit
		$original_time_limit = ini_get( 'max_execution_time' );
		// phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Necessary for long-running cron tasks.
		set_time_limit( $queue_timeout + 30 );
		
		try {
			$this->queue_audits( $queue_timeout );
			
			// Update last run time
			update_option( 'audit_export_last_cron_run', time() );
			
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for cron debugging.
			error_log( 'Audit Export: Cron processing completed successfully.' );
			
		} catch ( Exception $e ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Intentional logging for cron debugging.
			error_log( 'Audit Export: Error during cron processing: ' . $e->getMessage() );
		} finally {
			// Restore original time limit
			// phpcs:ignore Squiz.PHP.DiscouragedFunctions.Discouraged -- Restoring original setting.
			set_time_limit( $original_time_limit );
		}
	}

	/**
	 * Queue audits for processing.
	 *
	 * @param int $timeout The timeout in seconds.
	 * @return int Number of items queued.
	 */
	public function queue_audits( $timeout = 120 ) {
		$audits = $this->audit_manager->get_audits();
		$items_queued = 0;
		
		foreach ( $audits as $audit_id => $audit_class ) {
			// Schedule audit processing
			wp_schedule_single_event( time(), 'audit_export_process_queue', array( $audit_id ) );
			$items_queued++;
		}
		
		return $items_queued;
	}

	/**
	 * Clear scheduled events.
	 */
	public function clear_scheduled_events() {
		wp_clear_scheduled_hook( 'audit_export_cron_hook' );
		wp_clear_scheduled_hook( 'audit_export_process_queue' );
	}

	/**
	 * Get next scheduled run time.
	 *
	 * @return int|false The timestamp of the next scheduled run or false if not scheduled.
	 */
	public function get_next_scheduled_run() {
		return wp_next_scheduled( 'audit_export_cron_hook' );
	}

	/**
	 * Force run cron now.
	 */
	public function force_run_now() {
		// Clear existing schedule
		wp_clear_scheduled_hook( 'audit_export_cron_hook' );
		
		// Run immediately
		$this->process_audits();
		
		// Reschedule
		$this->schedule_events();
	}
}