<?php
/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes
 */

/**
 * The core plugin class.
 */
class Audit_Export {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @access   protected
	 * @var      Audit_Export_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * The audit manager instance.
	 *
	 * @access   protected
	 * @var      Audit_Export_Audit_Manager    $audit_manager    Manages all audit types.
	 */
	protected $audit_manager;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 */
	public function __construct() {
		$this->version = AUDIT_EXPORT_VERSION;
		$this->plugin_name = AUDIT_EXPORT_NAME;

		$this->load_dependencies();
		$this->set_locale();
		$this->define_admin_hooks();
		$this->define_service_hooks();
		$this->register_audits();
		$this->define_cli_commands();
	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Audit_Export_Loader. Orchestrates the hooks of the plugin.
	 * - Audit_Export_i18n. Defines internationalization functionality.
	 * - Audit_Export_Admin. Defines all hooks for the admin area.
	 * - Audit_Export_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @access   private
	 */
	private function load_dependencies() {
		// Core classes
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'includes/class-audit-export-loader.php';
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'includes/class-audit-export-i18n.php';
		
		// Admin classes
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'admin/class-audit-export-admin.php';
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'admin/class-audit-export-settings.php';
		
		// Service classes
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'includes/services/class-audit-export-report.php';
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'includes/services/class-audit-export-cron.php';
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'includes/services/class-audit-export-queue.php';
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'includes/services/class-audit-export-remote-post.php';
		
		// Audit classes
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'includes/audits/abstract-audit-export-audit.php';
		require_once AUDIT_EXPORT_PLUGIN_DIR . 'includes/class-audit-export-audit-manager.php';
		
		// Load all audit implementations
		foreach ( glob( AUDIT_EXPORT_PLUGIN_DIR . 'includes/audits/class-audit-export-*.php' ) as $audit_file ) {
			require_once $audit_file;
		}

		$this->loader = new Audit_Export_Loader();
		$this->audit_manager = new Audit_Export_Audit_Manager();
	}

	/**
	 * Define the locale for this plugin for internationalization.
	 *
	 * Uses the Audit_Export_i18n class in order to set the domain and to register the hook
	 * with WordPress.
	 *
	 * @access   private
	 */
	private function set_locale() {
		$plugin_i18n = new Audit_Export_i18n();
		$this->loader->add_action( 'plugins_loaded', $plugin_i18n, 'load_plugin_textdomain' );
	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @access   private
	 */
	private function define_admin_hooks() {
		$plugin_admin = new Audit_Export_Admin( $this->get_plugin_name(), $this->get_version(), $this->audit_manager );
		$plugin_settings = new Audit_Export_Settings( $this->get_plugin_name(), $this->get_version() );

		// Admin scripts and styles
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

		// Admin menu
		$this->loader->add_action( 'admin_menu', $plugin_admin, 'add_admin_menu' );
		
		// Settings
		$this->loader->add_action( 'admin_init', $plugin_settings, 'register_settings' );
		
		// AJAX handlers
		$this->loader->add_action( 'wp_ajax_audit_export_process', $plugin_admin, 'ajax_process_audit' );
		$this->loader->add_action( 'wp_ajax_audit_export_download', $plugin_admin, 'ajax_download_report' );
	}

	/**
	 * Register all of the hooks related to the service functionality
	 * of the plugin.
	 *
	 * @access   private
	 */
	private function define_service_hooks() {
		$cron_service = new Audit_Export_Cron( $this->audit_manager );
		$queue_service = new Audit_Export_Queue( $this->audit_manager );
		$remote_post_service = new Audit_Export_Remote_Post();

		// Cron hooks
		$this->loader->add_action( 'audit_export_cron_hook', $cron_service, 'process_audits' );
		$this->loader->add_filter( 'cron_schedules', $cron_service, 'add_cron_interval' );
		
		// Queue processing
		$this->loader->add_action( 'audit_export_process_queue', $queue_service, 'process_queue_item' );
		
		// Remote post hooks
		$this->loader->add_action( 'audit_export_complete', $remote_post_service, 'maybe_post_audit', 10, 2 );
		
		// Schedule cron events on activation
		$this->loader->add_action( 'init', $cron_service, 'schedule_events' );
	}

	/**
	 * Register all audit types.
	 *
	 * @access   private
	 */
	private function register_audits() {
		// Register core audit types
		$this->audit_manager->register_audit( 'site_report', 'Audit_Export_Site_Report' );
		$this->audit_manager->register_audit( 'content_types', 'Audit_Export_Content_Types' );
		$this->audit_manager->register_audit( 'users_roles', 'Audit_Export_Users_Roles' );
		$this->audit_manager->register_audit( 'plugins', 'Audit_Export_Plugins' );
		$this->audit_manager->register_audit( 'themes', 'Audit_Export_Themes' );
		$this->audit_manager->register_audit( 'menus', 'Audit_Export_Menus' );
		$this->audit_manager->register_audit( 'widgets', 'Audit_Export_Widgets' );
		$this->audit_manager->register_audit( 'taxonomies', 'Audit_Export_Taxonomies' );
		
		// Allow other plugins to register audits
		do_action( 'audit_export_register_audits', $this->audit_manager );
	}

	/**
	 * Register WP-CLI commands if WP-CLI is available.
	 *
	 * @access   private
	 */
	private function define_cli_commands() {
		if ( defined( 'WP_CLI' ) && WP_CLI ) {
			require_once AUDIT_EXPORT_PLUGIN_DIR . 'includes/cli/class-audit-export-cli.php';
			WP_CLI::add_command( 'audit-export', 'Audit_Export_CLI' );
		}
	}

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @return    Audit_Export_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

	/**
	 * Get the audit manager instance.
	 *
	 * @return    Audit_Export_Audit_Manager    The audit manager.
	 */
	public function get_audit_manager() {
		return $this->audit_manager;
	}
}