<?php
/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes
 */

/**
 * Fired during plugin activation.
 */
class Audit_Export_Activator {

	/**
	 * Plugin activation routine.
	 *
	 * Create database tables and set default options.
	 */
	public static function activate() {
		self::create_tables();
		self::set_default_options();
		self::schedule_cron_events();
		
		// Clear the permalinks after the plugin has been activated.
		flush_rewrite_rules();
	}

	/**
	 * Create custom database tables.
	 */
	private static function create_tables() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();
		$table_name = $wpdb->prefix . 'audit_export_reports';

		$sql = "CREATE TABLE $table_name (
			id bigint(20) NOT NULL AUTO_INCREMENT,
			audit_name varchar(255) NOT NULL,
			audit_data longtext NOT NULL,
			created_by bigint(20) NOT NULL DEFAULT 0,
			created_at datetime DEFAULT CURRENT_TIMESTAMP,
			updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
			PRIMARY KEY (id),
			KEY audit_name (audit_name),
			KEY created_at (created_at)
		) $charset_collate;";

		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
		dbDelta( $sql );

		// Store the database version for future upgrades
		update_option( 'audit_export_db_version', '1.0.1' );
	}

	/**
	 * Set default plugin options.
	 */
	private static function set_default_options() {
		// Core settings
		add_option( 'audit_export_settings', array(
			'save_to_filesystem' => false,
			'filesystem_type' => 'uploads',
			'filesystem_path' => 'audit-export',
			'enable_cron' => false,
			'run_on_every_cron' => false,
			'cron_frequency' => 1440, // 24 hours in minutes
			'queue_timeout' => 120, // 2 minutes in seconds
		) );

		// Remote post settings
		add_option( 'audit_export_remote_post_settings', array(
			'enable_remote_post' => false,
			'remote_url' => '',
			'site_name' => get_bloginfo( 'name' ),
			'authentication_type' => 'none',
			'username' => '',
			'password' => '',
			'token' => '',
			'timeout' => 300,
			'verify_ssl' => true,
			'debug_mode' => false,
		) );

		// Capabilities
		self::add_capabilities();
	}

	/**
	 * Add plugin capabilities to roles.
	 */
	private static function add_capabilities() {
		$admin_role = get_role( 'administrator' );
		
		if ( $admin_role ) {
			$admin_role->add_cap( 'audit_export_view_reports' );
			$admin_role->add_cap( 'audit_export_process_reports' );
			$admin_role->add_cap( 'audit_export_manage_settings' );
		}

		$editor_role = get_role( 'editor' );
		
		if ( $editor_role ) {
			$editor_role->add_cap( 'audit_export_view_reports' );
		}
	}

	/**
	 * Schedule cron events.
	 */
	private static function schedule_cron_events() {
		if ( ! wp_next_scheduled( 'audit_export_cron_hook' ) ) {
			wp_schedule_event( time(), 'daily', 'audit_export_cron_hook' );
		}
	}
}