<?php
/**
 * Widgets audit implementation.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/audits
 */

/**
 * Widgets audit class.
 */
class Audit_Export_Widgets extends Audit_Export_Audit {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->label = __( 'Widgets', 'audit-export' );
		$this->description = __( 'Audit information about widgets and sidebars.', 'audit-export' );
		$this->group = 'system';
		$this->data_type = 'flat';
		$this->identifier = 'sidebar_id';
		$this->headers = array(
			__( 'Sidebar ID', 'audit-export' ),
			__( 'Sidebar Name', 'audit-export' ),
			__( 'Widget Count', 'audit-export' ),
			__( 'Widget Types', 'audit-export' ),
			__( 'Active', 'audit-export' ),
		);
	}

	/**
	 * Prepare data for the audit.
	 *
	 * @return array
	 */
	public function prepare_data() {
		global $wp_registered_sidebars;
		
		$sidebars = array();
		foreach ( $wp_registered_sidebars as $sidebar_id => $sidebar ) {
			$sidebars[] = array(
				'id' => $sidebar_id,
				'info' => $sidebar,
			);
		}
		
		return $sidebars;
	}

	/**
	 * Process a single data item.
	 *
	 * @param array $sidebar The sidebar data.
	 * @return array
	 */
	public function process_data( $sidebar ) {
		$sidebar_id = $sidebar['id'];
		$sidebar_info = $sidebar['info'];

		// Get widgets in this sidebar using the option directly instead of wp_get_sidebars_widgets().
		$widgets = get_option( 'sidebars_widgets', array() );
		$sidebar_widgets = isset( $widgets[ $sidebar_id ] ) ? $widgets[ $sidebar_id ] : array();
		
		// Get widget types
		$widget_types = array();
		foreach ( $sidebar_widgets as $widget_id ) {
			$widget_type = $this->get_widget_type( $widget_id );
			if ( ! in_array( $widget_type, $widget_types ) ) {
				$widget_types[] = $widget_type;
			}
		}
		
		// Check if sidebar is active
		$is_active = ! empty( $sidebar_widgets ) ? __( 'Yes', 'audit-export' ) : __( 'No', 'audit-export' );

		return array(
			$sidebar_id,
			$sidebar_info['name'],
			count( $sidebar_widgets ),
			implode( ', ', $widget_types ),
			$is_active,
		);
	}

	/**
	 * Get widget type from widget ID.
	 *
	 * @param string $widget_id The widget ID.
	 * @return string The widget type.
	 */
	private function get_widget_type( $widget_id ) {
		$widget_type = preg_replace( '/-\d+$/', '', $widget_id );
		return $widget_type;
	}
}