<?php
/**
 * Site Report audit implementation.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/audits
 */

/**
 * Site Report audit class.
 */
class Audit_Export_Site_Report extends Audit_Export_Audit {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->label = __( 'Site Report', 'audit-export' );
		$this->description = __( 'A general batch of items to check in a report.', 'audit-export' );
		$this->group = 'general';
		$this->data_type = 'process';
		$this->identifier = 'report_name';
		$this->headers = array(
			__( 'Report', 'audit-export' ),
			__( 'Description', 'audit-export' ),
			__( 'Status', 'audit-export' ),
		);
	}

	/**
	 * Prepare data for the audit.
	 *
	 * @return array
	 */
	public function prepare_data() {
		$reports = array();

		$reports[] = array(
			'name' => 'php_version',
			'label' => __( 'PHP Version', 'audit-export' ),
			'description' => __( 'Return PHP version that\'s currently installed on the environment.', 'audit-export' ),
			'process' => 'check_php_version',
		);

		$reports[] = array(
			'name' => 'wordpress_version',
			'label' => __( 'WordPress Version', 'audit-export' ),
			'description' => __( 'The current WordPress version for the site.', 'audit-export' ),
			'process' => 'check_wordpress_version',
		);

		$reports[] = array(
			'name' => 'site_url',
			'label' => __( 'Site URL', 'audit-export' ),
			'description' => __( 'The URL of the WordPress site.', 'audit-export' ),
			'process' => 'check_site_url',
		);

		$reports[] = array(
			'name' => 'active_theme',
			'label' => __( 'Active Theme', 'audit-export' ),
			'description' => __( 'The currently active theme.', 'audit-export' ),
			'process' => 'check_active_theme',
		);

		$reports[] = array(
			'name' => 'multisite',
			'label' => __( 'Multisite', 'audit-export' ),
			'description' => __( 'Check if WordPress is running as multisite.', 'audit-export' ),
			'process' => 'check_multisite',
		);

		$reports[] = array(
			'name' => 'debug_mode',
			'label' => __( 'Debug Mode', 'audit-export' ),
			'description' => __( 'Check if WordPress debug mode is enabled.', 'audit-export' ),
			'process' => 'check_debug_mode',
		);

		$reports[] = array(
			'name' => 'timezone',
			'label' => __( 'Timezone', 'audit-export' ),
			'description' => __( 'The configured timezone for the site.', 'audit-export' ),
			'process' => 'check_timezone',
		);

		$reports[] = array(
			'name' => 'ssl_enabled',
			'label' => __( 'SSL Enabled', 'audit-export' ),
			'description' => __( 'Check if the site is using SSL.', 'audit-export' ),
			'process' => 'check_ssl',
		);

		return $reports;
	}

	/**
	 * Process a single data item.
	 *
	 * @param array $item The item to process.
	 * @return array
	 */
	public function process_data( $item ) {
		$report_data = array();
		
		if ( ! empty( $item['process'] ) && method_exists( $this, $item['process'] ) ) {
			$report_data = call_user_func( array( $this, $item['process'] ) );
		}

		return array(
			$item['label'],
			$item['description'],
			! empty( $report_data['status'] ) ? $report_data['status'] : __( 'Unknown', 'audit-export' ),
		);
	}

	/**
	 * Check PHP version.
	 *
	 * @return array
	 */
	private function check_php_version() {
		return array( 'status' => PHP_VERSION );
	}

	/**
	 * Check WordPress version.
	 *
	 * @return array
	 */
	private function check_wordpress_version() {
		global $wp_version;
		return array( 'status' => $wp_version );
	}

	/**
	 * Check site URL.
	 *
	 * @return array
	 */
	private function check_site_url() {
		return array( 'status' => get_site_url() );
	}

	/**
	 * Check active theme.
	 *
	 * @return array
	 */
	private function check_active_theme() {
		$theme = wp_get_theme();
		return array( 'status' => $theme->get( 'Name' ) . ' (' . $theme->get( 'Version' ) . ')' );
	}

	/**
	 * Check if multisite.
	 *
	 * @return array
	 */
	private function check_multisite() {
		return array( 'status' => is_multisite() ? __( 'Yes', 'audit-export' ) : __( 'No', 'audit-export' ) );
	}

	/**
	 * Check debug mode.
	 *
	 * @return array
	 */
	private function check_debug_mode() {
		return array( 'status' => WP_DEBUG ? __( 'Enabled', 'audit-export' ) : __( 'Disabled', 'audit-export' ) );
	}

	/**
	 * Check timezone.
	 *
	 * @return array
	 */
	private function check_timezone() {
		return array( 'status' => wp_timezone_string() );
	}

	/**
	 * Check SSL.
	 *
	 * @return array
	 */
	private function check_ssl() {
		return array( 'status' => is_ssl() ? __( 'Yes', 'audit-export' ) : __( 'No', 'audit-export' ) );
	}
}