<?php
/**
 * Content Types audit implementation.
 *
 * @package    Audit_Export
 * @subpackage Audit_Export/includes/audits
 */

/**
 * Content Types audit class.
 */
class Audit_Export_Content_Types extends Audit_Export_Audit {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->label = __( 'Content Types', 'audit-export' );
		$this->description = __( 'Audit information about post types, including post counts and status.', 'audit-export' );
		$this->group = 'content';
		$this->data_type = 'flat';
		$this->identifier = 'post_type';
		$this->headers = array(
			__( 'Post Type Label', 'audit-export' ),
			__( 'Post Type Name', 'audit-export' ),
			__( 'Published', 'audit-export' ),
			__( 'Draft', 'audit-export' ),
			__( 'Private', 'audit-export' ),
			__( 'Trash', 'audit-export' ),
			__( 'Total', 'audit-export' ),
			__( 'Public', 'audit-export' ),
			__( 'Hierarchical', 'audit-export' ),
		);
	}

	/**
	 * Prepare data for the audit.
	 *
	 * @return array
	 */
	public function prepare_data() {
		$post_types = get_post_types( array(), 'objects' );
		$types_to_audit = array();

		foreach ( $post_types as $post_type ) {
			// Skip built-in types we don't want to audit
			if ( in_array( $post_type->name, array( 'revision', 'nav_menu_item', 'custom_css', 'customize_changeset', 'oembed_cache', 'user_request', 'wp_block', 'wp_template', 'wp_template_part', 'wp_global_styles', 'wp_navigation' ) ) ) {
				continue;
			}

			$types_to_audit[] = $post_type;
		}

		return $types_to_audit;
	}

	/**
	 * Process a single data item.
	 *
	 * @param WP_Post_Type $post_type The post type object.
	 * @return array
	 */
	public function process_data( $post_type ) {
		// Get post counts by status
		$count_posts = wp_count_posts( $post_type->name );
		
		$published = isset( $count_posts->publish ) ? $count_posts->publish : 0;
		$draft = isset( $count_posts->draft ) ? $count_posts->draft : 0;
		$private = isset( $count_posts->private ) ? $count_posts->private : 0;
		$trash = isset( $count_posts->trash ) ? $count_posts->trash : 0;
		
		// Calculate total
		$total = $published + $draft + $private + $trash;
		
		// Add pending and future if they exist
		if ( isset( $count_posts->pending ) ) {
			$draft += $count_posts->pending;
			$total += $count_posts->pending;
		}
		if ( isset( $count_posts->future ) ) {
			$draft += $count_posts->future;
			$total += $count_posts->future;
		}

		return array(
			$post_type->label,
			$post_type->name,
			$published,
			$draft,
			$private,
			$trash,
			$total,
			$post_type->public ? __( 'Yes', 'audit-export' ) : __( 'No', 'audit-export' ),
			$post_type->hierarchical ? __( 'Yes', 'audit-export' ) : __( 'No', 'audit-export' ),
		);
	}
}